/************************************************************************/
/* This is the grid map set of functions, which enables the robot's     */
/* path and findings to be marked in a map and stored.  There is a map  */
/* for planned paths and a map for actual paths.  Many of these         */
/* functions are not essential for the nav planner yet, since we are    */
/* not using the map for planning or marking targets yet.  The mark_map */
/* function does take care of the TOT_AREA area covered value.          */
/************************************************************************/
#include <stdio.h>
#include <math.h>
#include "navplan.h"
#include "grid.h"


/*************************************************************************/
/* initialize grid block, returning grid block pointer                   */
/*************************************************************************/
grid_block* init_grid_block(void) {
  grid_block* newblock;
  int i, j;

  newblock = (grid_block *) malloc(sizeof(grid_block));

  for (i=0; i<PP_BLOCK_SIZE; ++i) {
    for (j=0; j<PP_BLOCK_SIZE; ++j) {
      newblock->grid[i][j].i = i;
      newblock->grid[i][j].j = j;
      newblock->grid[i][j].elevation = 0.0;
      SETBIT(CONT_TARGET, newblock->grid[i][j].contents, 0);
      SETBIT(CONT_DANGER, newblock->grid[i][j].contents, 0);
      SETBIT(CONT_SNOW, newblock->grid[i][j].contents, 0);
      SETBIT(CONT_BOUND, newblock->grid[i][j].contents, 0);
      SETBIT(SENS_PANO, newblock->grid[i][j].sensors, 0);
      SETBIT(SENS_RADAR, newblock->grid[i][j].sensors, 0);
      SETBIT(SENS_CAMERA, newblock->grid[i][j].sensors, 0);
      SETBIT(SENS_SPEC, newblock->grid[i][j].sensors, 0);
      SETBIT(SENS_MAG, newblock->grid[i][j].sensors, 0);
      SETBIT(SENS_ROV, newblock->grid[i][j].sensors, 0);
      SETBIT(SENS_MET, newblock->grid[i][j].sensors, 0);
      SETBIT(SENS_PLAN, newblock->grid[i][j].sensors, 0);
      
      newblock->grid[i][j].when = 0;
      newblock->grid[i][j].block = newblock;
    }
  }

  newblock->corner.x = 0.0;
  newblock->corner.y = 0.0;
  newblock->corner.z = 0.0;
  newblock->corner.roll = 0.0;
  newblock->corner.yaw = 0.0;

  for (i=0; i<8; ++i)
    newblock->connect[i] = NULL;

  return(newblock);
}

/*************************************************************************/
/* initialize Map                                                        */
/*************************************************************************/
void init_map(planner_map* map, MP_POSE curr) {
  map->center = init_grid_block();

  /* Current position is center of map, so subtract half of
     grid_block to get corner coordinate */
  map->center->corner.x = curr.x - (float)PP_BLOCK_SIZE/2.0 * PP_PIXEL_SIZE;
  map->center->corner.y = curr.y - (float)PP_BLOCK_SIZE/2.0 * PP_PIXEL_SIZE;
}

/*************************************************************************/
/* add and link new grid block to map, given current grid block address  */
/* and (8-connected) direction in which to add (0 = up, increasing in    */
/* clockwise order), returns status (0 if new block, -1 if old block)    */
/*************************************************************************/
int add_grid_block(planner_map* map, grid_block* curr, int dir) {
  grid_block *newblock;
  int status, exists;
  map_pixel* pixel;
  MP_POSE corner;

  /* Set the coordinates of the corner pixel */
  if (dir == 0 || dir == 4)
    corner.x = curr->corner.x;
  if (dir == 1 || dir == 2 || dir == 3)
    corner.x = curr->corner.x + (float)PP_BLOCK_SIZE * PP_PIXEL_SIZE;
  if (dir >= 5)
    corner.x = curr->corner.x - (float)PP_BLOCK_SIZE * PP_PIXEL_SIZE;

  if (dir == 2 || dir == 6)
    corner.y = curr->corner.y;
  if (dir == 7 || dir == 0 || dir == 1)
    corner.y = curr->corner.y + (float)PP_BLOCK_SIZE * PP_PIXEL_SIZE;
  if (dir == 3 || dir == 4 || dir == 5)
    corner.y = curr->corner.y - (float)PP_BLOCK_SIZE * PP_PIXEL_SIZE;

  /* find if this corner is in the map already */
  pixel = find_pixel_addr(*map, corner, &exists);
  if (exists == -1) {
    /* no block exists - add */
    status = 0;
    newblock = init_grid_block();
    curr->connect[dir] = newblock;
    if (dir <=3)
      newblock->connect[dir + 4] = curr;
    else
      newblock->connect[dir - 4] = curr;
    newblock->corner = corner;
  }

  else {
    /* just add links */
    status = -1;
    curr->connect[dir] = pixel->block;
    if (dir <=3)
      pixel->block->connect[dir + 4] = curr;
    else
      pixel->block->connect[dir - 4] = curr;
  }


  return(status);
}


/*************************************************************************/
/* add a grid block that contains given GPS coordinates, adding whatever */
/* grid blocks are needed to get to that grid block.                     */
/*************************************************************************/
void GPS_grid_block(planner_map* map, MP_POSE curr) {
  int i, j;
  float xdif, ydif;
  grid_block *currblock;
  int dir = 0;

  /* comparing curr to central Map grid block, find the corner coordinates
     of the grid block in which curr should be */
  xdif = curr.x - map->center->corner.x;
  ydif = curr.y - map->center->corner.y;
  /* watch out for floating point errors around 0? */
  if (fabs(xdif) < 0.00001)
    xdif = 0.0;
  if (fabs(ydif) < 0.00001)
    ydif = 0.0;
  i = (int)floor(floor(xdif / PP_PIXEL_SIZE) / PP_BLOCK_SIZE);
  j = (int)floor(floor(ydif / PP_PIXEL_SIZE) / PP_BLOCK_SIZE);

  /* now travel through Map to i,j block, adding grids when needed */
  /* travel along x direction first */
  currblock = map->center;
  if (i < 0)
    dir = 6;
  else if (i > 0)
    dir = 2;
  while (i != 0) {
    add_grid_block(map, currblock, dir);
    currblock = currblock->connect[dir];
    if (i < 0) ++i;
    else --i;
  }
  /* now travel along y direction */
  if (j < 0)
    dir = 4;
  else if (j > 0)
    dir = 0;
  while (j != 0) {
    add_grid_block(map, currblock, dir);
    currblock = currblock->connect[dir];
    if (j < 0) ++j;
    else --j;
  }
}

/*************************************************************************/
/* print out the planner map to a text file, so we can show it later     */
/*************************************************************************/
void save_planner_map(planner_map map, char name[255]) {
  FILE *outp;
  ppqueue search, store;
  grid_block* block;
  int i, j, k;
  int stored;

  outp = fopen(name, "w");


  /* read out grids in order, printing contents and sensors chars for
     each grid pixel, row by row */
  search = ppinit_queue();
  store = ppinit_queue();

  ppenqueue(&search, map.center);
  ppenqueue(&store, map.center);

  /* print out the center block */
  fprintf(outp,"%f %f\n", map.center->corner.x, map.center->corner.y);
  for (i = 0; i < PP_BLOCK_SIZE; i++) {
    for (j = 0; j < PP_BLOCK_SIZE; j++) {
      fprintf(outp,"%c", map.center->grid[i][j].contents);
      fprintf(outp,"%c", map.center->grid[i][j].sensors);
    }
    fprintf(outp,"\n");
  }

  block = (grid_block*)ppdequeue(&search);
  while (block != NULL) {
    /* add each connecting block, if not previously added */
    for (k=0; k<8; ++k) {
      stored = ppin_queue(&store, block->connect[k]);

      if (block->connect[k] != NULL && !stored) {
        ppenqueue(&search, block->connect[k]);
	ppenqueue(&store, block->connect[k]);

        /* print out this new block that's been found */
        fprintf(outp,"%d\n", k);
        fprintf(outp,"%f %f\n", block->connect[k]->corner.x,
                block->connect[k]->corner.y);
        for (i = 0; i < PP_BLOCK_SIZE; i++) {
          for (j = 0; j < PP_BLOCK_SIZE; j++) {
            fprintf(outp,"%c", block->connect[k]->grid[i][j].contents);
            fprintf(outp,"%c", block->connect[k]->grid[i][j].sensors);
          }
	  fprintf(outp,"\n");
        }
      }
    }
    block = (grid_block*)ppdequeue(&search);
  }

  fprintf(outp,"-99\n");

  ppfree_queue(&search);
  ppfree_queue(&store);
  fclose(outp);
}


/************************************************************************/
/* print out the actual map to a text file, so we can show it later     */
/************************************************************************/
void save_command_map(planner_map map, char name[255]) {
  FILE *outp;
  ppqueue search, store;
  grid_block* block;
  int i, j, k;
  int stored;

  outp = fopen(name, "w");

  /* read out grids in order, printing contents and sensors chars for
     each grid pixel, row by row */
  search = ppinit_queue();
  store = ppinit_queue();

  ppenqueue(&search, map.center);
  ppenqueue(&store, map.center);

  /* print out the center block */
  fprintf(outp,"%f %f\n", map.center->corner.x, map.center->corner.y);
  for (i = 0; i < PP_BLOCK_SIZE; i++) {
    for (j = 0; j < PP_BLOCK_SIZE; j++) {
      fprintf(outp,"%c", map.center->grid[i][j].contents);
      fprintf(outp,"%c", map.center->grid[i][j].sensors);
    }
    fprintf(outp,"\n");
  }

  block = (grid_block*)ppdequeue(&search);
  while (block != NULL) {
    /* add each connecting block, if not previously added */
    for (k=0; k<8; ++k) {
      stored = ppin_queue(&store, block->connect[k]);

      if (block->connect[k] != NULL && !stored) {
        ppenqueue(&search, block->connect[k]);
	ppenqueue(&store, block->connect[k]);

        /* print out this new block that's been found */
        fprintf(outp,"%d\n", k);
        fprintf(outp,"%f %f\n", block->connect[k]->corner.x,
                block->connect[k]->corner.y);
        for (i = 0; i < PP_BLOCK_SIZE; i++) {
          for (j = 0; j < PP_BLOCK_SIZE; j++) {
            fprintf(outp,"%c", block->connect[k]->grid[i][j].contents);
            fprintf(outp,"%c", block->connect[k]->grid[i][j].sensors);
          }
	  fprintf(outp,"\n");
        }
      }
    }
    block = (grid_block*)ppdequeue(&search);
  }

  fprintf(outp,"-99\n");

  ppfree_queue(&search);
  ppfree_queue(&store);
  fclose(outp);
}


/*************************************************************************/
/* find a pixel's location address, given GPS coordinates (NULL if not   */
/* in map), returns status (0 if OK, -1 if error)                        */
/*************************************************************************/
map_pixel* find_pixel_addr(planner_map map, MP_POSE curr, int* status) {
  map_pixel* currpix;
  float xdif, ydif;
  float goalx, goaly;
  grid_block* block;
  int i, j;
  ppqueue search, store;

  /* comparing curr to central Map grid block, find the corner coordinates
     of the grid block in which curr should be */
  xdif = curr.x - map.center->corner.x;
  ydif = curr.y - map.center->corner.y;
  /* watch out for floating point errors around 0? */
  if (fabs(xdif) < 0.00001)
    xdif = 0.0;
  if (fabs(ydif) < 0.00001)
    ydif = 0.0;
  i = (int)floor(floor(xdif / PP_PIXEL_SIZE) / PP_BLOCK_SIZE);
  j = (int)floor(floor(ydif / PP_PIXEL_SIZE) / PP_BLOCK_SIZE);

  goalx = map.center->corner.x + i * (float)PP_BLOCK_SIZE * PP_PIXEL_SIZE;
  goaly = map.center->corner.y + j * (float)PP_BLOCK_SIZE * PP_PIXEL_SIZE;

  search = ppinit_queue();
  store = ppinit_queue();
  ppenqueue(&search, map.center);
  ppenqueue(&store, map.center);
  *status = 0;

  /* Now do a no-loops breadth-first search of map, starting with center,
     until we find the grid block with goalx, goaly */
  block = (grid_block*)ppdequeue(&search);
  while (block != NULL && (block->corner.x != goalx ||
			   block->corner.y != goaly)) {
    /* add each connecting block, if not previously added */
    for (i=0; i<8; ++i) {
      if (block->connect[i] != NULL &&
	  !ppin_queue(&store, block->connect[i])) {
	ppenqueue(&search, block->connect[i]);
	ppenqueue(&store, block->connect[i]);
      }
    }
    block = (grid_block*)ppdequeue(&search);
  }

  if (block != NULL) {   /* we must have found the match */
    /* find which pixel of this grid block we want */
    i = (int)floor((curr.x - block->corner.x) / PP_PIXEL_SIZE);
    j = (int)floor((curr.y - block->corner.y) / PP_PIXEL_SIZE);
    currpix = &(block->grid[i][j]);
  }
  else {  /* that location isn't in our map */
    *status = -1;
    currpix = NULL;
  }

  ppfree_queue(&search);
  ppfree_queue(&store);

  return(currpix);
}


/*************************************************************************/
/* find GPS coordinates, given current pixel location address            */
/*************************************************************************/
MP_POSE find_pixel_GPS(map_pixel* curr) {
  MP_POSE corner, value;

  corner = curr->block->corner;
  value.x = corner.x + curr->i * PP_PIXEL_SIZE;
  value.y = corner.y + curr->j * PP_PIXEL_SIZE;
  value.z = curr->elevation;
  value.roll = 0.0;
  value.yaw = 0.0;

  return(value);
}


/*************************************************************************/
/* find all 8 adjacent pixels, given current pixel location address;     */
/* adds new grid blocks if needed; returns an array of the pixel         */
/* location addresses, in clockwise order starting at the top middle     */
/* pixel                                                                 */
/*************************************************************************/
adj_pix_array find_adj_pixels(planner_map* map, map_pixel* curr) {
  adj_pix_array value;
  int i, j;
  grid_block* block;
  int newu, newd, newl, newr;

  i = curr->i;
  j = curr->j;
  block = curr->block;

  newu = newd = newl = newr = 0;

  if (j+1 >= PP_BLOCK_SIZE) {
    /* add a new grid block above current one */
    newu = 1;
    add_grid_block(map, block, UP);
    if (i+1 >= PP_BLOCK_SIZE)
      /* add a new grid block to the upper right of current one */
      add_grid_block(map, block, UPPERRIGHT);
  }
  if (i+1 >= PP_BLOCK_SIZE) {
    /* add a new grid block to the right of current one */
    newr = 1;
    add_grid_block(map, block, RIGHT);
    if (j-1 < 0)
      /* add a new grid block to the lower right of current one */
      add_grid_block(map, block, LOWERRIGHT);
  }
  if (j-1 < 0) {
    /* add a new grid block below current one */
    newd = 1;
    add_grid_block(map, block, DOWN);
    if (i-1 < 0)
      /* add a new grid block to the lower left of current one */
      add_grid_block(map, block, LOWERLEFT);
  }
  if (i-1 < 0) {
    /* add a new grid block to the left of current one */
    newl = 1;
    add_grid_block(map, block, LEFT);
    if (j+1 >= PP_BLOCK_SIZE)
      /* add a new grid block to the upper left of current one */
      add_grid_block(map, block, UPPERLEFT);
  }

  /* UP */
  if (newu)
    value.pixels[UP] = &(block->connect[UP]->grid[i][0]);
  else
    value.pixels[UP] = &(block->grid[i][j+1]);

  /* UPPERRIGHT */
  if (newu) {
    if (newr)
      value.pixels[UPPERRIGHT] = &(block->connect[UPPERRIGHT]->grid[0][0]);
    else
      value.pixels[UPPERRIGHT] = &(block->connect[UP]->grid[i+1][0]);
  }
  else if (newr)
    value.pixels[UPPERRIGHT] = &(block->connect[RIGHT]->grid[0][j+1]);
  else
    value.pixels[UPPERRIGHT] = &(block->grid[i+1][j+1]);

  /* RIGHT */
  if (newr)
    value.pixels[RIGHT] = &(block->connect[RIGHT]->grid[0][j]);
  else
    value.pixels[RIGHT] = &(block->grid[i+1][j]);

  /* LOWERRIGHT */
  if (newr) {
    if (newd)
      value.pixels[LOWERRIGHT] = &(block->connect[LOWERRIGHT]->grid[0][PP_BLOCK_SIZE-1]);
    else
      value.pixels[LOWERRIGHT] = &(block->connect[RIGHT]->grid[0][j-1]);
  }
  else if (newd)
    value.pixels[LOWERRIGHT] = &(block->connect[DOWN]->grid[i+1][PP_BLOCK_SIZE-1]);
  else
    value.pixels[LOWERRIGHT] = &(block->grid[i+1][j-1]);

  /* DOWN */
  if (newd)
    value.pixels[DOWN] = &(block->connect[DOWN]->grid[i][PP_BLOCK_SIZE-1]);
  else
    value.pixels[DOWN] = &(block->grid[i][j-1]);

  /* LOWERLEFT */
  if (newd) {
    if (newl)
      value.pixels[LOWERLEFT] = &(block->connect[LOWERLEFT]->grid[PP_BLOCK_SIZE-1][PP_BLOCK_SIZE-1]);
    else
      value.pixels[LOWERLEFT] = &(block->connect[DOWN]->grid[i-1][PP_BLOCK_SIZE-1]);
  }
  else if (newl)
    value.pixels[LOWERLEFT] = &(block->connect[LEFT]->grid[PP_BLOCK_SIZE-1][j-1]);
  else
    value.pixels[LOWERLEFT] = &(block->grid[i-1][j-1]);

  /* LEFT */
  if (newl)
    value.pixels[LEFT] = &(block->connect[LEFT]->grid[PP_BLOCK_SIZE-1][j]);
  else
    value.pixels[LEFT] = &(block->grid[i-1][j]);

  /* UPPERLEFT */
  if (newl) {
    if (newu)
      value.pixels[UPPERLEFT] = &(block->connect[UPPERLEFT]->grid[PP_BLOCK_SIZE-1][0]);
    else
      value.pixels[UPPERLEFT] = &(block->connect[LEFT]->grid[PP_BLOCK_SIZE-1][j+1]);
  }
  else if (newu)
    value.pixels[UPPERLEFT] = &(block->connect[UP]->grid[i-1][0]);
  else
    value.pixels[UPPERLEFT] = &(block->grid[i-1][j+1]);

  return(value);
}


/*************************************************************************/
/* free Map memory                                                       */
/*************************************************************************/
void free_map(planner_map map) {
  ppqueue search, store;
  grid_block* block;
  int i;

  search = ppinit_queue();
  store = ppinit_queue();
  ppenqueue(&search, map.center);
  ppenqueue(&store, map.center);

  /* Now do a no-loops breadth-first search of map, starting with center,
     until we have added all grid blocks to the queue */
  block = (grid_block*)ppdequeue(&search);
  while (block != NULL) {
    /* add each connecting block, if not previously added */
    for (i=0; i<8; ++i) {
      if (block->connect[i] != NULL &&
	  !ppin_queue(&store, block->connect[i])) {
	ppenqueue(&search, block->connect[i]);
	ppenqueue(&store, block->connect[i]);
      }
    }
    block = (grid_block*)ppdequeue(&search);
  }

  /* Now go through stored queue and free all grid blocks */
  block = (grid_block*)ppdequeue(&store);
  while (block != NULL) {
    free(block);
    block = (grid_block*)ppdequeue(&store);
  } 

  ppfree_queue(&search);
  ppfree_queue(&store);
}


/**************************************************************************/
/* Marks the map with an obstacle                                         */
/**************************************************************************/
void mark_obstacle(planner_map* map, MP_POSE curr) {
  int status;
  map_pixel* currpix;

  currpix = find_pixel_addr(*map, curr, &status);
  if (currpix == NULL) {
    /* add new grid */
    GPS_grid_block(map, curr);
    currpix = find_pixel_addr(*map, curr, &status);
  }

  SETBIT(CONT_DANGER, currpix->contents, 1);
}


/**************************************************************************/
/* Marks the planning map with sensor/visited information                 */
/**************************************************************************/
void mark_planning_map(planner_map* map, NPState state, int time, float* area)
{
  int i, j;
  int curri, currj;
  map_pixel *currpix, *pix;
  grid_block *currblock;
  int status;
  MP_POSE curr;
  float width = 0.0;

  if (state.type == STRAIGHT_TYPE) {
    curr = state.STRAIGHT_state.curr;
    width = state.STRAIGHT_state.sensWidth;
  }
  else if (state.type == SPIRAL_TYPE) {
    curr = state.SPIRAL_state.curr;
    width = state.SPIRAL_state.sensWidth;
  }
  else if (state.type == WAYPOINT_TYPE) {
    curr = state.PATH_state.curr;
    width = state.PATH_state.sensWidth;
  }
  else if (state.type == MANEUVER_TYPE) {
    curr = state.MAN_state.curr;
    width = state.MAN_state.sensWidth;
  }
  else
    printf("Error: Incorrect state type for mark_map\n");

  /* if current location not in map, add a grid block */
  currpix = find_pixel_addr(*map, curr, &status);
  if (currpix == NULL) {
    /* add new grid */
    GPS_grid_block(map, curr);
    currpix = find_pixel_addr(*map, curr, &status);
  }

  currblock = currpix->block;
  curri = currpix->i;
  currj = currpix->j;

  /* mark current pixel as visited */
  SETBIT(SENS_ROV, currpix->sensors, 1);

  /* mark pixels as seen by panospheric if within a circle of radius
     sens_width around robot */
  for (i = -(int)(width/PP_PIXEL_SIZE);
       i <= (int)(width/PP_PIXEL_SIZE); ++i) {
    for (j = -(int)(width/PP_PIXEL_SIZE);
	 j <= (int)(width/PP_PIXEL_SIZE); ++j) {
      if (sqr(i*PP_PIXEL_SIZE) + sqr(j*PP_PIXEL_SIZE) <= sqr(width)) {
	/* find the pixel to mark, making sure we're still within
	   the current grid block */
	if (currj+j >= PP_BLOCK_SIZE) {
	  if (curri+i < 0) {
	    /* add in direction 7 */
	    add_grid_block(map, currblock, 7);
	    pix = &(currblock->connect[7]->grid[PP_BLOCK_SIZE+curri+i][currj+j-PP_BLOCK_SIZE]);
	  }
	  else if (curri+i >= PP_BLOCK_SIZE) {
	    /* add in direction 1 */
	    add_grid_block(map, currblock, 1);
	    pix = &(currblock->connect[1]->grid[curri+i-PP_BLOCK_SIZE][currj+j-PP_BLOCK_SIZE]);
	  }
	  else {
	    /* add in direction 0 */
	    add_grid_block(map, currblock, 0);
	    pix = &(currblock->connect[0]->grid[curri+i][currj+j-PP_BLOCK_SIZE]);
	  }
	}
	else if (currj+j < PP_BLOCK_SIZE && currj+j >= 0) {
	  if (curri+i < 0) {
	    /* add in direction 6 */
	    add_grid_block(map, currblock, 6);
	    pix = &(currblock->connect[6]->grid[PP_BLOCK_SIZE+curri+i][currj+j]);
	  }
	  else if (curri+i >= PP_BLOCK_SIZE) {
	    /* add in direction 2 */
	    add_grid_block(map, currblock, 2);
	    pix = &(currblock->connect[2]->grid[curri+i-PP_BLOCK_SIZE][currj+j]);
	  }
	  else {
	    /* don't need to add a block */
	    pix = &(currblock->grid[curri+i][currj+j]);
	  }
	}
	else {
	  if (curri+i < 0) {
	    /* add in direction 5 */
	    add_grid_block(map, currblock, 5);
	    pix = &(currblock->connect[5]->grid[PP_BLOCK_SIZE+curri+i][PP_BLOCK_SIZE+currj+j]);
	  }
	  else if (curri+i >= PP_BLOCK_SIZE) {
	    /* add in direction 3 */
	    add_grid_block(map, currblock, 3);
	    pix = &(currblock->connect[3]->grid[curri+i-PP_BLOCK_SIZE][PP_BLOCK_SIZE+currj+j]);
	  }
	  else {
	    /* add in direction 4 */
	    add_grid_block(map, currblock, 4);
	    pix = &(currblock->connect[4]->grid[curri+i][PP_BLOCK_SIZE+currj+j]);
	  }
	}

	/* pix->when is used to check if plan bit was set this planning
	   cycle or another - set to a constant value for this cycle */
	if (pix->when != time) {
	  if (!GETBIT(SENS_PLAN, pix->sensors))
	    SETBIT(SENS_PLAN, pix->sensors, 1);
	  pix->when = time;
	  /* increment area covered */
	  *area = *area + PP_PIXEL_SIZE * PP_PIXEL_SIZE;
	}
      }
    }
  }
}

/**************************************************************************/
/* Marks the actual map with sensor/visited information                   */
/**************************************************************************/
void mark_map(planner_map* map, NPState state, float* area) {
  int i, j;
  int curri, currj;
  map_pixel *currpix, *pix;
  grid_block *currblock;
  int status;
  MP_POSE curr;
  float width = 0.0;

  if (state.type == STRAIGHT_TYPE) {
    curr = state.STRAIGHT_state.curr;
    width = state.STRAIGHT_state.sensWidth;
  }
  else if (state.type == SPIRAL_TYPE) {
    curr = state.SPIRAL_state.curr;
    width = state.SPIRAL_state.sensWidth;
  }
  else if (state.type == WAYPOINT_TYPE) {
    curr = state.PATH_state.curr;
    width = state.PATH_state.sensWidth;
  }
  else if (state.type == MANEUVER_TYPE) {
    curr = state.MAN_state.curr;
    width = state.MAN_state.sensWidth;
  }
  else
    printf("Error: Incorrect state type for mark_map\n");

  /* if current location not in map, add a grid block */
  currpix = find_pixel_addr(*map, curr, &status);
  if (currpix == NULL) {
    /* add new grid */
    GPS_grid_block(map, curr);
    currpix = find_pixel_addr(*map, curr, &status);
  }

  currblock = currpix->block;
  curri = currpix->i;
  currj = currpix->j;

  /* this will eventually be linked with Morphin map */
  /* mark current pixel as visited */
  SETBIT(SENS_ROV, currpix->sensors, 1);

  /* mark pixels as seen by panospheric if within a circle of radius
     sens_width around robot */
  for (i = -(int)(width/PP_PIXEL_SIZE);
       i <= (int)(width/PP_PIXEL_SIZE); ++i) {
    for (j = -(int)(width/PP_PIXEL_SIZE);
	 j <= (int)(width/PP_PIXEL_SIZE); ++j) {
      if (sqr(i*PP_PIXEL_SIZE) + sqr(j*PP_PIXEL_SIZE) <= sqr(width)) {
	/* find the pixel to mark, making sure we're still within
	   the current grid block */
	if (currj+j >= PP_BLOCK_SIZE) {
	  if (curri+i < 0) {
	    /* add in direction 7 */
	    add_grid_block(map, currblock, 7);
	    pix = &(currblock->connect[7]->grid[PP_BLOCK_SIZE+curri+i][currj+j-PP_BLOCK_SIZE]);
	  }
	  else if (curri+i >= PP_BLOCK_SIZE) {
	    /* add in direction 1 */
	    add_grid_block(map, currblock, 1);
	    pix = &(currblock->connect[1]->grid[curri+i-PP_BLOCK_SIZE][currj+j-PP_BLOCK_SIZE]);
	  }
	  else {
	    /* add in direction 0 */
	    add_grid_block(map, currblock, 0);
	    pix = &(currblock->connect[0]->grid[curri+i][currj+j-PP_BLOCK_SIZE]);
	  }
	}
	else if (currj+j < PP_BLOCK_SIZE && currj+j >= 0) {
	  if (curri+i < 0) {
	    /* add in direction 6 */
	    add_grid_block(map, currblock, 6);
	    pix = &(currblock->connect[6]->grid[PP_BLOCK_SIZE+curri+i][currj+j]);
	  }
	  else if (curri+i >= PP_BLOCK_SIZE) {
	    /* add in direction 2 */
	    add_grid_block(map, currblock, 2);
	    pix = &(currblock->connect[2]->grid[curri+i-PP_BLOCK_SIZE][currj+j]);
	  }
	  else {
	    /* don't need to add a block */
	    pix = &(currblock->grid[curri+i][currj+j]);
	  }
	}
	else {
	  if (curri+i < 0) {
	    /* add in direction 5 */
	    add_grid_block(map, currblock, 5);
	    pix = &(currblock->connect[5]->grid[PP_BLOCK_SIZE+curri+i][PP_BLOCK_SIZE+currj+j]);
	  }
	  else if (curri+i >= PP_BLOCK_SIZE) {
	    /* add in direction 3 */
	    add_grid_block(map, currblock, 3);
	    pix = &(currblock->connect[3]->grid[curri+i-PP_BLOCK_SIZE][PP_BLOCK_SIZE+currj+j]);
	  }
	  else {
	    /* add in direction 4 */
	    add_grid_block(map, currblock, 4);
	    pix = &(currblock->connect[4]->grid[curri+i][PP_BLOCK_SIZE+currj+j]);
	  }
	}

	if (!GETBIT(SENS_PANO, pix->sensors)) {
	  SETBIT(SENS_PANO, pix->sensors, 1);
	  /* increment area covered */
	  *area = *area + PP_PIXEL_SIZE * PP_PIXEL_SIZE;
	}
      }
    }
  }
}

