#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include "mplanner.h"
#include "nplanner.h"
#include "solar.h"


/**************************************************************************/
/*  Main body functions:                                                  */
/*    processStraight                                                     */
/*    processSpiral                                                       */
/*    processWaypt                                                        */
/*    processNewMsg                                                       */
/**************************************************************************/

/************************************************************************/
/* This function takes a navPlanUpdateSet message and the robot's       */
/* current position and sets up a farming plan.  The values from the    */
/* message are transferred to the NPState structure, and the coverage   */
/* area is defined.  init_farmer is called to finish the initialization.*/
/************************************************************************/
PlannerInfo processStraight(PlanParams msg) {
  PlannerInfo Status;
  NPState state;
  plan_command command;
  MP_POSE vertex;
  polygon* temp;
  int i, complete = 0;
  float area = 0.0;
  float tottime = 0.0;
  float totenergy = 0.0;
  double Jdate;
  int status;
  map_pixel* pixel;
  int foo;

  if (npVERBOSE) printf("Processing straight rows pattern\n");

  Jdate = msg.startdate;

  state.type = STRAIGHT_TYPE;
  state.STRAIGHT_state.speed = msg.speed;
  state.STRAIGHT_state.sensWidth = msg.cameraFOV;
  state.STRAIGHT_state.minTurnRad = msg.minTurningRadius;
  state.STRAIGHT_state.rowWidth = msg.straightTask.rowWidth;
  state.next = NULL;

  state.STRAIGHT_state.area = (polygon *) malloc (sizeof(polygon));
  temp = state.STRAIGHT_state.area;

  if (msg.straightTask.type == MP_STRAIGHT_TYPE_LENGTH) {
    /* user has entered row length and area width */
    state.STRAIGHT_state.hand = msg.straightTask.direction;

    /* vertex 1 */
    if (msg.useAbsCoord == 1)  /* Absolute DGPS */
      temp->vertex = msg.start;
    else {                      /* Relative */
      vertex = msg.start;
      temp->vertex = (RelToGlo(vertex, PlanningPose));
    }
    if (npVERBOSE) printf("Starting area corner = %.2f, %.2f\n",
			  temp->vertex.x, temp->vertex.y);
    temp->next = (polygon *) malloc(sizeof(polygon));
    temp = temp->next;

    /* vertex 2 */
    vertex.x = 0.0;
    vertex.y = msg.straightTask.length;
    vertex.yaw = 0.0;
    temp->vertex = RelToGlo(vertex, state.STRAIGHT_state.area->vertex);
    if (npVERBOSE) printf("2nd corner = %.2f, %.2f\n", temp->vertex.x, temp->vertex.y);
    temp->next = (polygon *) malloc(sizeof(polygon));
    temp = temp->next;

    /* vertex 3 */
    if (state.STRAIGHT_state.hand == 1)
      vertex.x = -1.0 * msg.straightTask.width;
    else
      vertex.x = msg.straightTask.width;
    vertex.y = msg.straightTask.length;
    vertex.yaw = 0.0;
    temp->vertex = RelToGlo(vertex, state.STRAIGHT_state.area->vertex);
    if (npVERBOSE) printf("3rd corner = %.2f, %.2f\n", temp->vertex.x, temp->vertex.y);
    temp->next = (polygon *) malloc(sizeof(polygon));
    temp = temp->next;

    /* vertex 4 */
    if (state.STRAIGHT_state.hand == 1)
      vertex.x = -1.0 * msg.straightTask.width;
    else
      vertex.x = msg.straightTask.width;
    vertex.y = 0.0;
    vertex.yaw = 0.0;
    temp->vertex = RelToGlo(vertex, state.STRAIGHT_state.area->vertex);
    if (npVERBOSE) printf("4th corner = %.2f, %.2f\n", temp->vertex.x, temp->vertex.y);
    temp->next = NULL;
  }

  else if (msg.useAbsCoord) {
    /* global coordinates entered */
    temp->vertex = msg.start;
    for (i = 1; i < msg.straightTask.numVert; i++) { 
      temp->next = (polygon *) malloc(sizeof(polygon));
      temp = temp->next;
      temp->vertex = msg.straightTask.vertices[i];
    }
    temp->next = NULL; }

  else {
    /* relative coordinates entered */
    vertex = msg.start;
    temp->vertex = RelToGlo(vertex, PlanningPose);
    for (i = 1; i < msg.straightTask.numVert; i++) {
      temp->next = (polygon *) malloc(sizeof(polygon));
      temp = temp->next;
      vertex = msg.straightTask.vertices[i];
      vertex.yaw = 0.0;
      temp->vertex = RelToGlo(vertex, PlanningPose);
    }
    temp->next = NULL;
  }

  init_straight(&(state.STRAIGHT_state), &command);
  if (npVERBOSE) printf("Straight rows pattern initialized\n");


  /* Now initiate pattern */
  while (!complete) {
    mark_planning_map(&PlanMap, state, PLAN_TIME, &area);
    complete = straightrows(&(state.STRAIGHT_state), &command, npVERBOSE);
    totenergy = totenergy + solar(state.STRAIGHT_state.curr, ROB_XAX, ROB_LAT,
				ROB_LON, Jdate, PlanMap, RecordSun) * 1.0;
    Jdate = Jdate + 1.0 / 86400.0;  /* add 1 second */
    tottime = tottime + 1.0;
    arc_to_pos(&(state.STRAIGHT_state.curr), command.turn, command.speed, 1.0);
    pixel = find_pixel_addr(PlanMap, state.STRAIGHT_state.curr, &status);
    state.STRAIGHT_state.curr.z = pixel->elevation;
    if (npVERBOSE) printf("new position = %.5f, %.5f, %.5f, yaw = %.1f\n",
			  state.STRAIGHT_state.curr.x, state.STRAIGHT_state.curr.y,
			  state.STRAIGHT_state.curr.z,
			  state.STRAIGHT_state.curr.yaw * 180 / PI);
  }

  /* Clean up and send results to Mission Planner */
  free_polygon(state.STRAIGHT_state.area);
  Status.taskMode = MP_STRAIGHT_ROWS_PATTERN;
  Status.navPlannerStatus = NAV_MAN_SUCCESSFUL;
  Status.timeCost = tottime;
  Status.areaCovered = area;
  Status.energyGen = totenergy;
  save_planner_map(PlanMap, PMAPNAME);

  return(Status);
}


/************************************************************************/
/* This function takes a navPlanUpdateSet message and the robot's       */
/* current position and sets up a spiral plan.  The values from the     */
/* message are transferred to the NPState structure, and init_spiral is */
/* called to finish the initialization.                                 */
/************************************************************************/
PlannerInfo processSpiral(PlanParams msg) {
  PlannerInfo Status;
  NPState state;
  plan_command command;
  int complete = 0;
  float area = 0.0;
  float tottime = 0.0;
  float totenergy = 0.0;
  double Jdate;
  int status;
  map_pixel* pixel;

  if (npVERBOSE) printf("Processing spiral plan\n");

  Jdate = msg.startdate;

  state.type = SPIRAL_TYPE;
  state.SPIRAL_state.speed = msg.speed;
  state.SPIRAL_state.sensWidth = msg.cameraFOV;
  state.SPIRAL_state.minTurnRad = msg.minTurningRadius;
  state.SPIRAL_state.rowWidth = msg.spiralTask.rowWidth;
  state.next = NULL;

  /* Set up center of spiral */
  if (msg.useAbsCoord == 1)
    state.SPIRAL_state.curr = msg.start;
  else
    state.SPIRAL_state.curr = RelToGlo(msg.start, PlanningPose);
  state.SPIRAL_state.currseg.dir = msg.spiralTask.direction;
  state.SPIRAL_state.maxrad = msg.spiralTask.maxRadius;
  init_spiral(&(state.SPIRAL_state));

  /* Now initiate pattern */
  while (!complete) {
    mark_planning_map(&PlanMap, state, PLAN_TIME, &area);
    complete = spiral(&(state.SPIRAL_state), &command, npVERBOSE);
    totenergy = totenergy + solar(state.SPIRAL_state.curr, ROB_XAX, ROB_LAT,
				  ROB_LON, Jdate, PlanMap, RecordSun) * 1.0;
    Jdate = Jdate + 1.0 / 86400.0;
    tottime = tottime + 1.0;
    arc_to_pos(&(state.SPIRAL_state.curr), command.turn, command.speed, 1.0);
    pixel = find_pixel_addr(PlanMap, state.SPIRAL_state.curr, &status);
    state.SPIRAL_state.curr.z = pixel->elevation;
    if (npVERBOSE) printf("new position = %.5f, %.5f, %.5f, yaw = %.1f\n",
			  state.SPIRAL_state.curr.x,
			  state.SPIRAL_state.curr.y,
			  state.SPIRAL_state.curr.z,
			  state.SPIRAL_state.curr.yaw * 180 / PI);
  }

  /* Clean up and send results to Mission Planner */
  Status.taskMode = MP_SPIRAL_PATTERN;
  Status.navPlannerStatus = NAV_MAN_SUCCESSFUL;
  Status.timeCost = tottime;
  Status.areaCovered = area;
  Status.energyGen = totenergy;
  save_planner_map(PlanMap, PMAPNAME);

  return(Status);
}


/************************************************************************/
/* This function takes a navPlanUpdateSet message and the robot's       */
/* current position and sets up a waypoint plan.  The values from the   */
/* message are transferred to the NPState structure.                    */
/************************************************************************/
PlannerInfo processWaypt(PlanParams msg) {
  PlannerInfo Status;
  MP_POSE tempcoords;
  NPState state;
  plan_command command;
  int complete = 0;
  float area = 0.0;
  float tottime = 0.0;
  float totenergy = 0.0;
  double Jdate;
  int status;
  map_pixel* pixel;

  Jdate = msg.startdate;

  if (npVERBOSE) printf("Processing waypoint plan\n");
  state.type = WAYPOINT_TYPE;
  state.PATH_state.speed = msg.speed;
  state.PATH_state.sensWidth = msg.cameraFOV;
  state.PATH_state.minTurnRad = msg.minTurningRadius;
  state.next = NULL;
  state.PATH_state.curr = msg.start;
  tempcoords = msg.wayptTask.goal;

  /* global coords */
  if (msg.useAbsCoord) state.PATH_state.goal_type = 1;
  /* relative coords - convert to global */
  else {
    state.PATH_state.goal_type = 2;
    tempcoords = RelToGlo(tempcoords, PlanningPose);
  }

  state.PATH_state.goal = tempcoords;
  state.PATH_state.tolerance = msg.wayptTask.tolerance;

  /* Now initiate plan */
  while (!complete) {
    mark_planning_map(&PlanMap, state, PLAN_TIME, &area);
    complete = pplanner(&(state.PATH_state), &command, npVERBOSE);
    totenergy = totenergy + solar(state.PATH_state.curr, ROB_XAX, ROB_LAT,
				  ROB_LON, Jdate, PlanMap, RecordSun) * 1.0;
    Jdate = Jdate + 1.0 / 86400.0;
    tottime = tottime + 1.0;
    arc_to_pos(&(state.PATH_state.curr),command.turn,command.speed,1.0);
    pixel = find_pixel_addr(PlanMap, state.PATH_state.curr, &status);
    state.PATH_state.curr.z = pixel->elevation;
    if (npVERBOSE) printf("new position = %.5f, %.5f, %.5f, yaw = %.1f\n",
			  state.PATH_state.curr.x, state.PATH_state.curr.y,
			  state.PATH_state.curr.z,
			  state.PATH_state.curr.yaw * 180 / PI);
  }

  /* Clean up and send results to Mission Planner */
  Status.taskMode = MP_WAYPOINT;
  Status.navPlannerStatus = NAV_MAN_SUCCESSFUL;
  Status.timeCost = tottime;
  Status.areaCovered = area;
  Status.energyGen = totenergy;
  save_planner_map(PlanMap, PMAPNAME);

  return(Status);
}


/************************************************************************/
/* This function takes a navPlanUpdateSet message and the robot's       */
/* current position and sets up a maneuver plan.  The values from the   */
/* message are transferred to the NPState structure.                    */
/************************************************************************/
PlannerInfo processMan(PlanParams msg) {
  PlannerInfo Status;
  MP_POSE tempcoords;
  NPState state;
  plan_command command;
  float tempyaw;
  int value;
  int complete = 0;
  float area = 0.0;
  float tottime = 0.0;
  float totenergy = 0.0;
  float number;
  double Jdate;
  maneuver_step* temp;
  FILE *planp, *posefile;
  int RECORD = 1;
  int status;
  map_pixel* pixel;

  Jdate = msg.startdate;

  if (npVERBOSE) printf("Processing maneuver plan\n");
  if (msg.startNewPlan == 0) {
    /* just enact maneuver plan */
    state.MAN_state.speed = msg.speed;
    state.MAN_state.sensWidth = msg.cameraFOV;
    state.MAN_state.minTurnRad = msg.minTurningRadius;
    state.next = NULL;
    if (msg.useAbsCoord)
      state.MAN_state.curr = msg.start;
    else
      state.MAN_state.curr = RelToGlo(msg.start, PlanningPose);
    state.type = MANEUVER_TYPE;
    state.MAN_state.ManPlan = NULL;

    /* read plan into state.MAN_state.manplan */
    planp = fopen(msg.manplanTask.filename, "r");
    if (planp == NULL) {
      printf("Error:  No file named '%s'\n", msg.manplanTask.filename);
      complete = 1; }
    else {
      fscanf(planp, "%f", &number);
      if (number != -99.0) {
	state.MAN_state.step = 1;
	temp = (maneuver_step *)malloc(sizeof(maneuver_step));
	state.MAN_state.ManPlan = temp;
	temp->turn = number;
	fscanf(planp, "%f %f %f %f", &(temp->speed), &(temp->desired.x),
	       &(temp->desired.y), &(temp->desired.yaw));
	fscanf(planp, "%f", &number); }
      while (number != -99.0) {
	temp->next = (maneuver_step *)malloc(sizeof(maneuver_step));
	temp = temp->next;
	temp->turn = number;
	fscanf(planp, "%f %f %f %f", &(temp->speed), &(temp->desired.x),
	       &(temp->desired.y), &(temp->desired.yaw));
	fscanf(planp, "%f", &number); }
      temp->next = NULL;
      fclose(planp);
    }
  }
  else {  /* creating new maneuver plan */
    state.type = MANEUVER_TYPE;
    state.MAN_state.speed = msg.speed;
    state.MAN_state.sensWidth = msg.cameraFOV;
    state.MAN_state.minTurnRad = msg.minTurningRadius;
    state.next = NULL;
    if (msg.useAbsCoord)
      state.MAN_state.curr = msg.start;
    else
      state.MAN_state.curr = RelToGlo(msg.start, PlanningPose);
    state.MAN_state.ManPlan = NULL;
    state.MAN_state.man_type = msg.manplanTask.type;
    tempcoords = msg.manplanTask.goal;

    /* global coords */
    if (msg.useAbsCoord == 0)
      tempcoords = RelToGlo(tempcoords, PlanningPose);

    state.MAN_state.Goal.pos.x = tempcoords.x;
    state.MAN_state.Goal.pos.y = tempcoords.y;
    state.MAN_state.Goal.pos.yaw = tempcoords.yaw;
    state.MAN_state.Goal.disttol = msg.manplanTask.distanceTol;
    state.MAN_state.Goal.angtol = msg.manplanTask.angularTol;
    state.MAN_state.ManPlan = NULL;

    /* Find plan - may take a while */
    value = man_planner(&(state.MAN_state));

    if (value == -1 && npVERBOSE) {
      printf("Error:  No plan found\n");
      complete = 1; }
    else {
      /* print out the plan to a file */
      temp = state.MAN_state.ManPlan;
      planp = fopen(msg.manplanTask.filename, "w");
      if (planp == NULL)
	printf("Error:  No file named '%s' to write plan to\n",msg.manplanTask.filename);
      else {
	while (temp != NULL) {
	  printf("command: turn %.3f, speed %.2f\n", temp->turn,temp->speed);
	  fprintf(planp, "%f %f %f %f %f\n", temp->turn, temp->speed,
		  temp->desired.x, temp->desired.y, temp->desired.yaw);
	  temp = temp->next; }
	fprintf(planp, "-99\n");
	fclose(planp); }
    }
  }

  /* Now initiate plan */
  while (!complete) {
    mark_planning_map(&PlanMap, state, PLAN_TIME, &area);
    complete = maneuver(&(state.MAN_state), &command, npVERBOSE);
    totenergy = totenergy + solar(state.MAN_state.curr, ROB_XAX, ROB_LAT,
				  ROB_LON, Jdate, PlanMap, RecordSun)*1.0;
    Jdate = Jdate + 1.0 / 86400.0;
    tottime = tottime + 1.0;
    if (!complete) {
      arc_to_pos(&(state.MAN_state.curr),command.turn,command.speed,1.0);
      pixel = find_pixel_addr(PlanMap, state.MAN_state.curr, &status);
      state.MAN_state.curr.z = pixel->elevation;
      if (npVERBOSE) printf("new position = %.5f, %.5f, %.5f, yaw = %.1f\n",
			    state.MAN_state.curr.x,
			    state.MAN_state.curr.y, state.MAN_state.curr.z,
			    state.MAN_state.curr.yaw * 180 / PI);
    }

    if (RECORD) {
      posefile = fopen("posefile.txt", "a");
      fprintf(posefile, "command %.3f at %.2f => %f, %f, %f\n", command.turn,
	      command.speed, state.MAN_state.curr.x, state.MAN_state.curr.y,
	      state.MAN_state.curr.yaw);
      fclose(posefile); }

  }

  /* Clean up and send results to Mission Planner */
  Status.taskMode = MP_MANEUVER;
  Status.navPlannerStatus = NAV_MAN_SUCCESSFUL;
  Status.timeCost = tottime;
  Status.areaCovered = area;
  Status.energyGen = totenergy;
  save_planner_map(PlanMap, PMAPNAME);

  return(Status);
}



/************************************************************************/
/* This function processes navPlanUpdateSet messages, either from the   */
/* getNddsMsg function (which comes from the Mission Planner) or the    */
/* getMsg function (which is direct user input).  It decides whether to */
/* send the message to the farmer, spiral or waypoint processing        */
/* functions.                                                           */
/************************************************************************/
PlannerInfo processNewMsg(PlanParams message) {
  PlannerInfo Status;

  /* We need to set PLAN_TIME for the planning map */
  PLAN_TIME = PLAN_TIME + 10.0;

  if (message.task == MP_STRAIGHT_ROWS_PATTERN)
    Status = processStraight(message);
  else if (message.task == MP_SPIRAL_PATTERN)
    Status = processSpiral(message);
  else if (message.task == MP_WAYPOINT)
    Status = processWaypt(message);
  else if (message.task == MP_MANEUVER)
    Status = processMan(message);

  return(Status);
}




