#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <math.h>
#include "mplanner.h"
#include "nplanner.h"
#include "readmap.h"

/* modes */
#define MODE_User      0
#define MODE_Search    1
#define MODE_Target    2

/* stages for target mode */
#define TS_Evaluate    1
#define TS_Maneuver    2
#define TS_WaitingMan  3
#define TS_PoseCheck   4
#define TS_Sensors     5
#define TS_WaitingSens 6
#define TS_CleanUp     7
#define TS_Resume      8

/* Globals */
int MODE = MODE_User;
int TargetStage = TS_Evaluate;
MP_POSE LastPose;

/* Planning variables */
int PLAN_TIME;  /* time used in marking planning map */
char PMAPNAME[255];
char RecordSun[40];
planner_map PlanMap;
MP_POSE PlanningPose;

/* Time/date solar variables */
float ROB_LAT, ROB_LON, ROB_XAX; /* coords of DGPS base station */

/* Global variable tags for how nplanner runs */
int npVERBOSE;


/**************************************************************************/
/*  Utility functions:                                                    */
/*    mpGPS_diff                                                          */
/*    getdate                                                             */
/**************************************************************************/

/****************************************************************************/
/****************************************************************************/
float mpGPS_diff(MP_POSE one, MP_POSE two) {
  float diff;

  diff = sqrt(sqr(one.x - two.x) + sqr(one.y - two.y));

  return(diff);
}

/****************************************************************************/
/* getdate: calculates the Julian date of a particular calendar date and    */
/*  time (time is given in Universal Time)                                  */
/****************************************************************************/
double getdate() {
  int a, b;
  int month, day, year;
  float time;
  double Jdate, timefrac;

  printf("Enter the month (e.g. Jan = 1, Nov = 11): ");
  scanf("%d",&month);
  printf("Enter the day: ");
  scanf("%d",&day);
  printf("Enter the year (e.g. 1997, 2001): ");
  scanf("%d",&year);
  printf("Enter the time (Universal Time -- 8:00am = 800, 3:30pm = 1550): ");
  scanf("%f",&time);

  if (month < 3) {
    year = year - 1;
    month = month + 12; }

  a = (int)(year/100);
  b = 2 - a + (int)(a/4);

  Jdate = (int)(365.25*(year+4716))+(int)(30.6001*(month+1))+day+b-1524.5;
  timefrac = time/2400.0;
  Jdate = Jdate + timefrac;

  return(Jdate);
}


/************************************************************************/
/* This function checks to see that the robot is within the workspace   */
/* of the given target, for a given sensor.                             */
/************************************************************************/
int check_workspace(MP_POSE robot, MP_POSE target, int sensor) {
  int valid;
  float angle;

  if (sensor == SENSOR_PAN_TILT_CAMERA) {
    if (mpGPS_diff(robot, target) < 3.5) {
      angle = atan2(target.y - robot.y, target.x - robot.x) - PI/2.0;
      if (fabs(robot.yaw - angle) < PI/4.0) valid = 1;
      else valid = 0;
    }
    else valid = 0;
  }
  else if (sensor == SENSOR_SPECTROMETER) {
    if (mpGPS_diff(robot, target) < 3.0) {
      angle = atan2(target.y - robot.y, target.x - robot.x) - PI/2.0;
      if (fabs(robot.yaw - angle) < PI / 2.0) valid = 1;
      else valid = 0;
    }
    else valid = 0;
  }
  else {
    printf("That is not a valid sensor type\n");
    valid = 1;
  }

  return(valid);
}


/************************************************************************/
/* This function converts an missionPlannerRequest message into a       */
/* navManRequest message.                                               */
/************************************************************************/
void storeNavCommands(PlanParams request, PlanParams* plan) {
  int i;

  plan->storage = 1;
  plan->minTurningRadius = request.minTurningRadius;
  plan->cameraFOV = request.cameraFOV;
  plan->speed = request.speed;
  plan->start = request.start;
  plan->startdate = request.startdate;
  plan->startNewPlan = request.startNewPlan;
  plan->useAbsCoord = request.useAbsCoord;

  if (request.task == MP_STRAIGHT_ROWS_PATTERN) {
    plan->task = MP_STRAIGHT_ROWS_PATTERN;
    plan->straightTask.rowWidth = request.straightTask.rowWidth;
    plan->straightTask.type = request.straightTask.type;
    plan->straightTask.length = request.straightTask.length;
    plan->straightTask.width = request.straightTask.width;
    plan->straightTask.direction = request.straightTask.direction;
    plan->straightTask.numVert = request.straightTask.numVert;
    for (i = 0; i < request.straightTask.numVert; ++i)
      plan->straightTask.vertices[i] = request.straightTask.vertices[i];
  }
  else if (request.task == MP_SPIRAL_PATTERN) {
    plan->task = MP_SPIRAL_PATTERN;
    plan->spiralTask.rowWidth = request.spiralTask.rowWidth;
    plan->spiralTask.maxRadius = request.spiralTask.maxRadius;
    plan->spiralTask.direction = request.spiralTask.direction;
  }
  else if (request.task == MP_WAYPOINT) {
    plan->task = MP_WAYPOINT;
    plan->wayptTask.goal = request.wayptTask.goal;
    plan->wayptTask.tolerance = request.wayptTask.tolerance;
  }
  else if (request.task == MP_MANEUVER) {
    plan->task = MP_MANEUVER;
    plan->manplanTask.enact = request.manplanTask.enact;
    sprintf(plan->manplanTask.filename, "%s", request.manplanTask.filename);
    plan->manplanTask.type = request.manplanTask.type;
    plan->manplanTask.goal = request.manplanTask.goal;
    plan->manplanTask.distanceTol = request.manplanTask.distanceTol;
    plan->manplanTask.angularTol = request.manplanTask.angularTol;
  }
}


/************************************************************************/
/* The following 4 functions are subscription callbacks:                */
/*   SensManCompleteCallback                                            */
/*   NavManCompleteCallback                                             */
/*   NddsPose                                                           */
/*   TargetFound                                                        */
/************************************************************************/

/************************************************************************/
/* This function is the callback function for sensManComplete messages. */
/* It updates the TargetStage of the mission.                           */
/* It is a subscription.                                                */
/************************************************************************/
RTIBool SensManCompleteCallback(NDDSRecvInfo *message) {
  sensManComplete *msg;
  sensor_list* temp, *CurrSensList;

  /* globals used here:  TargetStage */

  if (message->status == NDDS_FRESH_DATA) {
    printf("sensManComplete message received\n");
    msg = (sensManComplete*)message->instance;
    CurrSensList = (sensor_list *)message->callBackRtnParam;
    if (TargetStage == TS_WaitingSens) {
      if (msg->reqNum == CurrSensList->reqNum) {
	if (msg->status == SENS_MAN_SUCCESSFUL) {
	  if (CurrSensList->next == NULL) { /* we've done all sensors */
	    free(CurrSensList);
	    TargetStage = TS_CleanUp;
	  }
	  else { /* we need to use another sensor */
	    temp = CurrSensList;
	    CurrSensList = temp->next;
	    free(temp);
	    TargetStage = TS_Sensors;
	  }
	}
	else {
	  printf("Error: Status not good\n");
	  /* we should redo this sensor */
	  TargetStage = TS_Sensors;
	}
      }
      else printf("Sensor message is for different request number.");
      /* we just ignore this message and continue waiting */
    }
    else printf("Sensor message received when not waiting for one in autonomous mode\n");
    /* could be just a user request */

    return RTI_TRUE;
  }
  else {
    /* Error: Old sensManStatus data received */
    return RTI_FALSE;
  }
}



/************************************************************************/
/* This function is the callback function for navPlanComplete messages. */
/* It updates the state of the mission, and fills in the report         */
/* structure to publish in main.                                        */
/* It is a subscription.                                                */
/************************************************************************/
RTIBool NavManCompleteCallback(NDDSRecvInfo *message) {
  navManComplete *msg;

  /* globals used here:  MODE, TargetStage */

  if (message->status == NDDS_FRESH_DATA) {    
    printf("navManComplete message received\n");
    msg = (navManComplete *)message->instance;
    /* check to see what mplanner should do next */
    if (MODE == MODE_User)
      printf("We're in user mode, and just finished a navplan\n");
    else if (MODE == MODE_Target) {
      if (TargetStage == TS_WaitingMan) {
	printf("Finished maneuver, switching to TS_PoseCheck stage\n");
	TargetStage = TS_PoseCheck; }
      else printf("Finished some other NavPlan during target mode\n");
    }
    else if (msg->task == MP_STRAIGHT_ROWS_PATTERN) {
      printf("Finished straight rows pattern - switching to user mode\n");
      MODE = MODE_User; }
    else if (msg->task == MP_SPIRAL_PATTERN) {
      printf("Finished spiral pattern - switching to user mode\n");
      MODE = MODE_User; }
    else printf("Finished non-coverage plan in autonomous search mode\n");
    return RTI_TRUE;
  }
  else {
    /* Error: Old navPlanStatus data received */
    return RTI_FALSE;
  }
}


/************************************************************************/
/* This function is the callback function for posDerivedState messages. */
/* It puts the pose into the global MP_POSE variable LastPose.          */
/* It is a subscription.                                                */
/************************************************************************/
RTIBool NddsPose(NDDSRecvInfo *nddsPose) {
  posDerivedState *msg;
  MP_POSE *pose;
 
  if (nddsPose->status == NDDS_FRESH_DATA) {
    msg = (posDerivedState*)nddsPose->instance;
    pose = (MP_POSE *)nddsPose->callBackRtnParam;
    /* convert to MP_POSE */
    pose->x = (double)msg->posWest / -100.0;  /* x axis is East */
    pose->y = (double)msg->posNorth / 100.0;  /* y axis is North */
    pose->z = (double)msg->posUp / 100.0;
    pose->yaw = (double)msg->rotZ / 100.0;

    /* Convert from compass (angles go CW) to heading (angles go CCW) */
    pose->yaw = 2.0 * PI - pose->yaw;
    pose->roll = (double)msg->rotY / 100.0;
    pose->pitch = (double)msg->rotX / 100.0;
    return RTI_TRUE;
  }
  else {
    /* Error: Old posDerivedState data received */
    return RTI_FALSE;
  }
}

/************************************************************************/
/* This function is the callback for dbNotify messages.                 */
/* It adds the target found to the end of TargetList, and sets the      */
/* TARGET flag to 1.                                                    */
/* It is a subscription.                                                */
/************************************************************************/
RTIBool TargetFound(NDDSRecvInfo *message) {
  dbNotify* msg;
  target_list_item *temp, *targets;
  long ltime;

  /* globals used:  MODE */

  msg = (dbNotify*)message->instance;
  targets = (target_list_item *)message->callBackRtnParam;

  if (message->status == NDDS_FRESH_DATA) {
    /* check to see if DGPS field is the one that has changed */
    /* otherwise ignore this message */
    if (msg->changed.DGPS_coord == DB_ALTERED) {
      /* add this target to end of TargetList */
      temp = targets;
      while (temp != NULL) temp = temp->next;
      temp = (target_list_item*)malloc(sizeof(target_list_item*));
      temp->targetID = msg->record.targetID;
      temp->location.x = msg->record.DGPS_coord.x;
      temp->location.y = msg->record.DGPS_coord.y;
      temp->location.z = msg->record.DGPS_coord.z;
      time(&ltime);
      temp->timestamp = (double)ltime;
      temp->cost = -1;
      temp->complete = 0;
      temp->next = NULL;
      MODE = MODE_Target;
    }
    return RTI_TRUE;
  }
  else {
    /* Error: Old dbNotify data received */
    return RTI_FALSE;
  }
}


/************************************************************************/
/* This function is the callback function for mpUpdateSettings messages.*/
/* It determines which type of task is requested, and fills in the      */
/* appropriate structures to be sent in main.                           */
/* It send a reply to user interface when done.                         */
/* It is a server.                                                      */
/************************************************************************/
RTIBool RouteTaskMode(NDDSObjectInstance reply, NDDSObjectInstance request,
		      void *userData) {
  mPlannerReply   *mpReply   = (mPlannerReply *)reply;
  mPlannerRequest *mpRequest = (mPlannerRequest *)request;
  TaskMessage *newMessage = (TaskMessage *)userData;

  PlannerInfo Status;
  time_t ltime;
  MP_POSE start, curr;
  float dist;
  int i;

  /* globals used:  LastPose, MODE, TargetStage, PlanningPose */

  printf("mission planner request message received\n");

  if (mpRequest->type == MP_TYPE_AUTONOMOUS) {
    if (newMessage->sendNavMsg == 1) {
      /* already a request waiting to be sent */
      mpReply->status = MP_STATUS_BUSY;
    }
    else {
      printf("Entering autonomous search mode\n");
      /* initialize search stage parameters */
      MODE = MODE_Search;
      TargetStage = TS_Evaluate;

      /* ASKMIKE - answer questions */
      newMessage->targetMsg->function = TARGET_ACQ_START;
      newMessage->targetMsg->driverNum = 0;  /* figure out which driverNum is camera */
      newMessage->targetMsg->period = 0.01;  /* what is the shortest period? */
      newMessage->sendTargetMsg = 1;

      /* see if we're at starting point already, compare to LastPose */
      if (mpRequest->navReq.useAbsCoord != 1) {  /* compare with [0,0,0] */
	curr.x = 0.0; curr.y = 0.0; curr.z = 0.0;
	/* compare yaw too? roll? */
	dist = mpGPS_diff(mpRequest->navReq.start, curr);
      }
      else
	dist = mpGPS_diff(LastPose, mpRequest->navReq.start);
      if (fabs(dist) > 0.5) {
	/* set starting point as waypoint for 1st plan */
	newMessage->plan0.storage = 1;
	newMessage->plan0.minTurningRadius = mpRequest->navReq.minTurningRadius;
	newMessage->plan0.cameraFOV = mpRequest->navReq.cameraFOV;
	newMessage->plan0.speed = mpRequest->navReq.speed;
	newMessage->plan0.startdate = mpRequest->navReq.startdate;
	newMessage->plan0.startNewPlan = mpRequest->navReq.startNewPlan;
	newMessage->plan0.useAbsCoord = mpRequest->navReq.useAbsCoord;
	newMessage->plan0.task = MP_WAYPOINT;
	newMessage->plan0.wayptTask.goal = mpRequest->navReq.start;
	newMessage->plan0.wayptTask.tolerance = 1.5;  /* this should be close enough */
	if (mpRequest->navReq.useAbsCoord == 1)
	  newMessage->plan0.start = LastPose;
	else {
	  newMessage->plan0.start.x = 0.0;
	  newMessage->plan0.start.y = 0.0;
	  newMessage->plan0.start.z = 0.0;
	  newMessage->plan0.start.yaw = 0.0;
	  newMessage->plan0.start.roll = 0.0;
	  newMessage->plan0.start.pitch = 0.0;
	}
	
	/* then, set pattern as 2nd plan to enact */
	storeNavCommands(mpRequest->navReq, &(newMessage->plan1));
      }
      else  /* close enough already:  set pattern as first plan */
	storeNavCommands(mpRequest->navReq, &(newMessage->plan0));

      /* set flag to send this message */
      newMessage->sendNavMsg = 1;
      mpReply->status = MP_STATUS_RCVD;
    }
  }

  else {
    printf("Processing user command\n");
    MODE = MODE_User;
    if (mpRequest->planningMode == MP_SIMULATE_TASK) {
      if (mpRequest->type == MP_TYPE_SENSOR) {
	if (newMessage->sendSensMsg == 1) {
	  /* already a request waiting to be sent */
	  mpReply->status = MP_STATUS_BUSY;
	}
	else {
	  newMessage->sensMsg->mode = SENS_MAN_COST_REQUEST;
	  newMessage->sensMsg->targetID = mpRequest->sensReq.targetID;
	  newMessage->sensMsg->driverNum = mpRequest->sensReq.driverNum;
	  newMessage->sendSensMsg == 1;
          mpReply->status = MP_STATUS_RCVD;
	}
      }
      else {  /* MP_TYPE_NAVIGATE */
        if (newMessage->sendNavSim == 1) {
	  /* already a request waiting to be sent */
	  mpReply->status = MP_STATUS_BUSY;
	}
	else {
	  if (mpRequest->navReq.task == MP_MANEUVER && mpRequest->navReq.manplanTask.enact == 0) {
	    /* set the name of the maneuver plan we're generating here */
	    time(&ltime);
	    sprintf(mpRequest->navReq.manplanTask.filename, "mplan.%ld.txt", (long)ltime);
	  }
	  /* fix RelToGlo so it converts roll and pitch too? */
	  if (mpRequest->navReq.useAbsCoord == 0)             /* relative coords */
	    PlanningPose = RelToGlo(mpRequest->navReq.start, LastPose);
	  else                                                /* global coords */
	    PlanningPose = mpRequest->navReq.start;
	  storeNavCommands(mpRequest->navReq, &(newMessage->simplan));
	  newMessage->sendNavSim = 1;
	  mpReply->status = MP_STATUS_RCVD;
	}
      }
    }
    else {  /* planningMode == MP_COMMAND_TASK */
      if (mpRequest->type == MP_TYPE_SENSOR) {
	if (newMessage->sendSensMsg == 1) {
	  /* already a request waiting to be sent */
	  mpReply->status = MP_STATUS_BUSY;
	}
	else {
	  newMessage->sensMsg->mode = SENS_MAN_ACTION_REQUEST;
	  newMessage->sensMsg->targetID = mpRequest->sensReq.targetID;
	  newMessage->sensMsg->driverNum = mpRequest->sensReq.driverNum;
	  newMessage->sendSensMsg = 1;
          mpReply->status = MP_STATUS_RCVD;
	}
      }
      else { /* MP_TYPE_NAVIGATE */
	if (newMessage->sendNavMsg == 1) {
	  /* already a request waiting to be sent */
	  mpReply->status = MP_STATUS_BUSY;
	}
	else {
	  /* check here to see if we are at starting point - use LastPose */
	  if (mpRequest->navReq.useAbsCoord != 1) { /* compare with [0,0,0] */
	    curr.x = 0.0; curr.y = 0.0; curr.z = 0.0;
	    /* compare yaw too? roll? */
	    dist = mpGPS_diff(mpRequest->navReq.start, curr);
	  }
	  else dist = mpGPS_diff(mpRequest->navReq.start, LastPose);
	  if (fabs(dist) >= 0.5) {
	    /* if not close enough, alert user, ask if we should drive to start point first */
	    printf("Warning: requested start location is not current location\nFirst, drive to desired starting location\n");
	    mpReply->status = MP_STATUS_ERROR;
	  }
	  else {
	    storeNavCommands(mpRequest->navReq, &(newMessage->plan0));
	    newMessage->sendNavMsg = 1;
	    mpReply->status = MP_STATUS_RCVD;
	  }
	}
      }
    }
  }

  return RTI_TRUE;
}

/************************************************************************/
/* The following 4 functions set up the NDDS communications protocols:  */
/*   subscriberSetup                                                    */
/*   publisherSetup                                                     */
/*   serverSetup                                                        */
/*   clientSetup                                                        */
/************************************************************************/

/************************************************************************/
/* This function sets up the NDDS subscriber.                           */
/* It gathers info from sensor manager, database, nav manager, and      */
/* realtime modules.                                                    */
/************************************************************************/
void subscriberSetup(target_list_item* TargetList, MP_POSE* LastPose,
		     sensor_list* CurrSensList)
{
  NDDSSubscriber MissionPlannerSubs;
  NDDSSubscription sensorSubs, databaseSubs, navSubs, realtimeSubs;
  sensManComplete *item1;
  dbNotify *item2;
  navManComplete *item3;
  posDerivedState *item4;
  float deadline = 10.0;
  float min_separation = 0.0;

  item1 = (sensManComplete *)calloc(1, sizeof(sensManComplete));
  item2 = (dbNotify *)calloc(1, sizeof(dbNotify));
  item3 = (navManComplete *)calloc(1, sizeof(navManComplete));
  item4 = (posDerivedState *)calloc(1, sizeof(posDerivedState));

  MissionPlannerSubs = NddsSubscriberCreate();

  sensorSubs = NddsSubscriptionCreate(NDDS_SUBSCRIPTION_IMMEDIATE,
      "sensManComplete", sensManCompletePublicationType, item1,
      deadline, min_separation, SensManCompleteCallback, CurrSensList);
  databaseSubs = NddsSubscriptionCreate(NDDS_SUBSCRIPTION_IMMEDIATE,
      "dbNotify", dbNotifyPublicationType, item2,
      deadline, min_separation, TargetFound, TargetList);
  navSubs = NddsSubscriptionCreate(NDDS_SUBSCRIPTION_IMMEDIATE,
      "navManComplete", navManCompletePublicationType, item3,
      deadline, min_separation, NavManCompleteCallback, NULL);
  realtimeSubs = NddsSubscriptionCreate(NDDS_SUBSCRIPTION_IMMEDIATE,
      "posDerivedState", posDerivedStatePublicationType, item4,
      deadline, min_separation, NddsPose, LastPose);

  NddsSubscriberSubscriptionAdd(MissionPlannerSubs, sensorSubs);
  NddsSubscriberSubscriptionAdd(MissionPlannerSubs, databaseSubs);
  NddsSubscriberSubscriptionAdd(MissionPlannerSubs, navSubs);
  NddsSubscriberSubscriptionAdd(MissionPlannerSubs, realtimeSubs);
}


/************************************************************************/
/* This function sets up the NDDS client.                               */
/* It requests info/actions from sensor manager, target acquisition,    */
/* and nav manager modules.                                             */
/************************************************************************/
void clientSetup(NDDSClient* SensManClient, NDDSClient* TargetAcqClient,
		 NDDSClient* NavManClient)
{
  *TargetAcqClient = NddsClientCreate("targetAcqService", 
	targetAcqReplyPublicationType, targetAcqRequestPublicationType);
  *SensManClient = NddsClientCreate("sensManService",
	sensManReplyPublicationType, sensManRequestPublicationType);
  *NavManClient = NddsClientCreate("navManService",
	navManReplyPublicationType, navManRequestPublicationType);
}


/* NDDSchange - develop structure for missionReport */
/************************************************************************/
/* This function sets up the NDDS publisher.                            */
/* It produces (undefined) Mission Planner status messages.             */
/************************************************************************/
void publisherSetup(NDDSPublication* reportPublication, mPlannerReport* message)
{
  NDDSObjectInstance reportInstance;
  float persistence = 15.0;
  float strength = 1.0;

  /* do we just cast newReport into the proper type? */
  reportInstance = (NDDSObjectInstance)message;

  /* NDDSchange - how do we use newReport and reportInstance? */
  *reportPublication = NddsPublicationCreate("mPlannerReport",
     mPlannerReportPublicationType, reportInstance, persistence, strength);
}

/************************************************************************/
/* This function sets up the NDDS server.                               */
/* It responds to user requests.                                        */
/************************************************************************/
void serverSetup(TaskMessage *newMessage) {
  NDDSServer server = NULL; /* this isn't really used */
  mPlannerReply* mpReply;
  mPlannerRequest* mpRequest;
  float strength = 1.0;

  mpReply = (mPlannerReply *)calloc(1, sizeof(mPlannerReply));
  mpRequest = (mPlannerRequest *)calloc(1, sizeof(mPlannerRequest));

  NddsServerCreate(MPLANNER_SERVICE, NDDS_SERVER_IMMEDIATE, strength, mpReply,
		   mpRequest, mPlannerReplyPublicationType,
		   mPlannerRequestPublicationType, RouteTaskMode,
		   newMessage);
}


/************************************************************************/
/* Main                                                                 */
/************************************************************************/
void main(void) {
  time_t ltime;
  MP_POSE curr;
  int num, i;
  target_list_item *CurrTarget, *tempTarget;
  target_list_item* TargetList;
  sensor_list *tempSensor, *CurrSensList = NULL;
  int costs;  /* should be float? */
  PlanParams manMsg;
  PlannerInfo Status;
  int valid = 1;
  long REFTIME;
  double REFJULDATE;
  float minWaitTime    = 2.0;
  float maxWaitTime    = 5.0;

  /* NDDS client variables */
  targetAcqReply   *targetAcqItemReply;
  sensManReply     *sensManItemReply;
  navManReply      *navManItemReply;
  TaskMessage      *newMessage;
  missionReport    *newReport;
  navManRequest    *newNavMsg;

  NDDSClient SensManClient, TargetAcqClient, NavManClient;
  NDDSClientReplyStatus clientStatus;
  NDDSPublication reportPublication;

  /* Allocate memory for message data */
  targetAcqItemReply   = (targetAcqReply *)calloc(1, sizeof(targetAcqReply));
  sensManItemReply = (sensManReply *)calloc(1, sizeof(sensManReply));
  navManItemReply = (navManReply *)calloc(1, sizeof(navManReply));
  newReport = (missionReport *)calloc(1, sizeof(missionReport));
  newReport->message = (mPlannerReport *)calloc(1, sizeof(mPlannerReport));
  newMessage = (TaskMessage *)calloc(1, sizeof(TaskMessage));
  newMessage->sensMsg = (sensManRequest *)calloc(1, sizeof(sensManRequest));
  newMessage->targetMsg = (targetAcqRequest *)calloc(1, sizeof(targetAcqRequest));
  newNavMsg = (navManRequest *)calloc(1, sizeof(navManRequest));

  /* NDDSchange - pass in domain instead of using 1 */
  NddsInit(0, NULL);

  mPlannerReportNddsRegister();
  mPlannerReplyNddsRegister();
  mPlannerRequestNddsRegister();
  targetAcqReplyNddsRegister();
  targetAcqRequestNddsRegister();
  sensManReplyNddsRegister();
  sensManRequestNddsRegister();
  sensManCompleteNddsRegister();
  navManReplyNddsRegister();
  navManRequestNddsRegister();
  navManCompleteNddsRegister();
  dbNotifyNddsRegister();
  posDerivedStateNddsRegister();

  clientSetup(&SensManClient, &TargetAcqClient, &NavManClient);
  subscriberSetup(TargetList, &LastPose, CurrSensList);
  serverSetup(newMessage);
  publisherSetup(&reportPublication, newReport->message);

  PLAN_TIME = 0.0;
  /* Initialize LastPose to 0 in case telemetry is not running */
  LastPose.x = 0.0;
  LastPose.y = 0.0;
  LastPose.z = 0.0;
  LastPose.roll = 0.0;
  LastPose.pitch = 0.0;
  LastPose.yaw = 0.0;

  newReport->sendReportMsg = 0;
  newMessage->sendSensMsg = 0;
  newMessage->sendNavMsg = 0;
  newMessage->sendTargetMsg = 0;

  printf("Enter the coordinates of the robot's DGPS base station.\n");
  printf("  Latitude: ");
  scanf("%f", &ROB_LAT);
  ROB_LAT = ROB_LAT * PI / 180.0;
  printf("  Longitude: ");
  scanf("%f", &ROB_LON);
  ROB_LON = ROB_LON * PI / 180.0;
  ROB_XAX = 0.0;

  /* set reference for converting computer clock time to Julian date */
  printf("Enter the current date and time:\n");
  REFJULDATE = getdate();
  time(&ltime);
  REFTIME = (long)ltime;

  /* Set up verbosity level */
  npVERBOSE = -1;
  while (npVERBOSE != 0 && npVERBOSE != 1) {
    printf("Do you want verbose readout? (0 = no, 1 = yes) ");
    scanf("%d", &npVERBOSE); }

  /* we may want to publish this instead of printing it out */
  /* file RecordSun will contain the following information:
     robot.x, robot.y, robot.z, robot.yaw, power1, power2, altitude, azimuth,
     Jdate */
  num = -1;
  while (num != 0 && num != 1) {
    printf("Do you want to record the solar information? (0 = no, 1 = yes) ");
    scanf("%d", &num); }
  if (num == 0) sprintf(RecordSun, "none");
  else {
    printf("Enter the name of the file in which to record: ");
    scanf("%s", RecordSun); }

  /* set up the file names to use for saving maps and plans */
  sprintf(PMAPNAME, "pmap.%ld.txt", (long)ltime);

  /* initialize the planning map */
  curr.x = 0.0; curr.y = 0.0; curr.yaw = 0.0;
  init_map(&PlanMap, curr);
  /* readElevMap("cratermap.ppm", &PlanMap, 3.125, -40.0); */

  /*
  if (NddsClientServerWait(TargetAcqClient, 2.0, 5, 1) == RTI_FALSE)
    printf("error...can't find target acquisition server!\n");
  if (NddsClientServerWait(SensManClient, 2.0, 5, 1) == RTI_FALSE)
    printf("error...can't find sensor manager server!\n"); */
  if (NddsClientServerWait(NavManClient, 2.0, 5, 1) == RTI_FALSE)
  printf("error...can't find navigation manager server!\n");

  printf("Ready to accept commands.\n");
  while (1) {
    /* check for new messages to relay - reset flags */
    if (newMessage->sendNavMsg) {
      newNavMsg->plan0 = newMessage->plan0;
      newNavMsg->plan1 = newMessage->plan1;
      newNavMsg->plan2 = newMessage->plan2;
      newNavMsg->plan3 = newMessage->plan3;
      newNavMsg->plan4 = newMessage->plan4;
      clientStatus = NddsClientCallAndWait(NavManClient, navManItemReply,
		     newNavMsg, minWaitTime, maxWaitTime);
      if (clientStatus == NDDS_CLIENT_RECV_REPLY)
	printf("navManRequest message sent\n");
      else if (clientStatus == NDDS_CLIENT_ERROR)
	printf("error sending navManRequest message\n");
      /* if error, should we try again a certain number of times? */
      newMessage->sendNavMsg = 0;
      newMessage->plan0.storage = 0;
      newMessage->plan1.storage = 0;
      newMessage->plan2.storage = 0;
      newMessage->plan3.storage = 0;
      newMessage->plan4.storage = 0;
    }
    if (newMessage->sendSensMsg) {
      clientStatus = NddsClientCallAndWait(SensManClient, sensManItemReply,
		     newMessage->sensMsg, minWaitTime, maxWaitTime);
      if (clientStatus == NDDS_CLIENT_RECV_REPLY)
	printf("sensManRequest message sent\n");
      else if (clientStatus == NDDS_CLIENT_ERROR)
	printf("error sending sensManRequest message\n");
      /* if error, should we try again a certain number of times? */
      newMessage->sendSensMsg = 0;
    }
    if (newMessage->sendTargetMsg) {
      clientStatus = NddsClientCallAndWait(TargetAcqClient, targetAcqItemReply,
		     newMessage->targetMsg, minWaitTime, maxWaitTime);
      if (clientStatus == NDDS_CLIENT_RECV_REPLY)
	printf("targetAcqRequest message sent\n");
      else if (clientStatus == NDDS_CLIENT_ERROR)
	printf("error sending targetAcqRequest message\n");
      /* if error, should we try again a certain number of times? */
      newMessage->sendTargetMsg = 0;
    } 
    if (newMessage->sendNavSim) {
      Status = processNewMsg(newMessage->simplan);
      /* we should print this somewhere else? to a variable? a file? */
      printf("taskMode = %d\n", Status.taskMode);
      printf("status = %d\n", Status.navPlannerStatus);
      printf("time cost = %.3f\n", Status.timeCost);
      printf("area covered = %.3f\n", Status.areaCovered);
      printf("energy generated = %.3f\n", Status.energyGen);
      if (newMessage->simplan.task == MP_MANEUVER && newMessage->simplan.manplanTask.enact == 0)
	printf("maneuver plan saved in file '%s'\n", newMessage->simplan.manplanTask);
      newMessage->sendNavSim = 0;
    }
    if (newReport->sendReportMsg) {
      NddsPublicationSend(reportPublication);
      newReport->sendReportMsg = 0;
    }

    if (MODE == MODE_User) {
      /* we are in user mode, only do what user says to do */
      /* listen for user callback */
      NddsUtilitySleep(1.0);
      /* user can tell us to enter non-user and/or searching mode */
      /* when pattern has finished, we switch to user mode */
    }
    else {  /* we're in autonomous search mode */
      if (MODE == MODE_Target) {
	if (TargetStage == TS_Evaluate) {
	  /* stop robot - send nav command */
	  /* Note: this will override any navigation plan waiting to be sent */
       	  newNavMsg->plan0.storage = 1;
	  newNavMsg->plan0.minTurningRadius = 4.0;
	  newNavMsg->plan0.cameraFOV = 3.0;
	  newNavMsg->plan0.speed = 0.15;
	  newNavMsg->plan0.start.x = 0.0;
          newNavMsg->plan0.start.y = 0.0;
          newNavMsg->plan0.start.z = 0.0;
          newNavMsg->plan0.start.roll = 0.0;
          newNavMsg->plan0.start.pitch = 0.0;
          newNavMsg->plan0.start.yaw = 0.0;
	  time(&ltime);
	  newNavMsg->plan0.startdate = REFJULDATE + (double)((long)ltime - REFTIME)/86400.0;
	  newNavMsg->plan0.startNewPlan = 1;
	  newNavMsg->plan0.useAbsCoord = 0;
	  newNavMsg->plan0.task = MP_WAYPOINT;
	  newNavMsg->plan0.wayptTask.goal.x = 0.0;
          newNavMsg->plan0.wayptTask.goal.y = 0.0;
          newNavMsg->plan0.wayptTask.goal.z = 0.0;
          newNavMsg->plan0.wayptTask.goal.roll = 0.0;
          newNavMsg->plan0.wayptTask.goal.pitch = 0.0;
          newNavMsg->plan0.wayptTask.goal.yaw = 0.0;
	  newNavMsg->plan0.wayptTask.tolerance = 2.0;
	  printf("Sending message to (effectively) stop robot\n");
	  clientStatus = NddsClientCallAndWait(NavManClient, navManItemReply,
		         newNavMsg, minWaitTime, maxWaitTime);
	  /* should check clientStatus and resend if error */
	  newMessage->sendNavMsg = 0;

	  newMessage->targetMsg->function = TARGET_ACQ_STOP;
	  newMessage->targetMsg->driverNum = 0;  /* figure out which driverNum is camera */
	  newMessage->targetMsg->period = 0.01;  /* what is the shortest period? */
	  clientStatus = NddsClientCallAndWait(TargetAcqClient, targetAcqItemReply,
				   newMessage->targetMsg, minWaitTime, maxWaitTime);
	  if (clientStatus == NDDS_CLIENT_RECV_REPLY)
	    printf("targetAcqRequest message sent\n");
	  else if (clientStatus == NDDS_CLIENT_ERROR)
	    printf("error sending targetAcqRequest message\n");
	  newMessage->sendTargetMsg = 0;

	  /* calculate costs and determine if we should go to target & which
	     sensors to use - right now, always do incomplete targets */
	  tempTarget = TargetList;
	  while (tempTarget != NULL && tempTarget->complete == 1)
	    tempTarget = tempTarget->next;
	  if (tempTarget == NULL) costs = 0;
	  else costs = 1;

	  /* costs should be set such that if cheaper to wait until
	     next row, costs are too high for current row */
	  /* should also determine best target, not just first incomplete */

	  if (costs > 0) {
	    /* set CurrTarget to point to current target */
	    CurrTarget = tempTarget;
	    CurrTarget->cost = 1;

	    /* make sure sensor list is NULL */
	    tempSensor = CurrSensList;
	    while (tempSensor != NULL) {
	      CurrSensList = tempSensor->next;
	      free(tempSensor);
	      tempSensor = CurrSensList; }
	    CurrSensList = (sensor_list*)malloc(sizeof(sensor_list*));
	    tempSensor = CurrSensList;
            /* ASKMIKE - this should be a driver number, not a sensor name */
	    tempSensor->driver = SENSOR_PAN_TILT_CAMERA;
	    tempSensor->next = (sensor_list*)malloc(sizeof(sensor_list*));
	    tempSensor = tempSensor->next;
            /* ASKMIKE - this should be a driver number, not a sensor name */
	    tempSensor->driver = SENSOR_SPECTROMETER;
	    tempSensor->next = NULL;
	    TargetStage = TS_Maneuver; }
	  else TargetStage = TS_Resume;
	}

	if (TargetStage == TS_Maneuver) {
	  /* generate waypoint/maneuver plan */
	  manMsg.storage = 1;
	  manMsg.minTurningRadius = 4.0;
	  manMsg.cameraFOV = 3.0;
	  manMsg.speed = 0.15;
	  manMsg.start = LastPose;
	  time(&ltime);
	  manMsg.startdate = REFJULDATE + (double)((long)ltime - REFTIME)/86400.0;
	  manMsg.startNewPlan = 1;
	  manMsg.useAbsCoord = 1;
	  manMsg.task = MP_MANEUVER;
	  manMsg.manplanTask.enact = 0;  /* generate new plan */
	  time(&ltime);
	  sprintf(manMsg.manplanTask.filename, "mplan.%ld.txt", (long)ltime);
	  manMsg.manplanTask.type = 1;
	  manMsg.manplanTask.goal = CurrTarget->location;
	  /* find workspace relevant tolerances */
          /* ASKMIKE */
          /* need to ask for tolerances for this driver number, or else just use some default */
	  if (CurrSensList->driver == SENSOR_PAN_TILT_CAMERA) {
	    manMsg.manplanTask.distanceTol = 3.0;
	    manMsg.manplanTask.angularTol = PI/4.0;
	  }
	  else if (CurrSensList->driver == SENSOR_SPECTROMETER) {
	    manMsg.manplanTask.distanceTol = 2.0;
	    manMsg.manplanTask.angularTol = PI/2.0;
	  }
	  else {
	    printf("Incorrect sensor type?\n");
	    manMsg.manplanTask.distanceTol = 4.0;
	    manMsg.manplanTask.angularTol = PI;
	  }
	  Status = processNewMsg(manMsg);
	  if (Status.navPlannerStatus == NAV_MAN_SUCCESSFUL)
	    printf("Found a plan: time to move = %.3f\n", Status.timeCost);

	  /* enact maneuver plan */
	  newNavMsg->plan0.storage = 1;
	  newNavMsg->plan0.minTurningRadius = 4.0;
	  newNavMsg->plan0.cameraFOV = 3.0;
	  newNavMsg->plan0.speed = 0.15;
	  newNavMsg->plan0.start = LastPose;
	  time(&ltime);
	  newNavMsg->plan0.startdate = REFJULDATE + (double)((long)ltime - REFTIME)/86400.0;
	  newNavMsg->plan0.startNewPlan = 1;
	  newNavMsg->plan0.useAbsCoord = 1;
	  newNavMsg->plan0.task = MP_MANEUVER;
	  newNavMsg->plan0.manplanTask.enact = 1;
	  sprintf(newNavMsg->plan0.manplanTask.filename, "%s", manMsg.manplanTask.filename);
	  clientStatus = NddsClientCallAndWait(NavManClient, navManItemReply,
		         newNavMsg, minWaitTime, maxWaitTime);
	  newMessage->sendNavMsg = 0;

	  /* move to next stage */
	  TargetStage = TS_WaitingMan;
	}

	if (TargetStage == TS_WaitingMan) /* listen for navPlanStatus */
	  NddsUtilitySleep(1.0);

	if (TargetStage == TS_PoseCheck) {
	  /* check to see we're in proper position, within workspace */
          /* ASKMIKE - this should be a request of sensor manager? */
	  valid = check_workspace(LastPose, CurrTarget->location,
				  CurrSensList->driver);
	  if (valid) TargetStage = TS_Sensors;
	  else TargetStage = TS_Maneuver; }

	if (TargetStage == TS_Sensors) {
	  newMessage->sensMsg->mode = SENS_MAN_ACTION_REQUEST;
	  newMessage->sensMsg->targetID = CurrTarget->targetID;
	  /* ASKMIKE - figure out driverNum */
	  newMessage->sensMsg->driverNum = CurrSensList->driver;
	  printf("Sending sensManRequest message\n");
	  clientStatus = NddsClientCallAndWait(SensManClient, sensManItemReply,
		         newMessage->sensMsg, minWaitTime, maxWaitTime);
          /* should we check to see that clientStatus is good? redo if not? */
          CurrSensList->reqNum = sensManItemReply->reqNum;
	  newMessage->sendSensMsg = 0;

	  /* move to next stage */
	  TargetStage = TS_WaitingSens; }

	if (TargetStage == TS_WaitingSens) /* listen for sensManStatus */
	  NddsUtilitySleep(1.0);

	if (TargetStage == TS_CleanUp) {
	  /* set current target as complete */
	  CurrTarget->complete = 1;
	  /* check to make sure no other incomplete targets */
	  tempTarget = TargetList;
	  while (tempTarget != NULL && tempTarget->complete == 1)
	    tempTarget = tempTarget->next;
	  if (tempTarget->complete == 0) TargetStage = TS_Evaluate;
	  else TargetStage = TS_Resume; }

	if (TargetStage == TS_Resume) {
	  /* resume pattern - send nav command */
       	  newNavMsg->plan0.storage = 1;
	  newNavMsg->plan0.minTurningRadius = 4.0;
	  newNavMsg->plan0.cameraFOV = 3.0;
	  newNavMsg->plan0.speed = 0.15;
	  newNavMsg->plan0.start.x = 0.0;
          newNavMsg->plan0.start.y = 0.0;
          newNavMsg->plan0.start.z = 0.0;
          newNavMsg->plan0.start.roll = 0.0;
          newNavMsg->plan0.start.pitch = 0.0;
          newNavMsg->plan0.start.yaw = 0.0;
	  time(&ltime);
	  newNavMsg->plan0.startdate = REFJULDATE + (double)((long)ltime - REFTIME)/86400.0;
	  newNavMsg->plan0.startNewPlan = 0;
	  newNavMsg->plan0.useAbsCoord = 0;
	  printf("Sending message to resume pattern\n");
	  clientStatus = NddsClientCallAndWait(NavManClient, navManItemReply,
		         newNavMsg, minWaitTime, maxWaitTime);
	  newMessage->sendNavMsg = 0;

	  /* move out of target mode */
	  TargetStage = TS_Evaluate;
	  MODE = MODE_Search;

	  newMessage->targetMsg->function = TARGET_ACQ_START;
	  newMessage->targetMsg->driverNum = 0;  /* figure out which driverNum is camera */
	  newMessage->targetMsg->period = 0.01;  /* what is the shortest period? */
	  clientStatus = NddsClientCallAndWait(TargetAcqClient, targetAcqItemReply,
				   newMessage->targetMsg, minWaitTime, maxWaitTime);
	  if (clientStatus == NDDS_CLIENT_RECV_REPLY)
	    printf("targetAcqRequest message sent\n");
	  else if (clientStatus == NDDS_CLIENT_ERROR)
	    printf("error sending targetAcqRequest message\n");
	  newMessage->sendTargetMsg = 0;
	}
      }

      else {  /* we're just doing pattern */
	NddsUtilitySleep(1.0);
	/* evaluate current list of targets/tasks - every second? */
	tempTarget = TargetList;
	while (tempTarget != NULL && tempTarget->complete == 1)
	  tempTarget = tempTarget->next;
	if (tempTarget == NULL) costs = 0;  /* all targets complete */
	else costs = 1;                     /* incomplete target(s) */

	/* costs should be set such that if cheaper to wait until
	   next row, costs are too high for current row */
	if (costs > 0) {
	  TargetStage = TS_Evaluate;
	  MODE = MODE_Target; }
      }
    }
  }
  return;
}



