/* FILE: dbServer.cpp
   AUTHOR: Michael Wagner
   CREATED: April 22, 1999
   DESCRIPTION: This is code that uses a database object. It serves as a link
     between this object and NDDS.
*/

#include <iostream.h>
#include <stdio.h>
#include <stdlib.h>
#include "ndds/NDDS.h"
#include "dbMsgs.h"
#include "database.h"
#include "SAS_ServiceNames.h"
#include "dbServer.h"
#include "queue.h"
#include "nddsMsgNames.h"
#include "SAS_Config.h"

/* Returns RTI_TRUE when a reply is to be sent,
   RTI_FALSE if not. */
RTIBool ServerRoutine(NDDSObjectInstance reply,
		      NDDSObjectInstance request,
		      void *userData) 
{
  dbReply   *itemReply   = (dbReply *)reply;
  dbRequest *itemRequest = (dbRequest *)request;
  dbServerParam *param = (dbServerParam *)userData;
  database *db = param->db;
  queue<dbNotify *> *q = param->q;
  int verbose = param->verbose;

  /* Use the request here. */
  dbRecord getRec, searchRec;
  dbNotify *notifyMsg;
  int firstID, lastID;

  switch(itemRequest->function) {

  case DB_INSERT_FUNCTION:
    itemReply->status = db->insertRecord(itemReply->record.targetID);

    // For dbNotify
    notifyMsg = (dbNotify *)calloc(1, sizeof(dbNotify));
    notifyMsg->record.targetID = itemReply->record.targetID;
    notifyMsg->function = DB_INSERT_FUNCTION;
    q->add(notifyMsg);

    if(verbose) {
      cerr << "[dbServer] Inserted record #" << itemReply->record.targetID << ", returned " << itemReply->status << endl;
    }
    break;

  case DB_INSERT_MULTI_FUNCTION: 
    if(db->insertRecordMulti(itemRequest->numIDs, firstID, lastID) == DB_INSERT_ERROR) {
      itemReply->status = DB_INSERT_ERROR;
    } else {
      itemReply->status = DB_OK;
      itemReply->IDs[0] = firstID;
      itemReply->IDs[1] = lastID;
    }

    // For dbNotify
    for(int i=firstID; i <= lastID; i++) {
      notifyMsg = (dbNotify *)calloc(1, sizeof(dbNotify));
      notifyMsg->record.targetID = i;
      notifyMsg->function = DB_INSERT_FUNCTION;
      q->add(notifyMsg);
    }

    if(verbose) {
      cerr << "[dbServer] Inserted records #" << firstID << " through " << lastID << ", returned " << itemReply->status << endl;
    }
    break;

  case DB_GET_FUNCTION:
    itemReply->status = db->getRecord(itemRequest->record.targetID, getRec);
    memcpy(&itemReply->record, &getRec, sizeof(dbRecord));
    if(verbose) {
      cerr << "[dbServer] Got record #" << itemRequest->record.targetID << ", returned " << itemReply->status << endl;
    }
    break;

  case DB_PUT_FUNCTION:
    itemReply->status = db->putRecord(itemRequest->record, itemRequest->recordBitMask);

    // For dbNotify
    notifyMsg = (dbNotify *)calloc(1, sizeof(dbNotify));
    memcpy(&notifyMsg->record, &itemRequest->record, sizeof(dbRecord));
    memcpy(&notifyMsg->changed, &itemRequest->recordBitMask, sizeof(dbRecordBitMask));
    notifyMsg->function = DB_PUT_FUNCTION;
    q->add(notifyMsg);

    if(verbose) {
      cerr << "[dbServer] Put record #" << itemRequest->record.targetID << ", returned " << itemReply->status << endl;
    }
    break;

  case DB_SEARCH_MULTI_FUNCTION:
    itemReply->status = db->searchRecordMulti(itemRequest->record, itemRequest->recordBitMask, itemRequest->numIDs, searchRec);
    memcpy(&itemReply->record, &searchRec, sizeof(dbRecord));
    if(verbose) {
      cerr << "[dbServer] Search multiple returned " << itemReply->status << endl;
    }
    break;
 
  case DB_SEARCH_FUNCTION:
    itemReply->status = db->searchRecord(itemRequest->record, itemRequest->recordBitMask, searchRec);
    memcpy(&itemReply->record, &searchRec, sizeof(dbRecord));
    if(verbose) {
      cerr << "[dbServer] Search returned " << itemReply->status << endl;
    }
    break;

  case DB_SAVE_FUNCTION:
    itemReply->status = db->saveDB(itemRequest->name);
    if(verbose) {
      cerr << "[dbServer] DB save returned status " << itemReply->status << endl;
    }
    break;

  case DB_RESTORE_FUNCTION:
    itemReply->status = db->restoreDB(itemRequest->name);
    if(verbose) {
      cerr << "[dbServer] DB restore returned status " << itemReply->status << endl;
    }
    break;

  case DB_ARCHIVE_FUNCTION:
    itemReply->status = db->saveDB(itemRequest->name);
    if(itemReply->status == DB_OK) {
      itemReply->status = db->archiveDB(itemRequest->name);
    }
    if(verbose) {
      cerr << "[dbServer] DB archive returned status " << itemReply->status << endl;
    }
    break;

  case DB_SM_DRIVER_REG_FUNCTION:
    itemReply->status = db->smDriverRegister(itemRequest->name);
    if(verbose) {
      cerr << "[dbServer] Sensor manager driver registration returned " << itemReply->status << endl;
    }
    break;

  case DB_GET_SM_DRIVER_NAME_FUNCTION:
    itemReply->status = db->getSmDriverName(itemRequest->driverNum, itemReply->name);
    if(verbose) {
      cerr << "[dbServer] Get sensor manager driver #" << itemRequest->driverNum << " returned " << itemReply->status << endl;
    }
    break;

  case DB_GET_NUM_RECORDS_FUNCTION:
    itemReply->status = db->getNumRecords(itemReply->numRecords);
    if(verbose) {
      cerr << "[dbServer] Get num records returned " << itemReply->status << endl;
    }
    break;

  default:
    return(RTI_FALSE);
  }

  return(RTI_TRUE);
}


int main(int argc, char *argv[]) {
  // Set up local variables
  database *db = new database();
  queue<dbNotify *> *dbQ = new queue<dbNotify *>();
  dbServerParam *param = (dbServerParam *)calloc(1, sizeof(dbServerParam));
  param->db = db;
  param->q = dbQ;
  param->verbose = 0;
  dbRequest *dbRequestMsg;
  dbReply *dbReplyMsg;
  dbNotify *dbNotifyMsg;

  if(argc != 1 && argc != 2) {
    cerr << "Usage: dbServer [-v]" << endl;
    return(-1);
  }
  if(argc == 2) {
    if(strcmp(argv[1], "-v") == 0) {
      cerr << "[dbServer] Running in verbose mode" << endl;
      param->verbose = 1;
    } 
  }

  int nddsDomain;
  if(!getSAS_Config(NDDS_DOMAIN, nddsDomain)) {
    cerr << "[dbServer] ERROR: Cannot read " << NDDS_DOMAIN << " from config file!" << endl;
    return(-1);
  }
  NddsInit(nddsDomain, NULL);
  dbReplyNddsRegister();
  dbRequestNddsRegister();
  dbNotifyNddsRegister();

  cerr << "[dbServer] Finished NDDS initialization, starting server on domain " << nddsDomain << "...";
  
  dbReplyMsg   = (dbReply *)calloc(1, sizeof(dbReply));
  dbRequestMsg = (dbRequest *)calloc(1, sizeof(dbRequest));
  
  NddsServerCreate(DB_SERVICE, NDDS_SERVER_IMMEDIATE, DB_SERVER_STRENGTH, 
		   dbReplyMsg, dbRequestMsg, 
		   dbReplyPublicationType, 
		   dbRequestPublicationType, 
		   ServerRoutine, (void *)param);
  
  cerr << "done!" << endl;

  cerr << "[dbServer] Creating dbNotify publisher...";

  dbNotifyMsg  = (dbNotify *)calloc(1, sizeof(dbNotify));
  NDDSPublication dbNotifyPublication = NddsPublicationCreate(DB_NOTIFY_MSG_NAME, dbNotifyPublicationType,
							      dbNotifyMsg, DB_NOTIFY_PERSISTANCE,
							      DB_NOTIFY_STRENGTH);
  cerr << "done!" << endl;

  // Variables used inside while loop
  dbNotify *tmpNotifyMsg;
  long count = 0; // When count reaches MAX_COUNT_UNTIL_ARCHIVE, the database will automatically archive itself.
  int saveStatus;

  while(1) {
    // Check to see if archival is necessary
    if(count > MAX_COUNT_UNTIL_ARCHIVE) {
      count = 0;
      if((saveStatus = db->saveDB("db_autosave")) != DB_OK) {
	cerr << "[dbServer] WARNING: Unable to save database to file \"db_autosave\", saveDB returned: " << saveStatus << endl;
      }
    } 

    // Check for dbNotify messages that need to be sent out.
    if(!dbQ->isEmpty()) {
      tmpNotifyMsg = (dbNotify *)dbQ->remove();
      if(tmpNotifyMsg != NULL) {
	memcpy(dbNotifyMsg, tmpNotifyMsg, sizeof(dbNotify));
	NddsPublicationSend(dbNotifyPublication);
      }
    }

    NddsUtilitySleep(0.1);
    count++;
  }

  delete(db);

  cout << "[dbServer] ERROR: main() should not return!" << endl;

  return(0);
}
