#include <iostream.h>
#include <string.h>
#include "ndds/NDDS.h"
#include "database.h"
#include "dbDef.h"
#include "dbClient.h"
#include "dbServer.h"
#include "dbClientDef.h"
#include "SAS_Config.h"
#include "smDriverDef.h"
#include "nddsMsgNames.h"
#include "SAS_ServiceNames.h"
#include "queue.h"

#ifndef MIN
#define MIN(_a_, _b_) (_a_ < _b_) ? _a_ : _b_
#endif

RTIBool dbNotifyCallback(NDDSRecvInfo *issue) {
  dbNotify *item = (dbNotify *)issue->instance;
  queue<dbNotify *> *dbQ = (queue<dbNotify *> *)issue->callBackRtnParam;

  if(issue->status == NDDS_FRESH_DATA) {
    dbNotify *tmp = (dbNotify *)calloc(1, sizeof(dbNotify));
    memcpy(tmp, item, sizeof(dbNotify));
    dbQ->add(tmp);
  }

  return(RTI_TRUE);
}


dbClient::dbClient(int verbose) {
  this->verbose = verbose;
  dbReplyMsg   = (dbReply *)calloc(1, sizeof(dbReply));
  dbRequestMsg = (dbRequest *)calloc(1, sizeof(dbRequest));
  dbNotifySendMsg = (dbNotify *)calloc(1, sizeof(dbNotify));
  dbNotifyRecvMsg = (dbNotify *)calloc(1, sizeof(dbNotify));
  q = new queue<dbNotify *>();
}
  

dbClient::dbClient() {
  this->verbose = 0;
  dbReplyMsg   = (dbReply *)calloc(1, sizeof(dbReply));
  dbRequestMsg = (dbRequest *)calloc(1, sizeof(dbRequest));
  dbNotifySendMsg = (dbNotify *)calloc(1, sizeof(dbNotify));
  dbNotifyRecvMsg = (dbNotify *)calloc(1, sizeof(dbNotify));
  q = new queue<dbNotify *>();
}
  

dbClient::~dbClient() {
  free(dbReplyMsg);
  free(dbRequestMsg);
  free(dbNotifySendMsg);
  delete(q);
}


int dbClient::startNDDS(int verbose) {
  this->verbose = verbose;
  startNDDS();
}

int dbClient::startNDDS() {
  int nddsDomain;

  if(!getSAS_Config(NDDS_DOMAIN, nddsDomain)) {
    if(verbose) {
      cerr << "[dbClient] ERROR: Cannot read " << NDDS_DOMAIN << " from config file!" << endl;
    }
    return(0);
  }

  NddsInit(nddsDomain, NULL);

  dbReplyNddsRegister();
  dbRequestNddsRegister();
  dbNotifyNddsRegister();

  if(verbose) {
    cerr << "[dbClient] Finished NDDS initialization, starting client...";
  }

  dbClient = NddsClientCreate(DB_SERVICE, dbReplyPublicationType,
			      dbRequestPublicationType);

  if(NddsClientServerWait(dbClient, 2.0, 5, 1) == RTI_FALSE) { 
    cerr << "[dbClient] ERROR: \"" << DB_SERVICE << "\" server missing!" << endl;
    return(-1);
  } 

  if(verbose) {
    cerr << "done!" << endl;
    
    cerr << "[dbClient] Creating dbNotify subscription and publication...";
  }

  dbNotifySub = NddsSubscriptionCreate(NDDS_SUBSCRIPTION_IMMEDIATE,
  				       DB_NOTIFY_MSG_NAME, dbNotifyPublicationType,
  				       dbNotifyRecvMsg, DB_NOTIFY_DEADLINE,
  				       DB_NOTIFY_MIN_SEPARATION, dbNotifyCallback, q);

  dbNotifyPub = NddsPublicationCreate(DB_NOTIFY_MSG_NAME, dbNotifyPublicationType,
				      dbNotifySendMsg, DB_NOTIFY_PERSISTANCE,
				      DB_NOTIFY_STRENGTH);


  if(verbose) {
    cerr << "done!" << endl;
  }

  return(1);
}


int dbClient::databaseHasNewInfo(dbRecord &record) {
  if(!q->isEmpty()) {
    dbNotify *msg = q->remove();
    if(msg != NULL) {
      memcpy(&record, &msg->record, sizeof(dbRecord));
      free(msg);
      return(1);
    }
  }

  return(0);
}


int dbClient::getRecord(int targetID, dbRecord &record) {
  dbRequestMsg->function = DB_GET_FUNCTION;
  dbRequestMsg->record.targetID = targetID;
      
  dbClientStatus = NddsClientCallAndWait(dbClient, dbReplyMsg, dbRequestMsg,
					 DB_MIN_WAIT_TIME, DB_MAX_WAIT_TIME);
  if(dbClientStatus == NDDS_CLIENT_RECV_REPLY) {
    if(dbReplyMsg->status != DB_OK) {
      if(verbose) {
	cerr << "[dbClient] ERROR: Get record failed! Returned " << dbReplyMsg->status << endl;
      }
    }
    memcpy(&record, &dbReplyMsg->record, sizeof(dbRecord));
    return(dbReplyMsg->status);
  } else {
    if(verbose) {
      cerr << "[dbClient] ERROR: Database did not respond to get record request!" << endl;
    }
    return(DB_DID_NOT_RESPOND_ERROR);
  }
}


int dbClient::putRecord(int targetID, const dbRecord &record, const dbRecordBitMask &bitMask) {
  dbRequestMsg->function = DB_PUT_FUNCTION;
  memcpy(&dbRequestMsg->record, &record, sizeof(dbRecord));
  memcpy(&dbRequestMsg->recordBitMask, &bitMask, sizeof(dbRecordBitMask));
  dbRequestMsg->record.targetID = targetID; /* redundant, but hey */
      
  dbClientStatus = NddsClientCallAndWait(dbClient, dbReplyMsg, dbRequestMsg,
					 DB_MIN_WAIT_TIME, DB_MAX_WAIT_TIME);
  if(dbClientStatus == NDDS_CLIENT_RECV_REPLY) {
    if(dbReplyMsg->status != DB_OK) {
      if(verbose) {
	cerr << "[dbClient] ERROR: Put record failed! Returned " << dbReplyMsg->status << endl;
      }
    }
    return(dbReplyMsg->status);
  } else {
    if(verbose) {
      cerr << "[dbClient] ERROR: Database did not respond to put record request!" << endl;
    }
    return(DB_DID_NOT_RESPOND_ERROR);
  }
}


int dbClient::putRecord(const dbRecord &record, const dbRecordBitMask &bitMask) {
  dbRequestMsg->function = DB_PUT_FUNCTION;
  memcpy(&dbRequestMsg->record, &record, sizeof(dbRecord));
  memcpy(&dbRequestMsg->recordBitMask, &bitMask, sizeof(dbRecordBitMask));
     
  dbClientStatus = NddsClientCallAndWait(dbClient, dbReplyMsg, dbRequestMsg,
					 DB_MIN_WAIT_TIME, DB_MAX_WAIT_TIME);
  if(dbClientStatus == NDDS_CLIENT_RECV_REPLY) {
    if(dbReplyMsg->status != DB_OK) {
      if(verbose) {
	cerr << "[dbClient] ERROR: Put record failed! Returned " << dbReplyMsg->status << endl;
      }
    }
    return(dbReplyMsg->status);
  } else {
    if(verbose) {
      cerr << "[dbClient] ERROR: Database did not respond to put record request!" << endl;
    }
    return(DB_DID_NOT_RESPOND_ERROR);
  }
}


int dbClient::insertRecord(int &newID) {
  dbRequestMsg->function = DB_INSERT_FUNCTION;
  
  dbClientStatus = NddsClientCallAndWait(dbClient, dbReplyMsg, dbRequestMsg,
					 DB_MIN_WAIT_TIME, DB_MAX_WAIT_TIME);
  if(dbClientStatus == NDDS_CLIENT_RECV_REPLY) {
    if(dbReplyMsg->status != DB_OK) {
      if(verbose) {
	cerr << "[dbClient] ERROR: Insert record failed! Returned " << dbReplyMsg->status << endl;
      }
    }
    return(dbReplyMsg->status);
  } else {
    if(verbose) {
      cerr << "[dbClient] ERROR: Database did not respond to insert request!" << endl;
    }
    return(DB_DID_NOT_RESPOND_ERROR);
  }
}


int dbClient::insertRecords(int numRecords, int &firstID, int &lastID) {
  dbRequestMsg->function = DB_INSERT_MULTI_FUNCTION;
  dbRequestMsg->numIDs = numRecords;
  dbClientStatus = NddsClientCallAndWait(dbClient, dbReplyMsg, dbRequestMsg,
					 DB_MIN_WAIT_TIME, DB_MAX_WAIT_TIME);
  if(dbClientStatus == NDDS_CLIENT_RECV_REPLY) {
    if(dbReplyMsg->status != DB_OK) {
      if(verbose) {
	cerr << "[dbClient] ERROR: Insert record multiple failed! Returned " << dbReplyMsg->status << endl;
      }
    } 
    firstID = dbReplyMsg->IDs[0];
    lastID = dbReplyMsg->IDs[1];
    return(dbReplyMsg->status);
  } else {
    if(verbose) {
      cerr << "[dbClient] ERROR: Database did not respond to insert multiple request!" << endl;
    }
    return(DB_DID_NOT_RESPOND_ERROR);
  }
}


int dbClient::getNumRecords(int &num) {
  dbRequestMsg->function = DB_GET_NUM_RECORDS_FUNCTION;

  dbClientStatus = NddsClientCallAndWait(dbClient, dbReplyMsg, dbRequestMsg,
					 DB_MIN_WAIT_TIME, DB_MAX_WAIT_TIME);
  if(dbClientStatus == NDDS_CLIENT_RECV_REPLY) {
    if(dbReplyMsg->status != DB_OK) {
      if(verbose) {
	cerr << "[dbClient] ERROR: Get num records failed! Returned " << dbReplyMsg->status << endl;
      }
    } 
    num = dbReplyMsg->numRecords;
    return(dbReplyMsg->status);
  } else {
    if(verbose) {
      cerr << "[dbClient] ERROR: Database did not respond to get num records request!" << endl;
    }
    return(DB_DID_NOT_RESPOND_ERROR);
  }
}


int dbClient::driverRegister(char *driverName) {
  dbRequestMsg->function = DB_SM_DRIVER_REG_FUNCTION;
  strcpy(dbRequestMsg->name, driverName);

  dbClientStatus = NddsClientCallAndWait(dbClient, dbReplyMsg, dbRequestMsg,
					 DB_MIN_WAIT_TIME, DB_MAX_WAIT_TIME);
  if(dbClientStatus == NDDS_CLIENT_RECV_REPLY) {
    if(dbReplyMsg->status != DB_OK) {
      if(verbose) {
	cerr << "[dbClient] ERROR: Driver register failed! Returned " << dbReplyMsg->status << endl;
      }
    } 
    return(dbReplyMsg->status);
  } else {
    if(verbose) {
      cerr << "[dbClient] ERROR: Database did not respond to register driver request!" << endl;
    }
    return(DB_DID_NOT_RESPOND_ERROR);
  }
}


int dbClient::getDriverName(int driverNum, char *driverName) {
  dbRequestMsg->function = DB_GET_SM_DRIVER_NAME_FUNCTION;
  dbRequestMsg->driverNum = driverNum;

  dbClientStatus = NddsClientCallAndWait(dbClient, dbReplyMsg, dbRequestMsg,
					 DB_MIN_WAIT_TIME, DB_MAX_WAIT_TIME);
  if(dbClientStatus == NDDS_CLIENT_RECV_REPLY) {
    if(dbReplyMsg->status != DB_OK) {
      if(verbose) {
	cerr << "[dbClient] ERROR: Driver get name failed! Returned " << dbReplyMsg->status << endl;
      }
    } else {
      strcpy(driverName, dbReplyMsg->name);
    }
    return(dbReplyMsg->status);
  } else {
    if(verbose) {
      cerr << "[dbClient] ERROR: Database did not respond to driver get name request!" << endl;
    }
    return(DB_DID_NOT_RESPOND_ERROR);
  }
}


int dbClient::saveDB(char *filename) {
  dbRequestMsg->function = DB_SAVE_FUNCTION;
  strcpy(dbRequestMsg->name, filename);
  dbClientStatus = NddsClientCallAndWait(dbClient, dbReplyMsg, dbRequestMsg,
					 DB_MIN_WAIT_TIME, DB_MAX_WAIT_TIME);
  if(dbClientStatus == NDDS_CLIENT_RECV_REPLY) {
    if(dbReplyMsg->status != DB_OK) {
      if(verbose) {
	cerr << "[dbClient] ERROR: Save failed! Returned " << dbReplyMsg->status << endl;
      }
    }
    return(dbReplyMsg->status);
  } else {
    if(verbose) {
      cerr << "[dbClient] ERROR: Database did not respond to save request!" << endl;
    }
    return(DB_DID_NOT_RESPOND_ERROR);
  }
}


int dbClient::restoreDB(char *filename) {
  dbRequestMsg->function = DB_RESTORE_FUNCTION;
  strcpy(dbRequestMsg->name, filename);
  dbClientStatus = NddsClientCallAndWait(dbClient, dbReplyMsg, dbRequestMsg,
					 DB_MIN_WAIT_TIME, DB_MAX_WAIT_TIME);
  if(dbClientStatus == NDDS_CLIENT_RECV_REPLY) {
    if(dbReplyMsg->status != DB_OK) {
      if(verbose) {
	cerr << "[dbClient] ERROR: Restore failed! Returned " << dbReplyMsg->status << endl;
      }
    } 
    return(dbReplyMsg->status);
  } else {
    if(verbose) {
      cerr << "[dbClient] ERROR: Database did not respond to restore request!" << endl;
    }
    return(DB_DID_NOT_RESPOND_ERROR);
  }
}


int dbClient::archiveDB(char *filename) {
  dbRequestMsg->function = DB_ARCHIVE_FUNCTION;
  strcpy(dbRequestMsg->name, filename);
  dbClientStatus = NddsClientCallAndWait(dbClient, dbReplyMsg, dbRequestMsg,
					 DB_MIN_WAIT_TIME, DB_MAX_WAIT_TIME);
  if(dbClientStatus == NDDS_CLIENT_RECV_REPLY) {
    if(dbReplyMsg->status != DB_OK) {
      if(verbose) {
	cerr << "[dbClient] ERROR: Archive failed! Returned " << dbReplyMsg->status << endl;
      }
    } 
    return(dbReplyMsg->status);
  } else {
    if(verbose) {
      cerr << "[dbClient] ERROR: Database did not respond to archive request!" << endl;
    }
    return(DB_DID_NOT_RESPOND_ERROR);
  }
}


void dbClient::displayRecord(const dbRecord &record) {
  char timeStampChar[50];
  cout << "** Record #" << record.targetID << ":" << endl;
  
  strcpy(timeStampChar, ctime(&record.timeStamp));
  cout << "   Acquisition time: " << timeStampChar;
  
  cout << "   DGPS position estimate: " << record.DGPS_coord.x << ", " << record.DGPS_coord.y 
       << ", " << record.DGPS_coord.z << endl;
  
  cout << "   Class member probabilities: ";
  for(int i=0; i < NUM_CLASSES; i++) {
    cout << record.probClassMember[i] << " ";
  }
  cout << endl;
  
  cout << "   " << record.numSensors << " sensors are registered to the database" << endl;
  for(int i=0; i < MIN(record.numSensors, MAX_SENSORS); i++) {
    
    cout << "   Sensor #" << i << " data:" << endl;
    
    cout << "      Estimated info gain: " << record.sensorData[i].estInfoGain << endl;
    
    cout << "      Sensor #" << i << " has " << record.sensorData[i].numReadings << " readings" << endl;
    for(int j=0; j < MIN(record.sensorData[i].numReadings, MAX_SENSOR_READINGS); j++) {
      cout << "      Reading #" << j << ":" << endl;
      
      strcpy(timeStampChar, ctime(&record.sensorData[i].readings[j].timeStamp));
      cout << "         Reading time: " << timeStampChar;
      
      cout << "         Data filename: " << record.sensorData[i].readings[j].filename << endl;
      
      cout << "         Robot position at reading: " 
	   << record.sensorData[i].readings[j].robotPosition.x 
	   << ", " << record.sensorData[i].readings[j].robotPosition.y
	   << ", " << record.sensorData[i].readings[j].robotPosition.z << endl;
      
      cout << "         Robot pose at reading: "
	   << "r = " << record.sensorData[i].readings[j].robotPose.roll << ", "
	   << "p = " << record.sensorData[i].readings[j].robotPose.pitch << ", "
	   << "y = " << record.sensorData[i].readings[j].robotPose.yaw << endl;
      
      cout << "         Feature vector: ";
      for(int k=0; k < MAX_FEATURE_VECTOR_SIZE; k++) {
	cout << record.sensorData[i].readings[j].features.featureVector[k] << " ";
      }
      cout << endl;
      
      cout << "         Optional data: ";
      for(int k=0; k < MAX_OPTIONAL_DATA_SIZE; k++) {
	cout << record.sensorData[i].readings[j].optionalData.optionalData[k] << " ";
      }
      cout << endl;
    }
  }
}


void dbClient::runUI(void) {
  char option;
  int done = 0;
  int targetID, numRecords, status, sensorNum, readingNum, classNum;
  dbRecord tmpRec;
  dbRecordBitMask tmpRecBitMask;
  dbNotify *tmpNotify;
  char timeStampChar[50];
  int returnVal;
  char name[SM_DRIVER_MAX_ID_LENGTH];
  database db;

  int oldVerbose = verbose;
  verbose = 1;

  while(!done) {

    cout << endl;
    cout << "*** SAS Database Client ***" << endl;
    cout << "Options:" << endl;
    cout << "  (i)nsert target" << endl;
    cout << "  (g)et record" << endl;
    cout << "  (p)ut record" << endl;
    cout << "  get the (n)umber of records" << endl;
    cout << "  register sensor manager (d)river" << endl;
    cout << "  (s)ave database" << endl;
    cout << "  (r)estore database" << endl;
    cout << "  (a)rchive database" << endl;
    cout << "  (t)ransmit dbNotify" << endl;
    cout << "  (l)ist dbNotify data" << endl;
    cout << "  (q)uit UI" << endl;
    cin >> option;

    switch(option) {
      
    case 'q':
      cout << "[dbClient] **** Quitting dbClient UI! Are you sure! (y/n): ";
      cin >> option;
      if(option == 'y') {
	done = 1;
      }
      break;


    case 'l':
      option = 'y';
      while(!q->isEmpty() && option == 'y') {
	cout << "Remove queue element (y/n)? ";
	cin >> option;
	if(option == 'y' || option == 'Y') {
	  tmpNotify = q->remove();
	  if(tmpNotify->function == DB_PUT_FUNCTION) {
	    cout << "Function: PUT" << endl;
	  } else if(tmpNotify->function == DB_INSERT_FUNCTION) {
	    cout << "Function: INSERT" << endl;
	  } else {
	    cout << "Function: " << tmpNotify->function << endl;
	  }
	  displayRecord(tmpNotify->record);
	}
      }
      break;

    case 'i':
      insertRecord(targetID);
      break;
      
    case 'g':
      cout << "[dbClient] Target ID: "; 
      cin >> targetID;

      status = getRecord(targetID, tmpRec);
      if(status == DB_OK) {
	cout << endl;
	displayRecord(tmpRec);
      } else {
	cout << "[dbClient] ERROR: Get record request returned " << status << endl;;
      }
      break;

    case 'n':
      getNumRecords(numRecords);
      cout << "[dbClient] There are " << numRecords << " records in the database." << endl;
      break;

    case 'd':
      cout << "[dbClient] Please enter new sensor manager driver ID: ";
      cin >> name;
      driverRegister(name);
      break;
             
    case 's':
      cout << "[dbClient] Please enter save filename: ";
      cin >> name;
      saveDB(name);
      break;

    case 'r':
      cout << "[dbClient] Please enter restore filename: ";
      cin >> name;
      restoreDB(name);
      break;

    case 'a':
      cout << "[dbClient] Please enter archive filename: ";
      cin >> name;
      archiveDB(name);
      break;

    case 't':
      cout << "[dbClient] Send dbNotify of which target ID: ";
      cin >> targetID;

      status = getRecord(targetID, tmpRec);

      if(status == DB_OK) {
	db.createUnalteredBitMask(tmpRecBitMask);
	option = ' ';

	while(option != 'x') {
	  cout << "Altered fields:"  << endl;
	  cout << "  (1) DGPS coordinates [" << tmpRecBitMask.DGPS_coord << "]" << endl;
	  cout << "  (2) Sensor data" << endl;
	  cout << "  (3) Class member probabilities [" << tmpRecBitMask.probClassMember << "]" << endl;
	  cout << "  (x) Finished -- transmit dbNotify" << endl;
	  cin >> option;
	  
	  switch(option) {
	  case '1':
	    tmpRecBitMask.DGPS_coord = 
	      (tmpRecBitMask.DGPS_coord == DB_UNALTERED) ? DB_ALTERED : 
		DB_UNALTERED;
	    break;
	    
	  case '2':
	    sensorNum = -1;
	    while(sensorNum < 0 || sensorNum >= MAX_SENSORS) {
	      cout << "  Which sensor number (0 - " << MAX_SENSORS-1 << "): ";
	      cin >> sensorNum;
	    }
	    cout << "    (1) Estimated info gain [" << tmpRecBitMask.sensorData[sensorNum].estInfoGain << "]" << endl;
	    cout << "    (2) Reading data ";
	    for(int i=0; i < MAX_SENSOR_READINGS; i++) {
	      cout << "[" << tmpRecBitMask.sensorData[sensorNum].readings[i].filename << "]" ;
	    }
	    cout << endl;
	    cout << "    (3) Number of readings [" << tmpRecBitMask.sensorData[sensorNum].numReadings << "]" << endl;
	    cin >> option;
	    
	    switch(option) {
	    case '1':
	      tmpRecBitMask.sensorData[sensorNum].estInfoGain = (tmpRecBitMask.sensorData[sensorNum].estInfoGain == DB_UNALTERED) ? DB_ALTERED : DB_UNALTERED;
	      break;
	      
	    case '2':
	      readingNum = -1;
	      while(readingNum < 0 || readingNum >= MAX_SENSOR_READINGS) {
		cout << "    Which reading (0 - " << MAX_SENSOR_READINGS-1 << "): ";
		cin >> readingNum;
	      }
	      
	      tmpRecBitMask.sensorData[sensorNum].readings[readingNum].filename = 
		(tmpRecBitMask.sensorData[sensorNum].readings[readingNum].filename == DB_UNALTERED) ? DB_ALTERED : DB_UNALTERED;
	      tmpRecBitMask.sensorData[sensorNum].readings[readingNum].features = tmpRecBitMask.sensorData[sensorNum].readings[readingNum].filename;
	      tmpRecBitMask.sensorData[sensorNum].readings[readingNum].optionalData = tmpRecBitMask.sensorData[sensorNum].readings[readingNum].optionalData;
	      break;
	      
	    case '3':
	      tmpRecBitMask.sensorData[sensorNum].numReadings = (tmpRecBitMask.sensorData[sensorNum].numReadings == DB_UNALTERED) ? DB_ALTERED : DB_UNALTERED;
	      break;
	    }

	    break;

	  case '3':
	    tmpRecBitMask.probClassMember = (tmpRecBitMask.probClassMember == DB_UNALTERED) ? DB_ALTERED : DB_UNALTERED;
	    break;
	    
	  }
	}

	memcpy(&dbNotifySendMsg->record, &tmpRec, sizeof(dbRecord));
	memcpy(&dbNotifySendMsg->changed, &tmpRecBitMask, sizeof(dbRecordBitMask));
	dbNotifySendMsg->function = DB_PUT_FUNCTION;
	NddsPublicationSend(dbNotifyPub);

      }
      break;

    case 'p':
      cout << "[dbClient] Target ID: "; cin >> targetID;

      status = getRecord(targetID, tmpRec);

      if(status == DB_OK) {
	db.createUnalteredBitMask(tmpRecBitMask);

	cout << "Do you want to alter: " << endl;
	cout << "  (1) DGPS position estimate" << endl;
	cout << "  (2) Class member probabilities" << endl;
	cout << "  (3) Sensor data" << endl;
	cin >> option;

	switch(option) {
	case '1':
	  cout << "  DGPS [" << tmpRec.DGPS_coord.x << " " << tmpRec.DGPS_coord.y << " " << tmpRec.DGPS_coord.z << "]: ";
	  cin >> tmpRec.DGPS_coord.x >> tmpRec.DGPS_coord.y >> tmpRec.DGPS_coord.z;
	  tmpRecBitMask.DGPS_coord = DB_ALTERED;
	  break;
	case '2':
	  classNum = -1;
	  while(classNum < 0 || classNum >= NUM_CLASSES) {
	    cout << "  Alter which class' probability (0 - " << NUM_CLASSES-1 << "): ";
	    cin >> classNum;
	  }
	  cout << "  Class #" << classNum << " [" << tmpRec.probClassMember[classNum] << ": ";
	  cin >> tmpRec.probClassMember[classNum];
	  tmpRecBitMask.probClassMember = DB_ALTERED;
	  break;
	case '3':
	  sensorNum = -1;
	  while(sensorNum < 0 || sensorNum >= MAX_SENSORS) {
	    cout << "  Alter which sensor number (0 - " << MAX_SENSORS-1 << "): ";
	    cin >> sensorNum;
	  }
	  cout << "  Do you want to alter:" << endl;
	  cout << "    (1) Estimated info gain" << endl;
	  cout << "    (2) Reading data" << endl;
	  cout << "    (3) Number of readings" << endl;
	  cin >> option;
	  
	  switch(option) {
	  case '1':
	    cout << "    Estimated info gain [" << tmpRec.sensorData[sensorNum].estInfoGain << "]: ";
	    cin >> tmpRec.sensorData[sensorNum].estInfoGain;
	    tmpRecBitMask.sensorData[sensorNum].estInfoGain = DB_ALTERED;
	    break;
	  case '2':
	    readingNum = -1;
	    while(readingNum < 0 || readingNum >= MAX_SENSOR_READINGS) {
	      cout << "    Alter which reading (0 - " << MAX_SENSOR_READINGS-1 << "): ";
	      cin >> readingNum;
	    }

	    cout << "        Data filename [" << tmpRec.sensorData[sensorNum].readings[readingNum].filename << "]: ";
	    cin >> tmpRec.sensorData[sensorNum].readings[readingNum].filename;
	    tmpRecBitMask.sensorData[sensorNum].readings[readingNum].filename = DB_ALTERED;

	    cout << "        Feature vector [";
	    for(int i=0; i < MAX_FEATURE_VECTOR_SIZE; i++) {
	      cout << tmpRec.sensorData[sensorNum].readings[readingNum].features.featureVector[i] << " ";
	    }
	    cout << "]: ";
	    for(int i=0; i < MAX_FEATURE_VECTOR_SIZE; i++) {
	      cin >> tmpRec.sensorData[sensorNum].readings[readingNum].features.featureVector[i];
	    }
	    tmpRecBitMask.sensorData[sensorNum].readings[readingNum].features = DB_ALTERED;

	    cout << "        Optional data [";
	    for(int i=0; i < MAX_OPTIONAL_DATA_SIZE; i++) {
	      cout << tmpRec.sensorData[sensorNum].readings[readingNum].optionalData.optionalData[i] << " ";
	    }
	    cout << "]: ";
	    for(int i=0; i < MAX_OPTIONAL_DATA_SIZE; i++) {
	      cin >> tmpRec.sensorData[sensorNum].readings[readingNum].optionalData.optionalData[i];
	    }
	    tmpRecBitMask.sensorData[sensorNum].readings[readingNum].optionalData = DB_ALTERED;
	    break;

	  case '3':
	    cout << "        Number of readings for sensor #" << sensorNum << " [" << tmpRec.sensorData[sensorNum].numReadings << "]: ";
	    cin >> tmpRec.sensorData[sensorNum].numReadings;
	    tmpRecBitMask.sensorData[sensorNum].numReadings = DB_ALTERED;
	    break;
	  }
	}
	
	status = putRecord(tmpRec, tmpRecBitMask);

	if(status != DB_OK) {
	  cerr << "[dbClient] ERROR: Could not put record " << tmpRec.targetID << "!" << endl;
	}

      } else {
	cerr << "[dbClient] ERROR: Could not get record " << tmpRec.targetID << "!" << endl;
      }
    }
  }

  verbose = oldVerbose;
  return;
}


#ifdef DB_CLIENT_MAIN
int main(void) {
  dbClient *dbc = new dbClient();
  dbc->startNDDS();
  dbc->runUI();
  delete dbc;
  return(0);
}
#endif /* DB_CLIENT_MAIN */
