/* FILE: database.h
   CREATED: 4/22/99
   AUTHOR: Michael Wagner
   DESCRIPTION: This file defines the SAS database class. See 
     databaseMain.cpp for the hooks into NDDS. The database class contains
     a linked list of dbRecordNode objects. We'll be simply inserting new
     objects in the front, because we can assume the number of records will
     be relatively small most the time (<200). Therefore, linear searches
     will be acceptable. However, if this assumption doesn't hold, a binary
     search tree could be implemented. When a dbRecord is returned, the
     appropriate sensor's data (indexed by sensorNum) is copied into
     dbRecordNode.record.sensorData, even though the dbRecordNode has a whole
     array of sensor data. This fits in better with NDDS.
*/

#ifndef __DATABASE_H__
#define __DATABASE_H__

#include "dbRecord.h"
#include "dbMsgs.h"
#include "dbRecordDef.h"
#include "dbDef.h"

#define DB_ARCHIVE_SCRIPT_NAME "./dbArchive"

#ifndef EXEC_OK
#define EXEC_OK 0
#endif


class dbRecordNode {
 public:
  dbRecordNode(int ID, dbRecordNode *next) {
    this->record.targetID = ID;
    this->next = next;
  }
  dbRecord record;
  dbRecordNode *next;
};

class database {
 public:
  database();
  ~database();

  /* Return DB_OK on success, something else on failure. */

  int getNumRecords(int &numRecords);
  
  int smDriverRegister(char *driverID);
  int getSmDriverName(int driverNum, char *driverName);

  int insertRecord(int &ID);
  int insertRecord(int &ID, long timeStamp);
  int insertRecordMulti(int numRecords, int &firstID, int &lastID);
  int getRecord(int targetID, dbRecord &record);
  int putRecord(dbRecord &record, dbRecordBitMask &altered);
  int searchRecord(dbRecord &searchCriteria, dbRecordBitMask &searchFields, dbRecord &results);
  int searchRecordMulti(dbRecord &searchCriteria, dbRecordBitMask &searchFields, int numRecords, dbRecord &results);
  int archiveDB(char *filename); /* Archive a data packet...include a database
				    save file along with all data files 
				    referenced in sensor data. NOTE: saveDB(filename) must be called first. */
  int saveDB(char *filename);    /* Save current DB state. */
  int restoreDB(char *filename); /* Restore previously saved DB state. */

  void createUnalteredBitMask(dbRecordBitMask &mask); /* Returns a bit mask with
							 all fields DB_UNALTERED. */
  void createAlteredBitMask(dbRecordBitMask &mask); /* Returns a bit mask with
						       all fields DB_ALTERED. */

 protected:
  dbRecordNode *head;
  int numRecords;
  char *smDriverIDs[MAX_SENSORS];
  int numDrivers;

  /* This method checks to see if 'a' matches 'b', but only checks the fields specified in 'mask'. */
  int maskedRecordMatch(dbRecord &a, dbRecord &b, dbRecordBitMask &mask);			

};

#endif /* __DATABASE_H__ */

