/* FILE: database.cpp
   CREATED: 4/22/99
   AUTHOR: Michael Wagner
   DESCRIPTION: This file implements the SAS database class. See 
     databaseMain.cpp for the hooks into NDDS.
*/

#include <iostream.h>
#include <fstream.h>
#include "database.h"
#include "smDriverDef.h"

// Public database methods
database::database() {
  numRecords = -1;
  head = NULL;
  for(int i=0; i < MAX_SENSORS; i++) {
    smDriverIDs[i] = NULL;
  }
  numDrivers = 0;
}

database::~database() {
  numRecords = -1;
  delete(head);
  for(int i=0; i < MAX_SENSORS; i++) {
    delete(smDriverIDs[i]);
  }
}

int database::smDriverRegister(char *driverID) {

  if(strlen(driverID) > SM_DRIVER_MAX_ID_LENGTH) {
    return(DB_SM_DRIVER_REG_NAME_ERROR);
  }

  for(int i=0; i < MAX_SENSORS; i++) {
    if(smDriverIDs[i] == NULL) {
      smDriverIDs[i] = new char[strlen(driverID)];
      strcpy(smDriverIDs[i], driverID);
      numDrivers++;
      return(DB_OK);
    } else if(strcmp(smDriverIDs[i], driverID) == 0) {
      return(DB_SM_DRIVER_REG_DUPLICATE_ERROR); /* found a duplicate */
    } 
  }
       
  return(DB_SM_DRIVER_REG_LIMIT_ERROR); /* too many drivers already registered */
}

int database::getSmDriverName(int num, char *driverName) {
  if(num >=0 && num < MAX_SENSORS) {
    if(smDriverIDs[num] != NULL) {
      strcpy(driverName, smDriverIDs[num]);
      return(DB_OK); 
    } 
  }
  
  return(DB_GET_SM_DRIVER_NAME_ERROR);
}


int database::getNumRecords(int &num) {
  num = numRecords+1;
  return(DB_OK);
}


int database::insertRecordMulti(int numRecords, int &firstID, int &lastID) {
  if(insertRecord(firstID) == DB_INSERT_ERROR) {
    return(DB_INSERT_ERROR);
  }
  for(int i=1; i < numRecords; i++) {
    if(insertRecord(lastID) == DB_INSERT_ERROR) {
      return(DB_INSERT_ERROR);
    }
  }
  if(numRecords == 1) {
    lastID = firstID; // If it gets here, then we only inserted ONE target.
  }
  return(DB_OK);
}

int database::insertRecord(int &ID) {
  time_t ltime;
  time(&ltime);
  return(insertRecord(ID, (long)ltime));
}  

// This version is used when restoring data so time stamps can be kept.
int database::insertRecord(int &ID, long timeStamp) {
  time_t ltime;
  if(head == NULL) {
    // This is the first insertRecord.
    head = new dbRecordNode(numRecords, NULL);
    numRecords++;
    head->record.targetID = numRecords;
    // Now put time stamp in record
    head->record.timeStamp = timeStamp;
  } else {
    // Insert at the beginning.
    dbRecordNode *temp = new dbRecordNode(numRecords, head);
    if(temp != NULL) {
      head = temp;
      numRecords++;
      temp->record.targetID = numRecords; // Put new target ID in the record
      // Now put time stamp in record
      head->record.timeStamp = timeStamp;
    } else {
      return(DB_INSERT_ERROR);
    }
  }
    
  ID = numRecords;
  return(DB_OK);
}  

int database::putRecord(dbRecord &record, dbRecordBitMask &altered) {

  dbRecordNode *traverse = head;
  time_t ltime;

  while(traverse != NULL) {
    if(traverse->record.targetID == record.targetID) {
      // Found it!

      if(altered.DGPS_coord == DB_ALTERED) {
	memcpy(&traverse->record.DGPS_coord, &record.DGPS_coord, sizeof(DGPS));
      }

      for(int i=0; i < numDrivers; i++) {
	if(altered.sensorData[i].numReadings == DB_ALTERED) {
	  traverse->record.sensorData[i].numReadings = record.sensorData[i].numReadings;
	}

	if(altered.sensorData[i].estInfoGain == DB_ALTERED) {
	  traverse->record.sensorData[i].estInfoGain = record.sensorData[i].estInfoGain;
	}
	
	for(int j=0; j < record.sensorData[i].numReadings; j++) {
	  if(altered.sensorData[i].readings[j].filename == DB_ALTERED) {
	    strcpy(traverse->record.sensorData[i].readings[j].filename, record.sensorData[i].readings[j].filename);
	    // Have to update other stuff too
	    cerr << "sensor " << i  << " reading " << j << " time stamp = " << record.sensorData[i].readings[j].timeStamp << endl;
	    if(record.sensorData[i].readings[j].timeStamp == 0) {
	      time(&ltime);
	      traverse->record.sensorData[i].readings[j].timeStamp = (long)ltime;
	    } else {
	      traverse->record.sensorData[i].readings[j].timeStamp = record.sensorData[i].readings[j].timeStamp;
	    }
	    memcpy(&traverse->record.sensorData[i].readings[j].robotPosition, &record.sensorData[i].readings[j].robotPosition, sizeof(DGPS));
	    memcpy(&traverse->record.sensorData[i].readings[j].robotPose, &record.sensorData[i].readings[j].robotPose, sizeof(pose));
	  }
	  if(altered.sensorData[i].readings[j].features == DB_ALTERED) {
	    memcpy(&traverse->record.sensorData[i].readings[j].features, &record.sensorData[i].readings[j].features, sizeof(dbSensorFeatures));
	  }
	  if(altered.sensorData[i].readings[j].optionalData == DB_ALTERED) {
	    memcpy(&traverse->record.sensorData[i].readings[j].optionalData, &record.sensorData[i].readings[j].optionalData, sizeof(dbSensorOptionalData));
	  }
	}
      }

      if(altered.probClassMember == DB_ALTERED) {
	for(int i=0; i < NUM_CLASSES; i++) {
	  traverse->record.probClassMember[i] = record.probClassMember[i];
	}
      }
      return(DB_OK);
      
    } else {
      traverse = traverse->next;
    }
  }

  /* Either head == NULL, targetID was not found, or the targetID was found but all the
     data was marked as DB_UNALTERED. */
  return(DB_PUT_ERROR);
}

int database::getRecord(int targetID, dbRecord &record) {
  dbRecordNode *traverse = head;

  while(traverse != NULL) {
    if(traverse->record.targetID == targetID) {
      memcpy(&record, &traverse->record, sizeof(dbRecord));
      record.numSensors = numDrivers; // Keep it up to date.
      return(DB_OK);
    } else {
      traverse = traverse->next;
    }
  }

  // Either head == NULL or the targetID was not found.
  return(DB_INVALID_ID);
}

int database::searchRecordMulti(dbRecord &searchCriteria, dbRecordBitMask &searchFields, int numRecords, dbRecord &results) {
  // TBD: Implement this!
  return(DB_SEARCH_FAIL);
}

int database::searchRecord(dbRecord &searchCriteria, dbRecordBitMask &searchFields, dbRecord &results) {
#ifdef ZERO
  dbRecordNode *traverse = head;

  while(traverse != NULL) {
    if(maskedRecordMatch(record, traverse->record, searchFields)) {
      // We've found a match! 
      // TBD: Maybe return a dbRecordNode instead to allow a linked list of matches to be returned?
      memcpy(&results, &traverse->record, sizeof(dbRecord));
      return(DB_OK);
    } else {
      traverse = traverse->next;
    }
  }

  // Either head == NULL or no matches were found.
#endif
  return(DB_SEARCH_FAIL);
}


int database::archiveDB(char *filename) {
  char command[MAX_DATA_FILENAME_LENGTH * 2]; // NOTE: The 2 is just to be safe
  
  // Set up the main archive directory, copy the saved database file into it.
  // NOTE: THIS WILL OVERWRITE AN EXISTING DIRECTORY! 
  sprintf(command, "\\rm -r %s_archive", filename);
  system(command);
  sprintf(command, "mkdir %s_archive", filename);
  if(system(command) != EXEC_OK) {
    return(DB_ARCHIVE_ERROR);
  }
  sprintf(command, "cp %s %s_archive/.", filename, filename);
  if(system(command) != EXEC_OK) {
    return(DB_ARCHIVE_ERROR);
  }
  sprintf(command, "cp %s_XDR %s_archive/.", filename, filename);
  if(system(command) != EXEC_OK) {
    return(DB_ARCHIVE_ERROR);
  }
  
  // Set up the sensor manager driver directories
  for(int i=0; i < numDrivers; i++) {
    if(smDriverIDs[i] != NULL) {
      sprintf(command, "mkdir %s_archive/%s", filename, smDriverIDs[i]);
      if(system(command) != EXEC_OK) {
	return(DB_ARCHIVE_ERROR);
	}
    } else {
      return(DB_ARCHIVE_ERROR);
    }
  }
  
  for(dbRecordNode *traverse = head; traverse != NULL; traverse = traverse->next) {
    for(int i=0; i < numDrivers; i++) {
      for(int j=0; j < traverse->record.sensorData[i].numReadings; j++) {
	cerr << j << " " << traverse->record.sensorData[i].readings[j].filename << " " << filename << " " << smDriverIDs[i] << endl;
	sprintf(command, "cp %s %s_archive/%s/.", traverse->record.sensorData[i].readings[j].filename, filename, smDriverIDs[i]);
	if(system(command) != EXEC_OK) {
	  return(DB_ARCHIVE_ERROR);
	}
      }
    }
  }
  
  // Now create the archive file
  // TBD: Make sure it's less than a specified file size, and break it up if it is.
  sprintf(command, "tar zcvf %s_archive.tgz %s_archive", filename, filename);
  if(system(command) != EXEC_OK) {
    return(DB_ARCHIVE_ERROR);
  }
  
  // Clean up 
  sprintf(command, "\\rm -r %s_archive", filename);
  system(command);
  
  return(DB_OK);
}    

int database::saveDB(char *filename) {
  if(numDrivers == 0) {
    cerr << "[database] ERROR: Database has no registered sensor manager drivers!" << endl;
    return(DB_SAVE_ERROR);
  }
  if(head != NULL) {
    FILE *outFile = fopen(filename, "w");
    char xdrFilename[strlen(filename) + strlen("_XDR")];
    sprintf(xdrFilename, "%s%s", filename, "_XDR");
    FILE *xdrOutFile = fopen(xdrFilename, "w");
    XDR xdrStream;

    if(outFile == NULL || xdrOutFile == NULL) {
      cerr << "[database] ERROR: At least one of the output filenames are null!" << endl;
      return(DB_SAVE_ERROR);
    }

    xdrstdio_create(&xdrStream, xdrOutFile, XDR_ENCODE);

    // Output the number of records and the names of the sensor manager drivers 
    fprintf(outFile, "%d\n", numRecords+1);
    for(int i=0; i < MAX_SENSORS; i++) {
      if(smDriverIDs[i] != NULL) {
	fprintf(outFile, "%s\n", smDriverIDs[i]);
      } else {
	fprintf(outFile, "NULL\n");
      }
    }

    // Save the XDR data. Note that since records are inserted at 'head', this will
    // save them in reverse order.
    dbRecordNode *traverse = head;
    while(traverse != NULL) {
      if(!xdr_dbRecord(&xdrStream, &traverse->record)) {
	cerr << "[database] ERROR: XDR record encoding failed!" << endl;
	return(DB_SAVE_ERROR);
      }
      traverse = traverse->next;
    }

    fclose(outFile);
    fclose(xdrOutFile);
    return(DB_OK);
  } else {
    cerr << "[database] ERROR: Database empty!" << endl;
    return(DB_SAVE_ERROR);
  }
  return(DB_SAVE_ERROR);
}
    
int database::restoreDB(char *filename) {
  FILE *inFile = fopen(filename, "r");
  char xdrFilename[strlen(filename) + strlen("_XDR")];
  sprintf(xdrFilename, "%s%s", filename, "_XDR");
  FILE *xdrInFile = fopen(xdrFilename, "r");
  XDR xdrStream;

  if(inFile == NULL || xdrInFile == NULL) {
    return(DB_RESTORE_ERROR);
  }
    
  xdrstdio_create(&xdrStream, xdrInFile, XDR_DECODE);

  dbRecord tmpRecord;
  dbRecordBitMask tmpRecordBitMask;
  createAlteredBitMask(tmpRecordBitMask);
  char tmpSmDriverID[SM_DRIVER_MAX_ID_LENGTH];
  int restoreNumRecords;

  // Input the number of records and the names of the sensor manager drivers 
  fscanf(inFile, "%d", &restoreNumRecords);
  for(int i=0; i < MAX_SENSORS; i++) {
    fscanf(inFile, "%s", tmpSmDriverID);
    if(strcmp(tmpSmDriverID, "NULL") != 0) {
      if(smDriverRegister(tmpSmDriverID) == DB_SM_DRIVER_REG_LIMIT_ERROR) {
	return(DB_RESTORE_ERROR);
      }
    }
  }

  for(dbRecordNode *traverse = head; traverse != NULL; traverse = traverse->next);
  
  // Now read in XDR data
  int newID;
  for(int i=0; i < restoreNumRecords; i++) {
    if(!xdr_dbRecord(&xdrStream, &tmpRecord)) {
      return(DB_RESTORE_ERROR);
    }

    cerr << "saved target ID " << tmpRecord.targetID << " was acquired at " << tmpRecord.timeStamp << endl; 

    if(insertRecord(newID, tmpRecord.timeStamp) != DB_OK) {
      return(DB_RESTORE_ERROR);
    } 

    /* The XDR data is saved in reverse order, since records are inserted at 'head'
       Therefore restoring them by re-inserting them will reverse their order.
       So, since the new record is pointed to by 'head', let's just manually change
       its target ID */
    head->record.targetID = tmpRecord.targetID;

    if(putRecord(tmpRecord, tmpRecordBitMask) != DB_OK) {
      return(DB_RESTORE_ERROR);
    }
  }
   
  fclose(inFile);
  fclose(xdrInFile);
  return(DB_OK);
}

void database::createUnalteredBitMask(dbRecordBitMask &mask) {
  char *p = (char *)&mask;
  for(int i=0; i < sizeof(dbRecordBitMask); i++) {
    *p = DB_UNALTERED;
    p++;
  }
}

void database::createAlteredBitMask(dbRecordBitMask &mask) {
  char *p = (char *)&mask;
  for(int i=0; i < sizeof(dbRecordBitMask); i++) {
    *p = DB_ALTERED;
    p++;
  }
}

#ifdef ZERO
// Private database methods
int database::maskedRecordMatch(dbRecord &a, dbRecord &b, dbRecordBitMask &mask) {
  if(mask->DGPS_coord) {
  }

  for(int i=0; i
  if(mask->sensorData
}
#endif /* ZERO */
