/* FILE: ppm.cpp
   CREATED: 4/13/99
   AUTHOR: Michael Wagner
   DESCRIPTION: This file implements the ppm class, which is used to work
     with PPM files. Currently, only P5 and P6 format PPM files are 
     supported. 
*/

#include <iostream.h>
#include <fstream.h>
#include <stdio.h>
#include <string.h>
#include "ppm.h"

// Public functions

ppm::ppm() {
  red = blue = green = NULL;
}

ppm::~ppm() {
  delete(red);
  delete(green);
  delete(blue);
}

int ppm::read(char *filename) {
  fstream rgbIn(filename, ios::in);

  if(rgbIn.fail()) {
    return(0);
  }

  // Get the PPM header data
  rgbIn >> format;
  if(!((0 == strcmp(format, "P6")) || (0 == strcmp(format, "P5")))) {
    cerr << "[ppm] ERROR: PPM format " << format << " not supported!\n";
    return(0);
  }

  char c;
  char tmpComment[MAX_PPM_COMMENT_LENGTH];
  rgbIn.get(c); // Get the line return after the format.
  rgbIn.get(c); // Now get the first character of the next line.
  while((c == PPM_COMMENT_CHAR) &&
 	(strlen(comment) < MAX_PPM_COMMENT_LENGTH)) {
    rgbIn.getline(tmpComment, MAX_PPM_COMMENT_LENGTH, '\n');
    strcat(comment, "#");
    strcat(comment, tmpComment);
    strcat(comment, "\n");
    rgbIn.get(c);
  }
  rgbIn.seekg(-1, ios::cur); /* The last rgbIn.get(c) will have gotten one
				character too many. Here, "return" that 
				character, so it can be read into the number
				of rows. */

  rgbIn >> numCols;
  rgbIn >> numRows;
  rgbIn >> pixelDepth;

  // Get pixels
  int currPixel;

  if(0 == strcmp(format, "P6")) {
    // 24-bit color
    if(red != NULL) {
      delete(red);
    }
    red   = new unsigned char[numRows*numCols];
    if(green != NULL) {
      delete(green);
    }
    green = new unsigned char[numRows*numCols];
    if(blue != NULL) {
      delete(blue);
    }
    blue  = new unsigned char[numRows*numCols];

    rgbIn.get(red[0]); // Get some leftover junk.

    for(int i=0; i < numRows; i++) {
      for(int j=0; j < numCols; j++) {
	currPixel = i * numCols + j;
	rgbIn.get(red[currPixel]);
	rgbIn.get(green[currPixel]);
	rgbIn.get(blue[currPixel]);
      }
    }
  } else if(0 == strcmp(format, "P5")) {
    // 8-bit grayscale
    red = new unsigned char[numRows*numCols];
   
    for(int i=0; i < numRows; i++) {
      for(int j=0; j < numCols; j++) {
	currPixel = i * numCols + j;
	rgbIn.get(red[currPixel]);
      }
    }    
  }

  rgbIn.close();
  
  return(1);
}

int ppm::readMem(unsigned char *image, int numRows, int numCols, int numBytes) {
  if(numBytes != 3) {
    cerr << "[ppm] ERROR: readMem not yet implemented with numBytes != 3" << endl;
    return(0);
  } else if(image == NULL) {
    return(0);
  }

  strcpy(this->format, "P6"); // TBD: Support other formats
  this->pixelDepth = 255;

  if(numRows != this->numRows || numCols != this->numCols) {
    this->numRows = numRows;
    this->numCols = numCols;
    delete(red);
    delete(green);
    delete(blue);
    red   = new unsigned char[numRows*numCols];
    green = new unsigned char[numRows*numCols];
    blue  = new unsigned char[numRows*numCols];
  }
  
  int currPixel;
  for(int i=0; i < numRows; i++) {
    for(int j=0; j < numCols; j++) {
      currPixel = i * numCols + j;
      red[currPixel] = image[3*currPixel];
      green[currPixel] = image[3*currPixel+1];
      blue[currPixel] = image[3*currPixel+2];
    }
  }

  return(1);
}
  
int ppm::write(char *filename) {
  fstream rgbOut(filename, ios::out);
  
  // Write PPM header
  rgbOut << format << endl;
  if(strlen(comment) > 0) {
    rgbOut << comment;
  }
  rgbOut << numCols << " " << numRows << endl;
  rgbOut << pixelDepth << endl;
  
  // Write the pixels
  int currPixel;
  if(0 == strcmp(format, "P6")) {
    // 24-bit color
    for(int i=0; i < numRows; i++) {
      for(int j=0; j < numCols; j++) {
	currPixel = i * numCols + j;
	rgbOut.put(red[currPixel]);
	rgbOut.put(blue[currPixel]);
	rgbOut.put(green[currPixel]);
      }
    }
  } else if(0 == strcmp(format, "P5")) {
    // 8-bit grayscale
    for(int i=0; i < numRows; i++) {
      for(int j=0; j < numCols; j++) {
	currPixel = i * numRows + j;
	rgbOut.put(red[currPixel]);
      }
    }    
  }
  
  rgbOut.close();
  return(1);
}  

int ppm::threshhold(int minBlue, int minRed, int minGreen) {
  if(((NULL == red) || (NULL == green) || (NULL == blue) && 
      (0 == strcmp(format, "P6"))) || 
     ((NULL == red) && (0 == strcmp(format, "P5")))) {
    // Therefore, it's a color image and one of the buffers is NULL
    // or it's grayscale, and red is NULL.
    return(0);
  } else {
    int currPixel;
    if(0 == strcmp(format, "P6")) {
      for(int i=0; i < numRows; i++) {
	for(int j=0; j < numCols; j++) {
	  currPixel = i * numRows + j;
	  if((red[currPixel] < minRed) && 
	     (green[currPixel] < minGreen) &&
	     (blue[currPixel] < minBlue)) {
	    red[currPixel]   = 0;
	    green[currPixel] = 0;
	    blue[currPixel]  = 0;
	  } else {
	    red[currPixel]   = 255;
	    green[currPixel] = 255;
	    blue[currPixel]  = 255;
	  }
	}
      }
    } else if(0 == strcmp(format, "P5\n")) {
      for(int i=0; i < numRows; i++) {
	for(int j=0; j < numCols; j++) {
	  currPixel = i * numRows + j;
	  if(red[currPixel] < minRed) {
	    red[currPixel] = 0;
	  } else {
	    red[currPixel] = 255;
	  }
	}
      }
    }
    return(1);
  }
}

int ppm::threshhold(int minIntensity) {
  return(threshhold(minIntensity, minIntensity, minIntensity));
}

unsigned char *ppm::getRow(int rowNum) {
  int currPixel;
  unsigned char *row;
  if(0 == strcmp(format, "P6")) {
    // 24-bit color
    row = new unsigned char[numCols*3];
    for(int i=0; i < numCols; i++) {
      currPixel = rowNum * numRows + i;
      row[3*i]   = red[currPixel];
      row[3*i+1] = green[currPixel];
      row[3*i+2] = blue[currPixel];
    }
  } else {
    // 8-bit grayscale
    row = new unsigned char[numCols];
    for(int i=0; i < numCols; i++) {
      currPixel = rowNum * numRows + i;
      row[i] = red[currPixel];
    }
  }

  return(row);
}

unsigned char *ppm::getCol(int colNum) {
  int currPixel;
  unsigned char *col;
  if(0 == strcmp(format, "P6")) {
    // 24-bit color
    col = new unsigned char[numRows*3];
    for(int i=0; i < numRows; i=i+3) {
      currPixel = i * numRows + colNum;
      col[i]   = red[currPixel];
      col[i+1] = green[currPixel];
      col[i+2] = blue[currPixel];
    }
  } else {
    // 8-bit grayscale
    col = new unsigned char[numRows];
    for(int i=0; i < numRows; i++) {
      currPixel = i * numRows + colNum;
      col[i] = red[currPixel];
    }
  }

  return(col);
}

int ppm::getRowSize() {
  return(numCols);
}

int ppm::getColSize() {
  return(numRows);
}
