#include <iostream.h>
#include <stdio.h>
#include "ndds/NDDS.h"
#include "dbMsgs.h"
#include "dbRecord.h"
#include "dbDef.h"
#include "queue.h"
#include "ClassifierNddsComms.h"
#include "SAS_ServiceNames.h"
#include "SAS_Config.h"

RTIBool dbNotifyCallback(NDDSRecvInfo *issue) {
  dbNotify *item = (dbNotify *)issue->instance;
  queue<dbNotify *> *q = (queue<dbNotify *> *)issue->callBackRtnParam;
  dbNotify *copyNotify = (dbNotify *)calloc(1, sizeof(dbNotify));

  if(issue->status == NDDS_FRESH_DATA) {
    memcpy(copyNotify, item, sizeof(dbNotify));
    q->add(copyNotify);
  }

  return(RTI_TRUE);
}

ClassifierNddsComms::ClassifierNddsComms() {
  dbNotifyMsg = (dbNotify *)calloc(1, sizeof(dbNotify));
  dbRequestMsg = (dbRequest *)calloc(1, sizeof(dbRequest));
  dbRequestMsg->name = (char *)calloc(DB_MAX_SAVE_FILENAME_LENGTH, sizeof(char));
  dbReplyMsg = (dbReply *)calloc(1, sizeof(dbReply));
  dbReplyMsg->name = (char *)calloc(DB_MAX_SAVE_FILENAME_LENGTH, sizeof(char));
  q = new queue<dbNotify *>();
  dbNotifySub = NddsSubscriptionCreate(NDDS_SUBSCRIPTION_IMMEDIATE, DB_NOTIFY_MSG_NAME,
				       dbNotifyPublicationType, dbNotifyMsg, DB_NOTIFY_DEADLINE,
				       DB_NOTIFY_MIN_SEPARATION, dbNotifyCallback, q);
  dbClient = NddsClientCreate(DB_SERVICE, dbReplyPublicationType, dbRequestPublicationType);
}

ClassifierNddsComms::~ClassifierNddsComms() {
  free(dbNotifyMsg);
  free(dbRequestMsg->name);
  free(dbRequestMsg);
  free(dbReplyMsg->name);
  free(dbReplyMsg);
  delete(q);
}

int ClassifierNddsComms::StartNDDS() {
  cerr << "[ClassifierNddsComms] Starting up NDDS on domain " << getSAS_Config(NDDS_DOMAIN) << "...";
  if(NddsInit(getSAS_Config(NDDS_DOMAIN), NULL) != RTI_TRUE) {
    cerr << "failed!" << endl;
    return(0);
  }
  dbNotifyNddsRegister();
  dbRequestNddsRegister();
  dbReplyNddsRegister();
  cerr << "done!" << endl;

  if(NddsClientServerWait(dbClient, 2.0, 5, 1) == RTI_FALSE) {
    cerr << "[ClassifierNddsComms] ERROR: Database server cannot be found!" << endl;
    return(0);
  } 

  return(1);
}

int ClassifierNddsComms::DatabaseHasNewInfo(dbRecord &record, int &driverNum) {
  dbNotify *tmpNotify;
  if(!q->isEmpty()) {
    tmpNotify = (dbNotify *)q->remove();
    if((tmpNotify->function == DB_PUT_FUNCTION) && (tmpNotify->changed.sensorData == DB_ALTERED)) {
      memcpy(record, &tmpNotify->record, sizeof(dbRecord));
      driverNum = tmpNotify->driverNum;
      free(tmpNotify);
      return(1);
    } else {
      return(0);
    }
  } else {
    return(0);
  }
}

int ClassifierNddsComms::GetRecord(int targetID, int driverNum, dbRecord &record) {
  dbRequestMsg->function = DB_GET_FUNCTION;
  dbRequestMsg->record.targetID = targetID;
  dbRequestMsg->driverNum = driverNum;
  
  dbClientStatus = NddsClientCallAndWait(dbClient, dbReplyMsg, dbRequestMsg, 
					 DB_MIN_WAIT_TIME, DB_MAX_WAIT_TIME);
  if(dbClientStatus == NDDS_CLIENT_RECV_REPLY) {
    switch(dbReplyMsg->status) {
    case DB_OK:
      memcpy(record, &dbReplyMsg->record, sizeof(dbRecord));
      return(1);
    case DB_INVALID_ID:
      cerr << "[ClassifierNddsComms] ERROR: Database GET failed because of bad targetID or driver number!" << endl;
      return(0);
    case default:
      cerr << "[ClassifierNddsComms] ERROR: Database GET returned " << dbreplyMsg->status << endl;
      return(0);
    }
  } else {
    cerr << "[ClassifierNddsComms] ERROR: Database did not respond to GET request!" << endl;
    return(0);
  }				 
}

int ClassifierNddsComms::PutRecord(int targetID, int driverNum, const dbRecord &record) {
  dbRequestMsg->function = DB_PUT_FUNCTION;
  dbRequestMsg->record.targetID = targetID;
  dbRequestMsg->driverNum = driverNum;
  memcpy(&dbRequestMsg->record, record, sizeof(dbRecord));

  dbClientStatus = NddsClientCallAndWait(dbClient, dbReplyMsg, dbRequestMsg, 
					 DB_MIN_WAIT_TIME, DB_MAX_WAIT_TIME);
  if(dbClientStatus == NDDS_CLIENT_RECV_REPLY) {
    switch(dbReplyMsg->status) {
    case DB_OK:
      return(1);
    case DB_PUT_ERROR:
      cerr << "[ClassifierNddsComms] ERROR: Database PUT failed because of bad targetID or driver number!" << endl;
      return(0);
    case default:
      cerr << "[ClassifierNddsComms] ERROR: Database PUT returned " << dbreplyMsg->status << endl;
      return(0);
    }
  } else {
    cerr << "[ClassifierNddsComms] ERROR: Database did not respond to PUT request!" << endl;
    return(0);
  }				 
}

int ClassifierNddsComms::GetImageParameters(const dbRecord &record, int readingNum, 
					    float &originRow, float &originCol,
					    float &scaleRow, float &scaleCol /*, vector &features */) {
}

int ClassifierNddsComms::GetSpectrumParameters(const dbRecord &record, int readingNum, 
					       /*, vector &features */) {
}

int ClassifierNddsComms::GetMetalDetectorParameters(const dbRecord &record, int readingNum, 
					    /*, vector &features */) {
}

int ClassifierNddsComms::InsertFeatures(dbRecord &record, int readingNum /*, vector features */) {
}

int ClassifierNddsComms::GetReadingDataFile(const dbRecord &record, int readingNum, char &filename) {
  if((readingNum >= record->data.sensorData.numReadings) || (readingNum < 0)) {
    return(0);
  }

  char *traverse = record->data.sensorData.dataFilenames;
  for(int i=0; i < readingNum; i++) {
    if((traverse = strstr(traverse, DB_FILE_NAME_DELIMITER)) == NULL) {
      return(0);
    }
  }

  // If we've gotten this far, then traverse will be pointing at the beginning of the desired file name.
  // There will be a delimiter at the end of this file name. 
  char *end;
  if((end = strstr(traverse, DB_FILE_NAME_DELIMITER)) == NULL) {
    // Can't find the ending token -- dataFilenames must be messed up -- we already checked readingNum against the
    // number of readings! Therefore, let the user know that something weird is going on.
    cerr << "[ClassifierNddsComms] ERROR: Cannot parse reading #" << readingNum << " from dbRecord! Should be able to!" << endl;
    return(0);
  } 

  filename = new char[(end - traverse) * sizeof(char)];
  strncpy(filename, traverse, (end - traverse));
  return(1);
}


// Example usage:

int main(void) {
  ClassifierNddsComms *c = new ClassifierNddsComms();
  dbRecord record;
  int driverNum;
  char *filename;
  
  c->StartNDDS();

  while(1) {
    if(c->DatabaseHasNewInfo(&record, &driverNum)) {
      cout << "Target ID: " << record.targetID << endl;
      cout << "Sensor driver number: " << driverNum << endl;
      cout << "Number of sensor readings: " << record.sensorData.numReadings << endl;
      for(int i=0; i < record.sensorData.numReadings; i++) {
	if(c->GetReadingDataFile(record, i, filename)) {
	  cout << "Sensor reading #" << i << " filename: " << filename << endl;
	}
      }
      cout << "Optional data: ";
      for(int i=0; i < DB_MAX_RECORD_OPTIONAL_DATA; i++) {
	cout << record.sensorData.optionalData[i] << " ";
      }
      cout << endl;
    }
  }

  delete c;

  return(1);
}
