/* $Revision: 1.4 $ $Author: nomad $ $Date: 1997/05/05 18:34:54 $ */

/****************************************************************************
 *
 * PROJECT: Nomad
 *
 * MODULE:  imageSender
 *
 * FILE:    isParseArgs.cc
 *
 * DESCRIPTION: 
 *  Command line argument parsing for imageSend.cc
 *
 * EXPORTS:  
 *
 * NOTES:        Derived from imageReceiver/irParseArgs.c
 *
 * REVISION HISTORY:
 * $Log: isParseArgs.c,v $
 * Revision 1.4  1997/05/05 18:34:54  nomad
 * Arg.  Found another fix in PrintUsage.  -Kurt Schwehr
 *
 * Revision 1.3  1997/05/05 18:33:18  nomad
 * Minor fix to PrintUsage.  -Kurt Schwehr
 *
 * Revision 1.2  1997/05/05 04:05:56  nomad
 * Changed strings.h to string.h for WinNT.
 * MS Vis C++ does not have strncasecmp.  Bummer!
 *
 * Revision 1.1  1997/05/05 01:35:44  nomad
 * Initial version of isParseArgs.c:  Command line arg handler for
 * imageSend.  Derived from imageReceiver/irParseArgs.c.
 *   -Kurt Schwehr
 *
 *
 * (c) Copyright 1996 NASA. All rights reserved.
 ***************************************************************************/

/***************************************************************************
 * INCLUDES
 ***************************************************************************/
#include <assert.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <NDDS.h>

#include "nomadNddsDomain.h"	// This is local for windows NT

#include "isParseArgs.h"

/***************************************************************************
 * EXTERNS
 ***************************************************************************/

extern char* imageSendVersion;
extern char* imageSendDate;

/***************************************************************************
 * GLOBALS
 ***************************************************************************/
#ifndef TRUE
#define TRUE (1==1)
#endif
#ifndef FALSE
#define FALSE (1!=1)
#endif

/* Case insensitive str n equal */
// Case insensitive not available with Visual C++ 4.x?
//#define STRNCASEEQU(S1,S2,NUM) (0==strncasecmp(S1,S2,NUM))

#define STRNEQU(S1,S2,NUM) (0==strncmp(S1,S2,NUM))
#define STREQU(S1,S2) (0==strcmp(S1,S2))


/***************************************************************************
 *
 * FUNCTION: err_quit
 * DESCRIPTION: Print an error message and exit.
 * INPUTS:  Error message string.
 * OUTPUTS: 
 * EXCEPTIONS: 
 * DESIGN:  
 * NOTES:   
 *
 ***************************************************************************/

void
err_quit (char *msg)
{
    if (!msg) fprintf (stderr, "err_quit: error message is NULL???\n");
    else fprintf (stderr, "ERROR: %s\n",msg);
    exit (EXIT_FAILURE);
}


/***************************************************************************
 *
 * FUNCTION: InitArgValues
 * DESCRIPTION: Set the default values for all of the options.
 * INPUTS:  A pointer to a struct to fill in.
 * OUTPUTS: TRUE on success, FALSE on failure.
 * EXCEPTIONS: 
 * DESIGN:  
 * NOTES:   
 *
 ***************************************************************************/

int
InitArgValues (struct ArgValues *av)
{

    assert (av);
    av->nddsDomain = NOMAD_NDDS_DOMAIN;		/* nomadNddsDomain.h */

    av->multicastInfo.Domain = NOMAD_NDDS_MULTICAST_DOMAIN;
    av->multicastInfo.TTL = NOMAD_NDDS_MULTICAST_TTL;

    av->nddsVerb = 1;
    av->nddsErrorVerb = 1;
    av->nddsDebugVerb = 0;

    av->debug_level = 2;

    return TRUE;
}


/***************************************************************************
 *
 * FUNCTION: PrintArgs
 * DESCRIPTION: Print out the command line.
 * INPUTS:  argc, argv
 * OUTPUTS: None
 * EXCEPTIONS: 
 * DESIGN:  
 * NOTES:   
 *
 ***************************************************************************/

void
PrintArgs (int argc, char *argv[])
{
    int i;
    assert (argc > 0);
    assert (argv);
    for (i=0; i<argc; i++) {
	printf ("%s ",argv[i]);
    }

    return;
}


/***************************************************************************
 *
 * FUNCTION: PrintUsage
 * DESCRIPTION: Print out how to call the program.
 * INPUTS:  The name of the program as called in argv[0]
 * OUTPUTS: None
 * EXCEPTIONS: 
 * DESIGN:  
 * NOTES:   
 *
 ***************************************************************************/

void
PrintUsage (char *name)
{
    assert (name);

    printf ("Usage: %s [OPTION]...\n", name);
    printf ("Take images (probably compressed) and send them via ndds.\n");
    printf ("\n");

    printf ("  --domain=INT      NDDS Domain\n");
    printf ("  --TTL=INT         NDDS multicast time to live\n");
    printf ("  --mdomain=INT     NDDS multicast domain\n");
    printf ("\n");

    printf ("  --verb=INT        NDDS verbosity\n");
    printf ("  --error-verb=INT  NDDS error verbosity\n");
    printf ("  --debug-verb=INT  NDDS debug verbosity\n");
    printf ("\n");

    printf ("  --debug-level=INT non-NDDS debug level\n");
    printf ("  --version         report imageSend version\n");
    printf ("  -h, --help        produce this help message\n");

    return;
}


/***************************************************************************
 *
 * FUNCTION: CheckForHelp
 * DESCRIPTION: Check to see if the user asked for help.
 * INPUTS:  A pointer to a struct to fill in.
 * OUTPUTS: Returns FALSE if no --help, TRUE if found help request
 * EXCEPTIONS: 
 * DESIGN:  
 * NOTES:   
 *
 ***************************************************************************/

int
CheckForHelp (int argc, char *argv[])
{
    int i;

    assert (argv);
    if (1==argc) return FALSE;

    for (i=0; i<argc;i++) {
	//STRNCASEEQU
	if (STRNEQU("--help", argv[i], 6)  || STREQU("-h",argv[i])) {
	    PrintUsage(argv[0]);
	    return TRUE;	/* Found a cry for help */
	}
    }
    return FALSE;
}


/***************************************************************************
 *
 * FUNCTION: GetArgInt
 * DESCRIPTION: Extract the integer value after the "=" in an arg.
 * INPUTS:  one argv string
 * OUTPUTS: -1 on error, otherwise the int value of the arg.
 * EXCEPTIONS: 
 * DESIGN:  
 * NOTES:   
 *
 ***************************************************************************/
int
GetArgInt (const char *argv)
{
    int value=-1;
    char *str;
    assert (argv);

    str = strchr (argv, '=');
    if (NULL==str) {
	fprintf (stderr, "GetArgInt ERROR: no '=' found in argument\n");
	fprintf (stderr, "  %s\n", argv);
	return -1;
    }
    str++;
    if ('\0'==str[0]) {
	fprintf (stderr, "GetArgInt ERROR: no int found after '='\n");
	fprintf (stderr, "  %s\n", argv);
	return -1;
    }
    sscanf (str, "%d", &value);
    return value;
}



/***************************************************************************
 *
 * FUNCTION: SetArgInt
 * DESCRIPTION: Set the target to the arg if it is in range.
 * INPUTS:  Argv is one argv[] string.
 *            min/max are for range checking.
 * OUTPUTS: the value to set target to.  Out of range returns default.
 * EXCEPTIONS: 
 * DESIGN:  
 * NOTES:   
 *
 ***************************************************************************/

int
SetArgInt (int defaultVal, const char *argv, const int min, const int max)
{
    int value;
    assert (argv);

    value = GetArgInt(argv);

    if (!(min <= value && max >= value)) {
	value = defaultVal;
	fprintf (stderr,"WARNING: argument out of range.\n");
	fprintf (stderr,"  ARG> %s\n", argv);
	fprintf (stderr,"  Using default value: %d\n",value);
	fprintf (stderr,"\n");
    }

    return (value);
}


/***************************************************************************
 *
 * FUNCTION: ParseThem
 * DESCRIPTION: Step through the args and make the appropriate changes.
 * INPUTS:  argc, argv, argValues - a place to put the results.
 * OUTPUTS: argValues and an int result - TRUE on success, FALSE on failure
 * EXCEPTIONS: 
 * DESIGN:  
 * NOTES:   
 *
 ***************************************************************************/

// Not avail on NT w/ VizC++ 4.x?  STRNCASEEQU
#define MATCH_ARG(OPTION,ARGV) ( \
STRNEQU(OPTION,ARGV,strlen(OPTION)) \
)

int
ParseThem (int argc, char *argv[], struct ArgValues *val)
{
    int i;

    for (i=1; i<argc;i++)
    {
	char *a = argv[i];
	if (MATCH_ARG("-h",a) || MATCH_ARG("--help",a))  {
	    /* This should not happend.  Should have exited alread... */
	    err_quit ("ParseThem->Should not see help options here.\n");
	}

	if (MATCH_ARG("--domain",a)) {
	    val->nddsDomain = SetArgInt (val->nddsDomain, a, 2001, 64000);
	    continue;	/* for loop */
	}
	if (MATCH_ARG("--TTL",a)) {
	    val->multicastInfo.TTL = SetArgInt(val->multicastInfo.TTL,a,0,255);
	    continue;	/* for loop */
	}
	if (MATCH_ARG("--mdomain",a)) {
	    val->multicastInfo.Domain = SetArgInt(val->multicastInfo.Domain,a,
						  2001,64000);
	    continue;	/* for loop */
	}
	if (MATCH_ARG("--verb",a)) {
	    val->nddsVerb = SetArgInt(val->nddsVerb,a,-1,5);
	    continue;	/* for loop */
	}
	if (MATCH_ARG("--error-verb",a)) {
	    val->nddsErrorVerb = SetArgInt(val->nddsErrorVerb,a,-1,5);
	    continue;	/* for loop */
	}
	if (MATCH_ARG("--debug-verb",a)) {
	    val->nddsDebugVerb = SetArgInt(val->nddsDebugVerb,a,-1,5);
	    continue;	/* for loop */
	}
	if (MATCH_ARG("--debug-level",a)) {
	    val->debug_level = SetArgInt(val->debug_level,a,0,20);
	    continue;	/* for loop */
	}
	if (MATCH_ARG("--version",a)) {
	    /* From imageSend.cc */
	    printf ("imageSend: %s -- %s\n", imageSendVersion, imageSendDate);
	    continue;	/* for loop */
	}

	/* No Match... this is bad. */
	fprintf (stderr,"ERROR: bad command line argument\n");
	fprintf (stderr,"  %d arg = '%s' not recognized.\n", i, a);
	PrintArgs(argc,argv);
	fprintf (stderr,"\n");

	fflush (NULL);
	PrintUsage(argv[0]);

	exit (EXIT_FAILURE);
    }


    return TRUE;
}



/***************************************************************************
 *
 * FUNCTION: ISParseArgs
 * DESCRIPTION: Main function to call to handle command line args.
 * INPUTS:  argc, argv, argValues - a place to put the results.
 * OUTPUTS: argValues and an int result - TRUE on success, FALSE on failure
 * EXCEPTIONS: 
 * DESIGN:  
 * NOTES:   
 *
 ***************************************************************************/

int
ISParseArgs (struct ArgValues *argValues)
{
    InitArgValues (argValues);			/* Set the defaults */
//   if (1==argc) {
//	return TRUE;				/* Nothing to do.  No args. */
//    }

 //   if (TRUE == CheckForHelp(argc, argv)) {	/* See the user wants help */
//	exit (EXIT_FAILURE);
 //   }

 //   ParseThem (argc,argv, argValues);		/* Do all the work. */

    return TRUE;
}
