/* $Revision: 1.5 $ */

/****************************************************************************
 *
 * PROJECT:     Nomad
 *
 * MODULE:      NT Windows yuck (imageSender)
 *
 * FILE:        imageSend.cpp
 *
 * DESCRIPTION: Paced image sender for Windows NT.  Uses a Multimedia timer
 *  and a separate thread to send pkts using a semaphore to sync to the timer.
 *
 * EXPORTS: 
 *
 * NOTES:   Needs image compression code added into the main loop.
 *
 *
 * REVISION HISTORY:
 *
 * $Log: imageSend.cpp,v $
 * Revision 1.5  1997/05/05 18:47:11  nomad
 * I think I've gotten rid of the extra packet bug.  -Kurt Schwehr
 *
 * Revision 1.4  1997/05/05 04:11:11  nomad
 * MAJOR CLEANUP.
 * Added command line argument parsing.
 * Stopped MS Vis C++ from name mangling imageSendVersion & imageSendDate
 * Added lots of NddsDebug statements to help with debugging on nomad.
 * Fixed it so the main loop runs until "Running" is set FALSE.
 *     Still no mechanism to change the state of Running.
 *  -Kurt Schwehr
 *
 * Revision 1.3  1997/04/25 01:35:22  nomad
 * Lots of cleanup of old dead code.
 * Works with ndds 1.11d.
 * Added some debugging code.
 *  -Kurt Schwehr
 *
 * Revision 1.2  1997/04/07 19:11:55  hjt
 * Added multicasting capablity under NDDS 1.11c. Works.
 *
 * Revision 1.3  1997/04/07 18:07:42  schwehr
 * Packets now have y2 set to the number of packets in the image.
 *
 * Revision 1.2  1997/04/03 20:26:21  schwehr
 * Patched for NDDS 1.11b multicast.  Not tested yet on NT, but will
 * compile on IRIX.
 *
 * Revision 1.1.1.1  1997/04/02 00:35:43  schwehr
 * Initial version of the code for the CVS tree.  Some functionality is
 * still missing.  Code needs cleanup.  Runs only on Windows NT.
 *
 * Initial Version by Kurt Schwehr
 *
 *
 ***************************************************************************/

/***************************************************************************
 * INCLUDES
 ***************************************************************************/

#include <assert.h>

#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include <memory.h>		// memcpy()

#include <NDDS.h>
#include "ImagePacket.h"	// nddsgen file + tweaks to fix problems.

#ifdef WIN32
#  include <windows.h>
#endif

#include "fifo.h"		// Dan Christians queue package
#include "imageSend.h"
#include "isParseArgs.h"	// Command line arguments
#include "globals.h"

/**********************************************************************/
// Macros and typedefs
/**********************************************************************/


#define NDDSDomain int

struct tmpPkt {
    int imgNum;
    int pktNum;
    int pktTot;		// Total number of packets in the image.
    int numData;
	int typeflag;
	int timestamp[2];
    char data[IMG_PKT_BLOCK_SIZE];
};

boolean firstTime=TRUE;

/**********************************************************************/
/* Debugging levels... the higher the # the more output you get. */

#define ALWAYS 0	/* Hint: if you use always, it won't work without -g */
#define TERSE 1
#define TRACE 4
#define VERBOSE 8
#define BOMBASTIC 20	/* Major spew... when life sucks. */

#ifdef NDEBUG
#  define NddsDebug(n,s)		/* expand to nothing */
#else
   int debug_level = TERSE;
#  define NddsDebug(n,s)  (((n) <= debug_level) ? printf s :0)
#endif

/**********************************************************************/
// Function Prototypes
/**********************************************************************/

void StartTimer (void);
void StopTimer (void);
//void printTimeCaps (void);
void PASCAL TimerCallback (UINT wTimerID, UINT msg,
			   DWORD dwUser, DWORD dw1, DWORD dw2);
//int StartNdds (void);
void SetupNddsLocal (void);

void SendBuf (u_char *buf, int length, int typeflag, long *timestamp);
void printPkt (ImagePacket p);
void PrepareNddsSend();

int CalcPacketDelay (const int bps);

DWORD ThreadTransmit (LPDWORD lpdwParam);
void PanoError(int flag, char *message);

/***************************************************************************
 * GLOBALS
 ***************************************************************************/

/* MS Visual C++ mangles variable names too... Grrr! */
extern "C" {
  char *imageSendVersion = "$Revision: 1.5 $";
  char *imageSendDate = "$Date: 1997/05/05 18:47:11 $";
} /* extern "C" */

ImagePacket pkt;
NDDSProducer pktProducer;

HANDLE hMutex;
HANDLE hSendSem;	// Send Semaphore... give this in the timer.

// Hopefully...
static int Running=TRUE;	// Set to FALSE and everything shuts down.
FIFO_ID sendQueue = NULL;

int period=2;	// timer period in milli sec

MMRESULT timerID;

// ************************************************************
// Timer code to pace transmittions.
// ************************************************************

#ifndef NDEBUG
// Print out the timer's capabilities.
void
PrintTimeCaps (void) {
    TIMECAPS tc;
    if ( TIMERR_NOCANDO == timeGetDevCaps (&tc, sizeof(tc)) ) {
	printf ("timeGetDevCaps: ERROR -> no can do!\n");
	exit (EXIT_FAILURE);
    }
    printf ("timeGetDevCaps:\n");
    printf ("  wPeriodMin = %d\n",tc.wPeriodMin);
    printf ("  wPeriodMax = %d\n",tc.wPeriodMax);
    return;
}
#endif

// Fast stats in timer loop.
#if 0
static int tCount=0;
static int tDeltaTotal=0;
static int tDevTotal = 0;
#endif

// This was taken from the docs... I didn't come up with the PASCAL thing
void
PASCAL TimerCallback (UINT wTimerID, UINT msg,
		      DWORD dwUser, DWORD dw1, DWORD dw2)
{
#if 0
    // Code to monitor timing accuracy.
    static boolean started = FALSE;
    static DWORD tickLast;
    DWORD tickCur;
    tickCur = GetTickCount();
    printf ("wTID=%u, msg=%u, dwU=%d, dw1=%d, dw2=%d\n",
	    wTimerID, msg, dwUser, dw1, dw2
	    );
    tCount++;
    if (started) {
	tDeltaTotal += tickCur-tickLast;
	tDevTotal = abs( DELAY - abs(tickCur-tickLast));
	//printf ("  %d\n", tickCur - tickLast);
        //printf (" tick diff = %d ms\n", tickCur - tickLast);
    } else
	started = TRUE;

    tickLast = tickCur;
#endif

    // Signal the Transmit Thread
    ReleaseSemaphore (hSendSem, 1, NULL);

    return;
}

// Setup all that we need for timing.
void
StartTimer (struct ArgValues *args) {
    NddsDebug (TRACE,("StartTimer.\n"));

#ifndef NDEBUG    
    if (VERBOSE <= debug_level) PrintTimeCaps ();
#endif

    if (TIMERR_NOCANDO == timeBeginPeriod (period)) {
	fprintf (stderr,"timeBeginPeriod: ERROR, period %d ms out of range\n",
		period);
	exit (EXIT_FAILURE);
    }

    // Place the delay between packets in uDelay.
	if (!ndds_send_msecdelay)
    ndds_send_msecdelay = CalcPacketDelay (ndds_send_bps);
    UINT uResolution=1;	// timer res ms, 0=="greatest pos acc"
    DWORD dwUser=1;		// Callback user data.
    UINT fuEvent = TIME_PERIODIC;
	//NddsDebug (TERSE,("Timer Params:\n"));
   //NdddsDebug (TERSE,("  bps: %d\n",args->bps));
    //NddsDebug (TERSE,("  uDelay: %d\n",(int)uDelay));
    //NddsDebug (TERSE,("  uResolution: %d\n",(int)uResolution));
    printf("Params:\n");
    printf("  bps: %d\n",ndds_send_bps);
    printf("  uDelay: %d\n",(int)ndds_send_msecdelay);
    printf("  uResolution: %d\n",(int)uResolution);
#ifdef WIN32
    if (0 == (timerID=timeSetEvent(ndds_send_msecdelay, uResolution,
				   (LPTIMECALLBACK)TimerCallback,
				   dwUser, fuEvent))
	)
#endif
    {
	fprintf (stderr,"ERROR: could not setup periodic timer event.\n");
	fprintf (stderr,"  errVal = %d\n", timerID);
	exit (EXIT_FAILURE);
    }
    NddsDebug (VERBOSE,("  Timer ID = %d\n",timerID));
    NddsDebug (TRACE,("StartTimer.  DONE\n"));

    return;
}

// Clean up after the timer.
void
StopTimer (void) {

#if 0
    // Performance Summary
    printf ("Timer summary:\n");
    printf ("  tCount   = %d\n",tCount);
    printf ("  tDetaTot = %d\n",tDeltaTotal);
    printf ("  tDetaTot = %d\n",tDevTotal);
    printf ("  ave ticks btwn = %f\n", (float)tDeltaTotal/(float)tCount);
    printf ("  ave ticks dev  = %f\n", (float)tDevTotal/(float)tCount);
#endif

    timeKillEvent(timerID);
    timeEndPeriod (period);	// Docs say must match each begin w/ an end

    return;
}

// ************************************************************
// NDDS setup code
// ************************************************************

int
StartNdds (struct ArgValues *args) {

    NddsDebug (TRACE, ("StartNdds\n"));

    /* Safety check.  I've been bitten by this too many times.
	{ 
	char *s;
	s = getenv ("NDDS_PEER_HOSTS");
	NddsDebug (TERSE,("***\n"));
	printf ("***     getenv(NDDS_PEER_HOSTS): %s\n", s);
	NddsDebug (TERSE,("***\n"));
	NddsDebug (TERSE,("***\n"));
	NddsDebug (VERBOSE,("Watch out for NDDS_PEER_HOST_LIST files!!!\n"));
    }
	*/

    NddsDebug (TERSE, ("NddsInit: domain=%d, mcast domain=%d, TTL=%d\n",
		   args->nddsDomain, args->multicastInfo.Domain,
		   args->multicastInfo.TTL));

    NddsInit(args->nddsDomain, &args->multicastInfo);

    NddsVerbositySet      (args->nddsVerb);
    NddsVerbosityErrorSet (args->nddsErrorVerb);
    NddsVerbosityDebugSet (args->nddsDebugVerb);

	NddsPeerHostsSet("146.155.14.241:172.20.3.129:172.20.1.123:192.204.240.206:172.20.1.200:128.102.113.196:128.102.113.197:128.102.113.198:128.102.113.199");
#ifndef NDEBUG
    if (TRACE <= debug_level) {
	printf ("NddsVerbositySet = %d\n",      NddsVerbositySet(-1));
	printf ("NddsErrorVerbositySet = %d\n", NddsVerbosityErrorSet(-1));
	printf ("NddsDebugVerbositySet = %d\n", NddsVerbosityDebugSet(-1));
    }
#endif

    NddsDebug (TRACE, ("StartNdds DONE.  Connected to NDDS\n"));

    return OK;
}


void
printPkt (ImagePacket p) {
    printf ("  P: rl=%d, pn=%d, l=%d, Id=%d, (%d,%d)-(%d,%d)\n",
	    p->row_len, p->packet_num,
	    p->line_data.line_data_len,
	    p->headerId,
	    p->x1, p->y1, p->x2, p->y2);
    return;
}


/*
 * Input: max bits per second on the connection
 *  	bps = 10000000;	=> 10BaseT or 10Base2 Ethernet
 * Output: milliseconds delay that should be between packets
 */

int
CalcPacketDelay (const int bps) {
    const int bitsIP = 8*20;
    const int bitsUDP = 8*10;
    const int bitsNDDS = 8*20;		/* WAG */
    const int bitsImgOverhead = 8*8;	/* XDR other than image */
    const int bitsData = 8 * IMG_PKT_SIZE;
    /* Bits per packet */
    const int bpp = bitsIP+bitsUDP+bitsNDDS+bitsImgOverhead+bitsData;

    const int bpms = bps / 1000; /* Bits per milli second */

    /* ms to wait for packet to go out. */
    /* The 2 is a WAG.. I was loosing lots of pkts */
    const int msec = 2 * (int)(1 + (ceil)((float )bpp / (float)bpms));

    NddsDebug(BOMBASTIC,("CalcPacketDelay info:\n"));
    NddsDebug(BOMBASTIC,("  bitsIP: %d\n",bitsIP));
    NddsDebug(BOMBASTIC,("  bitsUDP: %d\n",bitsUDP));
    NddsDebug(BOMBASTIC,("  bitsNDDS: %d\n",bitsNDDS));
    NddsDebug(BOMBASTIC,("  bitsImgOverhead: %d\n",bitsImgOverhead));
    NddsDebug(BOMBASTIC,("  bitsData: %d\n",bitsData));
    NddsDebug(BOMBASTIC,("  bpp: %d\n",bpp));
    NddsDebug(BOMBASTIC,("  bpms: %d\n",bpms));
    NddsDebug(BOMBASTIC,("  msec: %d\n",msec));

    NddsDebug(TRACE,(" %d bps => %d ms delay btwn NDDS samples\n", bps, msec));
    return msec;
}

void
SendBuf (u_char *buf, int length, int typeflag, long *timestamp)
{
    int i;
    static int imgCnt = 0;	// How many images have we sent?
    static int pktCnt = 0;	// How many packets have been sent? 
    static struct tmpPkt p;
	boolean fullerror=FALSE;
    int numPkts = length / IMG_PKT_SIZE;
    
	if ( 0 != length % IMG_PKT_SIZE ) {
	numPkts++;	// There is a partial packet.
    }

	PanoImageNumPkts=numPkts;
	

    if (imgCnt%10==0) printf("SendBuf:  len=%d, imgCnt=%d, pktCnt=%d, numPkts=%d\n",
	    length, imgCnt, pktCnt, numPkts);
    assert (buf);
    assert (length > 0);
    assert (numPkts > 0);

    p.imgNum = imgCnt++;
    p.numData = IMG_PKT_SIZE;
    p.pktTot = numPkts;
	p.typeflag = typeflag;
	p.timestamp[0]= timestamp[0];
	p.timestamp[1]= timestamp[1];
    
    for (i = 0; i < numPkts-1; i++)
	{
		const int offset = i * IMG_PKT_SIZE;
		assert (offset >= 0);
		assert (offset < length);

		p.pktNum = pktCnt++;
	
		NddsDebug(BOMBASTIC,("SendBuf: i= %d  offset= %d\n", i, offset));

		void *dataPtr = (void*)memcpy (&p.data,&buf[offset],IMG_PKT_SIZE);
		assert (dataPtr);

RetryFifoSend:
		if (-1==FifoSend (sendQueue, (char*)&p, sizeof (struct tmpPkt)))
		{
			fullerror=TRUE;
			Sleep(50);
			goto RetryFifoSend;
		}
	}

    // Handle last packet... probably not full.
    const int offset = i * IMG_PKT_SIZE;
    p.pktNum = pktCnt++;
    p.numData = (length % IMG_PKT_SIZE);
    if (0 == p.numData) p.numData = IMG_PKT_SIZE;

    NddsDebug(BOMBASTIC,("SendBuf: Last i= %d  offset= %d\n", i, offset));

    void *dataPtr = (void*)memcpy (&p.data, &buf[offset], p.numData);
    assert (dataPtr);								 
RetryLastSend:
    if (-1==FifoSend (sendQueue, (char*)&p, sizeof (struct tmpPkt)))
	{
		fullerror=TRUE;
		Sleep(50);
		goto RetryLastSend;
	}
	if (fullerror)
	{
		PanoError(PE_ERROR,"Fifo Was full.");
//		AcqWait=ndds_send_msecdelay*numPkts;
//		if (PanoRun) SetEvent(AcqColorEvent[ACQCOLOR_BLOCKGET]);
	}
		
	return;
}


// Debugging code to just read a file to ship.
int
LoadFile (u_char *buf, char *fileName) {
    FILE *fp;
    int count=0;
    assert (buf);
    assert (fileName);

    NddsDebug (TRACE,("Opening file: '%s'\n",fileName));

    fp = fopen (fileName,"rb");  assert (fp);

    count = fread (buf, sizeof(char), MAX_FILE_SIZE, fp);

    if ( feof  (fp) ) NddsDebug (VERBOSE,("feof... ok!\n"));
    if ( ferror(fp) ) {
	perror("Could not read data from file");
	exit (EXIT_FAILURE);
    }
    assert (count <= MAX_FILE_SIZE);
    assert (count > 0);

    NddsDebug (TRACE,("loadFile: returning %d bytes read from file\n",count));
    fclose (fp);
    return count;
}


void
SetupNddsLocal (void) {
    const float per = 15.0f;
    const float str = 1.0f;

    NddsDebug (TRACE,("SetupNddsLocal\n"));
    NddsDebug (VERBOSE,(" per = %.1f, str = %.1f\n",per,str));

    ImagePacketNddsRegister();    /* Register types */
    /* Allocate space for NDDS types */
    pkt = ImagePacketAllocate ( (const char *)NULL,
				(const char *)NULL, 
				(void *)IMG_PKT_SIZE
			      );
    /* Allocate producers and consumers */
    pktProducer = NddsProducerCreate ("imgPktProducer",
				      NDDS_SYNCHRONOUS, 
				      per, str);
    /* Setup messages to send out */
    NddsProducerProductionAdd (pktProducer, "ImagePacket", "imgPacket", pkt, 
			       0, NULL, NULL, NULL, NULL);

    NddsDebug(TRACE,("SetupNddsLocal: Sleep 1 sec while Ndds sets up\n"));
#ifndef NDEBUG
    fflush(stdout);
#endif
    Sleep(1000);	/* Give up CPU, so NDDS can setup */

#ifndef NDEBUG
    if (debug_level >= BOMBASTIC)
	NddsDBaseListAll();		/* Useful when life sucks. */
#endif

    return;
}


DWORD
ThreadTransmit (LPDWORD lpdwParam)
{
    DWORD event;

    NddsDebug (TRACE,("Starting sender thread.\n"));

    /* negative value indicates not a normal image. */
    /* Will only check x1 */
    pkt->x1 = pkt->x2 = pkt->y1 = pkt->y2 = -1;

    while (Running) {
	event = WaitForSingleObject (hSendSem, 100000);

	if (-1 == event) {
	    fprintf (stderr,"ERROR: Wait on semaphore returned error.\n");
	    exit (EXIT_FAILURE);
	}

	switch (event) {
	case WAIT_ABANDONED: printf ("Thread: WAIT_ABANDONED\n"); break;
	case WAIT_OBJECT_0:  /*printf ("Thread: WAIT_OBJECT_0\n");*/ break;
	case WAIT_TIMEOUT:   printf ("Thread: WAIT_TIMEOUT\n"); break;
	default:
	    //printf ("Thread: Don't know what the event was: %d\n",event);
	    break;
	}

	// If it timed out, something is wrong.
	if (WAIT_TIMEOUT == event) continue;

	// Avoid copying data, so peek at the buffer.
	struct tmpPkt *chunk;
	int len = FifoPeek (sendQueue, (char **)&chunk);
	if (len > 0) {

	    pkt->headerId = chunk->imgNum;
	    pkt->packet_num = chunk->pktNum;
	    pkt->line_data.line_data_len = chunk->numData;
	    pkt->line_data.line_data_val = chunk->data;

	    pkt->y2 = chunk->pktTot;	// Total # of packets in image.

	    NddsDebug (BOMBASTIC,(" %d\n", chunk->pktNum));
#ifndef NDEBUG
	    if (BOMBASTIC<=debug_level) printPkt (pkt);
#endif
	    NddsProducerSample (pktProducer);
	    FifoReceive (sendQueue, NULL, IMG_PKT_BLOCK_SIZE); //free entry
	} else {
	    // No new data to send
	}
    }
    NddsDebug (TERSE,("Exiting Thread Sender.\n"));

    return 0;
}

void PrepareNddsSend()
{
	DWORD dwThreadId, dwThreadParam = 1;
    HANDLE hThread;

    struct ArgValues argValues;
    ISParseArgs (&argValues);

#ifndef NDEBUG
    debug_level = argValues.debug_level;
    NddsDebug(TERSE,("debug_level = %d\n", debug_level));
#endif

    NddsDebug (TRACE,("\n"));
    if (!sendQueue) {
	sendQueue = FifoCreate (NUM_CACHE_PACKETS, IMG_PKT_BLOCK_SIZE);
	if (!sendQueue) {
	    fprintf (stderr,"ERROR: sendQueue create failed.\n");
	    exit (EXIT_FAILURE);
	}
    } else {
	fprintf (stderr, "ERROR: sendQueue already allocated... what?\n");
	exit (EXIT_FAILURE);
    }

    // Give this semaphore in the timer interupt callback to
    // let it send one NDDS packet.
    if (NULL == (hSendSem = CreateSemaphore (NULL, 0, 1, "SendSem")) ) {
	fprintf (stderr, "ERROR: Could not create Semaphore.\n");
	exit (EXIT_FAILURE);
    }

    StartNdds(&argValues);	// Generic NDDS setup
    if (firstTime) { SetupNddsLocal();firstTime=FALSE;}		// Stuff just for this program.
    NddsDebug(TRACE,("Main: Sleep 1 sec while Ndds sets up\n"));
#ifndef NDEBUG
    fflush(stdout);
#endif
    Sleep (1000);

#ifdef WIN32
    NddsDebug (TRACE,("Calling CreateThread\n"));
    hThread = CreateThread ( NULL, 0,
	(LPTHREAD_START_ROUTINE) ThreadTransmit,
	&dwThreadParam,	0, &dwThreadId
	);
#else
    hThread = -1;	// Throw a random bad value in when not on NT.
#endif
    if (NULL == hThread ) {
	fprintf (stderr,"CreateThread error\n");
	exit (EXIT_FAILURE);
    }

    Sleep (100);
    StartTimer(&argValues);

    NddsDebug(TRACE,("Main: Sleep 1 sec while timer gets set up\n"));
#ifndef NDEBUG
    fflush(stdout);
#endif
    Sleep (1000);
	return;
}