// This a simple FIFO implementation that can be safely called by a
// single reader and a single writer asyncronously
// It handles entries of variable sizes
// All memory for the FIFO is allocated on creation
// This is mostly compatible with VxWorks MsgQ library


#include <stdio.h>			// printf()
#include <stdlib.h>			// malloc()
#include <memory.h>			// memcpy()

#include "fifo.h"

#ifndef MIN
#define MIN(x,y)	((x) < (y) ? (x) : (y))
#endif

typedef struct _msg_entry {
    unsigned int	len;
    char	*data;
} msg_entry;

struct _fifo {
    int	maxMsgs;
    unsigned int	maxMsgLen;
    msg_entry	*head;			// insert point
    msg_entry	*tail;			// removal point
    msg_entry	*store;			// data storeage
    msg_entry	*end;			// past the end of store
    char	*dataStore;		// the whole data store as one block
};
    
int 	FifoLibInit (void)
{
    return 0;
}

FIFO_ID FifoCreate (int maxMsgs, int maxMsgLength)
{
    FIFO_ID	m;
    m = (FIFO_ID)malloc (sizeof (struct _fifo));
    if (!m) return NULL;
    m->maxMsgs = maxMsgs;
    m->maxMsgLen = maxMsgLength;
    m->store = (msg_entry *)calloc (m->maxMsgs, sizeof (msg_entry));
    if (!m->store) {
	free (m);
	printf ("Failed to create %d x %d FIFO structure\n",
		m->maxMsgs, sizeof (msg_entry));
	return NULL;
    }
    m->dataStore = (char *)calloc (m->maxMsgs, m->maxMsgLen);
    if (!m->store) {
	free (m->store);
	free (m);
	printf ("Failed to create %d x %d FIFO store\n",
		m->maxMsgs, m->maxMsgLen);
	return NULL;
    }
    m->end = m->store + m->maxMsgs;
    m->head = m->tail = m->store;
    for (char *dp = m->dataStore;	// initialize storage area
	 m->head < m->end;
	 ++m->head, dp+=m->maxMsgLen) {
	m->head->len = 0;
	m->head->data = dp;
    }
    m->head = m->tail = m->store;
    return m;
}

int 	FifoDelete (FIFO_ID m)
{
    if (!m) return ERROR;
    free (m->dataStore);
    free (m->store);
    free (m);
	m=NULL;
    return 0;
}

/* Insert into queue */
int 	FifoSend (FIFO_ID m, char *buffer, unsigned int nBytes)
{
    if (!m) return ERROR;
    if (m->head->len) return ERROR;	// full!
    m->head->len = MIN (nBytes, m->maxMsgLen); // mark as full
    memcpy (m->head->data, buffer, m->head->len);
    ++m->head;				// advance FIFO
    if (m->head >= m->end) m->head = m->store;
    return OK;
}

// Get the size and pointer to the top message without removing it
int 	FifoPeek (FIFO_ID m, char **ptr)
{
    if (!m) return ERROR;
    if (!m->tail->len) return ERROR;	// empty
    if (ptr) *ptr = m->tail->data;	// peek at the data
    return m->tail->len;
}

// buffer may be NULL to save the copy
int 	FifoReceive (FIFO_ID m, char *buffer,
			     unsigned int maxNBytes)
{
    if (!m) return ERROR;
    if (!m->tail->len) return ERROR;	// empty
    int len = MIN (maxNBytes, m->tail->len);
    if (buffer)
	memcpy (buffer, m->tail->data, len);
    m->tail->len = 0;			// mark as empty
    ++m->tail;				// advance FIFO
    if (m->tail >= m->end) m->tail = m->store;
    return len;
}

int 	FifoNumMsgs (FIFO_ID m)
{
    if (!m) return ERROR;
    if (m->tail->len) return 1;		// 1 or more in queue
    return 0;				// empty
}

