#include "globals.h"
#include "R2Api.h"

// --------------------------
void PrepareAcqColor();
void InitAcqColorThread();
void AcqColorThread();
void StartStreamAcq();
void StopStreamAcq();
void EndAcqColorThread();
BFBOOL HardBoardReset();
BFBOOL OpenRdRnBoard();
void ResetExposure(int value);
void Colorize(int bufnum);
void LowerExposure(double maxMax);
void RaiseExposure(int maxAvg);
void SaveRawToFile(PBFU16 rawImage);
void StreamRawToFile(PBFU16 rawImage);
void FillTimestamp0();
void SetTimestamp();
void QuickImageGrab();
void PanoError(int flag, char *message);

// --------------------------

RdRn hBoard;
BFU32 timeout;
int newtimestamp[2];
R2SIGNAL AcqColorSignal;
int shiftamount;

// --------------------------
void PrepareAcqColor()
{
	R2RC rv;
	HardBoardReset();
	OpenRdRnBoard();

	// fill global timeout varaible from camera config file (reset once things are running)
	timeout=0;
	R2BrdInquire(hBoard,R2CamInqAqTimeout,&timeout);
	timeout=timeout/2;
	// adjust the exposure initially
	R2CTabFill(hBoard,1,0x8000-1,R2VCTabVCon2,0xffff);
	ResetExposure(RdRnExposure);
	R2ConSwTrig(hBoard,R2TrigA);
	rv=R2SignalNextWait(hBoard,&AcqColorSignal,timeout);
	if (rv!=R2_OK) PanoError(PE_ERROR,"RdRn: problems getting first image");

	return;
}

void InitAcqColorThread()
{
	DWORD AcqColorThreadId;
	shiftamount=4;
	
	hAcqColorThread = CreateThread( 
        NULL,						// no security attributes        
        0,							// use default stack size        
        (LPTHREAD_START_ROUTINE) AcqColorThread, // thread function       
        NULL,						// argument to thread function   
        0,							// use default creation flags    
        &AcqColorThreadId);			// returns the thread identifier 
	return;
}

void AcqColorThread()
{
	R2RC rv;
	DWORD evnt;
	BFBOOL ImageDone;
		// create the events
	PanoError(PE_LOG,"Starting AcqColorThread.\n");
	AcqColorEvent[ACQCOLOR_END]=CreateEvent (NULL,FALSE, FALSE, "ExitAcqColorThread");
	AcqColorEvent[ACQCOLOR_BLOCKGET]=CreateEvent (NULL,FALSE, FALSE, "AcqColorBlockGet");
	AcqColorEvent[ACQCOLOR_GETIMAGE]=CreateEvent (NULL,FALSE, FALSE, "AcqColorGetImage");
	AcqColorEvent[ACQCOLOR_TOCOMP0]=CreateEvent (NULL,FALSE, FALSE, "AcqColorToComp0");
	AcqColorEvent[ACQCOLOR_TOCOMP1]=CreateEvent (NULL,FALSE, FALSE, "AcqColorToComp1");
	AcqColorEvent[ACQCOLOR_EXPOSE]=CreateEvent (NULL,FALSE, FALSE, "AcqColorExpose");
	ImageDone=TRUE;
	
	// begin the main thread loop communicating with the compression threads
	while (1)	
	{	
		evnt = WaitForMultipleObjects(6,AcqColorEvent, FALSE, INFINITE)-WAIT_OBJECT_0;
		switch (evnt)
		{
		case ACQCOLOR_END: return;

		case ACQCOLOR_GETIMAGE:
			R2ConSwTrig(hBoard,R2TrigA);
			rv=R2SignalNextWait(hBoard,&AcqColorSignal,timeout);
			if (rv == R2_SIGNAL_TIMEOUT) 
			{
				//We assume that we have a problem here
				R2AqCleanUp(hBoard);
				R2BrdClose(hBoard);
				Sleep(50);
				PrepareAcqColor();
				R2ConSwTrig(hBoard,R2TrigA);
				rv=R2SignalNextWait(hBoard,&AcqColorSignal,timeout);
				PanoError(PE_ERROR,"Image timeout, restarted.\n");
			}
			SetTimestamp();
			Sleep(UMIN(ndds_send_msecdelay*PanoImageNumPkts,20000));
			ImageDone=FALSE;
			break;
		case ACQCOLOR_BLOCKGET:
			Sleep(AcqWait);
			AcqWait=0;
			break;
		case ACQCOLOR_EXPOSE:
			ResetExposure(RdRnExposure);
			break;

		case ACQCOLOR_TOCOMP0:
			if (!PanoRun) break;
			if (ImageDone) {SetEvent(AcqColorEvent[ACQCOLOR_TOCOMP0]);break;}
		
			Colorize(0);
		
			LastTimestamp[0][0]=newtimestamp[0];
			LastTimestamp[0][1]=newtimestamp[1];
			
			SetEvent(AcqColorEvent[ACQCOLOR_GETIMAGE]);
			
			ImageDone=TRUE;
			if (OldRGBColor) SetEvent(Comp0Event[COMPRESS_OLD]);
			else SetEvent(Comp0Event[COMPRESS_GO]);
		
			break;

		case ACQCOLOR_TOCOMP1:
			if (!PanoRun) break;
			if (ImageDone) {SetEvent(AcqColorEvent[ACQCOLOR_TOCOMP1]);break;}
			Colorize(1);
			LastTimestamp[1][0]=newtimestamp[0];
			LastTimestamp[1][1]=newtimestamp[1];
			SetEvent(AcqColorEvent[ACQCOLOR_GETIMAGE]);
			ImageDone=TRUE;
			if (OldRGBColor) SetEvent(Comp1Event[COMPRESS_OLD]);
			else SetEvent(Comp1Event[COMPRESS_GO]);
			break;
		}
	}
	return;
}

void FillTimestamp0()
{
	LastTimestamp[0][0]=newtimestamp[0];
	LastTimestamp[0][1]=newtimestamp[1];
	return;
}

void SetTimestamp()
{
	struct _timeb tstruct; 
	_ftime( &tstruct );
	newtimestamp[0]=tstruct.time;
	newtimestamp[1]=tstruct.millitm;
	return;
}

void QuickImageGrab()
{
	R2ConSwTrig(hBoard,R2TrigA);
	R2SignalNextWait(hBoard,&AcqColorSignal,timeout);
return;
}

void StartStreamAcq()
{
	PanoRun=TRUE;
	SetEvent(AcqColorEvent[ACQCOLOR_GETIMAGE]);
	SetEvent(AcqColorEvent[ACQCOLOR_TOCOMP0]);
	SetEvent(AcqColorEvent[ACQCOLOR_TOCOMP1]);
	return;
}

void StopStreamAcq()
{
	PanoRun=FALSE;
	Sleep(5000);
}

void EndAcqColorThread()
{
	DWORD returnValue;
	SetEvent (AcqColorEvent[0]);
	returnValue = WaitForSingleObject(hAcqColorThread, INFINITE)-WAIT_OBJECT_0;
	if (returnValue)
	{
		PanoError,(PE_ERROR,"Wait for thread to complete failed");
	}
	PanoError(PE_LOG,"AcqColorThread ended");

	// clean up the RdRn board
	if (!R2SignalCancel(hBoard,&AcqColorSignal)) R2SignalFree(hBoard,&AcqColorSignal);
	R2AqCommand(hBoard,R2ConFreeze,R2ConWait,R2QTabBank0);
	R2AqCleanUp(hBoard);
	R2BrdClose(hBoard);
	return;
}

BFBOOL HardBoardReset()
{
	R2ENTRY Rentry;
	RdRn RhBoard;
	BFU32 ICount,ECount;

	R2SysBoardFindByNum(0,&Rentry);
     
	if (R2BrdOpen(&Rentry, &RhBoard, R2SysNoCameraOpen|R2SysNoIntThread|R2SysNoAlreadyOpenMess))
	{
		PanoError(PE_ERROR,"Could not open board.");
		return FALSE;
	}

	R2_OpenStatus(RhBoard,R2SysResetOpenCounts,&ICount,&ECount);

	R2BrdClose(RhBoard);
	return TRUE;
}

BFBOOL OpenRdRnBoard()
{
	R2ENTRY entry;
	BFU32 error;
	
	if (R2SysBoardFindByNum(0,&entry))
	{
		PanoError(PE_ERROR,"RdRn: Couldn't find board 0");
		return FALSE;
	}
	
	if (error=(R2BrdOpen(&entry, &hBoard, R2SysInitialize)))
	{
		PanoError(PE_ERROR,"RdRn: Couldn't open board");
	    return FALSE;
	}


	// 12 bit lut has only two lanes, program both the same
	if (R2LutRamp(hBoard,R2Lut12Bit,R2LutBank0,(R2LutLane0|R2LutLane1),0,4095,0,4095)!=R2_OK)
	{
		PanoError(PE_ERROR,"RdRn: Error in R2LutRamp");
		return FALSE;
	}

	
	if (R2AqSetup(hBoard,(PBFVOID)RawImageBuf, RAWIMSIZE,-2048,R2DMADataMem,
			R2LutBank0,R2Lut12Bit,R2QTabBank0,TRUE)!= R2_OK)
	{
		PanoError(PE_ERROR,"RdRn: AqSetup failed");
		return FALSE;
	}

	R2SignalCreate(hBoard,R2IntTypeQuadDone,&AcqColorSignal);
 
  	return TRUE;
}

void Colorize(int bufnum)
{
	int avgRed,avgBlue,avgGreen,maxAvg,maxRed,maxBlue,maxGreen;
	double maxMax;
	int thresh,i;

	for (thresh=64,i=0;i<1014;i++)
	{
		if (RawImageBuf[1024*255+i]<thresh &&
			RawImageBuf[1024*255+i+3]<thresh &&
			RawImageBuf[1024*255+i+5]<thresh)
		{

			if (i==0) break;
			PanoError(PE_ERROR,"Image was Split, Correcting...");
			R2AqCleanUp(hBoard);
			R2BrdClose(hBoard);
			Sleep(50);
			PrepareAcqColor();

		break;
		}
	}
	

	//This section of the code is designed to actually
	//add in a white-balance feature so that we are
	//always actually using a true white base, rather
	//than a precompiled algorithm. We used four places
	//640,460  360,460  320,660  680,660 as selected points
	int redbal, rgrbal,bgrbal, blubal,maxbal;
	redbal=(RawImageBuf[640+IMAGEWIDTH*640]+
			RawImageBuf[360+IMAGEWIDTH*460]+2+
			RawImageBuf[320+IMAGEWIDTH*660]+
			RawImageBuf[680+IMAGEWIDTH*660])/4;

	rgrbal=(RawImageBuf[640+IMAGEWIDTH*641]+
			RawImageBuf[360+IMAGEWIDTH*461]+2+
			RawImageBuf[320+IMAGEWIDTH*661]+
			RawImageBuf[680+IMAGEWIDTH*661])/4;

	bgrbal=(RawImageBuf[641+IMAGEWIDTH*640]+
			RawImageBuf[361+IMAGEWIDTH*460]+2+
			RawImageBuf[321+IMAGEWIDTH*660]+
			RawImageBuf[681+IMAGEWIDTH*660])/4;
	
	blubal=(RawImageBuf[641+IMAGEWIDTH*641]+
			RawImageBuf[361+IMAGEWIDTH*461]+2+
			RawImageBuf[321+IMAGEWIDTH*661]+
			RawImageBuf[681+IMAGEWIDTH*661])/4;
	maxbal=UMAX(UMAX(rgrbal,bgrbal),UMAX(blubal,redbal));

//	printf("balances: r=%d,rg=%d,bg=%d,b=%d\n",redbal,rgrbal,bgrbal,blubal);
//	printf("m/r:%f,m/rg:%f,m/bg:%f,m/b:%f\n",
//		maxbal/(float)redbal,maxbal/(float)rgrbal,maxbal/(float)bgrbal,maxbal/(float)blubal);

if (OldRGBColor)
	{
	int j,k;
	maxRed=maxBlue=maxGreen=avgRed=avgGreen=avgBlue=0;
	maxMax=0.0;
	BFU16 red,rgreen,bgreen,blue,realgreen;
	PBFU8 rgbImage=RGBOldBufs[bufnum];
	for (k=0,i=0;i<IMAGESIZE;i+=2*IMAGEWIDTH)
	{
	for (j=0;j<IMAGEWIDTH;)
		{
			// map in the rgb components from the weird filter
			// Each four pixels is of the form:
			//			rG	B	  upsidedown-    R   bG		
			//			R	bG			         rG  B
			// So, we convert this into one RGB triple
			// Note that order matters here due to the inc's:
		
			//we would like to color balance here, but we can 
		    //cause some pixels to supersaturate.
			rgreen = (BFU16)((int)(RawImageBuf[(i+IMAGEWIDTH)+j  ]));//*maxbal/(float)rgrbal));
			red	   = (BFU16)((int)(RawImageBuf[ i +j++]));//*maxbal/(float)redbal));				
			blue   = (BFU16)((int)(RawImageBuf[(i+IMAGEWIDTH)+j]));//*maxbal/(float)blubal));
			realgreen = bgreen   = (BFU16)((int)(RawImageBuf[ i +j++]));//*maxbal/(float)bgrbal));
		
			
			//To color, we selectively choose rgreen or bgreen as appropriate.
			// we have blue color, so use the red green
			if (((red<blue) && abs(blue-bgreen)>abs(red-rgreen)) ||
			// we have yellow color, so use the red green
			(abs(blue-bgreen)>abs(red-rgreen)) && (blue<red)) realgreen=rgreen;
			
			avgRed+=red;
			maxRed+=(red>4050)?1:0;
			avgGreen+=realgreen;
			maxGreen+=(realgreen>4050)?1:0;
			avgBlue+=blue;
			maxBlue+=(blue>>4050)?1:0;
	
			RGBOldBufs[bufnum][k++]=(BFU8)(red>>shiftamount);
			RGBOldBufs[bufnum][k++]=(BFU8)(realgreen>>shiftamount);
			RGBOldBufs[bufnum][k++]=(BFU8)(blue>>shiftamount);
		}	
	}
		maxAvg=avgBlue>avgGreen?avgBlue:avgGreen;
		maxAvg=(maxAvg>avgRed?maxAvg:avgRed)/(RGBIMWIDTH*RGBIMWIDTH);
		maxMax=maxBlue>maxGreen?maxBlue:maxGreen;
		maxMax=(maxMax>maxRed?maxMax:maxRed)/(RGBIMWIDTH*RGBIMWIDTH);
		if (shiftamount>0 && maxMax<=saturationThresh && RdRnExposure==EXPOSURE_MAXIMUM)
		{
			PanoError(PE_LOG,"ShiftAmount reduced");
			shiftamount=UMAX(0,shiftamount-1);
		}
		if (shiftamount<4 && maxAvg>avgpixelThresh && RdRnExposure==EXPOSURE_MINIMUM)
		{
			PanoError(PE_LOG,"ShiftAmount increased");
			shiftamount=UMIN(4,shiftamount+1);
		}
		if (bufnum==0) printf("buf:%d,maxMax:%lf,maxAvg:%d,exposure:0x%X\n",
				bufnum,maxMax,maxAvg,RdRnExposure);
		//SaveRawToFile(RawImageBuf);
		if (maxMax>saturationThresh && maxAvg>avgpixelThresh) LowerExposure(maxMax);
		else
			if (maxAvg<=avgpixelThresh && maxMax<=saturationThresh) RaiseExposure(maxAvg);
		
		
	}
	else
	{
		int j,k,tmp;
		maxRed=maxBlue=maxGreen=avgRed=avgGreen=avgBlue=0;
		maxMax=0.0;
		for (k=0,tmp=3*bufnum,i=0;i<IMAGESIZE;i+=2*IMAGEWIDTH)
			for (j=0;j<IMAGEWIDTH;j+=2,k+=2)
			{
			//fill into three separate 12 bit rgb buffers in little endian ordering for compression
					
				avgRed+=RawImageBuf[i+j];
				RGBImageBufs[tmp][k]=RawImageBuf[i+j] & 0xff;  //red
				RGBImageBufs[tmp][k+1]=(RawImageBuf[i+j++]>>8) & 0x0f;
				maxRed+=(RGBImageBufs[tmp][k+1]>>2)&0x03;

				avgGreen+=RawImageBuf[i+j];
				RGBImageBufs[tmp+1][k]=RawImageBuf[i+j] & 0xff;  //bluegreen
				RGBImageBufs[tmp+1][k+1]=(RawImageBuf[i+j]>>8)&0x0f;
				maxGreen+=(RGBImageBufs[tmp+1][k+1]>>2)&0x03;

				avgBlue+=RawImageBuf[i+IMAGEWIDTH+j];
				RGBImageBufs[tmp+2][k]=RawImageBuf[i+IMAGEWIDTH+j] &0xff; //blue
				RGBImageBufs[tmp+2][k+1]=(RawImageBuf[i+IMAGEWIDTH+j]>>8) &0x0f;
				maxBlue+=(RGBImageBufs[tmp+2][k+1]>>2)&0x03;
			}
		maxAvg=avgBlue>avgGreen?avgBlue:avgGreen;
		maxAvg=(maxAvg>avgRed?maxAvg:avgRed)>>18;
		maxMax=maxBlue>maxGreen?maxBlue:maxGreen;
		maxMax=(maxMax>maxRed?maxMax:maxRed)/(RGBIMWIDTH*RGBIMWIDTH*3);
		//printf("maxMax:%lf,maxAvg:%d,exposure:%d\n",maxMax,maxAvg,RdRnExposure);
		if (maxMax>saturationThresh) LowerExposure(maxMax);
		else if (maxAvg<avgpixelThresh) RaiseExposure(maxAvg);
	
	}
	return;
}

void LowerExposure(double maxMax)
{
	if (RdRnExposure==EXPOSURE_MINIMUM) return;
	int adjuster=(int)((maxMax-0.8*saturationThresh)*100);
	//printf("adjuster:%d.\n",adjuster);
	RdRnExposure-=UMAX(adjuster,adjustAmount);
	RdRnExposure=UMAX(EXPOSURE_MINIMUM,RdRnExposure);
	//PanoError(PE_LOG,"Exposure Lowered");
	ResetExposure(RdRnExposure);
	return;
}

void RaiseExposure(int maxAvg)
{
	if (RdRnExposure==EXPOSURE_MAXIMUM) return;
	int adjuster=12*(avgpixelThresh-maxAvg)/avgpixelThresh;
	//printf("adjuster:%d.\n",adjuster);
	RdRnExposure+=UMAX(adjuster,adjustAmount);
	RdRnExposure=UMIN(RdRnExposure,EXPOSURE_MAXIMUM);
	//PanoError(PE_LOG,"Exposure Raised");
	ResetExposure(RdRnExposure);
	return;
}

void ResetExposure(int value)
	{
		R2AqCommand(hBoard,R2ConAbort,R2ConWait,R2QTabBank0);
		R2CTabFill(hBoard,1,0x7ff-1,R2VCTabVCon2,0x0000);
		R2CTabFill(hBoard,1,value-1,R2VCTabVCon2,0xffff);
		R2AqCommand(hBoard,R2ConReset,R2ConWait,R2QTabBank0);
		R2AqCommand(hBoard,R2ConFreeze,R2ConWait,R2QTabBank0);
		Sleep(150);
			//This next commands are needed for the initialization to work...
			//if it is removed here it must be added to PrepareAcqColor
		R2AqCommand(hBoard,R2ConGrab,R2ConAsync,R2QTabBank0);
		R2ConSwTrig(hBoard,R2TrigA);
		R2SignalNextWait(hBoard,&AcqColorSignal,timeout);
	
		return;
	}

void SaveRawToFile(PBFU16 rawImage)
{
	FILE* fp;

	PanoError(PE_LOG,"Saving Raw File");
	fp=fopen("c:\\pano\\RawImage.pgm","wb");
	fprintf(fp,"P5\n1024 1024\n255\n");
	for (int i=0;i<1024*1024;i++)
	{
		putc((char)(rawImage[i]>>4),fp);
	}
	fclose(fp);

	return;
}
void StreamRawToFile(PBFU16 rawImage)
{
	FILE* fp;
	char filename[80];
	time_t ltime;
	struct _timeb tstruct;
    int tval;
	time( &ltime );
	PanoError(PE_LOG,"streaming raw to file");
	_ftime( &tstruct );
	tval = (ltime%1000000)*1000+tstruct.millitm;
	strcpy(filename,"C:\\panobin\\");
	strcat(filename,"PanStr");
	_itoa(tval,&filename[strlen(filename)],10);
	strcat(filename,".pgm");
	fp=fopen(filename,"wb");
	fprintf(fp,"P5\n1024 1024\n255\n");
	for (int i=0;i<1024*1024;i++)
			putc((char)(rawImage[i]>>4),fp);
	fclose(fp);
	return;
}