#include <stdio.h>
#include <stdlib.h>
#include <tcl.h>
#include <tk.h>
#include "ndds/NDDS.h"
#include "telemetryMsgs.h"
#include "posDerivedState.h"
#include "teleopMsgs.h"
#include "dbMsgs.h"
#include "nddsMsgNames.h"
#include "dbDef.h"
#include "SAS_Config.h"
#include "SAS_ServiceNames.h"

#define TCL_SCRIPT_FILE "/usr/local/pxc/pxc200/pxc_ui.tcl"
#define HI_RES_DRIVER_NUM 0 /* The driver number of smHiResDriver -- MAKE SURE THIS MATCHES! */

posDerivedState pos;
hiResDerivedState hiRes;
NDDSClient hiResClient;
NDDSClient armClient;
NDDSClient dbClient;

RTIBool posDerivedStateCallback(NDDSRecvInfo *issue) {
  posDerivedState *item = (posDerivedState *)issue->instance;

  if (issue->status == NDDS_FRESH_DATA) {
    memcpy(&pos, item, sizeof(posDerivedState));
    /*
    param->currPosition->x = item->posWest / -100.0;
    param->currPosition->y = item->posNorth / 100.0;
    param->currPosition->z = item->posUp / 100.0;
    param->currPose->pitch = item->rotX / 100.0;
    param->currPose->roll  = item->rotY / 100.0;
    param->currPose->yaw   = 2 * PI - item->rotZ / 100.0;
    */
  }

  return(RTI_TRUE);
}

RTIBool hiResDerivedStateCallback(NDDSRecvInfo *issue) {
  hiResDerivedState *item = (hiResDerivedState *)issue->instance;

  if (issue->status == NDDS_FRESH_DATA) {
    memcpy(&hiRes, item, sizeof(hiResDerivedState));
  }

  return(RTI_TRUE);
}

/* Returns a string "x y z pitch roll yaw". x is in -cm; y, z in cm; pitch, roll, yaw in 100ths of a radian */
int TclCmd_getPosDerivedState(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[]) {
  sprintf(interp->result, "%d %d %d %d %d %d", pos.posWest, pos.posNorth, pos.posUp, pos.rotX, pos.rotY, pos.rotZ);
  return TCL_OK;
}

/* Returns a string "pan tilt FOV focallength". pan, tilt, FOV in radians; focallength in mm */
int TclCmd_getHiResDerivedState(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[]) {
  sprintf(interp->result, "%f %f %f %f", hiRes.pan, hiRes.tilt, hiRes.FOV, hiRes.focalLength);
  return TCL_OK;
}

/* Takes arguments as "sendHiResRequest function pan tilt FOV" (see teleopMsgsDef.h for function description), 
   returns "status filename" (see sensorDef.h for status number meanings) */
int TclCmd_sendHiResRequest(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[]) {
  NDDSClientReplyStatus stat;
  hiResRequest *hiResRequestMsg = (hiResRequest *)calloc(1, sizeof(hiResRequest));
  hiResReply *hiResReplyMsg = (hiResReply *)calloc(1, sizeof(hiResReply));

  if(argc != 5) {
    interp->result = "Usage: sendHiResRequest function pan tilt FOV";
    return TCL_ERROR;
  }

  hiResReplyMsg->filename = (char *)calloc(SENSOR_MAX_FILENAME_LENGTH, sizeof(char));

  hiResRequestMsg->function = atoi(argv[1]);
  hiResRequestMsg->pan = atof(argv[2]);
  hiResRequestMsg->tilt = atof(argv[3]);
  hiResRequestMsg->FOV = atof(argv[4]);

  if(NddsClientServerWait(hiResClient, 2.0, 5, 1) == RTI_FALSE) { 
    interp->result = "ERROR: high-resolution camera sensor driver server missing!";
    return TCL_ERROR;
  }

  stat = NddsClientCallAndWait(hiResClient,
			       hiResReplyMsg, hiResRequestMsg,
			       0.0f, 1200.0f);
  if(stat != NDDS_CLIENT_RECV_REPLY) {
    interp->result = "ERROR: high-resolution camera sensor driver did not respond!";
    return TCL_ERROR;
  }
  
  sprintf(interp->result, "%d %s", hiResReplyMsg->status, hiResReplyMsg->filename);
  return TCL_OK;
} 

/* Takes arguments as "insertDbRecord DGPS_X DGPS_Y DGPS_Z" returns the targetID just inserted. */
int TclCmd_insertDbRecord(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[]) {
  NDDSClientReplyStatus stat;
  dbRequest *dbRequestMsg = (dbRequest *)calloc(1, sizeof(dbRequest));
  dbReply *dbReplyMsg = (dbReply *)calloc(1, sizeof(dbReply));

  if(argc != 4) {
    interp->result = "Usage: insertDbRecord DGPS_X DGPS_Y DGPS_Z";
    return TCL_ERROR;
  }

  if(NddsClientServerWait(dbClient, 2.0, 5, 1) == RTI_FALSE) { 
    interp->result = "ERROR: database server missing!";
    return TCL_ERROR;
  }

  dbRequestMsg->function = DB_INSERT_FUNCTION;

  stat = NddsClientCallAndWait(dbClient, dbReplyMsg, dbRequestMsg,
			       0.1f, 10.0f);
  if(stat != NDDS_CLIENT_RECV_REPLY) {
    interp->result = "ERROR: database did not respond!";
    return TCL_ERROR;
  } else if(dbReplyMsg->status != DB_OK) {
    interp->result = "ERROR: DB_INSERT_FUNCTION failed!";
    return TCL_ERROR;
  } else {
    sprintf(interp->result, "%d", dbReplyMsg->record.targetID);
  }
  
  dbRequestMsg->function = DB_PUT_FUNCTION;
  dbRequestMsg->record.targetID = dbReplyMsg->record.targetID;
  dbRequestMsg->record.DGPS_coord.x = atof(argv[1]);
  dbRequestMsg->record.DGPS_coord.y = atof(argv[2]);
  dbRequestMsg->record.DGPS_coord.z = atof(argv[3]);
  dbRequestMsg->recordBitMask.DGPS_coord = DB_ALTERED;

  stat = NddsClientCallAndWait(dbClient, dbReplyMsg, dbRequestMsg,
			       0.1f, 10.0f);
  if(stat != NDDS_CLIENT_RECV_REPLY) {
    interp->result = "ERROR: database did not respond!";
    return TCL_ERROR;
  } else if(dbReplyMsg->status != DB_OK) {
    interp->result = "ERROR: DB_PUT_FUNCTION failed!";
    return TCL_ERROR;
  }

  return TCL_OK;
} 

/* Takes arguments as "putHiResDbImage targetID fname" returns "success" on success. */
int TclCmd_putHiResDbImage(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[]) {
  NDDSClientReplyStatus stat;
  dbRequest *dbRequestMsg = (dbRequest *)calloc(1, sizeof(dbRequest));
  dbReply *dbReplyMsg = (dbReply *)calloc(1, sizeof(dbReply));
  int targetID, readingNum;

  if(argc != 3) {
    interp->result = "Usage: putHiResDbImage targetID filename";
    return TCL_ERROR;
  }

  targetID = atoi(argv[1]);

  if(NddsClientServerWait(dbClient, 2.0, 5, 1) == RTI_FALSE) { 
    interp->result = "ERROR: database server missing!";
    return TCL_ERROR;
  }

  dbRequestMsg->function = DB_GET_FUNCTION;
  dbRequestMsg->record.targetID = targetID;

  stat = NddsClientCallAndWait(dbClient, dbReplyMsg, dbRequestMsg,
			       0.1f, 10.0f);
  if(stat != NDDS_CLIENT_RECV_REPLY) {
    interp->result = "ERROR: database did not respond!";
    return TCL_ERROR;
  } else if(dbReplyMsg->status != DB_OK) {
    interp->result = "ERROR: Failed to get database record (bad target ID?)";
    return TCL_ERROR;
  }

  dbRequestMsg->function = DB_PUT_FUNCTION;
  dbRequestMsg->record.targetID = targetID;
  dbRequestMsg->record.sensorData[HI_RES_DRIVER_NUM].numReadings = dbReplyMsg->record.sensorData[HI_RES_DRIVER_NUM].numReadings+1;
  readingNum = dbRequestMsg->record.sensorData[HI_RES_DRIVER_NUM].numReadings - 1;
  strcpy(dbRequestMsg->record.sensorData[HI_RES_DRIVER_NUM].readings[readingNum].filename, argv[2]);
  dbRequestMsg->recordBitMask.sensorData[HI_RES_DRIVER_NUM].numReadings = DB_ALTERED;
  dbRequestMsg->recordBitMask.sensorData[HI_RES_DRIVER_NUM].readings[readingNum].filename = DB_ALTERED;

  stat = NddsClientCallAndWait(dbClient, dbReplyMsg, dbRequestMsg,
			       0.1f, 10.0f);
  if(stat != NDDS_CLIENT_RECV_REPLY) {
    interp->result = "ERROR: database did not respond!";
    return TCL_ERROR;
  } else if(dbReplyMsg->status != DB_OK) {
    interp->result = "ERROR: Failed DB_PUT_FUNCTION!";
    return TCL_ERROR;
  }

  strcpy(interp->result, "success\0");
  return TCL_OK;
} 

/* Takes arguments as "moveArm row col" returns "success" on success. */
int TclCmd_moveArm(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[]) {
  NDDSClientReplyStatus stat;
  armRequest *armRequestMsg = (armRequest *)calloc(1, sizeof(armRequest));
  armReply *armReplyMsg = (armReply *)calloc(1, sizeof(armReply));

  int row, col;

  if(argc != 3) {
    interp->result = "Usage: moveArm row col";
    return TCL_ERROR;
  }

  row = atoi(argv[1]);
  col = atoi(argv[2]);

  if(NddsClientServerWait(armClient, 2.0, 5, 1) == RTI_FALSE) { 
    interp->result = "ERROR: arm sensor driver missing!";
    return TCL_ERROR;
  }

  armRequestMsg->function = TELEOP_ARM_MOVE;
  armRequestMsg->row = row;
  armRequestMsg->col = col;

  stat = NddsClientCallAndWait(armClient, armReplyMsg, armRequestMsg,
			       0.1f, 10.0f);
  if(stat != NDDS_CLIENT_RECV_REPLY) {
    interp->result = "ERROR: arm sensor driver did not respond!";
    return TCL_ERROR;
  } else if(armReplyMsg->status != DB_OK) {
    interp->result = "ERROR: failed TELEOP_ARM_MOVE!";
    return TCL_ERROR;
  }

  strcpy(interp->result, "success\0");
  return TCL_OK;
} 

int main(int argc, char *argv[]) {

  NDDSSubscription posDerivedStateSub;
  posDerivedState *posDerivedStateMsg = NULL;
  NDDSSubscription hiResDerivedStateSub;
  hiResDerivedState *hiResDerivedStateMsg = NULL;
  
  //  if(argc != 3) {
  //    fprintf(stderr, "Usage: pxc_ui device nddsDomain\n");
  //    return(-1);
  //  }

  printf("[pxc_ui] Starting up NDDS telemetry subscribers on domain 0...");
  //  NddsInit(atoi(argv[1]), NULL);
  NddsInit(0, NULL);
  posDerivedStateNddsRegister();
  posDerivedStateMsg = (posDerivedState *)calloc(1, sizeof(posDerivedState));
  posDerivedStateSub = NddsSubscriptionCreate(NDDS_SUBSCRIPTION_IMMEDIATE,
					      POS_DERIVED_STATE_MSG_NAME, posDerivedStatePublicationType,
					      posDerivedStateMsg, 10.0f,
					      0.0f, posDerivedStateCallback, NULL);
  hiResDerivedStateNddsRegister();
  hiResDerivedStateMsg = (hiResDerivedState *)calloc(1, sizeof(hiResDerivedState));
  hiResDerivedStateSub = NddsSubscriptionCreate(NDDS_SUBSCRIPTION_IMMEDIATE,
						HI_RES_DERIVED_STATE_MSG_NAME, hiResDerivedStatePublicationType,
						hiResDerivedStateMsg, 10.0f,
						0.0f, hiResDerivedStateCallback, NULL);
  printf("done!\n");

  printf("[pxc_ui] Starting up NDDS clients on domain 0...");
  hiResRequestNddsRegister();
  hiResReplyNddsRegister();
  hiResClient = NddsClientCreate(TELEOP_HI_RES_SERVICE, hiResReplyPublicationType, hiResRequestPublicationType);
  armRequestNddsRegister();
  armReplyNddsRegister();
  armClient = NddsClientCreate(TELEOP_ARM_SERVICE, armReplyPublicationType, armRequestPublicationType);
  dbRequestNddsRegister();
  dbReplyNddsRegister();
  dbClient = NddsClientCreate(DB_SERVICE, dbReplyPublicationType, dbRequestPublicationType);
  printf("done!\n");

  //  Tcl_argv[0] = (char *)calloc(strlen(TCL_SCRIPT_FILE) + 1, sizeof(char));
  //  strcpy(Tcl_argv[0], TCL_SCRIPT_FILE);
  //  strcat(Tcl_argv[0], "\0");

  //  Tcl_argv[1] = (char *)calloc(strlen(argv[1]) + 1, sizeof(char));
  //  strcpy(Tcl_argv[1], argv[1]);
  //  strcat(Tcl_argv[1], "\0");

  Tk_Main(argc, argv, (Tcl_AppInitProc *)Tcl_AppInit); /* Should never return */

  //  free(Tcl_argv);

  return(0);
}

Tcl_AppInit(Tcl_Interp *interp) {
  /*
   * Initialize package
   * Tcl_Init sets up the Tcl library facility.
   */
  if(Tcl_Init(interp) == TCL_ERROR) {
    return(TCL_ERROR);
  }

  if (Tk_Init(interp) == TCL_ERROR) {
    printf ("Tk_Init failed!\n");
    return TCL_ERROR;
  }

  Tcl_StaticPackage(interp, "Tk", Tk_Init, (Tcl_PackageInitProc *) NULL);

  /*
   * Register application-specific commands.
   */
  Tcl_CreateCommand(interp, "getPosDerivedState", TclCmd_getPosDerivedState,
		    (ClientData)NULL, (Tcl_CmdDeleteProc *)NULL);
  Tcl_CreateCommand(interp, "getHiResDerivedState", TclCmd_getHiResDerivedState,
		    (ClientData)NULL, (Tcl_CmdDeleteProc *)NULL);
  Tcl_CreateCommand(interp, "sendHiResRequest", TclCmd_sendHiResRequest,
		    (ClientData)NULL, (Tcl_CmdDeleteProc *)NULL);
  Tcl_CreateCommand(interp, "insertDbRecord", TclCmd_insertDbRecord,
		    (ClientData)NULL, (Tcl_CmdDeleteProc *)NULL);
  Tcl_CreateCommand(interp, "putHiResDbImage", TclCmd_putHiResDbImage,
		    (ClientData)NULL, (Tcl_CmdDeleteProc *)NULL);
  Tcl_CreateCommand(interp, "moveArm", TclCmd_moveArm,
		    (ClientData)NULL, (Tcl_CmdDeleteProc *)NULL);

  return TCL_OK;
}


