/*
 * pxc_control.c -- a silly tool to send ioctl to the raw device
 *
 * Copyright (C) 1998,1999   rubini@prosa.it (Alessandro Rubini)
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <string.h>
#include <sys/ioctl.h>

#include "pxc200.h"

char *prgname;
char *devname = "/dev/pxc0ctl";

struct command {
    char *name;             /* symbolic name for cmdline interaction */
    char *macro;            /* actual name */
    unsigned int cmd;       /* ioctl number */
    unsigned int nargs;     /* number of arguments (numeric) */
    int (*fun)(int, int, unsigned long *, char *);
};

/* ---------------- functions in charge of dealing with the command */

int do_long(int fd, int cmd, unsigned long *args, char *outptr)
{
    int retval = ioctl(fd,cmd,args);

    if (retval < 0) {
	return retval;
    }

    if (_IOC_DIR(cmd) & _IOC_READ)
	    sprintf(outptr, "\"0x%08lx (%li)\" ",args[0],args[0]);
    return retval;
}

char pagebuffer[PAGE_SIZE];

int do_page(int fd, int cmd, unsigned long *args, char *outptr)
{
    int i, j;
    int retval = ioctl(fd,cmd,pagebuffer);

    if (retval < 0) {
        return retval;
    }

    for (j = PAGE_SIZE -1; j; j--) /* discard trailing zeros */
	if (pagebuffer[j])
	    break;
    for (i=0; i<j; i+=4)
	fprintf(stderr,"0x%04x: 0x%08x\n",i,*(int *)(pagebuffer+i));
    return retval;
}

/* ------------------------------------------------------ the table */
struct command commands[] = {
    {"reset",      "PX_IOCRESET",     PX_IOCRESET,     0, NULL},
    {"hardreset",  "PX_IOCHARDRESET", PX_IOCHARDRESET, 0, NULL},
    {"getflags",   "PX_IOCGFLAGS",    PX_IOCGFLAGS,    0, do_long},
    {"setflags",   "PX_IOCSFLAGS",    PX_IOCSFLAGS,    1, do_long},
    {"getrefv",    "PX_IOCGREFV",     PX_IOCGREFV,     0, do_long},
    {"setrefv",    "PX_IOCSREFV",     PX_IOCSREFV,     1, do_long},
    {"getdmasize", "PX_IOCGDMASIZE",  PX_IOCGDMASIZE,  0, do_long},
    {"getdmabuf",  "PX_IOCGDMABUF",   PX_IOCGDMABUF,   0, do_long},
    {"getirqcount","PX_IOCGIRQCOUNT", PX_IOCGIRQCOUNT, 0, do_long},
    {"getprograme","PX_IOCGPROGRAME", PX_IOCGPROGRAME, 0, do_page},
    {"getprogramo","PX_IOCGPROGRAMO", PX_IOCGPROGRAMO, 0, do_page},
    {"getriscadde","PX_IOCGRISCADDE", PX_IOCGRISCADDE, 0, do_long},
    {"getriscaddo","PX_IOCGRISCADDO", PX_IOCGRISCADDO, 0, do_long},
    {"setmux",     "PX_IOCSMUX",      PX_IOCSMUX,      1, do_long},
    {"getmux",     "PX_IOCGMUX",      PX_IOCGMUX,      0, do_long},
    {"getrefv",    "PX_IOCGREFV",     PX_IOCGREFV,     0, do_long},
    {"settrig",    "PX_IOCSTRIG",     PX_IOCSTRIG,     1, do_long},
    {"setnframes", "PX_IOCSACQLEN",   PX_IOCSACQLEN,   1, do_long},
    {"getnframes", "PX_IOCGACQLEN",   PX_IOCGACQLEN,   0, do_long},
    {"grabnow",    "PX_IOCACQNOW",    PX_IOCACQNOW,    0, do_long},
    {"waitvb",     "PX_IOCWAITVB",    PX_IOCWAITVB,    1, do_long},
    {"getbright",  "PX_IOCGBRIGHT",   PX_IOCGBRIGHT,   0, do_long},
    {"setbright",  "PX_IOCSBRIGHT",   PX_IOCSBRIGHT,   1, do_long},
    {"getcontrast","PX_IOCGCONTRAST", PX_IOCGCONTRAST, 0, do_long},
    {"setcontrast","PX_IOCSCONTRAST", PX_IOCSCONTRAST, 1, do_long},
    {"gethue",     "PX_IOCGHUE",      PX_IOCGHUE,      0, do_long},
    {"sethue",     "PX_IOCSHUE",      PX_IOCSHUE,      1, do_long},
    {"getsatu",    "PX_IOCGSATU",     PX_IOCGSATU,     0, do_long},
    {"setsatu",    "PX_IOCSSATU",     PX_IOCSSATU,     1, do_long},
    {"getsatv",    "PX_IOCGSATV",     PX_IOCGSATV,     0, do_long},
    {"setsatv",    "PX_IOCSSATV",     PX_IOCSSATV,     1, do_long},
    {NULL,0,0}
    };

int main(int argc, char **argv)
{
    FILE *f;
    struct command *ptr;
    int i, j, cmd, retval;
    unsigned long nums[4];
    char trash[24];
    char output[80];

    prgname = argv[0];

    if (argc > 1 && argv[1][0]=='/') {
	devname = argv[1];
	argc--; argv++;
    } else if (argc > 1 && argv[argc-1][0]=='/') {
	devname = argv[argc-1];
	argc--;
    } else if (getenv("PXCDEVICE")) {
        devname = getenv("PXCDEVICE");
    }

    if (argc<2) {
        fprintf(stderr,"Use: %s [command [arg] ...]\n", prgname);
	fprintf(stderr,"The device used is /dev/pxc0ctl, or $PXCDEVICE\n");
	fprintf(stderr,"Commands are:\n");
	for (ptr=commands; ptr->name; ptr++) {
	    fprintf(stderr,"  %-15s (%-20s, %2i numeric arg%s)\n",
		    ptr->name, ptr->macro, ptr->nargs,
		    ptr->nargs == 1 ? "" : "s");
	}
        exit(1);
    }

    f=fopen(devname,"r+");
    if (!f) {
        fprintf(stderr,"%s: %s: %s\n",prgname,devname,strerror(errno));
        exit(1);
    }

    for (i=1; i<argc; i++) {
	for (ptr=commands; ptr->name; ptr++) {
	    if (strcmp(ptr->name,argv[i]) && strcmp(ptr->macro,argv[i]))
		continue;

	    /* found */
	    cmd = ptr->cmd;
	    for (j=0; j<ptr->nargs; j++) {
		if (!argv[i+1+j]) {
		    fprintf(stderr,"%s: too few arguments for \"%s\"\n",
			    prgname, ptr->name);
		    exit(1);
		}
		if (sscanf(argv[i+1+j],"%li%s",&nums[j],trash)!=1) {
		    fprintf(stderr,"%s: not a number \"%s\"\n",
			    prgname, argv[i+1+j]);
		    exit(1);
		}
	    }
	    i+=j;
	    output[0] = '\0'; /* no output */
	    if (ptr->fun) /* do that via function */
		retval = ptr->fun(fileno(f),cmd,nums,output);
	    else
		retval = ioctl(fileno(f),cmd,nums);
	    fprintf(stderr,"ioctl(\"%s\", %s, ...) = %s%i", devname,
		   ptr->macro, output, retval);
	    if (retval < 0)
		fprintf(stderr," (%s)",strerror(errno));
	    fprintf(stderr,"\n");
	    break;
	}
	if (!ptr->name) {
	    fprintf(stderr,"%s: invalid command \"%s\"\n", prgname, argv[i]);
	    exit(1);
	}
    }

    fclose(f);
    exit(0);
}
