/*
 * main.c -- the core of pxc, driver for the pxc200 grabber
 *
 * Copyright (c) 1998,1999  Alessandro Rubini (rubini@prosa.it)
 * Copyright (c) 1999 Kazuhiko Shinozawa (shino@cslab.kecl.ntt.co.jp)
 * Copyright (c) 1999 Chanop Silpa-Anan (chanop.silpa-anan@faceng.anu.edu.au)
 *
 *   Minor parts come from bttv.c, Copyright (c) Ralph Metzler
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifndef __KERNEL__
#  define __KERNEL__
#endif
#ifndef MODULE
#  define MODULE
#endif

#include <linux/module.h>

#include <linux/kernel.h> /* printk() */
#include <linux/malloc.h> /* kmalloc() */
#include <linux/fs.h>     /* everything... */
#include <linux/errno.h>  /* error codes */
#include <linux/types.h>  /* size_t */
#include <linux/proc_fs.h>
#include <linux/fcntl.h>        /* O_ACCMODE */
#include <linux/bios32.h> /* pcibios_* */
#include <linux/pci.h>
#include <linux/delay.h>

#include <asm/system.h>   /* cli(), *_flags */
#include <asm/segment.h>  /* memcpy and such */
#include <asm/io.h>
#include <asm/page.h>
#include <asm/pgtable.h>

#include "pxc200.h"        /* local definitions */
#include "allocator.h"
#include "pxc200_mdw_defines.h" /* MDW: my definitions */

/* this is a load-time parameter */
int really_pxc200 = 0;

/* all variables, functions, etc are prefixed "px", to save typing */

/*
 * I don't use static symbols here, because modularization code hides
 * public symbols
 */

int px_major =   PX_MAJOR;

Px_Dev *px_devices; /* allocated in init_module */

/*
 * Different minors behave differently, so let's use multiple fops
 */

struct file_operations *px_fop_array[]={
    &px_bin_fops,      /* type  0: binary */
    &px_pnm_fops,      /* type  1: pgm */
    &px_ctl_fops,      /* type  2: control */
    &px_bin_fops,      /* type  3: hires binary */
    &px_pnm_fops,      /* type  4: hires pgm */
    &px_pnm_fops,      /* type  5: ppm (header + byteswap) */
    &px_bin_fops,      /* type  6: rgb (byteswap) */
    &px_bin_fops,      /* type  7: bgr (straight from DMA) */
    &px_pnm_fops,      /* type  8: hires ppm */
    &px_bin_fops,      /* type  9: hires rgb */
    &px_bin_fops       /* type 10: hires bgr */
};

/*---------------------------------------------------------- allocate DMA */
void px_free_dma(Px_Dev *dev)
{
    if (dev->dmaremap) {
        iounmap((void *)dev->dmaremap);
        dev->dmaremap=0;
    }
    if (dev->dmabuffer) {
        PDEBUG("freeing DMA buffer\n");
        allocator_free_dma(dev->dmabuffer);
        dev->dmabuffer =0;
        PDEBUG("freed DMA buffer\n");
    }
    dev->dmasize = 0;
    /* don't override other values, which are useful in reallocating */
}

int px_allocate_dma(Px_Dev *dev, int kilobytes, int prio)
{
    if (dev->dmasize >= kilobytes * 1024)
        return 0;
    if (dev->dmasize)
        px_free_dma(dev);
    dev->dmabuffer = allocator_allocate_dma(kilobytes, prio);
    PDEBUG("allocating: %i kilos, got 0x%08lx\n", kilobytes,dev->dmabuffer);
    if (!dev->dmabuffer)
        return -ENOMEM;
    dev->dmasize = kilobytes << 10;
    dev->dmaremap = (unsigned long) ioremap(dev->dmabuffer, dev->dmasize);

    return 0;
}

/*---------------------------------------------------------- allocate prg */
void px_free_program_pages(Px_Dev *dev)
{
    struct px_programpage *next, *ptr = dev->program;

    while (ptr) {
	next = ptr->next;
	free_page(ptr->page);
	kfree(ptr);
	ptr = next;
    }
    dev->program = NULL;
}

void px_free_program(Px_Dev *dev)
{
    if (dev->dmaprogramE) {
        free_page(dev->dmaprogramE);
        dev->dmaprogramE = 0;
    }
    if (dev->dmaprogramO) {
        free_page(dev->dmaprogramO);
        dev->dmaprogramO = 0;
    }
    if (dev->program) {
	px_free_program_pages(dev);
	dev->prglen = 0;
    }
}

int px_allocate_program2(Px_Dev *dev, int prio)
{
    dev->dmaprogramE = get_free_page(prio); /* clear it too */
    dev->dmaprogramO = get_free_page(prio); /* clear it too */
    if (!dev->dmaprogramE || !dev->dmaprogramO) {
        return -ENOMEM;
    }
    return 0;
}

int px_allocate_programN(Px_Dev *dev, int prio)
{
    struct px_programpage *ptr, **nextptr;
    int i = dev->prglen;

    if (dev->program)
	px_free_program_pages(dev);
    PDEBUG("allocating %i program pages\n",i);
    nextptr = &dev->program;
    while (i--) {
	ptr = kmalloc(sizeof(struct px_programpage), prio);
	if (!ptr) {
	    px_free_program_pages(dev); return -ENOMEM;
	}
	ptr->next = NULL;
	ptr->page = get_free_page(prio);
	if (!ptr->page) {
	    kfree(ptr);
	    px_free_program_pages(dev); return -ENOMEM;
	}
	*nextptr = ptr;
	nextptr = &ptr->next;
    }
    return 0;
}

/*---------------------------------------------------- low level: GPIO */
/* This part is PXC-specific (not related to the Bt848) */

int px_init_gpio(Px_Dev *dev, int testflag)
{
    int bitmask;
    /* NOTE: this only works for the LC grabber */

    PX_WRITE32(dev, 1<<13, BT848_GPIO_OUT_EN); /* only the reset pin */
    PX_WRITE32(dev, 0,     BT848_GPIO_DATA);
    udelay(3);
    PX_WRITE32(dev, 1<<13, BT848_GPIO_DATA); /* ok, done */

    /*
     * Ok, the DAC is reset now. Enable output pins as needed, according to
     * the cfg word.
     * Since GPIO inputs are pulled up, there's no need to drive the
     * reset line high any more.
     */
    bitmask = 0; /* no output pins by default */

    if (dev->cfgword & PX_CFG_ADPOT)
        bitmask |= 7<<4; /* the DAC */

    PX_WRITE32(dev, bitmask, BT848_GPIO_OUT_EN);

    /* Use direct input instead of strobed one */
    PX_WRITE32(dev, 0, BT848_GPIO_REG_INP);
    return 0;
}

/*---------------------------------------------------- low level: I2C */
/* This part is PXC-specific (not related to the Bt848) */

static inline int px_i2c_command(Px_Dev *dev, u8 addr, u8 command,
                                 u8 rdwr, u8 two_or_three, u8 third)
{
    unsigned long j = jiffies;
    u32 datum = (u32)addr << 25
        | (u32)rdwr << 24
        | (u32)command << 16
        | (u32)third << 8
        | PX_I2C_DIVISOR
        | BT848_I2C_SYNC  
        | (two_or_three == 3 ? BT848_I2C_W3B : 0)
        | BT848_I2C_SCL 
        | BT848_I2C_SDA;   


    PDEBUGG("i2c: 0x%08x ->\n",(int)datum);
    PX_WRITE32(dev, BT848_INT_I2CDONE, BT848_INT_STAT);
    PX_WRITE32(dev, datum, BT848_I2C);
    udelay(30);
    PDEBUGG("stat: 0x%08x\n",(int)PX_READ32(dev,BT848_INT_STAT));
    while (!(PX_READ32(dev,BT848_INT_STAT) & BT848_INT_I2CDONE)) {
        if (jiffies-j > 2) {
            printk(KERN_ERR PX_MSG "I2C command 0x%02x timed out\n",
                   command);
            return -EINVAL;
        }
        schedule(); /* therefore, i2c_command() can't run at interrupt time */
    }
    if (!(PX_READ32(dev,BT848_INT_STAT) & BT848_INT_RACK)) {
        PDEBUGG("stat: 0x%08x\n",(int)PX_READ32(dev,BT848_INT_STAT));
        PDEBUGG("I2C command 0x%02x not acknowledged\n", command);
        return -EINVAL;
    }
    PDEBUGG("stat: 0x%08x\n",(int)PX_READ32(dev,BT848_INT_STAT));
    datum = PX_READ32(dev, BT848_I2C);
    PDEBUGG("i2c:                0x%08x\n",(int)datum);
    return (datum>>8) & 0xFF;
}

/*
 * Reading is tricky: it consists of two cycles: one write and one read
 */
int px_i2c_read(Px_Dev *dev, u8 addr, u8 command, u8 two_or_three, u8 third)
{
    int retval=-1;
    int retries;

    for (retries=0; retval<0 && retries<4; retries++) {
        udelay(100); /* pedantic */
        retval = px_i2c_command(dev, addr, command,
                                PX_I2C_WRITE, two_or_three, third);
        if (retval >= 0)
            retval = px_i2c_command(dev, addr, command, PX_I2C_READ, 2, 0);
        if (retval<0) {
            unsigned long j = jiffies;
            while (j-jiffies < (20*HZ/1000 + 1))
                schedule(); /* wait at least 20ms */
        }
    }
    if (retval<0)
        PDEBUG("i2c: read (%02x) failed\n",command);
    return retval;
}

static inline int px_set_refv(Px_Dev *dev)
{
    int retval, i;
    int value = (int)dev->refv;

    /*
     * Setting the reference value depends on the type of dac being
     * used
     */
    if (!(dev->cfgword & PX_CFG_ADPOT)) { /* use the MAX527 on the i2c */
        PDEBUG("initializing MAX517 to %i\n", value);
        retval =  px_i2c_command(dev, PX_I2C_DAC, 0, PX_I2C_WRITE, 3, value);
        return retval<0 ? retval : 0;
    }
    /* Otherwise, configure the AD serial device */

    PDEBUG("initializing AD8400 to %i\n", value);
    /*
     * By default there are no output bits but these three. Maybe I'd better
     * cache bit values to dev in the future, to control individual bits.
     */
    PX_WRITE32(dev, PX_GPIO_CSL, BT848_GPIO_DATA); /* disable chip-select */
    for (i = 1<<9; i; i>>=1) {
        int bit = (value & i) ? PX_GPIO_SDI : 0;
        PX_WRITE32(dev, PX_GPIO_CSL | bit, BT848_GPIO_DATA);
        udelay(5);
        PX_WRITE32(dev, PX_GPIO_CSL | bit | PX_GPIO_CLK, BT848_GPIO_DATA);
        udelay(5);
    }
    PX_WRITE32(dev,0, BT848_GPIO_DATA); /* enable normal operation, csl=0 */
    return 0;
}
    

int px_init_i2c(Px_Dev *dev, int testflag)
{
    /* NOTE: this only works for the LC grabber */
    int res;

    PDEBUG("init i2c called\n");
    if (testflag) {
        /* test out the various PIC command */
        static struct commands {
            int cmd; int expect;
        } cmd[] = {
            { PX_I2C_CMD_E00, 0x00},
            { PX_I2C_CMD_Eff, 0xff},
            { PX_I2C_CMD_E55, 0x55},
            { PX_I2C_CMD_Eaa, 0xaa},
            { PX_I2C_CMD_E0f, 0x0f},
            { PX_I2C_CMD_Ef0, 0xf0},
            { PX_I2C_CMD_REV, -1},
            { PX_I2C_CMD_ID, -1},
            { PX_I2C_CMD_HNR, -1},
            { PX_I2C_CMD_LNR, -1},
            { PX_I2C_CMD_PIC, -1},
            { PX_I2C_CMD_STAT, -1},
            { -1, -1}
        };
        struct commands *ptr;

        for (ptr=cmd; ptr->cmd>=0; ptr++) {
            /* it looks like read commands use a 3-b write */
            res = px_i2c_read(dev,PX_I2C_PIC, ptr->cmd, 3, 0);
            if (ptr->expect == -1) {
                PDEBUG("cmd %02x -> %02x\n",ptr->cmd,res);
            } else {
                if (res != ptr->expect) {
                    printk(KERN_WARNING PX_MSG "i2c: unexpected result: "
                           "%02x -> %02x\n",ptr->cmd,res);
                    return -ENODEV;
                }
            }
        }
        /* Ok, now get the configuration word and save it */
        dev->cfgword = px_i2c_read(dev,PX_I2C_PIC, PX_I2C_CMD_CFG, 3, 0);
        PDEBUG("cfgword is 0x%02x\n",dev->cfgword);

    } /* if (testflag) */

    return 0;
}

/*---------------------------------------------------- interrupt handler */
static void px_interrupt(int irq, void *devptr, struct pt_regs *regs)
{
    Px_Dev *dev=devptr;
    u32 status, tmp;


    /* This is a shared handler: if not ours, return */
    status = PX_READ32(dev, BT848_INT_STAT);
    if ( status == 0) {
        PDEBUG("isr: irq %i: not mine\n",irq);
        return;
    }


    if (0) {
        struct timeval tv;
        do_gettimeofday(&tv);
        PDEBUG("isr at %08u.%06u\n",
                (int)(tv.tv_sec % 100000000),
                (int)(tv.tv_usec));
    }

    dev->irq_count++;

    if (dev->irq_count % 100 == 0)
	PDEBUG("isr %li: status: 0x%08x, flags 0x%08lx (next seq %i)\n",
	       dev->irq_count,(int)status,dev->flags,dev->seq_next);
    PX_WRITE32(dev,status,BT848_INT_STAT);  /* acknowledge */

    /*
     * Print out information about the interrupt (most are disabled)
     */
    if (status & BT848_INT_FMTCHG)
        PDEBUG("isr %li: format change\n",dev->irq_count);
    if (status & BT848_INT_VSYNC)
        PDEBUGG("isr %li: vsync\n",dev->irq_count);
    if (status & BT848_INT_HSYNC)
        PDEBUGG("isr %li: hsync\n",dev->irq_count);
    if (status & BT848_INT_OFLOW)
        PDEBUGG("isr %li: overflow\n",dev->irq_count);
    if (status & BT848_INT_VPRES)
        PDEBUG("isr %li: \"present\" changed\n",dev->irq_count);
    if (status & BT848_INT_GPINT)
        PDEBUG("isr %li: GPIO interrupt\n",dev->irq_count);
    if (status & BT848_INT_RISCI)
        PDEBUGG("isr %li: RISC interrupt\n",dev->irq_count);
    if (status & BT848_INT_FBUS) {
        PDEBUGG("isr %li: overrun\n",dev->irq_count);
	dev->hwoverrun++;
    }
    if (status & BT848_INT_FTRGT) {
        PDEBUG("isr %li: faulty target\n",dev->irq_count);
	dev->hwoverrun++;
    }
    if (status & BT848_INT_PPERR)
        PDEBUG("isr %li: parity error\n",dev->irq_count);
    if (status & BT848_INT_PABORT)
        PDEBUG("isr %li: PCI abort\n",dev->irq_count);
    if (status & BT848_INT_OCERR)
        PDEBUG("isr %li: instruction error\n",dev->irq_count);
    if (status & BT848_INT_FIELD)
        PDEBUGG("isr %li: field is even\n",dev->irq_count);
    if (status & BT848_INT_RISC_EN)
        PDEBUGG("isr %li: risc is enabled\n",dev->irq_count);
    PDEBUGG("isr %li: risc status is %i\n",dev->irq_count,
           status >> 28);

    if (0) {
	tmp = PX_READ32(dev, BT848_RISC_COUNT);
	PDEBUG("isr: risc is at 0x%08x (even 0x%08x, odd 0x%08x)\n",
	   (int)tmp,(int)dev->dmaprogramE, (int)dev->dmaprogramO);
    }

    /*
     * Interrupts are only useful when triggered by the RISC program.
     * Other causes of interrupts are not interesting, but we'd
     * better prepare to handle some problems....
     */
    if ((status & BT848_INT_RISCI) == 0) {

        if (status & BT848_INT_VPRES) {
            /* That's good */
            return;
        }

        /* Otherwise, print a warning */
	tmp = PX_READ32(dev, BT848_RISC_COUNT);
#if 0
	printk(KERN_WARNING PX_MSG 
               "Unexpected IRQ (status 0x%08lx, IP 0x%08lx\n",
	       (long)status, (long)tmp);
#endif
        return;
    }

    /*
     * Special case: are we in trigger mode?
     */
    if (dev->flags & PX_FLAG_TRIGMODE) {
	if (status & BT848_INT_GPINT) {
	    PDEBUG("isr should do trig, but it is not implemented\n"); 
	    /* enable acquisition from the pre-set IP */
	    /* disable TRIG mode */
	    /* FIXME */
	    return;
	}
	else 
	    PDEBUG("trig mode and different (%08lx) interrupt\n",
		   (long)status);
    }
    /*
     * Special case: are we in grab-n-images mode?
     */
    if (dev->prglen) {
	PDEBUG("isr should do grab-n, but it is not implemented\n"); 
	/* count this field, awake, and disable acq if we are done */
	/* FIXME */
	return;
    }

    /*
     * Default behaviour: awake anyone if a whole image is there
     */
    if (dev->isup < PX_UPTIME_MIN) {
        if (dev->flags & PX_FLAG_HIRES)
            dev->isup += !!(status & BT848_INT_FIELD); /* only even */
        else
            dev->isup++; /* low res: every field */
	}
    if (dev->queue && dev->isup >= PX_UPTIME_MIN) {
        wake_up_interruptible(&dev->queue);
    }

    /*
     * Special case (as an add-on): are we in "grab sequence" mode?
     * Use "while" and "break" to avoid goto.
     */
    while ((dev->flags & PX_FLAG_SEQUENCE) && dev->seq_pending) {
	/* while is used in order to break instead of goto */
	struct timeval tv1, tv2;
	int count = dev->xsize * dev->ysize * PX_PIXEL_SIZE(dev);
	/* Only copy data after after an odd field (even is better?) */
	if ( (status & BT848_INT_FIELD) == 0)  break;

	if (dev->irq_count < dev->seq_next)
	    break; /* Not this time */

	PDEBUG("sequence at isr %li\n",dev->irq_count);
 	wake_up_interruptible(&dev->seq_queue); /* right ahead, avoid loops */
	if ( ( dev->seq_head < dev->seq_tail)
	     && (dev->seq_head + count < dev->seq_tail)) {
	    printk(KERN_WARNING PX_MSG "sequence buffer full (irqnr %li)\n",
		   (long)dev->irq_count);
	    /* Only count this image once */
	    if (dev->seq_next == dev->irq_count) dev->seq_overrun++;
	    break;
	}
	do_gettimeofday(&tv1); /* Timestamping aid */
	memcpy(dev->seq_buffer + dev->seq_head, (void *)dev->dmaremap, count);
	dev->seq_pending--;
	do_gettimeofday(&tv2);
	PDEBUG("Copy took %i usec\n",
	       tv2.tv_usec-tv1.tv_usec + 1000000 * (tv2.tv_sec-tv1.tv_sec));
	dev->seq_head += count;
	if (dev->seq_head == dev->seq_bufsize) dev->seq_head = 0;
	dev->seq_next = dev->irq_count + dev->seq_step2;
	break;
    }




    return;

}

/*---------------------------------------------------------- up/down */
int px_shutdown(Px_Dev *dev)
{
    u16 cmd;
    int px_acq_off(Px_Dev *dev); /* declared later */

    if (dev->flags & PX_FLAG_PERSIST)
	return 0; /* no shutdown */

    dev->isup = 0;
    px_acq_off(dev);
    if(dev->irq_active) {
        free_irq(dev->irq, dev);
        dev->irq_active = 0; /* safe */
    }
    pcibios_read_config_word(dev->bus, dev->device, PCI_COMMAND, &cmd);
    if ( (cmd & PCI_COMMAND_MEMORY) == 0) /* no memory: already off */
        return 0;

    px_free_dma(dev);
    px_free_program(dev);
    /* disable memory */
    pcibios_write_config_word(dev->bus, dev->device, PCI_COMMAND,
                              cmd & ~PCI_COMMAND_MEMORY);
    return 0;
}

/*
 * When a grabber is first used, it must be powered on. Dma is still
 * kept disabled, but memory mapping is enabled
 */
int px_bringup(Px_Dev *dev)
{
    u16 cmd; int retval;

    if (dev->isup) return 0;
    if (dev->flags & PX_FLAG_PERSIST)
	return 0; /* no bringup */

    /* enable memory */
    pcibios_read_config_word(dev->bus, dev->device, PCI_COMMAND, &cmd);
    pcibios_write_config_word(dev->bus, dev->device, PCI_COMMAND,
                               cmd | PCI_COMMAND_MEMORY);

    retval = request_irq(dev->irq, px_interrupt, SA_INTERRUPT | SA_SHIRQ,
                         "pxc200", dev);
    if (retval) {
        PDEBUG("irq %i not available\n",dev->irq);
        px_shutdown(dev);
        return retval;
    }
    dev->irq_active = dev->irq;

    PDEBUG("resetting\n");
    PX_WRITE32(dev,0,BT848_SRESET);
    if (!(dev->flags & PX_FLAG_848ONLY)) {
        /* PXC200-specific initialization */
	PDEBUG("pxc200-specific\n");
        px_init_i2c(dev, 0 /* no test */);
        px_init_gpio(dev, 0);
        /* Put the reference voltage to the default */
        dev->refv = PX_DEFAULT_REF;
        px_set_refv(dev);
    }

    dev->flags &= PX_FLAG_848ONLY; /* Preserve this one */
    dev->irq_count = 0;
    dev->hwoverrun = 0;
    dev->seq_overrun = 0;
    dev->isup = 1;
    return 0;
}

/*-------------------------------------------------------- init the device */
/*
 * Device initialization. Before calling this function, at power on,
 * the registers are all set to 0. The received pointer holds
 * valid bus, device and phys_addr fields
 */
int px_init(Px_Dev *dev)
{
    u16 cmd;
    int retval, errorcount = 0;

    dev->regs = ioremap(dev->phys_add, 4096); /* one page (on x86, at least) */
    if (!dev->regs) return -ENOMEM;

    pcibios_read_config_byte(dev->bus, dev->device, PCI_INTERRUPT_LINE,
                             &dev->irq);

    /*
     * this overwrites registers, before checking the ID. But some of
     * the registers need to be set in order to access the ID. Bad.
     * At least, save its cfg status before acting on it -- but enable
     * bus mastering.
     */
    pcibios_read_config_word(dev->bus, dev->device, PCI_COMMAND, &cmd);
    cmd |= PCI_COMMAND_MASTER;
    pcibios_write_config_word(dev->bus, dev->device, PCI_COMMAND, cmd);

    PDEBUG("px_init\n");
    if ( (retval = px_init_i2c(dev, 1)) ) /* i2c first: gpio depends on it */
      errorcount++;
    if ( (retval = px_init_gpio(dev, 1)) ) 
      errorcount++;

    if (!really_pxc200 && errorcount) 
        dev->flags |= PX_FLAG_848ONLY;

    px_bringup(dev);

    px_shutdown(dev);

    /* restore cmd */
    pcibios_write_config_word(dev->bus, dev->device, PCI_COMMAND, cmd);
    if (dev->flags & PX_FLAG_848ONLY) {
        printk(KERN_INFO PX_MSG "found bt848  device: mem=08%08x, irq=%i\n",
               dev->phys_add, dev->irq);
    } else {
        printk(KERN_INFO PX_MSG "found pxc200 device: mem=08%08x, irq=%i\n",
               dev->phys_add, dev->irq);
    }
    return 0; /* success */
}


#ifdef PX_USE_PROC
/*---------------------------------------------------- the proc filesystem */

/*
 * The proc filesystem: function to read an entry
 */

int px_read_proc(char *buf, char **start, off_t offset,
                   int len, int unused)
{
    int i;
    Px_Dev *dev;

    #define LIMIT (PAGE_SIZE-80) /* don't print anymore after this size */
    len=0;
    for(dev=px_devices; dev; dev=dev->next) {
        u32 dword;
        u16 cmd;

	if (!dev->usage) px_bringup(dev);
        dev->usage++;

        len += sprintf(buf+len,
		       "\n\nDevice at %p (bus %i dev %i)\nPCI Registers:\n",
		       dev, dev->bus, dev->device);
        for (i=0; i<0x40; i+=4) {
            if (i%16 == 0)
                len += sprintf(buf+len, "0x%02x: ",i);
            pcibios_read_config_dword(dev->bus, dev->device, i, &dword);
	    /* cut to 8 bits, as most of them are just 8-bit wide */
            len += sprintf(buf+len, "0x%08x%c", dword,
			   (i+4)%16 ? ' ' : '\n');
            if (len > LIMIT) return len;
        }

        /* Save the current settings, enable memory mapping and snooze it */
        pcibios_read_config_word(dev->bus, dev->device, PCI_COMMAND, &cmd);
        pcibios_write_config_word(dev->bus, dev->device, PCI_COMMAND,
                               cmd | PCI_COMMAND_MEMORY);
        len += sprintf(buf+len, "\nBT848 registers:");
        for (i=0; i<BT848_NR_REGISTERS; i+=4) {
            if (i%16 == 0)
                len += sprintf(buf+len, "\n0x%04x: ", i);
            dword = ((volatile u32 *)(dev->regs))[i/4];
            len += sprintf(buf+len, "0x%08x%s", dword,
			   (i+4)%8 ? " " : "   ");
            if (len > LIMIT) break;
        }
	len += sprintf(buf+len,"\n");
        pcibios_write_config_word(dev->bus, dev->device, PCI_COMMAND, cmd);
        dev->usage--;
        if (dev->usage == 0) px_shutdown(dev);
    }
    #undef LIMIT
    return len;
}

struct proc_dir_entry px_proc_entry = {
        0,                 /* low_ino: the inode -- dynamic */
        3, "pxc",          /* len of name and name */
        S_IFREG | S_IRUGO, /* mode */
        1, 0, 0,           /* nlinks, owner, group */
        0, NULL,           /* size - unused; operations -- use default */
        &px_read_proc,     /* function used to read data */
        /* nothing more */
    };


#endif /* PX_USE_PROC */

/*-------------------------------------------------------- risc program */
unsigned long px_risc_program(Px_Dev *dev,  int sync, unsigned long page,
			      unsigned long dest, u32 **nextptr)
{
  /*
   * This function creates the dma program for one field, allocating
   * the program page if thus needed. The return value is the program page
   * or NULL in case of error. "nextptrptr" is used to return the address
   * of the last jump of this page.
   */
    u32 *ip;
    int increment, i;

    if (!page) 	page = get_free_page(GFP_ATOMIC); /* Bah! */
    if (!page)	return 0;
    ip = (u32 *)page; /* start from there */
    increment = 1; /* line count as DMA address */
    if (dev->flags & PX_FLAG_HIRES) increment = 2;

    /* SYNC to start of field */
    *(ip++) = BT848_RISC_SYNC | sync;                  *(ip++) = 0; /* resvd */
    *(ip++) = BT848_RISC_SYNC | BT848_FIFO_STATUS_FM1; *(ip++) = 0;

    i = 0;
    if ( ((sync & 0xf) == BT848_FIFO_STATUS_VRO) && (increment == 2) )
	i = 1; /* hires and odd: write to position 1 */

    PDEBUG("sync %08x, i %i, incr %i\n", sync, i, increment);
    for ( ; i < dev->ysize; i+=increment) {
        /* WRITE one line */
        *(ip++) = BT848_RISC_WRITE | (dev->xsize * PX_PIXEL_SIZE(dev))
            | BT848_RISC_SOL | BT848_RISC_EOL;
        *(ip++) = dest + dev->xsize * i * PX_PIXEL_SIZE(dev);
    }

    /* JUMP to next field, and interrupt */
    *(ip++) = BT848_RISC_JUMP | BT848_RISC_IRQ;
    *ip = page;     /* loop back */
    *nextptr = ip;  /* but tell the caller where is this ptr */

    return page;
}

/*-------------------------------------------------------- grab on/off */
int px_acq_on(Px_Dev *dev)
{
    u32 tmp;

    dev->isup = 1; /* "just-up" */

    /* enable DMA and interrupt reporting */
    PX_WRITE32(dev, BT848_CAP_CTL_CAPTURE_ODD 
                  | BT848_CAP_CTL_CAPTURE_EVEN, BT848_CAP_CTL);

    /* 0x7b820 are the interesting interrupts */
    tmp = PX_DEFAULT_IRQ_MASK & ~BT848_INT_GPINT;
    if (dev->flags & PX_FLAG_TRIGMODE)	tmp |= BT848_INT_GPINT;
    PX_WRITE32(dev, tmp, BT848_INT_MASK);

    PX_WRITE32(dev, dev->physdmaprog, BT848_RISC_STRT_ADD);

    tmp = PX_READ32(dev, BT848_RISC_COUNT);
    PDEBUG("acq on: risc now at 0x%08x\n",(int)tmp);

    /*
     * The RISC engine is only enabled on the rising edge, so first
     * lower all the interesting bits, then raise them.
     */
    tmp = PX_READ32(dev, BT848_GPIO_DMA_CTL);
    tmp &=  ~0xFF;
    PX_WRITE32(dev, tmp, BT848_GPIO_DMA_CTL);
    /* FIXME: no support for GPIO yet */
    tmp |= BT848_GPIO_DMA_CTL_FIFO_ENABLE
         | BT848_GPIO_DMA_CTL_RISC_ENABLE  /* enable risc and fifo */
         | BT848_GPIO_DMA_CTL_PKTP_4   /* and do DMA as soon as possible */
         | BT848_GPIO_DMA_CTL_PLTP1_4
         | BT848_GPIO_DMA_CTL_PLTP23_4;
    PX_WRITE32(dev, tmp, BT848_GPIO_DMA_CTL);

    tmp = PX_READ32(dev, BT848_RISC_COUNT);
    PDEBUG("acq on: risc now at 0x%08x\n",(int)tmp);

    dev->flags |= PX_FLAG_RUNNING;
    return 0;
}

int px_acq_off(Px_Dev *dev)
{
    u32 tmp = PX_READ32(dev, BT848_GPIO_DMA_CTL);

    tmp &=  ~0xFF;
    PX_WRITE32(dev, tmp, BT848_GPIO_DMA_CTL);
    /* disable DMA */
    PX_WRITE32(dev, 0, BT848_CAP_CTL);

    dev->flags &= ~PX_FLAG_RUNNING;
    dev->prgcurr = 0;
    return 0;
}

int px_prepare_device(Px_Dev *dev, int prio)
{
    int retval = 0;
    u32 *next;
    int size, i;

    px_acq_off(dev);

    if (!dev->prglen) { /* continuous grabbing */
	size = dev->xsize * dev->ysize * PX_PIXEL_SIZE(dev);
	if ( (retval = px_allocate_dma(dev, (size+1023)>>10, prio)) )
	     return retval;
	
	/* free program pages */
	if (dev->program)
	    px_free_program(dev);
	/* allocate dmaE and dmaO */
	if (!dev->dmaprogramE)
	    retval = px_allocate_program2(dev, prio);
	if (retval)
	    return retval;

	/*
	 * create the RISC program to control data xfer. In high resolution
	 * the two fields build up a single image, in low resolution we
	 * only grab the even field (Chanop)
	 * overwrite each other
	 */
	px_risc_program(dev, BT848_FIFO_STATUS_VRE | BT848_RISC_RESYNC,
			dev->dmaprogramE, dev->dmabuffer, &next);
	*next = dev->dmaprogramE;

	if (dev->flags & PX_FLAG_HIRES) {
	    *next = dev->dmaprogramO;
	    px_risc_program(dev, BT848_FIFO_STATUS_VRO,
			    dev->dmaprogramO, dev->dmabuffer, &next);
	    *next = dev->dmaprogramE;
	}
	
	/* start from even field */
	dev->physdmaprog = virt_to_bus((void *)dev->dmaprogramE);

    } else { /* count-the-images */
	struct px_programpage *prgptr;
	int flags;

	size = dev->xsize * dev->ysize * PX_PIXEL_SIZE(dev) * dev->prglen;
	/* prglen is "fields" */
	if (dev->flags & PX_FLAG_HIRES) size /= 2; 
	if ( (retval = px_allocate_dma(dev, (size+1023)>>10, prio)) )
	    return retval;

	/* free program pages */
	if (dev->dmaprogramE)
	    px_free_program(dev);
	/* allocate buffer */
	retval = px_allocate_programN(dev, prio);
	if (retval) return retval;

	/* build program */
	for (prgptr = dev->program, i=0; prgptr; prgptr = prgptr->next, i++) {
	    flags = (i&1) ? BT848_FIFO_STATUS_VRO : BT848_FIFO_STATUS_VRE;
	    if (!i) flags |= BT848_RISC_RESYNC;

	    if (dev->flags & PX_FLAG_HIRES)
		px_risc_program(dev, flags, prgptr->page, dev->dmabuffer+
				(i/2)*dev->xsize*dev->ysize, &next);
	    else
		px_risc_program(dev, flags, prgptr->page, dev->dmabuffer+
				i*dev->xsize*dev->ysize, &next);
	    if (prgptr->next)
		*next = prgptr->next->page;
	    else
		*next = dev->program->page; /* loop back */
	}
	
	/* start from there */
	dev->physdmaprog = virt_to_bus((void *)dev->program->page);
	
	/* set trigger in the device */
	/* FIXME */
    }
    return 0;
}

/*-------------------------------------------------------- setup dma */
/* Low level parameters */
int px_set_dma_parameters(Px_Dev *dev, int prio)
{
    u32 tmp;

#if 0
    /* paranoid */
    if (!dev->dmabuffer || !dev->xsize || !dev->ysize \
	|| !dev->dmaprogramE || !dev->dmaprogramO)
        return -EINVAL;
#endif

    /*
     * program the 848: set needed registers
     */
    PX_WRITE32(dev, dev->adelay, BT848_ADELAY);
    PX_WRITE32(dev, dev->bdelay, BT848_BDELAY); /* what's that? */

    PX_WRITE32(dev, dev->hscale >> 8, BT848_E_HSCALE_HI);
    PX_WRITE32(dev, dev->hscale >> 8, BT848_O_HSCALE_HI);
    PX_WRITE32(dev, dev->hscale&0xff, BT848_E_HSCALE_LO);
    PX_WRITE32(dev, dev->hscale&0xff, BT848_O_HSCALE_LO);

    /* FIXME: this was changed by NTT */
    tmp = PX_READ32(dev, BT848_E_VSCALE_HI);
    PX_WRITE32(dev, (tmp & 0xe0) | (dev->vscale >> 8), BT848_E_VSCALE_HI);
    tmp = PX_READ32(dev, BT848_O_VSCALE_HI);
    PX_WRITE32(dev, (tmp & 0xe0) | (dev->vscale >> 8), BT848_O_VSCALE_HI);
    PX_WRITE32(dev, dev->vscale&0xff, BT848_E_VSCALE_LO);
    PX_WRITE32(dev, dev->vscale&0xff, BT848_O_VSCALE_LO);

    PX_WRITE32(dev, dev->hactive&0xff, BT848_E_HACTIVE_LO);
    PX_WRITE32(dev, dev->hactive&0xff, BT848_O_HACTIVE_LO);
    PX_WRITE32(dev, dev->hdelay &0xff, BT848_E_HDELAY_LO);
    PX_WRITE32(dev, dev->hdelay &0xff, BT848_O_HDELAY_LO);

    PX_WRITE32(dev, dev->vactive&0xff, BT848_E_VACTIVE_LO);
    PX_WRITE32(dev, dev->vactive&0xff, BT848_O_VACTIVE_LO);
    PX_WRITE32(dev, dev->vdelay &0xff, BT848_E_VDELAY_LO);
    PX_WRITE32(dev, dev->vdelay &0xff, BT848_O_VDELAY_LO);

    tmp = ((dev->hactive >> 8) & 0x03) 
        | ((dev->hdelay  >> 6) & 0x0c)
        | ((dev->vactive >> 4) & 0x30)
        | ((dev->vdelay  >> 2) & 0xc0);
    PX_WRITE32(dev, tmp, BT848_E_CROP);
    PX_WRITE32(dev, tmp, BT848_O_CROP);

    if (dev->flags & PX_FLAG_COLOR) {
	/* tell that we are going rgb color 24 */
	PX_WRITE32(dev, BT848_COLOR_FMT_RGB24, BT848_COLOR_FMT);

	/* FIXME: check what exactly these commands do */
	PX_WRITE32(dev, BT848_ADC_SYNC_T, BT848_ADC);
	PX_WRITE32(dev, BT848_COLOR_CTL_RGB_DED, BT848_COLOR_CTL);
	//MDW 5/7 PX_WRITE32(dev, BT848_CONTROL_CON_MSB, BT848_E_CONTROL);
	//MDW 5/7 PX_WRITE32(dev, BT848_CONTROL_CON_MSB, BT848_O_CONTROL);

	/* Restore default values */
	PX_WRITE32(dev, 0x00, BT848_BRIGHT);
	PX_WRITE32(dev, 0xd8, BT848_CONTRAST_LO);
		
	/* Changed MDW 9/1/99 */
	//PX_WRITE32(dev, BT848_BRIGHT__MDW, BT848_BRIGHT);
	//PX_WRITE32(dev, BT848_CONTRAST_LO__MDW, BT848_CONTRAST_LO);
	PX_WRITE32(dev, BT848_HUE__MDW, BT848_HUE);
	PX_WRITE32(dev, BT848_SAT_U_LO__MDW, BT848_SAT_U_LO);
	PX_WRITE32(dev, BT848_SAT_V_LO__MDW, BT848_SAT_V_LO);
    } else {
	/* tell that we are going gray scale */
	PX_WRITE32(dev, BT848_COLOR_FMT_Y8, BT848_COLOR_FMT);
    }

    sprintf(dev->pnmheader, PX_HEADERFMT(dev), dev->xsize, dev->ysize);

#if 0
    /* first, write something silly to the buffer */
    {
    char *p = (char *) dev->dmaremap;
    int i, j;
    for (i=0; i<dev->ysize; i++)
      for (j=0; j<dev->xsize; j++)
	*(p++)=255-i;
    }
#endif

    px_prepare_device(dev, prio);
    PDEBUG("acq on: program at 0x%08lx\n",dev->physdmaprog);
    px_acq_on(dev);
    return 0;
}



/*
 * This code is common to the open() method of several devices.
 * It refuses to change parameters of a grabber that's actively used
 */

int px_initialize_operation(Px_Dev *dev, int flags, int prio)
{
    int result = 0;
    u32 format, tmp;

    if (dev->usage > 1) { /* usage already incremented */
	/* FIXME: allow using the control mode anyways */
        if ((dev->flags & PX_INIT_FLAGS) != (flags & PX_INIT_FLAGS))
            return -EBUSY;
        else
            return 0; /* already initialized */
    }
    /* preserve previous user flags */
    dev->flags = (dev->flags & (PX_USER_FLAGS | PX_FLAG_848ONLY)) | flags;

    /* Setup operation: must choose the image size */
    format = PX_READ32(dev,BT848_IFORM) & BT848_IFORM_NORM;
    PDEBUG("init: video format is %i\n",format);

    if (format == BT848_IFORM_AUTO) { /* auto: use pal or ntsc */
        tmp = PX_READ32(dev,BT848_DSTATUS);
        if (tmp & BT848_DSTATUS_NUML)
            format = BT848_IFORM_PAL_M;
        else
            format = BT848_IFORM_NTSC;
    }
    /* set information for square pixels (from data sheet, page 28) */
    if (flags & PX_FLAG_HIRES) {
        switch(format) {
          case BT848_IFORM_NTSC:
            dev->xsize = 640;
            dev->ysize = 480;
            dev->hscale = 0x02aa;
            dev->vscale = 0x0000;
            dev->hactive = dev->xsize;
            dev->hdelay = (dev->hactive * 135 / 754) & 0x3fe;
            dev->vactive = dev->ysize;
            dev->vdelay = 0x16;
            
            dev->adelay = 0x68;
            dev->bdelay = 0x5d;
            break;

          default: /* all of PAL */
            dev->xsize = 768;
            dev->ysize = 576;
            dev->hscale = 0x033c;
            dev->vscale = 0x0000;
            dev->hactive = dev->xsize;
            dev->hdelay = (dev->hactive * 186 / 922) & 0x3fe;
            dev->vactive = dev->ysize;
            dev->vdelay = 0x20;

            dev->adelay = 0x7f;
            dev->bdelay = 0x72;
            break;
        }
    } else { /* lowres: only one field */
        switch(format) {
          case BT848_IFORM_NTSC:
            dev->xsize = 320;
            dev->ysize = 240;
            dev->hscale = 0x1555;
            dev->vscale = 0x0000; /* single field */
            dev->hactive = dev->xsize;
            dev->hdelay = (dev->hactive * 135 / 754) & 0x3fe;
            dev->vactive = dev->ysize *2; /* active is frame-wise */
            dev->vdelay = 0x16;
            
            dev->adelay = 0x68;
            dev->bdelay = 0x5d;
            break;

          default: /* all of PAL */
            dev->xsize = 384;
            dev->ysize = 288;
            dev->hscale = 0x1679;
            dev->vscale = 0x0000; /* single field */
            dev->hactive = dev->xsize;
            dev->hdelay = (dev->hactive * 186 / 922) & 0x3fe;
            dev->vactive = dev->ysize * 2; /* active is frame-wise */
            dev->vdelay = 0x20;

            dev->adelay = 0x7f;
            dev->bdelay = 0x72;
            break;
        }
    }
    
    if (flags & PX_FLAG_DODMA) {  /* try to allocate the DMA buffer */
        int kb = (dev->xsize * dev->ysize * PX_PIXEL_SIZE(dev) + 1023) / 1024;

        px_allocate_dma(dev, kb, prio);
        if (!dev->dmabuffer) {
            return -ENOMEM;
        }
        PDEBUG("dmabuf %ikB: virt %lx, phys %lx\n", kb,
               dev->dmaremap,dev->dmabuffer);

        result = px_set_dma_parameters(dev, prio);
        if (result) {
            return result; /* no need to free DMA buffer, cache it */
        }
    }
    /* else no dmaflag: don't activate dma */

    return 0;
}

/*-------------------------------------------------------- sequence */

int px_release_sequence(Px_Dev *dev, struct file *filp)
{
    PDEBUG("release sequence...\n");
    if (dev->seq_filp && dev->seq_filp != filp)	return -EBUSY;
    dev->flags &= ~PX_IOCSEQUENCE;
    if (!dev->seq_buffer) return 0;
    dev->seq_filp = NULL;
    vfree(dev->seq_buffer); dev->seq_buffer = 0;
    PDEBUG("release sequence ok\n");
    return 0;
}

int px_new_sequence(Px_Dev *dev, struct file *filp, Px_AcqControl *ctrl)
{
    int retval;

    PDEBUG("new sequence...\n");
    if (dev->seq_filp) return -EBUSY;
    if (dev->flags & PX_FLAG_SEQUENCE) {
	printk(KERN_WARNING PX_MSG "hit unexpected driver bug\n");
	return -EBUSY;
    }
    if ( (retval = px_acq_on(dev)) ) return retval;
    if (!ctrl->count) ctrl->count = 1;
    dev->seq_bufsize = dev->xsize * dev->ysize * PX_PIXEL_SIZE(dev)
	* ctrl->buflen;
    dev->seq_buffer = vmalloc(dev->seq_bufsize);
    PDEBUG("buffer at 0x%08lx, 0x%08lx bytes\n",
	   (long)dev->seq_buffer, dev->seq_bufsize);
    if (!dev->seq_buffer) return -ENOMEM;
    dev->seq_head = dev->seq_tail  = 0;
    dev->seq_filp = filp;
    dev->seq_pending = ctrl->count;
    dev->seq_step2 = 2*ctrl->step;
    dev->seq_next = 0; /* force acquisition on first IRQ */
    dev->seq_overrun = 0;
    dev->flags |= PX_FLAG_SEQUENCE;
    PDEBUG("sequence: %i images, %i buffers, %i step\n",
	   dev->seq_pending, ctrl->buflen, dev->seq_step2);
    PDEBUG("new sequence ok\n");
    return 0;
}

/*-------------------------------------------------------- open */

int px_open (struct inode *inode, struct file *filp)
{
    int type = TYPE(inode->i_rdev);
    int num = NUM(inode->i_rdev);
    Px_Dev *dev = px_devices;
    int i, retval = 0;


    static int open_flags[] = {
 	/* pxc0     */ PX_FLAG_DODMA,  
        /* pxc0pgm  */ PX_FLAG_DODMA,
	/* pxc0ctl  */ 0,    
        /* pxc0H    */ PX_FLAG_DODMA | PX_FLAG_HIRES,
        /* pxc0Hpgm */ PX_FLAG_DODMA | PX_FLAG_HIRES,
        /* pxc0ppm  */ PX_FLAG_DODMA | PX_FLAG_COLOR,
        /* pxc0rgb  */ PX_FLAG_DODMA | PX_FLAG_COLOR,
        /* pxc0bgr  */ PX_FLAG_DODMA | PX_FLAG_COLOR,
        /* pxc0Hppm */ PX_FLAG_DODMA | PX_FLAG_COLOR | PX_FLAG_HIRES,
        /* pxc0Hrgb */ PX_FLAG_DODMA | PX_FLAG_COLOR | PX_FLAG_HIRES,
        /* pxc0Hbgr */ PX_FLAG_DODMA | PX_FLAG_COLOR | PX_FLAG_HIRES
    };

    if (type > PX_MAX_TYPE) return -ENODEV;
    for (i=0; dev && i<num; dev = dev->next, i++)
        /* nothing */;
    if (!dev)
        return -ENODEV;

    filp->private_data = dev;
    filp->f_op = px_fop_array[type];

    /* Ok, now bring it up */
    if (!dev->usage)
        px_bringup(dev);
    dev->usage++;
    MOD_INC_USE_COUNT;

#if 0
    /*
     * With previous versions of the driver, I used to have a multi-headed
     * open. I don't use this approach any more, because the only difference
     * now is in the flags argument
     */

    if (type != 0) /* else, fall through */
        if (filp->f_op->open)
            retval = (filp->f_op->open)(inode,filp); /* special open code */
        else {
            retval = -ENODEV; /* no open method: not implemented */
        }
#endif

    PDEBUG("open with flags 0x%08x\n",open_flags[type]);
    retval = px_initialize_operation(dev, open_flags[type], GFP_KERNEL);

    if (retval) {
        dev->usage--;
        if (!dev->usage)
            px_shutdown(dev);
        MOD_DEC_USE_COUNT;
    }
    return retval;
}

/*-------------------------------------------------------- close */

release_t px_close (struct inode *inode, struct file *filp)
{
    Px_Dev *dev = filp->private_data;

    if (dev->flags & PX_FLAG_SEQUENCE && dev->seq_filp == filp) {
	px_release_sequence(dev, filp);
    }

    MOD_DEC_USE_COUNT;
    dev->usage--;
    if (dev->usage==0) {
        px_shutdown(dev);
    }
    release_return(0);
}

/*-------------------------------------------------------- seek */
lseek_t px_lseek(struct inode *inode, struct file *filp, lseek_off_t off,
		 int whence)
{
    Px_Dev *dev = filp->private_data;
    lseek_t newpos;

    switch(whence) {
      case 0: /* SEEK_SET */
        newpos = off;
        break;

      case 1: /* SEEK_CUR */
        newpos = filp->f_pos + off;
        break;

      case 2: /* SEEK_END */
        newpos = dev->xsize * dev->ysize * PX_PIXEL_SIZE(dev) + off;
	switch (TYPE(inode->i_rdev)) {
	    case PX_PGM:
	    case PX_PPM:
	    case PX_HI_PGM:
	    case PX_HI_PPM:
	        newpos += PX_HEADERLEN;
	}
        break;

      default: /* can't happen */
        return -EINVAL;
    }
    if (newpos<0) return -EINVAL;
    filp->f_pos = newpos;
    return newpos;
}

/*-------------------------------------------------------- read */

/*
 * This function is used for the rgb and the ppm entry points.
 * Strangely, declaring it as inline looses 4usec at each invocation
 */
int px_convert_rgb(struct file *filp, Px_Dev *dev, char *buf,
		   unsigned int now, count_t count)
{
    #define LOCALBUFSIZE 0x600
    static u8 localbuf[LOCALBUFSIZE]; /* used to convert bgr->rgb */
    u8 *r, *b;
    
    unsigned long done = 0;

    switch (now % 3) { /* copy the first half-pixel */
        case 1:
	    PDEBUGG("0-2nd byte: 0x%02x\n",*(u8 *)(dev->dmaremap + now));
	    put_user(*((u8 *)dev->dmaremap + now), buf);
	    now++; done++; buf++; count--;
	    if (!count)
		break;
	    /* else, fall through */
	case 2:
	    PDEBUGG("0-3rd byte: 0x%02x\n",*(u8 *)(dev->dmaremap + now -2));
	    put_user(*((u8 *)dev->dmaremap + now -2), buf);
	    now++; done++; buf++; count--;
	    break;
	default:
	    break;
    }

    while (count > 2) {
	int localcount = (count > LOCALBUFSIZE) ? LOCALBUFSIZE : count;
	localcount = localcount / 3 * 3;
	memcpy(localbuf, (void *)(dev->dmaremap + now), localcount);
#if 0
	r = localbuf; b = localbuf+2;
	while (r-localbuf < localcount) {
            u8 tmp = *r; *r = *b; *b = tmp;
	    r+=3; b+=3;
	}
#else /* This is 1usec better for each 0x600 bytes of data (gcc 2.7.2.3) */
	r = localbuf; b = (u8 *)(dev->dmaremap + now);
	while (r-localbuf < localcount) {
	    *r = *(b+2);
	    *(r+2) = *b;
	    r+=3; b+=3;
	}
#endif

	copy_to_user(buf, (void *)localbuf, localcount);
	count -= localcount;
	now += localcount;
	buf += localcount;
	done += localcount;
    }

    if (count) { /* copy the trailing half-pixel */

	PDEBUGG("tail byte: 0x%02x\n",*(u8 *)(dev->dmaremap + now + 2));
	put_user(*((u8 *)dev->dmaremap + now + 2), buf);
	now++; done++; buf++; count--;
	if (count) {
	PDEBUGG("tai2 byte: 0x%02x\n",*(u8 *)(dev->dmaremap + now));
	    put_user(*((u8 *)dev->dmaremap + now), buf);
	    now++; done++; buf++; count--;
	}
    }

    filp->f_pos += done;
    return done;
    #undef LOCALBUFSIZE
}

/* Binary file: B/W and bgr */
read_write_t px_read (struct inode *inode, struct file *filp,
                char *buf, count_t count)
{
    Px_Dev *dev = filp->private_data;
    int type = TYPE(inode->i_rdev);

    unsigned int size = dev->xsize * dev->ysize * PX_PIXEL_SIZE(dev);
    unsigned int now = filp->f_pos;

    /*
     * Return a continuous flow of images. f_pos is kept within our limits,
     * because the % operator on a long long needs a lib function :(
     */

    if (size==0) return -EINVAL; /* shouldn't happen */

    if (now >= size)
        filp->f_pos = now = (unsigned long)filp->f_pos % size;

    if (now+count > size) count = size-now;

    /* wait for bringup to complete */
    if (dev->isup < PX_UPTIME_MIN) {
        PDEBUG("read: sleep\n");
        interruptible_sleep_on(&dev->queue);
    }

    if (type == PX_RGB || type == PX_HI_RGB) { /* Must convert bgr -> rgb */
	return px_convert_rgb(filp, dev, buf, now, count);
    }

    /* else, just copy raw data */
    copy_to_user(buf, (void *)(dev->dmaremap + now), count);
    filp->f_pos += count;
    return count;
}

/* pgm and ppm file */
read_write_t px_pnm_read (struct inode *inode, struct file *filp,
                char *buf, count_t count)
{
    Px_Dev *dev = filp->private_data;

    unsigned int size = dev->xsize * dev->ysize * PX_PIXEL_SIZE(dev);
    unsigned int now = filp->f_pos;
    int partialcount = 0;

#if 0 /* Sequence mode is not currently supported */
    /*
     * Special case: are we in sequence mode?
     */
    if (dev->flags & PX_FLAG_SEQUENCE && dev->seq_filp == filp) {
	unsigned long nexttail;
	if (dev->seq_tail == dev->seq_head && !dev->seq_pending) {
	    PDEBUG("Read at end of sequence\n");
	    return 0; /* End of sequence */
	}
	while (dev->seq_tail == dev->seq_head && dev->seq_pending) {
	    PDEBUG("read sequence: sleep with head 0x%08lx, tail 0x%08lx\n",
		   dev->seq_head, dev->seq_tail);
	    interruptible_sleep_on(&dev->seq_queue);
	    if (current->signal & ~current->blocked) /* a signal arrived */
		return -ERESTARTSYS; /* tell the fs layer to handle it */
	}

	PDEBUGG("read sequence: awake, head 0x%08lx, tail 0x%08lx\n",
	       dev->seq_head, dev->seq_tail);
	nexttail = dev->seq_head;
	if (nexttail < dev->seq_tail)
	    nexttail = dev->seq_bufsize;
	if (nexttail - dev->seq_tail > count)
	    nexttail = dev->seq_tail + count;
	count = nexttail - dev->seq_tail;
	copy_to_user(buf, (void *)(dev->seq_buffer + dev->seq_tail), count);
	if (nexttail == dev->seq_bufsize) nexttail = 0;
	dev->seq_tail = nexttail;
	return count;
    }
#endif

    /* first, the header */

    PDEBUGG("pnm read at %i\n",now);
    if (now < PX_HEADERLEN) {
        partialcount = (count+now >= PX_HEADERLEN) ? PX_HEADERLEN-now : count;
        copy_to_user(buf, dev->pnmheader+now, partialcount);
        filp->f_pos += partialcount;
        if (count == partialcount)
            return count; /* nothing more, yet */
        buf += partialcount;
        count -= partialcount;
        now = PX_HEADERLEN;
    }

    /* then, the image */
                                 
    now -= PX_HEADERLEN; /* don't count the header */
    if (now >= size)
	return 0; /* end of file */

    if (now+count > size) count = size-now;

    /* wait for bringup to complete */
    if (dev->isup < PX_UPTIME_MIN) {
        PDEBUG("pnm read: sleep\n");
        interruptible_sleep_on(&dev->queue);
    }

    if (dev->flags & PX_FLAG_COLOR) { /* Must convert bgr -> rgb */
	return partialcount + px_convert_rgb(filp, dev, buf, now, count);
    }
    /* else, pgm: just copy raw data */
    copy_to_user(buf, (void *)(dev->dmaremap + now), count);
    filp->f_pos += count;
    PDEBUGG("pgm read: %i returned\n",count+partialcount);
    return count + partialcount;
}

/* The control file */
read_write_t px_ctl_read (struct inode *inode, struct file *filp,
                char *buf, count_t count)
{
    Px_Dev *dev = filp->private_data;
    unsigned int now = filp->f_pos;
    int size = BT848_NR_REGISTERS;

    /* Return a continuous flow of registers (0x200 bytes each burst) */

    if (now >= size) /* problem: division of long-long requires lib */
        filp->f_pos = now = (unsigned long)filp->f_pos % size;

    if (now+count > size) count = size-now;

    copy_to_user(buf, (void *)(dev->regs + now), count);
    filp->f_pos += count;
    return count;
}

/*-------------------------------------------------------- write */

read_write_t px_write (struct inode *inode, struct file *filp,
                const char *buf, count_t count)
{
#if 0
    Px_Dev *dev = filp->private_data;
#endif
    return -EINVAL;
}

/*-------------------------------------------------------- ioctl */

int px_ioctl (struct inode *inode, struct file *filp,
                 unsigned int cmd, unsigned long arg)
{
    Px_Dev *dev = filp->private_data;
    u32 tmp;
    int size = _IOC_SIZE(cmd); /* the size bitfield in cmd */
    int i, retval = 0; /* success by default */
    #define IOC_BUFSIZE 128 /* some space, just random */
    static unsigned char localbuf[IOC_BUFSIZE];
    /* I'm lazy, and I define "alias" names for the buffer */
    void *karg = localbuf;
    unsigned long *klong = karg;

    /*
     * extract the type and number bitfields, and don't decode
     * wrong cmds: return EINVAL before verify_area()
     */
    if (_IOC_TYPE(cmd) != PX_IOC_MAGIC) return -EINVAL;
    if (_IOC_NR(cmd) > PX_IOC_MAXNR) return -EINVAL;

    /*
     * the direction is a bitmask, and VERIFY_WRITE catches R/W
     * transfers. `Dir' is user-oriented, while
     * verify_area is kernel-oriented, so the concept of "read" and
     * "write" is reversed
     */
    if (_IOC_DIR(cmd) & _IOC_READ) {
        if (!access_ok(VERIFY_WRITE, (void *)arg, size))
            return -EFAULT;
    }
    else if (_IOC_DIR(cmd) & _IOC_WRITE) {
        if (!access_ok(VERIFY_READ, (void *)arg, size))
            return -EFAULT;
    }

    PDEBUGG("ioctl: %08x %08lx: nr %i, size %i, dir %i\n",
            cmd, arg, _IOC_NR(cmd), _IOC_SIZE(cmd), _IOC_DIR(cmd));

    /* first, retrieve data from userspace */
    if ((_IOC_DIR(cmd) & _IOC_WRITE) && (size <= IOC_BUFSIZE))
        copy_from_user(karg, (void *)arg, size);

    /* and now the big switch... */
    switch(cmd) {

      case PX_IOCHARDRESET:
        while (MOD_IN_USE)
            MOD_DEC_USE_COUNT;
        MOD_INC_USE_COUNT;
        dev->usage = 1; /* fall through */
      case PX_IOCRESET:
        px_shutdown(dev);
        px_bringup(dev);
        break;

      case PX_IOCGFLAGS:
        *klong = dev->flags;
        break;
      case PX_IOCSFLAGS:         
	dev->flags = (dev->flags & ~PX_USER_FLAGS) |
	    (*klong & PX_USER_FLAGS);
        break;

      case PX_IOCGIRQCOUNT:
        *klong = dev->irq_count;
        break;

	/* get DMA information */
      case PX_IOCGDMASIZE:
        *klong = dev->dmasize;
        break;
      case PX_IOCGDMABUF:
        *klong = dev->dmabuffer;
        break;

	/* get RISC program information */
      case PX_IOCGRISCADDE:
        *klong = dev->dmaprogramE;
        break;
      case PX_IOCGRISCADDO:
        *klong = dev->dmaprogramO;
        break;
      case PX_IOCGPROGRAME:
	if (!dev->dmaprogramE) 
	    return -EAGAIN;
        retval = copy_to_user((void *)arg, (void *)dev->dmaprogramE, size);
        return retval;
      case PX_IOCGPROGRAMO:
	if (!dev->dmaprogramO) 
	    return -EAGAIN;
        retval = copy_to_user((void *)arg, (void *)dev->dmaprogramO, size);
        return retval;

	/* reference voltage */
      case PX_IOCGREFV:
	if (dev->flags & PX_FLAG_848ONLY)
	    return -ENOSYS;
        *klong = dev->refv;
        break;
      case PX_IOCSREFV:
	if (dev->flags & PX_FLAG_848ONLY)
	    return -ENOSYS;
        if (*klong & ~0xff)
            return -EINVAL;
        dev->refv = *klong;
        return px_set_refv(dev);

	/* input multiplexer (for multiple cameras) */
      case PX_IOCGMUX:
        *klong = (PX_READ32(dev, BT848_IFORM) & BT848_IFORM_MUXSEL)
	  >> BT848_IFORM_MUXSEL_S;
        /* ok, now turn 0,1,2,3 to 3,2,0,1 */
        if (*klong & 2) *klong ^= 2;
        else            *klong ^= 3;
        break;
      case PX_IOCSMUX:         
        tmp = PX_READ32(dev, BT848_IFORM) & ~BT848_IFORM_MUXSEL;
        /* ok, now turn 0,1,2,3 to 2,3,1,0 */
        if (*klong & 2) *klong ^= 3;
        else            *klong ^= 2;
	PX_WRITE32(dev, tmp | (*klong << BT848_IFORM_MUXSEL_S), BT848_IFORM);
        break;

	/* set the trig-related flags */
    case PX_IOCSTRIG:
	if (*klong & ~PX_TRIG_FLAGS) return -EINVAL;
	dev->flags = (dev->flags & ~PX_TRIG_FLAGS) | *klong;
	/* setup and disable active grabs */
	retval = px_prepare_device(dev, GFP_KERNEL);
	break;

	/* acquisition length */
    case PX_IOCSACQLEN:
	if (dev->flags & PX_FLAG_HIRES)
	    *klong *= 2; /* count fields */
	if (*klong > 255) return -EINVAL;
	dev->prglen = (u8)(*klong);
	retval = px_prepare_device(dev, GFP_KERNEL);
	break;
    case PX_IOCGACQLEN:
	*klong = (unsigned long)dev->prglen;
	if (dev->flags & PX_FLAG_HIRES)
	    *klong /= 2; /* return frames */
	break;

    case PX_IOCACQNOW:
	retval = px_acq_on(dev);
	break;

    case PX_IOCWAITVB:
	if (!(dev->flags & PX_FLAG_RUNNING))
	    return -EAGAIN; /* nothing to wait for */

	i = *klong; /* easier name... */
	if (dev->prglen) { 
	    if (dev->flags & PX_FLAG_HIRES)
		i *= 2; /* count frames, not fields */
	    while (dev->prgcurr < i) {
		interruptible_sleep_on(&dev->queue);
		if (current->signal & ~current->blocked) /* a signal arrived */
		    return -ERESTARTSYS; /* tell the fs layer to handle it */
	    }
	    i = dev->prgcurr;
	    if (dev->flags & PX_FLAG_HIRES)
		i /= 2; /* count frames, not fields */
	    *klong = i; /* return to caller */
	    break;
	}

	/* else, this is continous grabbing */
	if (!i) i++;
	while (i) {
	    /* wait for a VB */
	    interruptible_sleep_on(&dev->queue);
	    if (current->signal & ~current->blocked) /* a signal arrived */
		return -ERESTARTSYS; /* tell the fs layer to handle it */
	    i--;
	}
	break;

    case PX_IOCSEQUENCE:
	if (!klong) { /* Null pointer, release the sequence */
	    return px_release_sequence(dev, filp);
	}
	return px_new_sequence(dev, filp, (struct Px_AcqControl *)localbuf);

    case PX_IOCGHWOVERRUN:
        *klong = dev->hwoverrun;
	break;
    case PX_IOCGSWOVERRUN:
        *klong = dev->seq_overrun;
	break;

    case PX_IOCGWHOLEDEVICE:
        copy_to_user((void *)arg, dev, size);
	return 0;
	break;

    case PX_IOCGBRIGHT:
        *klong = PX_READ32(dev, BT848_BRIGHT);
        break;
    case PX_IOCSBRIGHT:
	if (*klong & ~0xff) return -EINVAL;
        PX_WRITE32(dev, *klong & 0x0ff, BT848_BRIGHT);
	break;

    case PX_IOCGCONTRAST:
	*klong = PX_READ32(dev, BT848_CONTRAST_LO) | 

	  (( PX_READ32(dev, BT848_E_CONTROL) & BT848_CONTROL_CON_MSB) << 6);
	break;
    case PX_IOCSCONTRAST:
	if (*klong & ~0x1ff) return -EINVAL;
	PX_WRITE32(dev, *klong & 0x0ff, BT848_CONTRAST_LO);
	tmp = PX_READ32(dev, BT848_E_CONTROL);
	tmp &= ~BT848_CONTROL_CON_MSB;
	tmp |= ( *klong & 0x0100 ) >> 6;
	PX_WRITE32(dev, tmp, BT848_E_CONTROL);
	PX_WRITE32(dev, tmp, BT848_O_CONTROL);      
	break;

    case PX_IOCGHUE:
	*klong = PX_READ32(dev, BT848_HUE);
	break;
    case PX_IOCSHUE:
        if (*klong & ~0xff) return -EINVAL;
	PX_WRITE32(dev, *klong & 0x0ff, BT848_HUE);
	break;

    case PX_IOCGSATU:
	*klong = PX_READ32(dev, BT848_SAT_U_LO) | 
	  (( PX_READ32(dev, BT848_E_CONTROL) & BT848_CONTROL_SAT_U_MSB) << 7);
	break;
    case PX_IOCSSATU:
        if (*klong & ~0x1ff) return -EINVAL;
	PX_WRITE32(dev, *klong & 0x0ff, BT848_SAT_U_LO);
	tmp = PX_READ32(dev, BT848_E_CONTROL);
	tmp &= ~BT848_CONTROL_SAT_U_MSB;
	tmp |= ( *klong & 0x0100 ) >> 7;
	PX_WRITE32(dev, tmp, BT848_E_CONTROL);
	PX_WRITE32(dev, tmp, BT848_O_CONTROL);      
	break;
    case PX_IOCGSATV:
	*klong = PX_READ32(dev, BT848_SAT_V_LO) | 
	  (( PX_READ32(dev, BT848_E_CONTROL) & BT848_CONTROL_SAT_V_MSB) << 8);
	break;
    case PX_IOCSSATV:
        if (*klong & ~0x1ff) return -EINVAL;
	PX_WRITE32(dev, *klong & 0x0ff, BT848_SAT_V_LO);
	tmp = PX_READ32(dev, BT848_E_CONTROL);
	tmp &= ~BT848_CONTROL_SAT_V_MSB;
	tmp |= ( *klong & 0x0100 ) >> 8;
	PX_WRITE32(dev, tmp, BT848_E_CONTROL);
	PX_WRITE32(dev, tmp, BT848_O_CONTROL);      
	break;

    default:
	PDEBUG("ioctl: invalid\n");
        return -EINVAL;
    }
    /* finally, copy data to user space and return */
    if (retval < 0) return retval;
    if ((_IOC_DIR(cmd) & _IOC_READ) && (size <= IOC_BUFSIZE))
        copy_to_user((void *)arg, karg, size);
    return retval; /* sometimes, positive is what I want */
}

/*-------------------------------------------------------- vma ops */

/*
 * the open and close methods are needed to keep track
 * of the usage count, nopage to accept mremap() calls
 */
void px_vma_open(struct vm_area_struct * area)
{
    MOD_INC_USE_COUNT;
}

void px_vma_close(struct vm_area_struct * area)
{
    MOD_DEC_USE_COUNT;
}

#if 0
unsigned long px_vma_nopage(struct vm_area_struct *vma,
                            unsigned long address, int write)
{
    pgd_t *pgd; pmd_t *pmd; pte_t *pte;

    /* int remap_page_range(virt_add, phys_add, size, protection); */
    remap_page_range(address & PAGE_MASK,
                     address - vma->vm_start + vma->vm_offset,
                     PAGE_SIZE, vma->vm_page_prot);
    /* now return its address to the caller */
    pgd = pgd_offset(current->mm, address);
    pmd = pmd_offset(pgd, page);
    pte = pte_offset(pmd, page);
    return pte_page(*pte);        /* this is the physical address */
}
#else
static unsigned long simple_pedantic_nopage(struct vm_area_struct *vma,
				     unsigned long address, int write_access)
{ return 0; /* send a SIGBUS */}
#endif
 

static struct vm_operations_struct px_vmops = {
   px_vma_open,
   px_vma_close,
   NULL,          /* unmap */
   NULL,          /* protect */
   NULL,          /* sync */
   NULL,          /* advise */
   simple_pedantic_nopage, /* px_vma_nopage */
};




/*-------------------------------------------------------- mmap */

int px_mmap(struct inode *inode, struct file *filp,
            struct vm_area_struct *vma)
{
    Px_Dev *dev = filp->private_data;

    unsigned long offset = vma->vm_offset;
    unsigned long physical = dev->dmabuffer + offset + PAGE_OFFSET;
    unsigned long vsize = vma->vm_end - vma->vm_start;
    unsigned long psize = dev->dmasize - offset;

    psize = (psize + PAGE_SIZE - 1) & PAGE_MASK;
    PDEBUG("mmap: psize: %li -> %li\n",dev->dmasize - offset, psize);

    if (offset & (PAGE_SIZE-1))
        return -ENXIO; /* need aligned offsets */
    if (offset > dev->dmasize || vsize > psize)
        return -EINVAL; /*  spans too high */
    if (vma->vm_ops)
            return -EINVAL; /* Hmm... shouldn't happen */

    /* int remap_page_range(virt_add, phys_add, size, protection); */
    if (remap_page_range(vma->vm_start, physical, vsize, vma->vm_page_prot))
        return -EAGAIN;

    vma->vm_ops = &px_vmops;
    px_vma_open(vma); /* housekeeping */
    vma->vm_inode = inode;
    inode->i_count++;

    /* don't return unless it's ready */
    if (dev->isup < PX_UPTIME_MIN) {
        PDEBUG("mmap: sleep\n");
        interruptible_sleep_on(&dev->queue);
    }
    return 0;
}


/*-------------------------------------------------------- the fops */

/* The binary node: b/w, bgr, rgb */
struct file_operations px_bin_fops = {
    px_lseek,
    px_read,
    px_write,
    NULL,          /* px_readdir */
    NULL,          /* px_select */
    px_ioctl,
    px_mmap,
    px_open,
    px_close,
    NULL,          /* px_fsync */
    NULL,          /* px_fasync */
                   /* nothing more, fill with NULLs */
};

/* The pnm node, used for pgm and ppm */
struct file_operations px_pnm_fops = {
    px_lseek,
    px_pnm_read,
    px_write,
    NULL,          /* px_readdir */
    NULL,          /* px_select */
    px_ioctl,
    px_mmap,
    px_open,
    px_close,
    NULL,          /* px_fsync */
    NULL,          /* px_fasync */
                   /* nothing more, fill with NULLs */
};

/* The control node does no DMA nor read/write (read returns control data) */
struct file_operations px_ctl_fops = {
    px_lseek,
    px_ctl_read,
    NULL,          /* no write */
    NULL,          /* px_readdir */
    NULL,          /* px_select */
    px_ioctl,
    NULL,          /* px_mmap */
    px_open,
    px_close,
    NULL,          /* nothing more... */
};

/*-------------------------------------------------------- modules */
/*
 * Finally, the module stuff. A free function, then init and cleanup
 */

void px_freethem(void)
{
    Px_Dev *ptr;

    for (ptr = px_devices; ptr; ) {
	ptr->flags = 0; /* Clear "persist", to allow shutdown */
	px_shutdown(ptr);
	px_free_dma(ptr);
	px_free_program(ptr);
	if (ptr->regs)
	  iounmap((void *)(ptr->regs));
        ptr = ptr->next;
        kfree(px_devices);
        px_devices = ptr;
    }
}

int init_module(void)
{
    int retval, i;
    struct Px_Dev *ptr=NULL, **ptrptr;

    /*
     * This will lead to an insmod error if CONFIG_PCI was not set in the
     * target machine.
     */
    if (!pcibios_present()) {
        printk(KERN_ERR PX_MSG "PCI bios needed: can't install\n");
        return -ENODEV;
    }

    /*
     * First of all allocate the device structures
     */

    for (i=0, ptrptr = &px_devices; ; i++) {
        u8 bus, device;

	/* look for the grabber */
        retval = pcibios_find_device(PCI_VENDOR_ID_BROOKTREE, 848,
				     i, &bus, &device);
	if (retval) {
	    if (retval == PCIBIOS_DEVICE_NOT_FOUND) {
	        retval = 0; break; /* this is good */
	    } 
	    printk(KERN_ERR PX_MSG "pcibios: %s\n",
		     pcibios_strerror(retval));
	    retval = -EIO;
	    break;
	}

	/* so, one is there */	
        PDEBUG("found at bus %i device %i\n",bus,device);

        retval = -ENOMEM;
        ptr = kmalloc(sizeof(Px_Dev), GFP_KERNEL);
        if (!ptr)
            goto fail_alloc;
        *ptrptr = ptr; /* the next pointer */
        memset(ptr,0,sizeof(Px_Dev));
	ptr->bus = bus;
	ptr->device = device;
        pcibios_read_config_dword(bus, device, 0x10, &ptr->phys_add);
	ptr->phys_add &= PCI_BASE_ADDRESS_MEM_MASK;

	if ( (retval = px_init(ptr)) ) {
	    if (retval == -ENODEV) {
                /* Not a frame grabber */
	        kfree(ptr); *ptrptr=NULL;
	        continue; /* try the next one */
	    }
	    else {
                /* Other errors are fatal */
		break;
	    }
	}
	ptrptr = &(ptr->next); /* store here  the next ptr */
    }

    if (!px_devices) {
        printk(KERN_ERR PX_MSG "no device found\n");
        retval = -ENODEV;
	goto fail_alloc;
    }
       
    /*
     * Register your major, and accept a dynamic number
     */
    retval = register_chrdev(px_major, "pxc200", px_fop_array[0]);
    if (retval < 0) {
        printk(KERN_WARNING PX_MSG "can't get major %d\n",px_major);
	goto fail_alloc;
    }
    if (px_major == 0) px_major = retval; /* dynamic */

#ifndef PX_DEBUG
    register_symtab(NULL); /* otherwise, leave global symbols visible */
#endif

#ifdef PX_USE_PROC
    proc_register_dynamic(&proc_root, &px_proc_entry);
#endif

    retval = allocator_init();
    if (!retval) {
        printk(KERN_INFO PX_MSG "driver loaded\n");
        return 0; /* succeed */
    }
    /* else return error */
#ifdef PX_USE_PROC
    proc_unregister(&proc_root, px_proc_entry.low_ino);
#endif
    unregister_chrdev(px_major, "pxc200");

  fail_alloc:
    px_freethem();
    return retval;
}

void cleanup_module(void)
{
    unregister_chrdev(px_major, "pxc200");
    px_freethem();
    allocator_cleanup();
#ifdef PX_USE_PROC
    proc_unregister(&proc_root, px_proc_entry.low_ino);
#endif
    printk(KERN_INFO PX_MSG "driver unloaded\n");
}
