/***********************************************************************
 *
 * fastgrab.c
 *
 * Mike Montemerlo
 * Field Robotics Center
 * Carnegie Mellon University
 * 5/7/99
 *
 * This program displays video from a PXC200 framegrabber under X Windows
 * The program works under the following combinations of X color depth
 * and framegrabber color depth.
 *
 * FRAME GRABBER         X SERVER
 * 24 color              24
 * 16 color              16
 * 8 grayscale           24, 16, 8
 *
 * NOTE: framegrabber driver has to be hacked in order for the device
 * to actually generate 16 bit color data.
 *
 * The constant DRIVER_COLOR_DEPTH should be set to 24 or 16 to signify
 * whether you have hacked the driver or not.
 *
 * The constant MONOCHROME should be set to 1 if you want B/W images
 *
 * The constant HIGH_RESOLUTION should be set to 1 if you want full sized
 * images.
 *
 * This program was written using version 0.21 of the PXC200 driver.
 * In order to use this program you need the X-Shm extension. The program
 * will only clean up the shared memory if you CTL-C to kill it.
 *
 * Word of caution: In order to display the images at video rate, I write
 * directly to the XImage data structure. This is a no no, because the
 * underlying implementation could be different on different machines with
 * different X servers. You may have to play with this part of the code
 * to get this program working on your system.  It works on my pentium
 * running RedHat 5.2 and XFree86.
 *
 ***********************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <unistd.h>
#include <errno.h>
#include <signal.h>
#include <sys/mman.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <sys/timeb.h>
#include <sys/ioctl.h>
#include <sys/shm.h>
#include <sys/ipc.h>
#include <sys/shm.h>

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/extensions/XShm.h>
#include <X11/keysym.h>

#include "pxc200.h"

#ifdef LOW_COLOR
#define        DRIVER_COLOR_DEPTH 8
#else
#define        DRIVER_COLOR_DEPTH 24
#endif

#ifndef MONOCHROME
#define        MONOCHROME         0
#endif

#ifndef HIGH_RESOLUTION
#define        HIGH_RESOLUTION    1
#endif

#if            HIGH_RESOLUTION
#if            MONOCHROME
#define        PXC_DEVICE         "/dev/pxc0H"
#else
#define        PXC_DEVICE         "/dev/pxc0Hrgb"
#endif
#define        IMAGE_ROWS         480
#define        IMAGE_COLS         640
#else
#if            MONOCHROME
#define        PXC_DEVICE         "/dev/pxc0"
#else
#define        PXC_DEVICE         "/dev/pxc0rgb"
#endif
#define        IMAGE_ROWS         240
#define        IMAGE_COLS         320
#endif
#if            MONOCHROME
#define        DMA_BUFF_SIZE      IMAGE_ROWS*IMAGE_COLS
#else
#define        DMA_BUFF_SIZE      IMAGE_ROWS*IMAGE_COLS*DRIVER_COLOR_DEPTH/8
#endif

#define map_color_fast(x) ((int)((x)/256.0*ncolors))

static Display* display;
static GC gc;
static Window win;
int default_depth;
static XImage *shmimage;
static XShmSegmentInfo shminfo;
unsigned long color_list[256];
unsigned long mapped_color_list[256];
int ncolors;

int time_to_quit=0;

double get_time_ms(void)
{
  struct timeval tv;

  gettimeofday(&tv, NULL);
  return tv.tv_sec+tv.tv_usec/1000000.0;
}

int setup_display()
{
  XGCValues gc_values;
  int screen_num;
  XVisualInfo visual_info;
  Visual *default_visual;
  Colormap default_cmap;
  int i;
  XColor exact_defs[256];

  display = XOpenDisplay(NULL);
  win = XCreateSimpleWindow(display, DefaultRootWindow(display),
			       0, 0, IMAGE_COLS, IMAGE_ROWS, 2,
			       BlackPixel(display, DefaultScreen(display)),
			       BlackPixel(display, DefaultScreen(display))); 
  XMapWindow(display, win);
  gc_values.graphics_exposures = False;
  gc = XCreateGC(display, win, GCGraphicsExposures, &gc_values);  

  screen_num=DefaultScreen(display);
  default_visual=DefaultVisual(display, screen_num);
  default_depth=DefaultDepth(display, screen_num);
  if((default_depth!=DRIVER_COLOR_DEPTH) && !MONOCHROME) {
    fprintf(stderr, "Invalid color depth combination.\n");
    return -1;
  }
  if(default_depth==8) {
    default_cmap=DefaultColormap(display, screen_num);
    if(!XMatchVisualInfo(display, screen_num, default_depth,
    			 GrayScale, &visual_info)) {
      fprintf(stderr, "Couldn't find a visual (1).\n");
      return -1;
    }
    ncolors=256;
    while(ncolors>0 && !XAllocColorCells(display, default_cmap, False,
					 NULL, 0, color_list, ncolors))
      ncolors--;
    if(ncolors==0) {
      fprintf(stderr, "Couldn't allocate any colors!\n");
      exit(1);
    }
    for(i=0; i<ncolors; i++) {
      exact_defs[i].red=(65535/(ncolors-1))*i;
      exact_defs[i].green=(65535/(ncolors-1))*i;
      exact_defs[i].blue=(65535/(ncolors-1))*i;
      exact_defs[i].flags=DoRed|DoGreen|DoBlue;
      exact_defs[i].pixel=color_list[i];
    }
    XStoreColors(display, default_cmap, exact_defs, ncolors);
    for(i=0; i<256; i++)
      mapped_color_list[i]=color_list[map_color_fast(i)];
    printf("\n");
  }
  else {
    if(!XMatchVisualInfo(display, screen_num, default_depth,
			 TrueColor, &visual_info)) {
      fprintf(stderr, "Couldn't find a visual (2).\n");
      return -1;
    }
  }

  shmimage = XShmCreateImage(display, default_visual, default_depth,
			     ZPixmap, NULL, &shminfo, IMAGE_COLS,
			     IMAGE_ROWS);
  shminfo.shmid = shmget(IPC_PRIVATE,
			 shmimage->bytes_per_line * shmimage->height,
			 IPC_CREAT | 0777);
  shminfo.shmaddr = (char *) shmat(shminfo.shmid, (void *) 0, 0);
  shmimage->data = shminfo.shmaddr;
  XShmAttach(display, &shminfo);
  return 0;
}

void caught_ctlc(int x)
{
  time_to_quit=1;
}

int main(int argc, char **argv)
{
  FILE *fp;
  int fd, count, i;
  char pix5, pix6;
  void *address;
  unsigned long arg=0;
  double end_time, start_time;
  char *screen_buff, *fg_buff;
  
  if(!(fp=fopen(PXC_DEVICE, "r"))) {
    fprintf(stderr, "%s: %s: %s\n", argv[0], PXC_DEVICE, strerror(errno));
    exit(1);
  }
  fd=fileno(fp);
  address=mmap(0, DMA_BUFF_SIZE, PROT_READ, 
	       MAP_FILE | MAP_PRIVATE, fd, 0);
  if(address == (void *)-1) {
    fprintf(stderr, "%s: mmap(): %s\n", argv[0], strerror(errno));
    exit(1);
  }

  signal(SIGINT, caught_ctlc);
  if(setup_display()) {
    fprintf(stderr, "%s: setup_display() failed.\n", argv[0]);
    fclose(fp);
    exit(1);
  }
  screen_buff=shmimage->data;
  fg_buff=address;

  count=0;
  start_time=get_time_ms();
  while(!time_to_quit) {
    ioctl(fd, PX_IOCWAITVB, &arg);
    count++;

    /* DANGER : This is device dependent.  You may have to fool with this
       to make things work on your system. */

    if(MONOCHROME) {
      if(default_depth==24)
	for(i=0; i<IMAGE_ROWS*IMAGE_COLS; i++) {
	  screen_buff[i*4]=fg_buff[i];
	  screen_buff[i*4+1]=fg_buff[i];
	  screen_buff[i*4+2]=fg_buff[i];
	}
      else if(default_depth==16) {
	for(i=0; i<IMAGE_ROWS*IMAGE_COLS; i++) {
	  pix5=((unsigned char)fg_buff[i])>>3;
	  pix6=((unsigned char)fg_buff[i])>>2;
	  screen_buff[i*2]=((unsigned char)(pix5)|(pix6<<5));
	  screen_buff[i*2+1]=((unsigned char)(pix5<<3)|(pix6>>3));
	}
      }
      else if(default_depth==8) {
	for(i=0; i<IMAGE_ROWS*IMAGE_COLS; i++)
	  screen_buff[i]=(char)(mapped_color_list[(unsigned char)fg_buff[i]]);
      }
    }
    else {
      if(default_depth==24)
	for(i=0; i<IMAGE_ROWS*IMAGE_COLS; i++)
	  memcpy(&(screen_buff[i*4]), &(fg_buff[i*3]), 3);
      else if(default_depth==16)
	memcpy(screen_buff, fg_buff, DMA_BUFF_SIZE);
    }

    /* END OF DANGER */

    XShmPutImage(display, win, gc, shmimage,
		 0, 0, 0, 0, IMAGE_COLS, IMAGE_ROWS, False);
    XSync(display, 0);
  }
  end_time=get_time_ms();
  printf("Grabbed %d images in %f seconds. (%f im/sec)\n", count, end_time-
	 start_time, count/(end_time-start_time));

  XShmDetach(display, &shminfo);
  XDestroyImage(shmimage);
  shmdt(shminfo.shmaddr);
  shmctl(shminfo.shmid, IPC_RMID, NULL);
  if(munmap(0, DMA_BUFF_SIZE)<0)
    fprintf(stderr, "%s: munmap(): %s\n", argv[0], strerror(errno));
  fclose(fp);
  return 0;
}



