/*************************************************************************
*****               PTU BINARY TEST DRIVER CODE FILE                 ****/
#define TEST_CODE_VERSION       "v1.09.8"
/****            (C)1995-1999, Directed Perception, Inc.             *****
*****                     All Rights Reserved.                       *****
*****                                                                *****
*****   Licensed users may freely distribute compiled code including *****
*****   this code and data. Source data and code may NOT be          *****
*****   distributed without the prior written consent from           *****
*****   Directed Perception, Inc.                                    *****
*****      Directed Perception, Inc. reserves the right to make      *****
*****   changes without further notice to any content herein to      *****
*****   improve reliability, function or design. Directed Perception *****
*****   shall not assume any liability arising from the application  *****
*****   or use of this code, data or function.                       *****
*****                                                                *****
**************************************************************************

CHANGE HISTORY:                             
    1/22/98:    v1.09.08   Added linuxser.c driver and made required small changes
                           to test.c for differences between Win/LINUX serial port
			   naming.
    9/23/98:    v1.09.07   Added speed_control_modes.
    7/15/98:    v1.08.02   Fixed SerialBytesOut to return TRUE/FALSE for DOS/Win16
    7/1/97:     v1.08.00   Made improvements to tests and feedback. 
    1/25/96:    v1.07.08d. Made correction to serial COM port selection
    2/19/95:    v1.07.04d. Generalized for Windows, DOS, & UNIX. Added networking. 
    10/12/94:	v1.07.03d. Pre-release working version.
	 		   XON/XOFF removed from PTU firmware to allow for binary mode.


**************************************************************************/

//#include <windows.h>

// #include <dos.h>
//#include <conio.h>
#include <stdio.h>      
#include <ctype.h>
//#include <string.h>
#include <stdlib.h>
//#include <time.h>
// #include <dir.h>
#ifdef _UNIX
#include "ptu.h"
#else
#include "..\include\ptu.h"
#endif


/********** MAIN PROGRAM *********/
#define TEST_ERROR			-1

unsigned short int uval, uval1, uval2, uval_in;
signed short int val, val1, val2, val3, val4, val_in;
long lval, lval_in, lval1, lval2;
char status, cval, cval_in;
char tmpChar;
  

char return_error ()
{ printf("(Enter 'c' to continue): ");
  tmpChar = 'f';
  while (tmpChar != 'c')
	    tmpChar = tolower(getchar());
  return(TEST_ERROR);
}


char return_error_status (char *failed_binary_op, char return_status)
{ printf("! %s failed with status %d\n", status);
  return ( return_error() );
}




char test_verbose_modes(void)
{   	/* test the ASCII verbose modes */

	 if ((status = set_mode(ASCII_VERBOSE_MODE,TERSE)) == TRUE)
		 { return (  return_error_status( "SET_VERBOSE_ASCII_OFF", status) );
				}
	 else printf("\nSET_VERBOSE_ASCII_OFF executed\n");
	 if ( (status = set_mode(ASCII_VERBOSE_MODE,QUERY_MODE)) != TERSE )
		 { return (  return_error_status( "SET_VERBOSE_ASCII_OFF", status) );
				}
	 else printf("SET_VERBOSE_ASCII_OFF verified\n");

	 if ((status = set_mode(ASCII_VERBOSE_MODE,VERBOSE))  == TRUE)
		 { return (  return_error_status( "SET_VERBOSE_ASCII_ON", status) );
				}
	 else printf("\nSET_VERBOSE_ASCII_ON executed\n");
	 if ( (status = set_mode(ASCII_VERBOSE_MODE,QUERY_MODE)) != VERBOSE )
		 { return (  return_error_status( "SET_VERBOSE_ASCII_ON", status) );
				}
	 else printf("SET_VERBOSE_ASCII_ON verified\n");
	 return(PTU_OK);
}

char test_ASCII_echo_modes(void)
{	/* test the ASCII echo modes */
	 if ((status = set_mode(ASCII_ECHO_MODE,OFF_MODE)) == TRUE)
		 { return (  return_error_status( "DISABLE_ECHO", status) );
				}
	 else printf("\nDISABLE_ECHO executed\n");
	 if ( (status = set_mode(ASCII_ECHO_MODE,QUERY_MODE)) != OFF_MODE )
		 { return (  return_error_status( "DISABLE_ECHO", status) );
				}
	 else printf("DISABLE_ECHO verified\n");

	 if ((status = set_mode(ASCII_ECHO_MODE,ON_MODE)) == TRUE)
		 { return (  return_error_status( "ENABLE_ECHO", status) );
				}
	 else printf("ENABLE_ECHO executed\n");
	 if ( (status = set_mode(ASCII_ECHO_MODE,QUERY_MODE)) != ON_MODE )
		 { return (  return_error_status( "ENABLE_ECHO", status) );
				}
	 else printf("ENABLE_ECHO verified\n");
	 return(PTU_OK);
}


char test_position_limits(void)
{   /* test the position limits modes */

	 if ((status = set_mode(POSITION_LIMITS_MODE,OFF_MODE)) == TRUE)
		 { return (  return_error_status( "DISABLE_POSITION_LIMITS", status) );
				}
	 else printf("\nDISABLE_POSITION_LIMITS executed\n");
	 if ( (status = set_mode(POSITION_LIMITS_MODE,QUERY_MODE)) != OFF_MODE )
		 { return (  return_error_status( "DISABLE_POSITION_LIMITS", status) );
				}
	 else printf("DISABLE_POSITION_LIMITS verified\n");

	 if ((status = set_mode(POSITION_LIMITS_MODE,ON_MODE)) == TRUE)
		 { return (  return_error_status( "ENABLE_POSITION_LIMITS", status) );
				}
	 else printf("ENABLE_POSITION_LIMITS executed\n");
	 if ( (status = set_mode(POSITION_LIMITS_MODE,QUERY_MODE)) != ON_MODE )
		 { return (  return_error_status( "ENABLE_POSITION_LIMITS", status) );
				}
	 else printf("ENABLE_POSITION_LIMITS verified\n");
	 return(PTU_OK);
}


char test_speed_commands(void)
{	 
	 halt(ALL); do_delay(1000);

     /* verify relative speed queries return the right value */
	 lval_in = get_current(PAN,SPEED);
	 if (lval_in != 0)
		 { printf("PAN_CURRENT_SPEED_QUERY failed: (%ld != 0)\n", lval_in);
			return ( return_error() );		}
	 printf("\nPAN_CURRENT_SPEED_QUERY issued and verified\n");

	 lval_in = get_current(TILT,SPEED);
	 if (lval_in != 0)
		 { printf("TILT_CURRENT_SPEED_QUERY failed: (%ld != 0)\n", lval_in);
			return ( return_error() );		}
	 printf("TILT_CURRENT_SPEED_QUERY issued and verified\n");


	 /* absolute speed command tests */
	 uval = 2500;

	 if ((status = set_desired(PAN, SPEED, (PTU_PARM_PTR *) &uval, ABSOLUTE)) == TRUE)
		 { return (  return_error_status( "PAN_SET_ABS_SPEED", status) );		}
	 else printf("\nPAN_SET_ABS_SPEED executed\n");
	 lval_in = get_desired(PAN,SPEED);
	 if (lval_in != (long) uval)
		 { printf("PAN_DESIRED_SPEED_QUERY failed: (set,get)=(%u,%ld)\n", uval, lval_in);
			return ( return_error() );		}
	 printf("PAN_DESIRED_SPEED_QUERY issued and verified\n");

	 if ((status = set_desired(TILT, SPEED, (PTU_PARM_PTR *) &uval, ABSOLUTE))  == TRUE)
		 { return (  return_error_status( "TILT_SET_ABS_SPEED", status) ); 		}
	 else printf("TILT_SET_ABS_SPEED executed\n");
	 lval_in = get_desired(TILT,SPEED);
	 if (lval_in != (long) uval)
		 { printf("TILT_DESIRED_SPEED_QUERY failed: (set,get)=(%u,%ld)\n", uval, lval_in);
			return ( return_error() );		}
	 printf("TILT_DESIRED_SPEED_QUERY issued and verified\n");


	 /* relative speed command tests (when current_speed=0) */
	 halt(ALL); do_delay(1000);
	 val = uval = 2000;
     
	 /* relative pan speed command test */
	 lval_in = get_current(PAN,SPEED);
	 if ((status = set_desired(PAN, SPEED, (PTU_PARM_PTR *) &val, RELATIVE)) == TRUE)
		  return (  return_error_status( "PAN_SET_REL_SPEED", status) );
	 else printf("\nPAN_SET_REL_SPEED executed\n");
	 /* relative tilt speed command test */
	 lval2 = get_desired(PAN,SPEED);
	 if (lval2 != (lval_in + val))
		  return ( return_error_status( "PAN_DESIRED_SPEED_QUERY", -1) );
	 else printf("PAN_DESIRED_SPEED_QUERY issued and verified\n");

	 /* query current tilt speed test */
	 lval_in = get_current(TILT,SPEED);
	 if ((status = set_desired(TILT, SPEED, (PTU_PARM_PTR *) &val, RELATIVE)) == TRUE)
		  return (  return_error_status( "TILT_SET_REL_SPEED", status) );
	 else printf("TILT_SET_REL_SPEED executed\n");

	 /* query current pan speed test */
	 lval2 = get_desired(TILT,SPEED);
	 if (lval2 != (lval_in + val))
		  return ( return_error_status("TILT_DESIRED_SPEED_QUERY", -1) );
	 else printf("TILT_DESIRED_SPEED_QUERY issued and verified\n");

	 return(PTU_OK);
}




char test_position_commands(void)
{ 	 /* pan position command */
	 val = -2500;

	 if ((status = set_desired(PAN, POSITION, (PTU_PARM_PTR *) &val, ABSOLUTE)) == TRUE)
		 { return (  return_error_status( "PAN_SET_ABS_POSITION", status) );
				}
	 else printf("\nPAN_SET_ABS_POSITION executed\n");

     if ((status = await_completion()) == PTU_OK)
          printf("AWAIT_COMPLETION executed (%d)\n", status);
	 else return (  return_error_status( "AWAIT_COMPLETION", status) );

	 lval_in = get_current(PAN,POSITION);
	 if (lval_in != val)
		 { printf("PAN_CURRENT_POSITION_QUERY failed: (%ld != %d)\n", lval_in, val);
			return ( return_error() );		}
	 printf("\nPAN_CURRENT_POSITION_QUERY issued and verified\n");

	 lval_in = get_desired(PAN,POSITION);
	 if (lval_in != val)
		 { printf("PAN_DESIRED_POSITION_QUERY failed: (%ld != %d)\n", lval_in, val);
			return ( return_error() );		}
	 printf("\nPAN_DESIRED_POSITION_QUERY issued and verified\n");


	 val = -val;
	 if ((status = set_desired(PAN, POSITION, (PTU_PARM_PTR *) &val, RELATIVE)) == TRUE)
		 { return (  return_error_status( "PAN_SET_REL_POSITION", status) );
				}
	 else printf("PAN_SET_REL_POSITION executed\n");

     if ((status = await_completion()) == PTU_OK)
          printf("AWAIT_COMPLETION executed (%d)\n", status);
	 else return (  return_error_status( "AWAIT_COMPLETION", status) );

	 /* tilt position command */
	 val = -600;

	 if ((status = set_desired(TILT, POSITION, (PTU_PARM_PTR *) &val, ABSOLUTE)) == TRUE)
		 { return (  return_error_status( "TILT_SET_ABS_POSITION", status) );
				}
	 else printf("\nTILT_SET_ABS_POSITION executed\n");

     if ((status = await_completion()) == PTU_OK)
          printf("AWAIT_COMPLETION executed (%d)\n", status);
	 else return (  return_error_status( "AWAIT_COMPLETION", status) );

	 lval_in = get_current(TILT,POSITION);
	 if (lval_in != val)
		 { printf("TILT_CURRENT_POSITION_QUERY failed: (%ld != %d)\n", lval_in, val);
			return ( return_error() );		}
	 printf("\nTILT_CURRENT_POSITION_QUERY issued and verified\n");

	 lval_in = get_desired(TILT,POSITION);
	 if (lval_in != val)
		 { printf("TILT_DESIRED_POSITION_QUERY failed: (%ld != %d)\n", lval_in, val);
			return ( return_error() );		}
	 printf("\nTILT_DESIRED_POSITION_QUERY issued and verified\n");

	 val = -val;
	 if ((status = set_desired(TILT, POSITION, (PTU_PARM_PTR *) &val, RELATIVE)) == TRUE)
		 { return (  return_error_status( "TILT_SET_ABS_POSITION", status) );
				}
	 else printf("TILT_SET_ABS_POSITION executed\n");

     if ((status = await_completion()) == PTU_OK)
          printf("AWAIT_COMPLETION executed (%d)\n", status);
	 else return (  return_error_status( "AWAIT_COMPLETION", status) );

	 return(PTU_OK);
}


char test_power_modes(void)
{   /* test power modes */
	 cval = PTU_LOW_POWER;

	 set_desired(PAN, HOLD_POWER_LEVEL, (PTU_PARM_PTR *) &cval, ABSOLUTE);
	 printf("\nPAN_HOLD_POWER_LEVEL set and query executed and verified\n");
	 lval_in = get_current(PAN, HOLD_POWER_LEVEL);
	 set_desired(PAN, HOLD_POWER_LEVEL, (PTU_PARM_PTR *) &cval, RELATIVE);
	 lval2   = get_desired(PAN, HOLD_POWER_LEVEL);
	 if ( (lval_in != cval) || (lval2 != cval) )
		 { printf("PAN_HOLD_POWER_LEVEL set and query failed: (set,get)=(%d,%ld)\n", cval, lval_in);
			return ( return_error() );		}

	 set_desired(TILT, HOLD_POWER_LEVEL, (PTU_PARM_PTR *) &cval, ABSOLUTE);
	 printf("TILT_HOLD_POWER_LEVEL set and query executed and verified\n");
	 lval_in = get_current(TILT, HOLD_POWER_LEVEL);
	 set_desired(TILT, HOLD_POWER_LEVEL, (PTU_PARM_PTR *) &cval, RELATIVE);
	 lval2   = get_desired(TILT, HOLD_POWER_LEVEL);
	 if ( (lval_in != cval) || (lval2 != cval) )
		 { printf("TILT_HOLD_POWER_LEVEL set and query failed: (set,get)=(%d,%ld)\n", cval, lval_in);
			return ( return_error() );		}

	 set_desired(PAN, MOVE_POWER_LEVEL, (PTU_PARM_PTR *) &cval, ABSOLUTE);
	 printf("PAN_MOVE_POWER_LEVEL set and query executed and verified\n");
	 lval_in = get_current(PAN, MOVE_POWER_LEVEL);
	 set_desired(PAN, MOVE_POWER_LEVEL, (PTU_PARM_PTR *) &cval, RELATIVE);
	 lval2   = get_desired(PAN, MOVE_POWER_LEVEL);
	 if ( (lval_in != cval) || (lval2 != cval) )
		 { printf("PAN_MOVE_POWER_LEVEL set and query failed: (set,get)=(%d,%ld)\n", cval, lval_in);
			return ( return_error() );		}

	 set_desired(TILT, MOVE_POWER_LEVEL, (PTU_PARM_PTR *) &cval, ABSOLUTE);
	 printf("TILT_MOVE_POWER_LEVEL set and query executed and verified\n");
	 lval_in = get_current(TILT, MOVE_POWER_LEVEL);
	 set_desired(TILT, MOVE_POWER_LEVEL, (PTU_PARM_PTR *) &cval, RELATIVE);
	 lval2   = get_desired(TILT, MOVE_POWER_LEVEL);
	 if ( (lval_in != cval) || (lval2 != cval) )
		 { printf("TILT_MOVE_POWER_LEVEL set and query failed: (set,get)=(%d,%ld)\n", cval, lval_in);
			return ( return_error() );		}

	 return(PTU_OK);
}


char test_base_speed_commands(void)
{ 	/* PAN set base testing */
	uval = 999;

	 if ((status = set_desired(PAN, BASE, (PTU_PARM_PTR *) &uval, ABSOLUTE)) == TRUE)
		 { return (  return_error_status( "PAN_SET_BASE_SPEED", status) );
				}
	 else printf("\nPAN_SET_BASE_SPEED executed\n");

	 lval_in = get_desired(PAN,BASE);
	 if (lval_in != (long) uval)
		 { printf("PAN_DESIRED_BASE_QUERY failed: (set,get)=(%u,%ld)\n", uval, lval_in);
			return ( return_error() );		}
	 printf("PAN_DESIRED_BASE_QUERY issued and verified\n");

	 lval_in = get_current(PAN,BASE);
	 if (lval_in != (long) uval)
		 { printf("PAN_CURRENT_BASE_QUERY failed: (set,get)=(%u,%ld)\n", uval, lval_in);
			return ( return_error() );		}
	 printf("PAN_CURRENT_BASE_QUERY issued and verified\n");


	/* tilt set base testing */
	uval = 1111;

	 if ((status = set_desired(TILT, BASE, (PTU_PARM_PTR *) &uval, RELATIVE)) == TRUE)
		 { return (  return_error_status( "TILT_SET_BASE_SPEED", status) );
				}
	 else printf("\nTILT_SET_BASE_SPEED executed\n");

	 lval_in = get_desired(TILT,BASE);
	 if (lval_in != (long) uval)
		 { printf("TILT_DESIRED_BASE_QUERY failed: (set,get)=(%u,%ld)\n", uval, lval_in);
			return ( return_error() );		}
	 printf("TILT_DESIRED_BASE_QUERY issued and verified\n");

	 lval_in = get_current(TILT,BASE);
	 if (lval_in != (long) uval)
		 { printf("TILT_CURRENT_BASE_QUERY failed: (set,get)=(%u,%ld)\n", uval, lval_in);
			return ( return_error() );		}
	 printf("TILT_CURRENT_BASE_QUERY issued and verified\n");

	 return(PTU_OK);
}




char test_speed_limits(void)
{ 	/* pan set lower speed limit testing */
	uval = 57;

	 if ((status = set_desired(PAN, LOWER_SPEED_LIMIT, (PTU_PARM_PTR *) &uval, ABSOLUTE)) == TRUE)
		 { return (  return_error_status( "PAN_SET_LOWER_SPEED_LIMIT", status) );
				}
	 else printf("\nPAN_SET_LOWER_SPEED_LIMIT executed\n");

	 lval_in = get_desired(PAN,LOWER_SPEED_LIMIT);
	 if (lval_in != (long) uval)
		 { printf("PAN_LOWER_SPEED_QUERY failed: (set,get)=(%u,%ld)\n", uval, lval_in);
			return ( return_error() );		}
	 printf("PAN_LOWER_SPEED_QUERY issued and verified\n");

	 if ((status = set_desired(PAN, LOWER_SPEED_LIMIT, (PTU_PARM_PTR *) &uval, RELATIVE)) == TRUE)
		 { return (  return_error_status( "PAN_SET_LOWER_SPEED_LIMIT", status) );
				}
	 else printf("PAN_SET_LOWER_SPEED_LIMIT executed\n");

	 lval_in = get_current(PAN,LOWER_SPEED_LIMIT);
	 if (lval_in != (long) uval)
		 { printf("PAN_LOWER_SPEED_QUERY failed: (set,get)=(%u,%ld)\n", uval, lval_in);
			return ( return_error() );		}
	 printf("PAN_LOWER_SPEED_QUERY issued and verified\n");


	/* tilt set lower speed limit testing */
	uval = 58;

	 if ((status = set_desired(TILT, LOWER_SPEED_LIMIT, (PTU_PARM_PTR *) &uval, ABSOLUTE)) == TRUE)
		 { return (  return_error_status( "TILT_SET_LOWER_SPEED_LIMIT", status) );
				}
	 else printf("\nTILT_SET_LOWER_SPEED_LIMIT executed\n");

	 lval_in = get_desired(TILT,LOWER_SPEED_LIMIT);
	 if (lval_in != (long) uval)
		 { printf("TILT_LOWER_SPEED_QUERY failed: (set,get)=(%u,%ld)\n", uval, lval_in);
			return ( return_error() );		}
	 printf("TILT_LOWER_SPEED_QUERY issued and verified\n");

	 if ((status = set_desired(TILT, LOWER_SPEED_LIMIT, (PTU_PARM_PTR *) &uval, RELATIVE)) == TRUE)
		 { return (  return_error_status( "TILT_SET_LOWER_SPEED_LIMIT", status) );
				}
	 else printf("TILT_SET_LOWER_SPEED_LIMIT executed\n");

	 lval_in = get_current(TILT,LOWER_SPEED_LIMIT);
	 if (lval_in != (long) uval)
		 { printf("TILT_LOWER_SPEED_QUERY failed: (set,get)=(%u,%ld)\n", uval, lval_in);
			return ( return_error() );		}
	 printf("TILT_LOWER_SPEED_QUERY issued and verified\n");

 	 /* pan set upper speed limit testing */
 	 uval = 2801;

	 if ((status = set_desired(PAN, UPPER_SPEED_LIMIT, (PTU_PARM_PTR *) &uval, ABSOLUTE)) == TRUE)
		 { return (  return_error_status( "PAN_SET_UPPER_SPEED_LIMIT", status) );
				}
	 else printf("\nPAN_SET_UPPER_SPEED_LIMIT executed\n");

	 lval_in = get_desired(PAN,UPPER_SPEED_LIMIT);
	 if (lval_in != (long) uval)
		 { printf("PAN_UPPER_SPEED_QUERY failed: (set,get)=(%u,%ld)\n", uval, lval_in);
		   return ( return_error() );		}
	 printf("PAN_UPPER_SPEED_QUERY issued and verified\n");


     halt(ALL); do_delay(1000);
	 lval_in = get_current(PAN,UPPER_SPEED_LIMIT);
	 
	 if ((status = set_desired(PAN, UPPER_SPEED_LIMIT, (PTU_PARM_PTR *) &uval, RELATIVE)) == TRUE)
		 { return (  return_error_status( "PAN_SET_UPPER_SPEED_LIMIT", status) );
				}
	 else printf("PAN_SET_UPPER_SPEED_LIMIT executed\n");

	 if (lval_in != (long) uval)
		 { printf("PAN_UPPER_SPEED_QUERY failed: (set,get)=(%u,%ld)\n", uval, lval_in);
			return ( return_error() );		}
	 printf("PAN_UPPER_SPEED_QUERY issued and verified\n");


	 /* tilt set UPPER speed limit testing */
	 uval = 2832;

	 if ((status = set_desired(TILT, UPPER_SPEED_LIMIT, (PTU_PARM_PTR *) &uval, ABSOLUTE))  == TRUE)
		 { return (  return_error_status( "TILT_SET_UPPER_SPEED_LIMIT", status) );
				}
	 else printf("\nTILT_SET_UPPER_SPEED_LIMIT executed\n");

	 lval_in = get_desired(TILT,UPPER_SPEED_LIMIT);
	 if (abs(lval_in - (long) uval) > 1)
		 { printf("TILT_UPPER_SPEED_QUERY failed: (set,get)=(%u,%ld)\n", uval, lval_in);
			return ( return_error() );		}
	 printf("TILT_UPPER_SPEED_QUERY issued and verified\n");

	 if ((status = set_desired(TILT, UPPER_SPEED_LIMIT, (PTU_PARM_PTR *) &uval, RELATIVE)) == TRUE)
		 { return (  return_error_status( "TILT_SET_UPPER_SPEED_LIMIT", status) );
				}
	 else printf("TILT_SET_UPPER_SPEED_LIMIT executed\n");

	 lval_in = get_current(TILT,UPPER_SPEED_LIMIT);
	 if (abs(lval_in - (long) uval) > 1)
		 { printf("TILT_UPPER_SPEED_QUERY failed: (set,get)=(%u,%ld)\n", uval, lval_in);
			return ( return_error() );		}
	 printf("TILT_UPPER_SPEED_QUERY issued and verified\n");

	 return(PTU_OK);
}


char test_position_bounds(void)
{ 	 /* Query minimum position testing */

	 lval_in = get_desired(PAN,MINIMUM_POSITION);
	 printf("\nPAN_MINIMUM_POSITION_QUERY RETURNED: %ld\n", lval_in);
	 lval_in = get_desired(PAN,MAXIMUM_POSITION);
	 printf("PAN_MAXIMUM_POSITION_QUERY RETURNED: %ld\n", lval_in);
	 lval_in = get_desired(TILT,MINIMUM_POSITION);
	 printf("TILT_MINIMUM_POSITION_QUERY RETURNED: %ld\n", lval_in);
	 lval_in = get_desired(TILT,MAXIMUM_POSITION);
	 printf("TILT_MAXIMUM_POSITION_QUERY RETURNED: %ld\n", lval_in);

	 printf("Are the above values correct? (Enter 'y' or 'n'): ");
     for (;;) {
		switch (tmpChar=tolower(getchar())) {
		  case 'y': printf("\nMin/Max position queries PASSED\n\n");
			        return(PTU_OK);
		  case 'n': return(TEST_ERROR);
		} }
}


char test_acceleration_commands(void)
{ 
	/* pan TEST SET AND QUERY ACCELERATION */
	lval = lval2 = 16666;

	 if ((status = set_desired(PAN, ACCELERATION, (PTU_PARM_PTR *) &lval, ABSOLUTE)) == TRUE)
		 { return (  return_error_status( "PAN_SET_ACCELERATION", status) );
				}
	 else printf("\nPAN_SET_ACCELERATION executed\n");

	 lval_in = get_desired(PAN,ACCELERATION);
	 if (lval_in != lval)
		 { printf("PAN_ACCELERATION_QUERY failed: (set,get)=(%ld,%ld)\n", lval, lval_in);
			return ( return_error() );		}
	 printf("PAN_ACCELERATION_QUERY issued and verified\n");

	 lval = 16777;
	 lval2 += lval;
	 if ((status = set_desired(PAN, ACCELERATION, (PTU_PARM_PTR *) &lval, RELATIVE)) == TRUE)
		 { return (  return_error_status( "PAN_SET_ACCELERATION", status) );
				}
	 else printf("PAN_SET_ACCELERATION executed\n");

	 lval_in = get_desired(PAN,ACCELERATION);
	 if (lval_in != lval2)
		 { printf("PAN_ACCELERATION_QUERY failed: (set,get)=(%ld,%ld)\n", lval2, lval_in);
			return ( return_error() );		}
	 printf("PAN_ACCELERATION_QUERY issued and verified\n");


	/* tilt TEST SET AND QUERY ACCELERATION */
	lval = lval2 = 16888;

	 if ((status = set_desired(TILT, ACCELERATION, (PTU_PARM_PTR *) &lval, ABSOLUTE)) == TRUE)
		 { return (  return_error_status( "TILT_SET_ACCELERATION", status) );
				}
	 else printf("\nTILT_SET_ACCELERATION executed\n");

	 lval_in = get_desired(TILT,ACCELERATION);
	 if (lval_in != lval)
		 { printf("TILT_ACCELERATION_QUERY failed: (set,get)=(%ld,%ld)\n", lval, lval_in);
			return ( return_error() );		}
	 printf("TILT_ACCELERATION_QUERY issued and verified\n");

	 lval = 16999;
	 lval2 += lval;
	 if ((status = set_desired(TILT, ACCELERATION, (PTU_PARM_PTR *) &lval, RELATIVE)) == TRUE)
		 { return (  return_error_status( "TILT_SET_ACCELERATION", status) );
				}
	 else printf("TILT_SET_ACCELERATION executed\n");

	 lval_in = get_desired(TILT,ACCELERATION);
	 if (lval_in != lval2)
		 { printf("TILT_ACCELERATION_QUERY failed: (set,get)=(%ld,%ld)\n", lval2, lval_in);
			return ( return_error() );		}
	 printf("TILT_ACCELERATION_QUERY issued and verified\n");

	return(PTU_OK);
}


char test_resolution_queries(void)
{ 
	/* test the resolution query commands */
	 lval_in = get_desired(PAN,RESOLUTION);
	 printf("\nPAN_RESOLUTION_QUERY RETURNED: %.3f arc min\n", (lval_in / 3600.0) );
	 lval_in = get_desired(TILT,RESOLUTION);
	 printf("TILT_RESOLUTION_QUERY RETURNED: %.3f arc min\n", (lval_in / 3600.0) );
	 lval_in = get_current(PAN,RESOLUTION);

	 printf("Are the above values correct? (Enter 'y' or 'n'): ");
     for (;;) {
		switch (tmpChar=tolower(getchar())) {
		  case 'y': printf("Resolution queries PASSED\n\n");
			        return(PTU_OK);
		  case 'n': return(TEST_ERROR);
		} }
}


char test_command_execution_modes(void)
{ 	/* test the command execution modes */

	 if ((status = set_mode(COMMAND_EXECUTION_MODE,EXECUTE_IMMEDIATELY)) == TRUE)
		 { return (  return_error_status( "SET_IMMEDIATE_COMMAND_MODE", status) );
				}
	 else printf("\nSET_IMMEDIATE_COMMAND_MODE executed\n");
	 val = 2000;
	 set_desired(PAN, POSITION, (PTU_PARM_PTR *) &val, ABSOLUTE);
	 do_delay(2000);
	 if ( (lval = get_current(PAN, POSITION)) != val )
		 { printf("! SET_IMMEDIATE_COMMAND_MODE verification failed: %ld != %d\n", lval, val);
			return ( return_error() );	}
	 else printf("SET_IMMEDIATE_COMMAND_MODE verified\n");

	 if ((status = set_mode(COMMAND_EXECUTION_MODE,EXECUTE_UPON_IMMEDIATE_OR_AWAIT)) == TRUE)
		 { return (  return_error_status( "SET_SLAVED_COMMAND_MODE", status) );
				}
	 else printf("SET_SLAVED_COMMAND_MODE executed\n");
	 val2 = 0;
	 set_desired(PAN, POSITION, (PTU_PARM_PTR *) &val2, ABSOLUTE);
	 do_delay(500);
	 if ( (lval = get_current(PAN, POSITION)) != val )
		 { printf("! SET_SLAVED_COMMAND_MODE verification failed: %ld != %d\n", lval, val);
			return ( return_error() );	}
	 else printf("SET_SLAVED_COMMAND_MODE verified\n");
	 set_mode(COMMAND_EXECUTION_MODE,EXECUTE_IMMEDIATELY);
	 if ((status = set_mode(COMMAND_EXECUTION_MODE,EXECUTE_UPON_IMMEDIATE_OR_AWAIT)) == TRUE)
		 { return (  return_error_status( "SET_SLAVED_COMMAND_MODE", status) );
				}
	 else printf("SET_SLAVED_COMMAND_MODE executed\n");

	 return(PTU_OK);
}


char test_halt_commands(void)
{ 	/* test the halt commands */
 { short int val1 = 3000;
	short int val2 = -3000;
	unsigned short int uval = 1000;
	short int val3 = -900;
	short int val4 =  600;
	long lval, lval_in;

	set_mode(COMMAND_EXECUTION_MODE,EXECUTE_IMMEDIATELY);
	set_desired(PAN,  SPEED,   (PTU_PARM_PTR *) &uval, ABSOLUTE);
	set_desired(TILT, SPEED,   (PTU_PARM_PTR *) &uval, ABSOLUTE);
	set_desired(PAN,  POSITION, (PTU_PARM_PTR *) &val1, ABSOLUTE);
	set_desired(TILT, POSITION, (PTU_PARM_PTR *) &val3, ABSOLUTE);
	await_completion();
	set_desired(PAN,  POSITION, (PTU_PARM_PTR *) &val2, ABSOLUTE);
	set_desired(TILT, POSITION, (PTU_PARM_PTR *) &val4, ABSOLUTE);
	do_delay(500);
	 if ((status = halt(ALL) ) == TRUE)
		 { return (  return_error_status( "HALT(ALL)", status) );
				}
	 else printf("\nHALT(ALL) executed\n");
	await_completion();
	if ((lval = get_current(PAN, POSITION)) == val2)
		 { printf("! HALT(ALL) verification failed to stop pan axis movement (%ld)\n", lval);
			return ( return_error() );	}
	else  printf("HALT(ALL) verified\n");

	/* test halt(tilt) */
	set_desired(PAN,  POSITION, (PTU_PARM_PTR *) &val1, ABSOLUTE);
	set_desired(TILT, POSITION, (PTU_PARM_PTR *) &val3, ABSOLUTE);
	await_completion();
	set_desired(PAN,  POSITION, (PTU_PARM_PTR *) &val2, ABSOLUTE);
	set_desired(TILT, POSITION, (PTU_PARM_PTR *) &val4, ABSOLUTE);
	do_delay(500);
	if   ((status = halt(TILT) ) == TRUE)
	     return (  return_error_status( "HALT(TILT)", status) );
	else printf("HALT(TILT) executed\n");
	await_completion();
	if ( (get_current(PAN, POSITION) != val2) ||
		  (get_current(TILT,POSITION) == val4) )
		 { printf("HALT(TILT) verification failed to stop pan axis movement or impeded tilt axis\n");
	printf("\nget_current(PAN, POSITION)=%d/%d, get_current(TILT,POSITION)=%d/%d\n",
		get_current(PAN, POSITION), val2, get_current(TILT,POSITION), val4);
		 	return ( return_error() ); }
	else  printf("HALT(TILT) verified\n");

	/* test halt(pan) */
	if ( set_desired(PAN,  POSITION, (PTU_PARM_PTR *) &val1, ABSOLUTE) ||
		  set_desired(TILT, POSITION, (PTU_PARM_PTR *) &val3, ABSOLUTE))
		  printf("\nOrig 1 position commanding error\n");
	await_completion();
	if ( set_desired(PAN,  POSITION, (PTU_PARM_PTR *) &val2, ABSOLUTE) ||
		  set_desired(TILT, POSITION, (PTU_PARM_PTR *) &val4, ABSOLUTE))
		  printf("\nOrig 2 position commanding error\n");
	do_delay(1000);
	 if ((status = halt(PAN) ) == TRUE)
		 { return (  return_error_status( "HALT(PAN)", status) );
				}
	 else printf("HALT(PAN) executed\n");
	await_completion();
	
	lval    = get_current(PAN,POSITION);
	lval_in = get_current(TILT,POSITION);
	if ( (lval == val2) || (lval_in != val4) )
		 { printf("! HALT(PAN) verification failed (P:%ld == %d || T:%ld != %d)\n",
					  lval, val2, lval_in, val4);
			return ( return_error() );	}
	else  printf("HALT(PAN) verified\n");

	return(PTU_OK);
 }         }


char test_reset(void)
{	if ( (status = reset_ptu())  == TRUE)
		{ return (  return_error_status( "RESET_PTU", status) );
		  	}
	else
		printf("\nRESET_PTU executed and verified\n");
	return(PTU_OK);
}


char test_default_operations(void)
{  unsigned short int uval;

	await_completion();
	lval = 9999;
	uval = 1692;
	set_desired(TILT, SPEED, (PTU_PARM_PTR *) &uval, ABSOLUTE);
	set_desired(PAN, ACCELERATION, (PTU_PARM_PTR *) &lval, ABSOLUTE);
	if ( (status = set_mode(DEFAULTS,SAVE_CURRENT_SETTINGS)) == TRUE)
		 { return (  return_error_status( "SAVE_DEFAULTS", status) );
				}
	else  printf("\nSAVE_DEFAULTS executed\n");

	if ( (status = set_mode(DEFAULTS,RESTORE_SAVED_SETTINGS)) == TRUE)
		 { return (  return_error_status( "RESTORE_SAVED_DEFAULTS", status) );
				}
	else  printf("RESTORE_SAVED_DEFAULTS executed\n");
	lval_in = get_desired(PAN,ACCELERATION);
	uval1 = (unsigned short) get_desired(TILT,SPEED);
	if ( (lval_in != lval ) || (uval1 != uval )    )
		 { printf("! RESTORE_SAVED_DEFAULTS verification failed (PA: %ld, TS: %u)\n",
					 lval_in, uval1 );
			return ( return_error() );	}
	else  printf("RESTORE_SAVED_DEFAULTS verified\n");

	if ( (status = set_mode(DEFAULTS,RESTORE_FACTORY_SETTINGS)) == TRUE)
		 { return (  return_error_status( "RESTORE_FACTORY_SETTINGS", status) );
				}
	else  printf("RESTORE_FACTORY_SETTINGS executed\n");
	if ( (get_desired(PAN,SPEED)         != 1000) ||
		  (get_desired(TILT,ACCELERATION) != 2000) )
		 { return (  return_error_status( "RESTORE_FACTORY_SETTINGS", 0) );
				}
	else  printf("RESTORE_FACTORY_SETTINGS verified\n");
	
	return(PTU_OK);
}



char rapid_cycle_test(void)
{  int i;
	long lval = 50000L;

	val1 = 0;
	val2 = 200;
	set_desired(PAN, ACCELERATION, (PTU_PARM_PTR *) &lval, ABSOLUTE);
	uval = 6600;
	set_desired(PAN, UPPER_SPEED_LIMIT, (PTU_PARM_PTR *) &uval, ABSOLUTE);
	uval = 1200;
	set_desired(PAN, BASE, (PTU_PARM_PTR *) &uval, ABSOLUTE);
	set_mode(COMMAND_EXECUTION_MODE,EXECUTE_IMMEDIATELY);
	for (i=0; i<50; i++)
	  {   await_completion();
			set_desired(PAN,  POSITION, (PTU_PARM_PTR *) &val1, ABSOLUTE);
			await_completion();
			set_desired(PAN, POSITION, (PTU_PARM_PTR *) &val2, ABSOLUTE);
	  }
	return(PTU_OK);
}



char test_pure_velocity_control_mode()
{ 	if (set_mode(SPEED_CONTROL_MODE, PTU_PURE_VELOCITY_SPEED_CONTROL_MODE) != PTU_OK)
       { printf("Speed control modes not supported in this PTU firmware version\n");
		 return(PTU_OK); }
    printf("\nSPEED_CONTROL_MODE set to PTU_PURE_VELOCITY_SPEED_CONTROL_MODE\n");

	val=600;
    set_desired(PAN,  POSITION, (PTU_PARM_PTR *) &val, ABSOLUTE);
    set_desired(TILT, POSITION, (PTU_PARM_PTR *) &val, ABSOLUTE);
	val = -1000;
    set_desired(PAN,  SPEED, (PTU_PARM_PTR *) &val, ABSOLUTE);
    set_desired(TILT, SPEED, (PTU_PARM_PTR *) &val, ABSOLUTE);
	await_completion();
	if ( ((val1=(short)get_desired(PAN,POSITION)) >= 0) || ((val2=(short)get_desired(TILT,POSITION)) >= 0) )
	   { printf("\nSPEED_CONTROL_MODE failed: signed velocity didn't override position command ([p,t]=[%d,%d]\n\n", val1, val2);
	     goto failed_exit;		}
	else
	   { printf("Absolute signed speed controls verified\n"); }

	val=600;
    set_desired(PAN,  POSITION, (PTU_PARM_PTR *) &val, ABSOLUTE);
    set_desired(TILT, POSITION, (PTU_PARM_PTR *) &val, ABSOLUTE);
	await_completion();
	val = -1000; /* so absolute speed should be 0 + -1000 = -1000 */
    set_desired(PAN,  SPEED, (PTU_PARM_PTR *) &val, RELATIVE);
    set_desired(TILT, SPEED, (PTU_PARM_PTR *) &val, RELATIVE);
	await_completion();
	if ( (get_desired(PAN,POSITION) >= 0) || (get_desired(TILT,POSITION) >= 0) )
	   { printf("\nSPEED_CONTROL_MODE failed: signed relative velocity didn't override position command ([p,t]=[%d,%d]\n\n", val1, val2);
	     goto failed_exit;		}
	else
	   { printf("Relative signed speed controls verified\n"); }

	set_mode(SPEED_CONTROL_MODE, PTU_PURE_VELOCITY_SPEED_CONTROL_MODE);
	printf("\n");
	return(PTU_OK);

failed_exit:
	set_mode(SPEED_CONTROL_MODE, PTU_PURE_VELOCITY_SPEED_CONTROL_MODE);
	return( return_error() );
}


int main()
	{ int COMportNum;
      char COMportName[256], tmpChar;
      portstream_fd COMstream;
#ifdef _UNIX
      char COMportPrefix[10] = "/dev/cua";
#else
      char COMportPrefix[10] = "COM";
#endif
	   
	 /* parse the input arguments */

         printf("\n\n\n****** PAN-TILT BINARY TEST PROGRAM, %s\n", TEST_CODE_VERSION); 
	 printf("****** Serial Port Interface Driver, %s\n", SERIAL_CODE_VERSION);
	 printf("****** (c)1999, Directed Perception, Inc. All Rights Reserved.\n");
	 COMportName[0] = ' ';
	 while (COMportName[0] == ' ') {
		 printf("\nEnter the %s port number the PTU is attached to: ", COMportPrefix);
		 scanf("%d", &COMportNum);
		 printf("You selected %s%d. Is this OK? (enter 'y' or 'n'): ", COMportPrefix, COMportNum);
		 tmpChar = 'f';
		 while ( (tmpChar != 'y') && (tmpChar != 'n') )
			   tmpChar = tolower(getchar());
		 if ( tmpChar == 'y' )
		    sprintf(COMportName, "%s%d", COMportPrefix, COMportNum);
	 }
         
	 /* initialize the serial port */
	 COMstream = open_host_port(COMportName);

	 if ( COMstream == PORT_NOT_OPENED )
		 { printf("\nSerial Port setup error.\n");
		   goto abnormal_exit;  }
	 printf("\nSerial port %s initialized\n", COMportName);


	 /* begin exercising the PTU command set */
     set_mode(DEFAULTS, RESTORE_FACTORY_SETTINGS);

	 switch ( reset_PTU_parser(5000) ) {
		case PTU_OK:						break;
		case PTU_FIRMWARE_VERSION_TOO_LOW:  printf("\nError(reset_PTU_parser): PTU FIRMWARE VERSION MATCH ERROR\n");
											goto abnormal_exit;
		case PTU_NOT_RESPONDING:			printf("\nError(reset_PTU_parser): PTU_NOT_RESPONDING\n");
											goto abnormal_exit;
	}
	printf("\nPending commands and input buffer flushed\n");


	printf("\n\nFirmware version: %s\nFirmware version command executed successfully\n",
			 firmware_version());

	if ( test_reset() ) goto abnormal_exit;
	if ( test_position_commands() ) goto abnormal_exit;
	if ( test_position_limits() ) goto abnormal_exit;
	if ( test_power_modes() ) goto abnormal_exit;
	if ( test_speed_commands() ) goto abnormal_exit;
	if ( test_position_bounds() ) goto abnormal_exit;
	if ( test_verbose_modes() ) goto abnormal_exit;
	if ( test_ASCII_echo_modes() ) goto abnormal_exit;
	if ( test_base_speed_commands() ) goto abnormal_exit;
	if ( test_acceleration_commands() ) goto abnormal_exit;
	if ( test_speed_limits() ) goto abnormal_exit;
	if ( test_resolution_queries() ) goto abnormal_exit;
	if ( test_command_execution_modes() ) goto abnormal_exit;
	if ( test_halt_commands() ) goto abnormal_exit;
	if ( test_default_operations() ) goto abnormal_exit;
    if ( test_pure_velocity_control_mode() ) goto abnormal_exit;

	printf("\n\nBINARY COMMAND SET TESTS FINISHED SUCCESSFULLY!!\n\n\n");

	/* rehome the PTU */
    set_mode(DEFAULTS, RESTORE_SAVED_SETTINGS);
	val = 0;
	set_desired(PAN,  POSITION, (PTU_PARM_PTR *) &val, ABSOLUTE);
	set_desired(TILT, POSITION, (PTU_PARM_PTR *) &val, ABSOLUTE);
	await_completion();


	goto exit;

  abnormal_exit:
	printf("\nABNORMAL EXIT: test failed\n\n");

  exit:
	/* FlushInputBuffer(); */
    return_error();
	close_host_port(COMstream);
	return(TRUE);

}


