/*
* Analog Input Example 1
*
* This example reads one sample from channel 0. 
*
* Channel 0 should be configured for bipolar RSE with no
* dithering.  No external multiplexers are present.  Sample
* analog channel 0 at a gain of 1 and read the unscaled 
* result.  Compare the unscaled result to the voltage.
* (This example illustrates the simplest way to sample a 
* channel.  Minimal DAQ-STC registers are programmed.
*/  

#include <linux/fs.h>
#include <linux/sched.h>

#include "eserrlp.h"
#include "daq_driver.h"
#include "misc.h"
/*
* Function declarations
*/

/*
mode=0: single point mode
mode=1: pune counerul SI la treaba
mode=2: single scan
*/

static int AI_Read_1pt(pd dev, unsigned short *);

static void AI_Add_Channel(pd dev, unsigned long x);

void MSC_Clock_Configure(pd dev);
static void AI_Reset_All(pd dev);
static void AI_Board_Personalize(pd dev);
static void AI_Initialize_Configuration_Memory_Output(pd dev);
static void AI_Board_Environmentalize(pd dev);
static void FIFO_Request(pd dev);
static void AI_Hardware_Gating(pd dev);
static void AI_Trigger_Signals(pd dev);
static void Number_of_Scans(pd dev, unsigned long no_scans);
static void AI_Scan_Start(pd dev, int mode, unsigned long scan_interv);
static void AI_End_of_Scan(pd dev);
static void Convert_Signal(pd dev, unsigned short spl_interv);
static void Clear_FIFO(pd dev);
static void AI_Interrupt_Enable(pd dev);
static void AI_Arming(pd dev, int mode);
static void AI_Start_Daq(pd dev, int mode);

int ai_read(pd dev, struct file * file, char  * buf, int count)
{
  int i,ret;
  unsigned short uValue;
  unsigned long flags;
  int blocked=1;
#ifdef DAQ_DEBUG
  printk ("calling ai_read with count %d\n", count);
#endif

  if(dev->irq==-1){                  /* POOLED AI */
    for(i=0; i<count/2;i++){
      if((ret=AI_Read_1pt(dev, &uValue))<0) return ret;
      copy_to_user (buf+i*2, &uValue, 2);
    }
    return i*2;
  } else {                           /* IRQ BASED AI*/
#ifdef DAQ_DEBUG
    printk ("doing interrupt based data transfers...\n");
#endif
    save_flags(flags);cli();
    for(i=0;i<count/2;i++){
      if (dev->ai_FIFO_oflow || dev->ai_BUFF_oflow)  {
#ifdef DAQ_DEBUG
	printk ("we have an overflow\n");
#endif
	break;
      }
      if (dev->ai_pread==dev->ai_pwrite) 
	if (!blocked)  {
#ifdef DAQ_DEBUG
	  printk ("we have blocked... returning an error...\n");
#endif
	  break;
	}
	else 
	  if (dev->ai_running) 
	    interruptible_sleep_on (&dev->ai_wait_queue);
	  else 
	    {
#ifdef DAQ_DEBUG
	      printk ("we aren't running....\n");
#endif
	      break;
	    }
      if (dev->ai_FIFO_oflow || dev->ai_BUFF_oflow)
	{
#ifdef DAQ_DEBUG
	  printk ("there has been an overflow...\n");
#endif
	  break;
	}
      uValue=dev->aibuff[dev->ai_pread++];
      if(dev->ai_pread==AIBUFFSIZE)dev->ai_pread=0;
      copy_to_user (buf+i*2, &uValue, 2);
    }
    restore_flags(flags);
    if(dev->ai_FIFO_oflow || dev->ai_BUFF_oflow) return -ETIME;
    return 2*i;
  }                                  /* END IRQ AI */
}

int ai_open(pd dev, struct file * file){

  dev->ai_running=0;
  AI_Reset_All(dev);
  return 0;
}

int ai_ioctl(pd dev, struct file *file, unsigned int cmd, unsigned long arg)
{
  int mode;

  //  printk("cmd=%d arg=%ld\n", _IOC_NR(cmd), arg);

  switch(_IOC_NR(cmd)){

  case _IOC_NR(NICTL_AI_RESET):
    AI_Reset_All(dev);
    /* Clear conf memory */
    DAQ_STC_Windowed_Mode_Strobe_Write(dev,Write_Strobe_0_Register,0x0001);
    /* Clear AI FIFO */
    DAQ_STC_Windowed_Mode_Strobe_Write(dev,Write_Strobe_1_Register,0x0001);    
    return 0;
    
  case _IOC_NR(NICTL_AI_SCAN_SETUP):
    mode=dev->ai_sc?1:2;
    MSC_Clock_Configure(dev);
    AI_Reset_All(dev);
    AI_Board_Personalize(dev);
    AI_Board_Environmentalize(dev);
    AI_Initialize_Configuration_Memory_Output(dev);
    FIFO_Request(dev);
    AI_Hardware_Gating(dev);
    AI_Trigger_Signals(dev);
    Number_of_Scans(dev, dev->ai_sc);
    AI_Scan_Start(dev, mode, dev->ai_si);
    AI_End_of_Scan(dev);
    Convert_Signal(dev, dev->ai_si2);
    return 0;

  case _IOC_NR(NICTL_AI_SCAN_START):
    mode=dev->ai_sc?1:2;
    Clear_FIFO(dev);
    AI_Interrupt_Enable(dev);
    AI_Arming(dev, mode);
    dev->ai_points_read=0;
    dev->ai_pread=dev->ai_pwrite=0;
    dev->ai_FIFO_oflow=dev->ai_BUFF_oflow=0;
    dev->ai_running=1;
    AI_Start_Daq(dev, mode);
    return 0;

  case _IOC_NR(NICTL_AI_ONEPT_SETUP):
    MSC_Clock_Configure(dev);
    AI_Reset_All(dev);
    AI_Board_Personalize(dev);
    AI_Board_Environmentalize(dev);
    AI_Initialize_Configuration_Memory_Output(dev);
    FIFO_Request(dev);
    AI_Hardware_Gating(dev);
    AI_Trigger_Signals(dev);
    /* Number_of_Scans(dev); nu e cazul*/
    AI_Scan_Start(dev, 0, 0);
    AI_End_of_Scan(dev);
    /* Convert_Signal(dev); nu e cazul: se da manual*/
    return 0;

  case _IOC_NR(NICTL_AI_CONVERT):
    Clear_FIFO(dev);  
    AI_Interrupt_Enable(dev);
    /* AI_Arming(dev);* nu e cazul */
    // dev->points_read=
    dev->ai_pread=dev->ai_pwrite=0;
    dev->ai_FIFO_oflow=dev->ai_BUFF_oflow=0;
    dev->ai_running=1;
    AI_Start_Daq(dev, 0);
  return 0;

  case _IOC_NR(NICTL_AI_ADDCH):
    AI_Add_Channel(dev, arg);
    return 0;

  case _IOC_NR(NICTL_AI_SET_SC):
    dev->ai_sc=arg;
    return 0;
  case _IOC_NR(NICTL_AI_SET_SI):
    dev->ai_si=arg;
    return 0;
  case _IOC_NR(NICTL_AI_SET_SI2):
    dev->ai_si2=(unsigned short)arg;
    return 0;
  }

  return -EINVAL;
}


int AI_Read_1pt(pd dev, unsigned short *uDataRead)
{
  unsigned  uStatus; /* flag to indicate FIFO not empty */
  unsigned  uValue=0;  /* binary voltage acquired */
  int i, err;

/* 
* Poll the FIFO not empty flag to determine if
* data is available.  If FIFO not empty then
* read the data.  
*/
  err=-ETIME;
  for(i=0;i<100000;i++) {  	
    uStatus = DAQ_STC_Windowed_Mode_Read(dev, AI_Status_1_Register);
    if (!((uStatus & 0x1000) == 0x1000))
      { 
	uValue = Board_Read(dev, ADC_FIFO_Data_Register);
	err=0;
	break;
	
      } /*  else  interruptible_sleep_on(&ni_pd[dev].wait_queue); */
    if(i%100==99)schedule();
  }
  *uDataRead=uValue;
  return err;
}     
        	
/*
* Call this function to configure board options.
*/

void AI_Add_Channel(pd dev, unsigned long x){
  Board_Write(dev, Configuration_Memory_High,(x>>16)&0xffff);
  Board_Write(dev, Configuration_Memory_Low,x&0xffff);
}

/* 
* Call this function to access the first value in the configuration
* FIFO.
*/

void AI_Initialize_Configuration_Memory_Output(pd dev)
{
  /*
  * Writing to register AI_Command_1_Register with address 8.
  *    AI_CONVERT_Pulse <= 1
  * New pattern = 0x0001
  */
  DAQ_STC_Windowed_Mode_Strobe_Write(dev, AI_Command_1_Register,0x0001); 
  return;
 }
  
/* 
* Call this function to stop any activities in progress.
*/

void AI_Reset_All(pd dev) /*** AS IN STC MANUAL *****/
{
  /* AI_Reset */
  DAQ_STC_Windowed_Mode_Strobe_Write(dev, Joint_Reset_Register,0x0001);

  /* AI_Configuration_Start */
  DAQ_STC_Windowed_Mode_Strobe_Write(dev, Joint_Reset_Register,0x0010);

  /* disable all AI interrupts */
  DAQ_STC_Windowed_Mode_Masked_Write(dev, Interrupt_A_Enable_Register, 0, 0x00BF);

  /*
  * Writing to register Interrupt_A_Ack_Register with address 2.
  *    AI_SC_TC_Error_Confirm <= 1
  *    AI_SC_TC_Interrupt_Ack <= 1
  *    AI_START1_Interrupt_Ack <= 1
  *    AI_START2_Interrupt_Ack <= 1
  *    AI_START_Interrupt_Ack <= 1
  *    AI_STOP_Interrupt_Ack <= 1
  *    AI_Error_Interrupt_Ack <= 1
  * New pattern = 0x3F80
  */
  DAQ_STC_Windowed_Mode_Masked_Write(dev, Interrupt_A_Ack_Register,0x3F80, 0x3F80);

  dev->stccpy[AI_Command_1_Register]=0;
  dev->stccpy[AI_Command_2_Register]=0;
  dev->stccpy[AI_Mode_1_Register]=0;
  dev->stccpy[AI_Mode_2_Register]=0;
  dev->stccpy[AI_Mode_3_Register]=0;
  dev->stccpy[AI_Output_Control_Register]=0;
  dev->stccpy[AI_Personal_Register]=0;
  dev->stccpy[AI_START_STOP_Select_Register]=0;
  dev->stccpy[AI_Trigger_Select_Register]=0;

  /*
  * Writing to register AI_Mode_1_Register with address 12.
  *    Reserved_One <= 1
  *    AI_Start_Stop <= 1
  * New pattern = 0x000C
  */
  DAQ_STC_Windowed_Mode_Masked_Write(dev, AI_Mode_1_Register,0x000C, 0x000C);

  /* AI_Configuration_End */
  DAQ_STC_Windowed_Mode_Strobe_Write(dev, Joint_Reset_Register,0x0100);
 
 return;
 }
 
/* 
* Call this function to setup the E-2.
*/

void AI_Board_Personalize(pd dev)   /*** AS IN STC MANUAL ***/
{
  /*
  * Writing to register Joint_Reset_Register with address 72.
  *    AI_Configuration_Start <= 1
  * New pattern = 0x0010
  */
  DAQ_STC_Windowed_Mode_Strobe_Write(dev, Joint_Reset_Register,0x0010);

  /*
  * Writing to register Clock_and_FOUT_Register with address 56.
  *    AI_Source_Divide_By_2 <= p->ai_source_divide_by_2 (0)
  *    AI_Output_Divide_By_2 <= p->ai_output_divide_by_2 (1)
  * New pattern = 0x1B80
  */
  DAQ_STC_Windowed_Mode_Masked_Write(dev, Clock_and_FOUT_Register,0x0080,0x00C0);

  /*
  * Writing to register AI_Personal_Register with address 77.
  *    AI_CONVERT_Pulse_Timebase <= p->ai_convert_pulse_timebase (0)
  *    AI_CONVERT_Pulse_Width <= p->ai_convert_pulse_width (1)
  *    AI_FIFO_Flags_Polarity <= p->ai_fifo_flags_polarity (0)
  *    AI_LOCALMUX_CLK_Pulse_Width <= p->ai_localmux_clk_pulse_width (1)
  *    AI_AIFREQ_Polarity <= p->ai_aifreq_polarity (0)
  *    AI_SHIFTIN_Polarity <= p->ai_shiftin_polarity (0)
  *    AI_SHIFTIN_Pulse_Width <= p->ai_shiftin_pulse_width (1)
  *    AI_EOC_Polarity <= p->ai_eoc_polarity (0)
  *    AI_SOC_Polarity <= p->ai_soc_polarity (1)
  *    AI_Overrun_Mode <= p->ai_overrun_mode (1)
  * New pattern = 0xA4A0
  */
  DAQ_STC_Windowed_Mode_Masked_Write(dev, AI_Personal_Register,0xA4A0, 0xFDB0);

  /*
  * Writing to register AI_Output_Control_Register with address 60.
  *    AI_CONVERT_Output_Select <= p->ai_convert_output_select (2)
  *    AI_SC_TC_Output_Select <= p->ai_sc_tc_output_select (3)
  *    AI_SCAN_IN_PROG_Output_Select <= p->ai_scan_in_prog_output_select (3)
  *    AI_LOCALMUX_CLK_Output_Select <= p->ai_localmux_clk_output_select (2)
  * New pattern = 0x032E
  */
  DAQ_STC_Windowed_Mode_Masked_Write(dev,AI_Output_Control_Register,0x032E,0x033F);

  /*
  * Writing to register Joint_Reset_Register with address 72.
  *    AI_Configuration_Start <= 0
  *    AI_Configuration_End <= 1
  * New pattern = 0x0100
  */
  DAQ_STC_Windowed_Mode_Strobe_Write(dev, Joint_Reset_Register,0x0100);
  return;
 }
 
/* 
* Call this function to setup for external hardware.
*/

void AI_Board_Environmentalize(pd dev) /* AS IN STC MANUAL */
{
  /*
  * Writing to register Joint_Reset_Register with address 72.
  *    AI_Configuration_Start <= 1
  * New pattern = 0x0010
  */
  DAQ_STC_Windowed_Mode_Strobe_Write(dev, Joint_Reset_Register,0x0010);

  /*
  * Writing to register AI_Mode_2_Register with address 13.
  *    AI_External_MUX_Present <= 0
  * New pattern = 0x0000
  */
  DAQ_STC_Windowed_Mode_Masked_Write(dev, AI_Mode_2_Register,0x0000, 0x1000);

  /*
  * Writing to register AI_Output_Control_Register with address 60.
  *    AI_EXTMUX_CLK_Output_Select <= p->ai_extmux_clk_output_select (0)
  * New pattern = 0x032E
  */
  DAQ_STC_Windowed_Mode_Masked_Write(dev,AI_Output_Control_Register,0x0000,0x00C0);

  /*
  * Writing to register Joint_Reset_Register with address 72.
  *    AI_Configuration_Start <= 0
  *    AI_Configuration_End <= 1
  * New pattern = 0x0100
  */
  DAQ_STC_Windowed_Mode_Strobe_Write(dev, Joint_Reset_Register,0x0100);
  return;
 }

/*
* Call this fuction to configure the FIFO condition for which
* interrupts and DMA requests will be generated.
*/

void FIFO_Request(pd dev) /* AS IN STC MANUAL */
{
  DAQ_STC_Windowed_Mode_Strobe_Write(dev, Joint_Reset_Register,0x0010);

  /* request when FIFO not empty */
  DAQ_STC_Windowed_Mode_Masked_Write(dev, AI_Mode_3_Register, 0x0000, 0x00C0);
 
  DAQ_STC_Windowed_Mode_Strobe_Write(dev, Joint_Reset_Register,0x0100);
 return;
}

/*
* Call this function to enable and disable gating.
*/ 

void AI_Hardware_Gating(pd dev) /***  AS IN STC MANUAL ****/
{

  DAQ_STC_Windowed_Mode_Strobe_Write(dev, Joint_Reset_Register,0x0010);

  /* ext gate disabled */
  DAQ_STC_Windowed_Mode_Masked_Write(dev, AI_Mode_3_Register, 0x0000, 0x003F);
 
  DAQ_STC_Windowed_Mode_Strobe_Write(dev, Joint_Reset_Register,0x0100);
  return;
}

/*
* Call this function to enable or disable retriggering.
*/

void AI_Trigger_Signals(pd dev) /** AS IN STC MANUAL */
{ 
  /*
  * Writing to register Joint_Reset_Register with address 72.
  *    AI_Configuration_Start <= 1
  * New pattern = 0x0010
  */
  DAQ_STC_Windowed_Mode_Strobe_Write(dev, Joint_Reset_Register,0x0010);

  /*
  * Writing to register AI_Mode_1_Register with address 12.
  *    AI_Trigger_Once <= 1
  * New pattern = 0x000D
  */
  DAQ_STC_Windowed_Mode_Masked_Write(dev, AI_Mode_1_Register,0x0001, 0x0001);

  /*
  * Writing to register AI_Trigger_Select_Register with address 63.
  *    AI_START1_Select <= 0
  *    AI_START1_Polarity <= 0
  *    AI_START1_Edge <= 1
  *    AI_START1_Sync <= 1
  * New pattern = 0x0060
  */
  DAQ_STC_Windowed_Mode_Masked_Write(dev,AI_Trigger_Select_Register,0x0060,0x807F);

  /*
  * Writing to register Joint_Reset_Register with address 72.
  *    AI_Configuration_Start <= 0
  *    AI_Configuration_End <= 1
  * New pattern = 0x0100
  */
  DAQ_STC_Windowed_Mode_Strobe_Write(dev, Joint_Reset_Register,0x0100);
  return;
 }

 
/*
* Call this function to select the number of scans.
*/

void Number_of_Scans(pd dev, unsigned long no_scans) /*** AS IN STC MANUAL */
{ 
 
  /* AI_Configuration_Start */
   DAQ_STC_Windowed_Mode_Strobe_Write(dev, Joint_Reset_Register,0x0010);

   /* AI_Continuous = 0 */
   DAQ_STC_Windowed_Mode_Masked_Write(dev, AI_Mode_1_Register,0x0000,0x0002);

   /* AI_Pretrigger = 0
     AI_SC_Initial_Load_Source =0
     AI_SC_Reload_Mode =0
   */
   DAQ_STC_Windowed_Mode_Masked_Write(dev, AI_Mode_2_Register,0x0000,0x2003);

   /*    AI_SC_Load_A <= p->ai_number_of_posttrigger_scans-1 (4) */
   DAQ_STC_Windowed_Mode_Write(dev, AI_SC_Load_A_Registers,(no_scans>>16)&0xff);
   DAQ_STC_Windowed_Mode_Write(dev, AI_SC_Load_A_Registers+1,no_scans & 0xffff);

   /* AI_SC_Load  */
   DAQ_STC_Windowed_Mode_Strobe_Write(dev, AI_Command_1_Register,0x0020);

   /* AI_Configuration_End */
   DAQ_STC_Windowed_Mode_Strobe_Write(dev, Joint_Reset_Register,0x0100);

   return;
}
 
/* 
Call this function to select the scan start event.
*/

void AI_Scan_Start(pd dev, int mode, unsigned long scan_interv) /* AS IN STC MAN */
{

  unsigned long special_ticks=1;

  /* AI_Configuration_Start */
  DAQ_STC_Windowed_Mode_Strobe_Write(dev, Joint_Reset_Register,0x0010);

  /* AI_SI_Special_Trigger_Delay = 0 */
  DAQ_STC_Windowed_Mode_Masked_Write(dev,AI_Mode_3_Register,0x0000,0x1000);

  /*
   * Writing to register AI_START_STOP_Select_Register with address 62.
   *    AI_START_Select <= 0
   *    AI_START_Edge <= 1
   *    AI_START_Sync <= 1
   *    AI_START_Polarity <= 0
   * New pattern = 0x0060
   */
  if(mode!=2)
    DAQ_STC_Windowed_Mode_Masked_Write(dev, AI_START_STOP_Select_Register,
				       0x0060, 0x807F);
  else DAQ_STC_Windowed_Mode_Masked_Write(dev, AI_START_STOP_Select_Register,
					  0x007F, 0x807F);

  /* AI_SI_Source_Select = 0 (AI_IN_TIMEBASE1)
     AI_SI_Source_Polarity = 0 */
  DAQ_STC_Windowed_Mode_Masked_Write(dev, AI_Mode_1_Register, 0x0000, 0x07D0);

    if(mode==1){
      /*    AI_SI_Load_B <= p->ai_si_special_ticks-1 (1) */
      DAQ_STC_Windowed_Mode_Write(dev, AI_SI_Load_B_Registers,
				  (special_ticks>>16)&0xff);
      DAQ_STC_Windowed_Mode_Write(dev, AI_SI_Load_B_Registers+1,
				  special_ticks&0xff);

      /*    AI_SI_Load_A <= p->ai_si_ordinary_ticks-1 (19999) */
      DAQ_STC_Windowed_Mode_Write(dev, AI_SI_Load_A_Registers,
				  (scan_interv>>16)&0xff);
      DAQ_STC_Windowed_Mode_Write(dev, AI_SI_Load_A_Registers+1,
				  scan_interv&0xffff);

      /* AI_SI_Initial_Load_Source=1 */
      DAQ_STC_Windowed_Mode_Masked_Write(dev, AI_Mode_2_Register, 0x0080, 0x0080);

      /*    AI_SI_Load */
      DAQ_STC_Windowed_Mode_Strobe_Write(dev, AI_Command_1_Register,0x0200);
      
      /* AI_SI_Initial_Load_Source=0 
       AI_SI_Reload_Mode = 6 (alternate first period on every SC_TC*/
      DAQ_STC_Windowed_Mode_Masked_Write(dev, AI_Mode_2_Register,0x0060,0x00F0);   
    }
    
    /*    AI_Configuration_End */
    DAQ_STC_Windowed_Mode_Strobe_Write(dev, Joint_Reset_Register,0x0100);
    return;
}
  
/*
* Call this function to select the end of scan event.
*/

void AI_End_of_Scan(pd dev) /*** AS IN STC MANUAL */
{
  /*
  * Writing to register Joint_Reset_Register with address 72.
  *    AI_Configuration_Start <= 1
  * New pattern = 0x0010
  */
  DAQ_STC_Windowed_Mode_Strobe_Write(dev, Joint_Reset_Register,0x0010);

  /*
  * Writing to register AI_START_STOP_Select_Register with address 62.
  *    AI_STOP_Select <= p->ai_stop_select (19)
  *    AI_STOP_Edge <= 0
  *    AI_STOP_Polarity <= p->ai_stop_polarity (0)
  *    AI_STOP_Sync <= 1
  * New pattern = 0x29E0
  */
  DAQ_STC_Windowed_Mode_Masked_Write(dev, AI_START_STOP_Select_Register,
				     0x2980,0x7F80);

  /*
  * Writing to register Joint_Reset_Register with address 72.
  *    AI_Configuration_Start <= 0
  *    AI_Configuration_End <= 1
  * New pattern = 0x0100
  */
  DAQ_STC_Windowed_Mode_Strobe_Write(dev, Joint_Reset_Register,0x0100);
  return;
 }
 
/*
* Call this function to select the convert signal.
*/

void Convert_Signal(pd dev, unsigned short spl_interv) /** AS IN STC MANUAL */
{ 
  unsigned short special_ticks=spl_interv;

  /* AI_Configuration_Start */
  DAQ_STC_Windowed_Mode_Strobe_Write(dev, Joint_Reset_Register,0x0010);

  /* AI_SC_Gate_Enable =0
     AI_Start_Stop_Gate_Enable =0 
     AI_SI2_Initial_Load_Source <= 0
     AI_SI2_Reload_Mode=1 */
  DAQ_STC_Windowed_Mode_Masked_Write(dev, AI_Mode_2_Register,0x0100,0x6300);

  /*    AI_SI2_Load_A <= p->ai_si2_special_ticks-1 (1999) */
  DAQ_STC_Windowed_Mode_Write(dev, AI_SI2_Load_A_Register,special_ticks);

  /*    AI_SI2_Load_B <= p->ai_si2_ordinary_ticks-1 (1999) */
  DAQ_STC_Windowed_Mode_Write(dev, AI_SI2_Load_B_Register,spl_interv);

  /*    AI_SI2_Load <= 1 */
  DAQ_STC_Windowed_Mode_Strobe_Write(dev, AI_Command_1_Register,0x0800);

  /*    AI_SI2_Initial_Load_Source <= 1 */
  DAQ_STC_Windowed_Mode_Masked_Write(dev, AI_Mode_2_Register,0x0200,0x0200);

  /*    AI_Configuration_End <= 1 */
  DAQ_STC_Windowed_Mode_Strobe_Write(dev, Joint_Reset_Register,0x0100);

  return;
}
 
/*
* Call this function to clear the AI FIFO.
*/

void Clear_FIFO(pd dev)
{
  /*
  * Writing to register Write_Strobe_1_Register with address 83.
  *    Write_Strobe_1 <= 1
  * New pattern = 0x0001
  */
  DAQ_STC_Windowed_Mode_Strobe_Write(dev, Write_Strobe_1_Register,0x0001);
  return;
 }

/*
* Call this function to enable interrupts.
*/

void AI_Interrupt_Enable(pd dev) /* SAME AS STC MANUAL **/
{

 /*
  * Writing to register Interrupt_A_Enable_Register with address 73.
  *   AI_FIFO_Interrupt_Enable <= p->ai_fifo_interrupt (1)
  *   AI_STOP_Interrupt_Enable <= p->ai_stop_interrupt (0)
  *   AI_START_Interrupt_Enable <= p->ai_start_interrupt (0)
  *   AI_SC_TC_Interrupt_Enable <= p->ai_sc_tc_interrupt (0)
  *   AI_START1_Interrupt_Enable <= p->ai_start1_interrupt (0)
  *   AI_START2_Interrupt_Enable <= p->ai_start2_interrupt (0)
  *   AI_Error_Interrupt_Enable <= p->ai_error_interrupt (1)
  * New pattern = 0x00A0
  */
  DAQ_STC_Windowed_Mode_Masked_Write(dev,Interrupt_A_Enable_Register,
				     0x00A0,0x00BF);
  return;
}

/*
* Call this function to arm the analog input counters.
*/

void AI_Arming(pd dev, int mode) /* AS IN STC MANUAL */
{

  if(mode==2) {
    DAQ_STC_Windowed_Mode_Strobe_Write(dev, AI_Command_1_Register, 0x0100);
    DAQ_STC_Windowed_Mode_Strobe_Write(dev, AI_Command_2_Register, 0x4000);
  }
  /*
  *    AI_SC_Arm <= 1
  *    AI_SI_Arm <= arm_si (1)
  *    AI_SI2_Arm <= arm_si2 (1)
  *    AI_DIV_Arm <= arm_div (1)
  * New pattern = 0x1540
  */
  DAQ_STC_Windowed_Mode_Strobe_Write(dev, AI_Command_1_Register,0x1540);
  return;
}

/* 
* Call this function to start the acquistion.
*/

void AI_Start_Daq(pd dev, int mode)
{
  switch (mode){
  case 0:
    /*
     * Writing to register AI_Command_1_Register with address 8.
     *    AI_CONVERT_Pulse <= 1
     * New pattern = 0x0001
     */
    DAQ_STC_Windowed_Mode_Strobe_Write(dev, AI_Command_1_Register,0x0001);
    break;

  case 1:
    /*
     * Writing to register AI_Command_2_Register with address 4.
     *    AI_START1_Pulse <= 1
     * New pattern = 0x0001
     */
    DAQ_STC_Windowed_Mode_Strobe_Write(dev, AI_Command_2_Register,0x0001);
    break;

  case 2:
    DAQ_STC_Windowed_Mode_Strobe_Write(dev, AI_Command_2_Register,0x0001);

    DAQ_STC_Windowed_Mode_Masked_Write(dev, AI_START_STOP_Select_Register,
				       0x8000, 0x8000);
    DAQ_STC_Windowed_Mode_Masked_Write(dev, AI_START_STOP_Select_Register,
				       0x0000, 0x8000);
    break;

  }
  return;

}

      
 






