#ifdef MODVERSIONS
#include <linux/modversions.h>
#endif
#include <linux/module.h>
#include <linux/kernel.h>
#include <asm/io.h>
#include <linux/ioport.h>
#include <linux/fs.h>
#include <linux/wrapper.h>
#include <asm/irq.h>
#include <linux/sched.h>
#include <linux/pci.h>
#include <linux/bios32.h>
#include <linux/malloc.h>
#include <linux/mm.h>

#include "daq_driver.h"
#include "eserrlp.h"
#include "misc.h"

static int major=0;
static char name[]="nidaq";

ni_private_data ni_pd[NI_MAX_BOARDS];

static u32 base0, base1, base2, base3;
static u16 irq0, irq1, irq2, irq3;
static u32 type0, type1, type2, type3;

int dio_read(pd dev, struct file * file, char * buf, int count);
int dio_write(pd dev, struct file * file, char const * buf, int count);
int dio_ioctl(pd dev, struct file *file, unsigned int cmd, unsigned long arg);
int dio_open(pd dev, struct file * file);

int ai_read(pd dev, struct file * file, char * buf, int count);
int ai_open(pd dev, struct file * file);
int ai_ioctl(pd dev, struct file *file, unsigned int cmd, unsigned long arg);

int ao_write(pd dev, struct file * file, char const * buf, int count);
int ao_open(pd dev, struct file * file);
int ao_ioctl(pd dev, struct file *file, unsigned int cmd, unsigned long arg);

int gpct_read(pd dev, int no, struct file * file, char * buf, int count);
int gpct_write(pd dev, int no, struct file * file, char const *buf, int count);
int gpct_ioctl(pd dev, int no, struct file *file, unsigned int cmd, unsigned long arg);
void gpct_interrupt(pd dev, int no);

static int daq_open(struct inode *inode, struct file * file)
{
  int minor,subdev;
  pd dev;

#ifdef DAQ_DEBUG
  printk ("calling daq_open\n");
#endif

  minor=MINOR (inode_get_rdev (inode));
  dev=ni_pd+(minor>>4); subdev=minor&0xf;

  if(!dev->type)return -EINVAL;

  switch(subdev){
  case SUBDEV_AIO:
    if((file->f_mode & FMODE_READ ) && (file->f_mode & FMODE_WRITE))
      if(dev->ai_in_use || dev->ao_in_use) return -EBUSY;
    if(file->f_mode & FMODE_READ){
      if(dev->ai_in_use) return -EBUSY;
      dev->ai_in_use++;
      ai_open(dev, file);
    }
    if(file->f_mode & FMODE_WRITE){
      if(dev->ao_in_use) return -EBUSY;
      dev->ao_in_use++;
      ao_open(dev, file);
    }
    break;
  case SUBDEV_DIO:
    if(dev->dio_in_use) return -EBUSY;
    dev->dio_in_use++;
    dio_open(dev, file);
    break;
  case SUBDEV_GPCT0:
  case SUBDEV_GPCT1:
    break;
  }
  MOD_INC_USE_COUNT;
  return 0;
}

static void daq_release (struct inode *inode, struct file * file)
{

  int minor,subdev;
  pd dev;

#ifdef DAQ_DEBUG
  printk ("calling daq_release\n");
#endif

  minor=MINOR (inode_get_rdev (inode));
  dev=ni_pd+(minor>>4); subdev=minor&0xf;

  MOD_DEC_USE_COUNT;
  
  switch(subdev){
  case SUBDEV_AIO:
    if(file->f_mode & FMODE_READ) dev->ai_in_use =0;
    if(file->f_mode & FMODE_WRITE) dev->ao_in_use =0;
    break;
  case SUBDEV_DIO:
    dev->dio_in_use =0;
    break;
  case SUBDEV_GPCT0:
  case SUBDEV_GPCT1:
    break;  
  }
}

static int daq_read (struct inode *inode, struct file *file, char *buf, int count)
{
  int minor,subdev;
  pd dev;
  char statbuff[500];

#ifdef DAQ_DEBUG
  printk ("calling daq_read\n");
#endif

  minor=MINOR (inode_get_rdev (inode));
  dev=ni_pd+(minor>>4); subdev=minor&0xf;

#ifdef DAQ_DEBUG
  printk ("minor version number is %d\n", minor);
#endif

  switch(subdev){
  case SUBDEV_AIO:
    return ai_read(dev,  file, buf, count);
  case SUBDEV_DIO:
    return dio_read(dev, file, buf, count);
  case SUBDEV_GPCT0:
    return gpct_read(dev, 0, file, buf, count);
  case SUBDEV_GPCT1:
    return gpct_read(dev, 1, file, buf, count);
  case SUBDEV_STATS:
    sprintf(statbuff,"Device %d\n no of interrupts=%ld\n ai points read %ld\naiuStatus=%x\naouStatus=%x\n",
	    minor>>4, (unsigned long)dev->no_of_interrupts, (unsigned long)dev->ai_points_read, dev->aiuStatus, dev->aouStatus);
    copy_to_user (buf,statbuff,strlen(statbuff));
    return strlen(statbuff);
  }
  return -EBADRQC;
}

static int daq_write (struct inode *inode, struct file * file, char const * buf, int count)
{
  int minor,subdev;
  pd dev;

#ifdef DAQ_DEBUG
  printk ("calling daq_write\n");
#endif

  minor=MINOR (inode_get_rdev (inode));
  dev=ni_pd+(minor>>4); subdev=minor&0xf;

  switch(subdev){
  case SUBDEV_AIO:
    return ao_write(dev, file, buf, count);
  case SUBDEV_DIO:
    return dio_write(dev, file, buf, count);
  case SUBDEV_GPCT0:
    return gpct_write(dev, 0, file, buf, count);
  case SUBDEV_GPCT1:
    return gpct_write(dev, 1, file, buf, count);
  }
  return -EBADRQC;
}

static int daq_ioctl (struct inode *inode, struct file *file, 
                      unsigned int cmd, unsigned long arg)
{
  int minor,subdev;
  pd dev;

#ifdef DAQ_DEBUG
  printk ("calling daq_ioctl\n");
#endif

  minor=MINOR (inode_get_rdev (inode));
  dev=ni_pd+(minor>>4); subdev=minor&0xf;
  
  if(_IOC_NR(cmd)==_IOC_NR(NICTL_DEVICE_INFO)){
    copy_to_user ((void *)arg, dev->dev_info, sizeof(device_info));
    return 0;
  }
  
  switch (subdev){
  case SUBDEV_AIO:
    if(_IOC_NR(cmd)<50)
      return dev->ai_in_use?ai_ioctl(dev, file, cmd, arg):-EINVAL;
    else
      return dev->ao_in_use?ao_ioctl(dev, file, cmd, arg):-EINVAL;
  case SUBDEV_DIO:
    return dev->dio_in_use?dio_ioctl(dev, file, cmd, arg):-EINVAL;
  case SUBDEV_GPCT0:
    return gpct_ioctl(dev, 0, file, cmd, arg);
  case SUBDEV_GPCT1:
    return gpct_ioctl(dev, 1, file, cmd, arg);
  }
  return -EINVAL;
}

/* VFS struct *********************************************************/
/* Again, sometimes I wish the Linux Kernel API was stable (*sigh*)
  
   So here someone modified the function pointers in fops by reordering
   and changing the arguments, so we get to declare special versions of
   these by kernel.  Fun.  */

#ifdef LINUX_2_1
ssize_t daq_read_wrapper (struct file *file, char *buf, size_t count, loff_t *junk)
{
  struct inode *inode = file->f_dentry->d_inode;
  return daq_read (inode, file, buf, count);
}
ssize_t daq_write_wrapper (struct file *file, const char *buf, size_t count, loff_t *junk)
{
  struct inode *inode = file->f_dentry->d_inode;
  return daq_write (inode, file, buf, count);
}
#else /* Kernel version less than 2.1.x */
int daq_read_wrapper (struct inode *inode, struct file *file, char *buf, int count)
{
  return daq_read (inode, file, buf, count);
}
int daq_write_wrapper (struct inode *inode, struct file *file, const char *buf, int count)
{
  return daq_write (inode, file, buf, count);
}
#endif /* LINUX_2_1 */ 

static struct file_operations fops = {
  read: daq_read_wrapper,
  write: daq_write_wrapper,
  ioctl: daq_ioctl,
  open: daq_open,
  release: daq_release
};


static void eser_daq_interrupt(int irq, void *dev_id, struct pt_regs *regs);

static int init_device(int brd, pd dev)
{
  static irqs[8]={3,4,5,7,10,11,12,15};
  int i, irq_type;

  dev->no_of_interrupts=0; if(!dev->irq)dev->irq=-1;
  if(dev->irq!=-1){
    if(dev->type & NI_PCI) {
      dev->irq_pin=0;
      irq_type=SA_SHIRQ | SA_INTERRUPT;
    } else {
      irq_type=SA_INTERRUPT;
      for(i=0;i<8;i++) if(irqs[i]==dev->irq)break;
      if(i!=8)dev->irq_pin=i; else {
	printk("board %d doesn't support irq %d\n", brd, dev->irq);
	return -1;
      }
    }
    if(request_irq(dev->irq, eser_daq_interrupt,irq_type,name, dev)) {
      printk("%s: unable to register int no %u\n", name, dev->irq);
      return -1;
    }
  }

  if(dev->type & NI_ESER){
    dev->ai_wait_queue=0;
    for(i=0;i<256;i++)dev->stccpy[i]=0;

    dev->aibuff=(u16 *)kmalloc(AIBUFFSIZE*2, GFP_KERNEL);
    dev->aobuff=(u16 *)kmalloc(AOBUFFSIZE*2, GFP_KERNEL);
    dev->gc[0].buff=(u32 *)kmalloc(GPCTBUFFSIZE*4, GFP_KERNEL);
    dev->gc[1].buff=(u32 *)kmalloc(GPCTBUFFSIZE*4, GFP_KERNEL);

    if(!dev->aibuff || !dev->aobuff || !dev->gc[0].buff || !dev->gc[1].buff){
      if(dev->aibuff)kfree(dev->aibuff);
      if(dev->aobuff)kfree(dev->aobuff);
      if(dev->gc[0].buff)kfree(dev->gc[0].buff);
      if(dev->gc[1].buff)kfree(dev->gc[1].buff);
      return -1;
    }

    //    DAQ_STC_Windowed_Mode_Strobe_Write(dev, Joint_Reset_Register,0x000F);
    DAQ_STC_Windowed_Mode_Masked_Write(dev,Interrupt_A_Enable_Register,0,0xFFFF);  DAQ_STC_Windowed_Mode_Masked_Write(dev,Interrupt_B_Enable_Register,0,0xFFFF);
    DAQ_STC_Windowed_Mode_Masked_Write(dev,Interrupt_A_Ack_Register,0xFFFF,0xFFFF);
    DAQ_STC_Windowed_Mode_Masked_Write(dev,Interrupt_B_Ack_Register,0xFFFF,0xFFFF);
    if(dev->type & NI_PCI) writel(0x0303, dev->virt_mite + 0x0a);
    if(dev->irq!=-1){
      MSC_IrqGroupEnable(dev, 0); /* Enable IRQ group A */
      MSC_IrqGroupEnable(dev, 1); /* Enable IRQ group B */ 
    }
    loadcalibration(dev);
  }
  return 0;
}

int init_module(void)
{
  int  ret, brd;
  pd dev;
  u16 vendor, device, pci_command;
  u8 pci_bus, pci_dev, pci_irq_line;
  u32 mite_address, board_address;
  device_info *dinfo;

#define FILLPARS(pd, t, b, i) \
  {pd.type=t; pd.base=b; pd.irq=i;}

  FILLPARS(ni_pd[0], type0, base0, irq0);
  FILLPARS(ni_pd[1], type1, base1, irq1);
  FILLPARS(ni_pd[2], type2, base2, irq2);
  FILLPARS(ni_pd[3], type3, base3, irq3);

  if((ret=module_register_chrdev(major, name, &fops))<0) {
    printk("unable to register %s device\n", name);
    return ret;
  } else {
    major = ret;
    printk("%s activated. Kernel assigned major: %d\n", name, major);
  }

  init_device_info_list();

  /* The ISA boards are not autodetected but their conf. is taken from
   the command line */

  for(brd=0; brd<NI_MAX_BOARDS; brd++){
    dev=&ni_pd[brd];
    if(!dev->type)continue;
    
    if(dev->type & NI_PCI) continue; /* not interested in PCI devices */
    dinfo=dev->dev_info=dev_info_by_type(dev->type);
    if(!dinfo){
      printk("cant find type=%lx in database\n",(unsigned long)dev->type);
      continue;
    }

    if(!dev->base) {
      printk("ISA board %d: has no base address specified\n",brd);
      continue;
    }
    dev->io_region_size=0x20;
    if((ret=check_region(dev->base, dev->io_region_size))) {
      printk("Unable to register i/o region 0x%.4lx, size=%lu\n",
	     (unsigned long)dev->base, (unsigned long)dev->io_region_size);
      continue;
    }
    request_region(dev->base, dev->io_region_size, name);
    if(init_device(brd, dev)<0){
      dev->type=0;
      release_region(dev->base, dev->io_region_size); continue;
    }
    printk("board %d: %s at io=%lx irq=%d\n", brd, dinfo->dev_name,
	   (unsigned long)dev->base, dev->irq);
  } /* END ISA DEVICES */


  /* Now the PCI devices are autodetected */
 
  if(pcibios_present()) for(pci_bus=0; pci_bus<1; pci_bus++) for(pci_dev=0; pci_dev<0xf8; pci_dev+=8){

    pcibios_read_config_word(pci_bus, pci_dev, PCI_VENDOR_ID,
			     &vendor);
    pcibios_read_config_word(pci_bus, pci_dev, PCI_DEVICE_ID, 
			     &device);

    if(vendor != 0x1093) continue; /* not a PCI board */

    dinfo=dev_info_by_deviceid(device);
    if(!dinfo){
      printk("cant find deviceid=%x in database\n", device);continue;}
    
    /* find an empty device */
    for(brd=0; brd<NI_MAX_BOARDS;brd++)
      if(!ni_pd[brd].type || ni_pd[brd].type==dinfo->type)break;
    if(brd==NI_MAX_BOARDS){printk("cant alloc device for NI PCI\n"); continue;}


    dev=&ni_pd[brd];
    
    pcibios_read_config_byte(pci_bus, pci_dev, PCI_INTERRUPT_LINE, 
			     &pci_irq_line);
    pcibios_read_config_word(pci_bus, pci_dev,
			     PCI_COMMAND, &pci_command);
    pcibios_write_config_word(pci_bus, pci_dev,
			      PCI_COMMAND, pci_command);
    pcibios_read_config_dword (pci_bus, pci_dev, 
			       PCI_BASE_ADDRESS_0, &mite_address);
    pcibios_read_config_dword (pci_bus, pci_dev,
			       PCI_BASE_ADDRESS_1, &board_address);
	
    dev->virt_mite= (u32) ioremap ((unsigned long)mite_address, (unsigned long)0x1000);
    writel (((0xffffff00 & board_address) | 0x00000080), 
	    dev->virt_mite + 0xc0);
    dev->base = (unsigned long)ioremap ((unsigned long)board_address, (unsigned long)0x1000);

    dev->type = dinfo->type;
    dev->dev_info=dinfo;
    dev->irq=pci_irq_line;
    //dev->irq=-1;
    if(init_device(brd, dev)<0) {dev->type=0; continue;} /* cant init device */
      
    printk("board %d: %s at memory=%lx irq=%d\n", brd, dinfo->dev_name,
	   (unsigned long)mite_address, dev->irq);

    } /* end PCI devices */
  
  return 0;
} 

void cleanup_module(void)
{
  pd dev;
  int brd;

  for(brd=0; brd<NI_MAX_BOARDS; brd++){
    dev=&ni_pd[brd];
    if(!dev->type)continue;
    if(dev->irq!=-1)free_irq(dev->irq, dev);
    if(dev->type&NI_ESER){
      kfree(dev->aibuff);
      kfree(dev->aobuff);
      kfree(dev->gc[0].buff);
      kfree(dev->gc[1].buff);
    }
    if(!(dev->type&NI_PCI))release_region(dev->base, dev->io_region_size);
    else { iounmap ((void *)dev->virt_mite); iounmap ((void *)dev->base); }
    printk("board %d uninstalled\n", brd);
  }

  if(module_unregister_chrdev(major, name))
        printk("unable to unregister %s device\n", name);
}

static int ai_wk;
static int eser_A_interrupt (pd dev) /* return 0 if it wasn't the cause  */
{
  u16 uValue;

  dev->aiuStatus=DAQ_STC_Windowed_Mode_Read(dev, AI_Status_1_Register);

  if(!(dev->aiuStatus&0x8000))return 0;

  if(dev->aiuStatus & 0x0004) gpct_interrupt(dev, 0); /* G0 Gate int */
  if(dev->aiuStatus & 0x0A00) { /* FIFO overflow or AI overrun */
    dev->ai_FIFO_oflow=1;ai_wk=1;
    DAQ_STC_Windowed_Mode_Masked_Write(dev, Interrupt_A_Ack_Register,
				       0x0080, 0x0080);
  }
  if (!(dev->aiuStatus & 0x1000)){   
    uValue = Board_Read(dev, ADC_FIFO_Data_Register);
    dev->ai_points_read++; ai_wk=1;
    dev->aibuff[dev->ai_pwrite++]=  uValue;
    if(dev->ai_pwrite==AIBUFFSIZE) dev->ai_pwrite=0;
    if(dev->ai_pread == dev->ai_pwrite) dev->ai_BUFF_oflow=1;
  }
  if(dev->aiuStatus & 0x0040) {dev->ai_running=0; ai_wk=1;} /* SC TC */
  return 1;
}

static int eser_B_interrupt (pd dev)
{
  dev->aouStatus=DAQ_STC_Windowed_Mode_Read(dev, AO_Status_1_Register);
  if(!(dev->aouStatus&0x8000))return 0;
  if(dev->aouStatus&0x0004)gpct_interrupt(dev, 1); /* G1 Gate int */
  return 1;
}

static void eser_daq_interrupt(int irq, void *dev_id, struct pt_regs *regs)
{
 
  pd dev;
 
  int i;

  dev=dev_id;
  dev->no_of_interrupts++;

  ai_wk=0;

  for(i=0;i<100000;i++) {
    if(!eser_A_interrupt(dev) && !eser_B_interrupt(dev)) break;
  }

  if(ai_wk) wake_up_interruptible(&dev->ai_wait_queue); 
}





