#include <stdio.h>
#include <math.h>

#include "xsimX.h"
#include "xtools.h"
#include "xtoolsP.h"

#define NUM_PATH_POINTS 15000
#define NUM_START_POINTS 10
#define NUM_WORLD_SEGMENTS 1000
#define NUM_SENSOR_PIXELS 100
#define NUM_BEACONS 100

#define BEACON_NOISE_ON 1
#define RECKON_NOISE_ON 1

static int   path_n;
static int   path_i;
static int   path_done = 0;
static float path_x[NUM_PATH_POINTS];
static float path_y[NUM_PATH_POINTS];

float goal_x;
float goal_y;
int   start_n;
float start_x[NUM_START_POINTS];
float start_y[NUM_START_POINTS];

int   world_n;
int   world_i;
int   world_r0;
int   world_c0;
int   world_dr;
int   world_dc;
float world_x0;
float world_y0;
float world_xf;
float world_yf;
int   world_pi[NUM_WORLD_SEGMENTS];
int   world_vi[NUM_WORLD_SEGMENTS];
float world_x1[NUM_WORLD_SEGMENTS];
float world_y1[NUM_WORLD_SEGMENTS];
float world_x2[NUM_WORLD_SEGMENTS];
float world_y2[NUM_WORLD_SEGMENTS];

float robot_width;
float robot_height;
float robot_x;
float robot_y;
float robot_th;

float sensor_x;
float sensor_y;
float sensor_yaw;
float sensor_maxr;
float sensor_fov;
int   sensor_pixels;

float K_len;
float N_len;
float K_gm;
float N_gm;
float K_r;
float N_r;
float K_alp;
float N_alp;
float K_phi;
float N_phi;

float beacon_x[NUM_BEACONS];
float beacon_y[NUM_BEACONS];
int beacon_n;
float beacon_minr;
float beacon_maxr;

xsim_object xtools_create_robot();
xsim_object xtools_create_hmmwv();

xsim_window world_window; 
int dmat[6][6];

static float gauss();
static float ran3();


static float normalize(a)
float a;
{
  while (a < 0.0)
      a += 2*M_PI;
  while (a >= 2*M_PI)
      a -= 2*M_PI;
  return(a);
}
static float normalize180(a)
float a;
{
  while (a <= -M_PI)
      a += 2*M_PI;
  while (a > M_PI)
      a -= 2*M_PI;
  return(a);
}

xsim_window xtools_get_window()
{
  return(world_window);
}

xtools_init(file)
char file[]; 
{
FILE *stream;
char ifile[80];
int i;
float x,y,w,h;

stream = fopen(file,"r");
if( !stream ) { printf("could not open config file\n"); return; }
fscanf(stream,"%s\n",ifile);
printf("scanning %s\n",ifile);
xtools_load_world_file(ifile);
world_window = xsim_create_window("World",world_r0,world_c0,world_dr,world_dc);
xsim_set_event_handler(world_window,xtools_event_proc);
sleep(2);

x = (world_x0+world_xf)/2.0; y = (world_y0+world_yf)/2.0;
w = (world_xf-world_x0);     h = (world_yf-world_y0);
xsim_set_viewport(world_window,x,y,w,h);
xtools_render_world();



fscanf(stream,"%s\n",ifile);
printf("scanning %s\n",ifile);
xtools_load_path_file(ifile);
xtools_render_path();

fscanf(stream,"%s\n",ifile);
printf("scanning %s\n",ifile);
xtools_load_robot_file(ifile);

fscanf(stream,"%s\n",ifile);
printf("scanning %s\n",ifile);
xtools_load_sensor_file(ifile);

fscanf(stream,"%s\n",ifile);
printf("scanning %s\n",ifile);
xtools_load_beacon_file(ifile);
xtools_render_beacons();

fscanf(stream,"%s\n",ifile);
printf("scanning %s\n",ifile);
xtools_load_error_file(ifile);

fscanf(stream,"%s\n",ifile);
printf("scanning %s\n",ifile);
xtools_load_plan_file(ifile);
}

xtools_load_world_file(file)
char file[];
{
FILE *stream;
int i;

stream = fopen(file,"r");
if( !stream ) { printf("could not open world file\n"); return; }
fscanf(stream,"world_n: %d\n",&world_n);
if( world_n > NUM_WORLD_SEGMENTS ) 
  { printf("too many world segments\n"); return; }
fscanf(stream,"world_r0: %d world_c0: %d world_dr: %d world_dc: %d\n",
       &world_r0,&world_c0,&world_dr,&world_dc);
fscanf(stream,"world_x0: %f world_y0: %f world_xf: %f world_yf: %f\n",
       &world_x0,&world_y0,&world_xf,&world_yf);

fscanf(stream,"following lines are x1 y1 x2 y2 line segments\n");

for(i=0;i<world_n;i++)
  {
  fscanf(stream,"%d %d %f %f %f %f\n",
    &(world_pi[i]),&(world_vi[i]),
    &(world_x1[i]),&(world_y1[i]),&(world_x2[i]),&(world_y2[i]));
  }
world_i = 0;
fclose(stream);
return;
}

xtools_load_path_file(file)
char file[];
{
FILE *stream;
int i;

stream = fopen(file,"r");
if( !stream ) { printf("could not open path file\n"); return; }
fscanf(stream,"path_n: %d\n",&path_n);
if( path_n > NUM_PATH_POINTS ) 
  { printf("too many path points\n"); return; }
fscanf(stream,"following lines are x y coordinates of each point\n");
for(i=0;i<path_n;i++)
  { fscanf(stream,"%f %f\n",&(path_x[i]),&(path_y[i])); }
path_i = 0;
fclose(stream);
}

xtools_load_plan_file(file)
char file[];
{
FILE *stream;
int i;

stream = fopen(file,"r");
if( !stream ) { printf("could not open plan file\n"); return; }
fscanf(stream,"goal_x: %f goal_y: %f\n",&goal_x,&goal_y);
fscanf(stream,"start_n: %d\n",&start_n);
if( start_n > NUM_START_POINTS ) 
  { printf("too many start points\n"); return; }
fscanf(stream,"following lines are x y coordinates of each start point\n");
for(i=0;i<start_n;i++)
  { fscanf(stream,"%f %f\n",&(start_x[i]),&(start_y[i])); }
fclose(stream);
}

xtools_dump_path_file(file)
char file[];
{
FILE *stream;
int i;

stream = fopen(file,"w");
if( !stream ) { printf("could not open path file\n"); return; }
fprintf(stream,"%d\n",path_n);
if( path_n > NUM_PATH_POINTS ) 
  { printf("too many path points\n"); return; }
for(i=0;i<path_n;i++)
  { fprintf(stream,"%f %f\n",path_x[i],path_y[i]); }
path_i = 0;
fclose(stream);
}


xtools_load_beacon_file(file)
char file[];
{
FILE *stream;
int i;

stream = fopen(file,"r");
if( !stream ) { printf("could not open beacon file\n"); return; }
fscanf(stream,"beacon_n: %d\n",&beacon_n);
fscanf(stream,"beacon_minr: %f\n",&beacon_minr);
fscanf(stream,"beacon_maxr: %f\n",&beacon_maxr);
if( beacon_n > NUM_BEACONS ) 
  { printf("too many start points\n"); return; }
for(i=0;i<beacon_n;i++)
  { fscanf(stream,"%f %f\n",&(beacon_x[i]),&(beacon_y[i])); }
fclose(stream);
}


xtools_load_robot_file(file)
char file[];
{
FILE *stream;
int i;

stream = fopen(file,"r");
if( !stream ) { printf("could not open robot file\n"); return; }

fscanf(stream,"robot_width: %f robot_height: %f\n",&robot_width,&robot_height);
fclose(stream);
}

xtools_load_error_file(file)
char file[];
{
FILE *stream;
int i;

stream = fopen(file,"r");
if( !stream ) { printf("could not open error file\n"); return; }

fscanf(stream,"K_len: %f\n",&K_len);
fscanf(stream,"N_len: %f\n",&N_len);
fscanf(stream,"K_gm: %f\n",&K_gm);
fscanf(stream,"N_gm: %f\n",&N_gm);
fscanf(stream,"K_r: %f\n",&K_r);
fscanf(stream,"N_r: %f\n",&N_r);
fscanf(stream,"K_alp: %f\n",&K_alp);
fscanf(stream,"N_alp: %f\n",&N_alp);
fscanf(stream,"K_phi: %f\n",&K_phi);
fscanf(stream,"N_phi: %f\n",&N_phi);

fclose(stream);
}

xtools_load_sensor_file(file)
char file[];
{
FILE *stream;
int i;

stream = fopen(file,"r");
if( !stream ) { printf("could not open sensor file\n"); return; }

fscanf(stream,"sensor_x: %f sensor_y: %f\n",&sensor_x,&sensor_y);
fscanf(stream,"sensor_yaw: %f sensor_fov: %f\n",&sensor_yaw,&sensor_fov);
sensor_yaw = sensor_yaw * M_PI/180.0;
sensor_fov = sensor_fov * M_PI/180.0;
fscanf(stream,"sensor_maxr: %f sensor_pixels: %d\n",
               &sensor_maxr,&sensor_pixels);
if( sensor_pixels > NUM_SENSOR_PIXELS ) 
  { printf("too many pixels\n"); return; }
fclose(stream);
}

xtools_load_path_from_mouse()
{
printf("drag left mouse button to enter path, release when done\n");
path_done = 0;
while(path_done == 0)
  {
  xsim_display_poll(world_window);
  }
smooth_path(path_x,path_y,&path_n);
}
/*
** Pure Pursuit Smoothing
*/
smooth_path(xi,yi,ni)
float xi[],yi[];
int   *ni;
{
float xo[NUM_PATH_POINTS],yo[NUM_PATH_POINTS];
float look = 2.0;
float step = 0.1;
int   ii=0,oo=0;

xo[0] = xi[0]; yo[0] = yi[0];

while(ii < *ni)
  {
  float x1,y1,x2,y2,th;
  while(sqrt(pow(xo[oo]-xi[ii],2.0)+pow(yo[oo]-yi[ii],2.0)) < look && ii<*ni)
    ii++;
  if( ii == *ni ) break;
  x1 = xo[oo]; y1 = yo[oo]; x2 = xi[ii]; y2 = yi[ii];
  th = atan2(y2-y1,x2-x1);
  oo++;
  if( oo == NUM_PATH_POINTS ) break;
  xo[oo] = xo[oo-1] + step * cos(th);
  yo[oo] = yo[oo-1] + step * sin(th);
  }
for(ii=0;ii<oo+1;ii++)
  {
  xi[ii] = xo[ii];
  yi[ii] = yo[ii];
  }
*ni = oo+1;
}

xtools_get_first_point(x,y,th)
float *x,*y,*th;
{
*x = path_x[0];
*y = path_y[0];
*th = atan2(path_y[1]-path_y[0],path_x[1]-path_x[0])-M_PI/2.0;
}
 
xtools_render_world()
{
int i;
static xsim_object world;
static float xx[2*NUM_WORLD_SEGMENTS],yy[2*NUM_WORLD_SEGMENTS];
static int ff[2*NUM_WORLD_SEGMENTS];

for(i=0 ; i<world_n ; i++)
  {
  xx[2*i  ]  = world_x1[i]; yy[2*i  ]  = world_y1[i]; ff[2*i  ]  = 0;
  xx[2*i+1]  = world_x2[i]; yy[2*i+1]  = world_y2[i]; ff[2*i+1]  = 1;
  }
world = xsim_create_polyline(xx,yy,ff,2*world_n);
xsim_draw_object(world_window,world,0.0,0.0,0.0,xsim_black);
}

/*
** Robot moves perfectly but is sensed incorrectly
*/

xtools_move_robot(s,gm,d)
float *s,*gm,d;
{
float x1,x2,x3,x4;
float y1,y2,y3,y4;
float sl,st;
float l12,t12;
float l14,t14;
float l32,t32;
int inc;

if( path_i >= path_n - 1 ) return(0);

x1 = path_x[path_i]; y1 = path_y[path_i];

inc = 1;

/*
** Find a path segment at least d long
*/
do
  {
  if( path_i >= path_n - inc ) return(0);
  x2 = path_x[path_i+inc]; y2 = path_y[path_i+inc];
  l12 = sqrt((x1-x2)*(x1-x2)+(y1-y2)*(y1-y2));
  t12 = atan2(y2-y1,x2-x1);
  inc++;
  }
while( l12 < d);
/*
** Now 1-2 is the current segment
*/

/*
** Move d units in direction of current segment from last position
*/
x3 = robot_x; y3 = robot_y;
x4 = x3 + d*cos(t12);
y4 = y3 + d*sin(t12);

l14 = sqrt((x1-x4)*(x1-x4)+(y1-y4)*(y1-y4));
/*t14 = atan2(y4-y1,x4-x1);*/

if( l14 > l12 )  /* at segment end, small step from 3 to 2 */
  {
  path_i++;
  l32 = sqrt((x3-x2)*(x3-x2)+(y3-y2)*(y3-y2));
  /*t32 = atan2(y2-y3,x2-x3);*/
  *s = l32;
  *gm =  normalize180(t12-(M_PI/2.0)- robot_th);
      /*t12;*/ /* not t32 because its backwards */  
  robot_x = x2;
  robot_y = y2;
  robot_th = t12-M_PI/2.0;
  }
else /* step from 3 to 4 */
  {
  *s = d;
  *gm = normalize180(t12-(M_PI/2.0)-robot_th);
         /* t12 */
  robot_x = x4;
  robot_y = y4;
  robot_th = t12-M_PI/2.0;

  }

if (RECKON_NOISE_ON) {
  if( *s > 1e-20 ) sl = K_len * pow(*s,N_len); else sl = 0.0;
  if( *gm > 1e-20) st = K_gm * pow(*gm,N_gm); else st = 0.0;
  sl *= gauss();
  st *= gauss();
/*  printf("sigma_l,sigma_gm = %f,%f\n",sl,st);*/
  
  *s = *s + sl;
  *gm = normalize180(*gm + st);
}

return(1);
}

xtools_init_robot_on_path()
{
float x1,x2,y1,y2;

if( path_n < 1 ) {printf("error-no path\n"); exit(0);}

x1 = path_x[0]; y1 = path_y[0];
x2 = path_x[1]; y2 = path_y[1];

robot_x = x1; 
robot_y = y1;
robot_th = atan2(y2-y1,x2-x1)-M_PI/2.0;

}

xtools_draw_robot(x,y,th)
float x,y,th;
{
static int robot_inited = 0;
static xsim_object robot;

if( ! robot_inited )
  {
  robot = xtools_create_robot();
/*  xsim_make_object_scroll_window(robot);*/
  robot_inited = 1;
  }
xtools_draw_object(robot,x,y,th,xsim_red);
}

xtools_draw_hmmwv(x,y,th)
float x,y,th;
{
static int robot_inited = 0;
static xsim_object robot;

if( ! robot_inited )
  {
  robot = xtools_create_hmmwv();
  xsim_make_object_scroll_window(robot);
  robot_inited = 1;
  }
xtools_draw_object(robot,x,y,th,xsim_red);
}

xtools_draw_object(obj,x,y,th,color)
xsim_object obj;
float x,y,th;
int color;
{
xsim_draw_object(world_window,obj,x,y,th,color);
}

xtools_clear_signal_window(win)
xsim_window win;
{
xsim_clear_window(win);
}

xsim_window xtools_make_signal_window(nam,row,col,rows,cols,xmin,xmax,ymin,ymax)
char nam[];
int  row,col,rows,cols;
float xmin,xmax,ymin,ymax;
{
float x,y,w,h;
xsim_window win;

win = xsim_create_window(nam,row,col,rows,cols);
x = (xmin+xmax)/2.0; y = (ymin+ymax)/2.0;
w = (xmax-xmin);     h = (ymax-ymin);
xsim_set_viewport(win,x,y,w,h);
xsim_clear_window(win);
return(win);
}

xtools_displ_signal_with_scroll(win,x,y,n,color)
xsim_window win;
float x[],y[];
int   n,color;
{
int i,*f;
static xsim_object path;

xsim_scroll_window(win,x[n-1],y[n-1],0.01,0.1);
f = (int *) malloc(n * sizeof(int));
for(i=0 ; i<n ; i++) { if( i==0 ) f[i] = 0; else f[i] = 1; }
path = xsim_create_polyline(x,y,f,n);
xsim_draw_object(win,path,0.0,0.0,0.0,color);
free(f);
}

xtools_displ_signal(win,x,y,n,color)
xsim_window win;
float x[],y[];
int   n,color;
{
int i,*f;
static xsim_object path;

f = (int *) malloc(n * sizeof(int));
for(i=0 ; i<n ; i++) { if( i==0 ) f[i] = 0; else f[i] = 1; }
path = xsim_create_polyline(x,y,f,n);
xsim_draw_object(win,path,0.0,0.0,0.0,color);
free(f);
}

xsim_object xtools_create_robot()
{
xsim_object robot;
float xx[25],yy[25];
int ff[25];
float x1,x2,x3,x4,x5,x6;
float y1,y2,y3,y4,y5,y6;

x1 = - 0.8   * robot_width/2.0;  
x2 =   0.8   * robot_width/2.0;
x3 = - 1.0   * robot_width/2.0;
x4 = - 0.825 * robot_width/2.0;
x5 =   0.825 * robot_width/2.0;
x6 =   1.0   * robot_width/2.0; 
y1 = - 1.0   * robot_height/2.0; 
y2 =   1.0   * robot_height/2.0;
y3 = - 0.33  * robot_height/2.0;
y4 = - 0.66  * robot_height/2.0;
y5 =   0.33  * robot_height/2.0;
y6 =   0.66  * robot_height/2.0;  

xx[ 0]  = x1; yy[ 0]  = y1; ff[ 0]  = 0;
xx[ 1]  = x2; yy[ 1]  = y1; ff[ 1]  = 1;
xx[ 2]  = x2; yy[ 2]  = y2; ff[ 2]  = 1;
xx[ 3]  = x1; yy[ 3]  = y2; ff[ 3]  = 1;
xx[ 4]  = x1; yy[ 4]  = y1; ff[ 4]  = 1;

xx[ 5]  = x3; yy[ 5]  = y3; ff[ 5]  = 0;
xx[ 6]  = x4; yy[ 6]  = y3; ff[ 6]  = 1;
xx[ 7]  = x4; yy[ 7]  = y4; ff[ 7]  = 1;
xx[ 8]  = x3; yy[ 8]  = y4; ff[ 8]  = 1;
xx[ 9]  = x3; yy[ 9]  = y3; ff[ 9]  = 1;

xx[10]  = x5; yy[10]  = y3; ff[10]  = 0;
xx[11]  = x6; yy[11]  = y3; ff[11]  = 1;
xx[12]  = x6; yy[12]  = y4; ff[12]  = 1;
xx[13]  = x5; yy[13]  = y4; ff[13]  = 1;
xx[14]  = x5; yy[14]  = y3; ff[14]  = 1;

xx[15]  = x3; yy[15]  = y5; ff[15]  = 0;
xx[16]  = x4; yy[16]  = y5; ff[16]  = 1;
xx[17]  = x4; yy[17]  = y6; ff[17]  = 1;
xx[18]  = x3; yy[18]  = y6; ff[18]  = 1;
xx[19]  = x3; yy[19]  = y5; ff[19]  = 1;

xx[20]  = x5; yy[20]  = y5; ff[20]  = 0;
xx[21]  = x6; yy[21]  = y5; ff[21]  = 1;
xx[22]  = x6; yy[22]  = y6; ff[22]  = 1;
xx[23]  = x5; yy[23]  = y6; ff[23]  = 1;
xx[24]  = x5; yy[24]  = y5; ff[24]  = 1;

robot = xsim_create_polyline(xx,yy,ff,25);
return(robot);
} 

xsim_object xtools_create_hmmwv()
{
static float xx[17],yy[17];
static int ff[17];

float x1,x2,x3,x4;
float y1,y2,y3,y4;

x1 = - robot_width/2.0;
x2 = - 0.6 * robot_width/2.0;
x3 =   0.6 * robot_width/2.0;
x4 =   robot_width/2.0;

y1 = - 0.1 * robot_height;
y2 =   0.8 * robot_height - 2.0;
y3 =   0.8 * robot_height - 0.8;
y4 =   0.8 * robot_height + 0.1 * robot_height;

xx[0]  = x1; yy[0]  = y1; ff[0]  = 1;
xx[1]  = x2; yy[1]  = y1; ff[1]  = 1;
xx[2]  = x3; yy[2]  = y1; ff[2]  = 1;
xx[3]  = x4; yy[3]  = y1; ff[3]  = 1;
xx[4]  = x4; yy[4]  = y3; ff[4]  = 1;
xx[5]  = x4; yy[5]  = y4; ff[5]  = 1;
xx[6]  = x1; yy[6]  = y4; ff[6]  = 1;
xx[7]  = x1; yy[7]  = y3; ff[7]  = 1;
xx[8]  = x1; yy[8]  = y1; ff[8]  = 1;

xx[9]  = x2; yy[9]  = y1; ff[9]  = 0;
xx[10] = x2; yy[10] = y2; ff[10] = 1;
xx[11] = x3; yy[11] = y2; ff[11] = 1;
xx[12] = x3; yy[12] = y1; ff[12] = 1;

xx[13] = x2; yy[13] = y2; ff[13] = 0; 
xx[14] = x1; yy[14] = y3; ff[14] = 1;
xx[15] = x4; yy[15] = y3; ff[15] = 1;
xx[16] = x3; yy[16] = y2; ff[16] = 1;

return(xsim_create_polyline(xx,yy,ff,17));
}

xtools_render_path()
{
int i;
static xsim_object path=NULL;
int    *flags;

flags = (int *) malloc(path_n * sizeof(int));
for(i=0 ; i<path_n ; i++) flags[i] = 1;
path = xsim_create_polyline(path_x,path_y,flags,path_n);
free(flags);
xsim_draw_object(world_window,path,0.0,0.0,0.0,xsim_black);
}

xtools_render_beacons()
{
int i;
static xsim_object *beacon_small;
static xsim_object *beacon_large;
int    flags[6];
float px[5],py[6];

beacon_small = (xsim_object *) malloc(beacon_n*sizeof(xsim_object));
beacon_large = (xsim_object *) malloc(beacon_n*sizeof(xsim_object));

flags[0] = flags[1]= 0; flags[2]=flags[3]=flags[4]=flags[5] = 1;
px[0] = py[0] = 0.0;
px[1] = 0.0; py[1] = BEACON_INNER_RADIUS;
px[2] = BEACON_INNER_RADIUS; py[2] = 0.0;
px[3] = 0.0; py[3] = -BEACON_INNER_RADIUS;
px[4] = -BEACON_INNER_RADIUS; py[4] = 0.0;
px[5] = 0.0; py[5] = BEACON_INNER_RADIUS;
for (i=0;i<beacon_n;i++)
    beacon_small[i] = xsim_create_polyline(px,py,flags,6);
px[1] = 0.0; py[1] = BEACON_OUTER_RADIUS;
px[2] = BEACON_OUTER_RADIUS; py[2] = 0.0;
px[3] = 0.0; py[3] = -BEACON_OUTER_RADIUS;
px[4] = -BEACON_OUTER_RADIUS; py[4] = 0.0;
px[5] = 0.0; py[5] = BEACON_OUTER_RADIUS;
for (i=0;i<beacon_n;i++)
    beacon_large[i] = xsim_create_polyline(px,py,flags,6);
for(i=0 ; i<beacon_n ; i+= 1) {
  xsim_draw_object(world_window,beacon_small[i],beacon_x[i],beacon_y[i],
		   0.0,xsim_black);
  xsim_draw_object(world_window,beacon_large[i],beacon_x[i],beacon_y[i],
		   0.0,xsim_black);
}
}

xtools_render_ellipse(x,y,cov,pr)
float  x,y,pr;
float cov[2][2];
{
float a,b;
float sx2,sy2,sxy;
float A,B,C,D,T;
float k,th;
float xx[40],yy[40];
int i;
xsim_object curve=NULL;
int *flags;

k = sqrt(-2.0*log(1.0-pr));

sx2 = cov[0][0]; sxy = cov[0][1]; sy2 = cov[1][1];
D = sx2*sy2-sxy*sxy;
if( D < 1.0e-9 ) { return; }
A = sy2/D; B = - sxy/D; C = sx2/D;
T = sqrt(A*A + C*C - 2.0*A*C +4.0*B*B);

a = sqrt( 2.0*k*k/(A+C+T));
b = sqrt( 2.0*k*k/(A+C-T));
th = 0.5 * atan2(2.0*B,A-C);
flags = (int *) malloc (40 * sizeof(int));
for(i=0 ; i<40 ; i++) 
  {
  flags[i] = 1;
  xx[i] = a * cos(2*M_PI*i/39.0);
  yy[i] = b * sin(2*M_PI*i/39.0);
  }
curve = xsim_create_polyline(xx,yy,flags,40);
xsim_render_object(world_window,curve,x,y,th,xsim_magenta,XSIM_XOR);
xsim_destroy_object(curve);
}

xtools_render_line(x1,y1,x2,y2,color,mode)
float x1,y1,x2,y2;
int color,mode;
{
float  xx[2],yy[2];
int    ff[2];
xsim_object line;
float  size=1.0;

xx[0] = x1; yy[0] = y1;
xx[1] = x2; yy[1] = y2;
ff[0] = 0;  ff[1] = 1;

line = xsim_create_polyline(xx,yy,ff,2);
xsim_render_object(world_window,line,0.0,0.0,0.0,color,mode);
xsim_destroy_object(line);
}

xtools_render_cross(x,y,size)
float x,y,size;
{
float  xx[4],yy[4];
int    ff[4];
xsim_object cross;

xx[0] = x-size; yy[0] = y; ff[0] = 1;
xx[1] = x+size; yy[1] = y; ff[1] = 1;
xx[2] = x; yy[2] = y-size; ff[2] = 0;
xx[3] = x; yy[3] = y+size; ff[3] = 1;

cross = xsim_create_polyline(xx,yy,ff,4);
xsim_render_object(world_window,cross,0.0,0.0,0.0,xsim_black,XSIM_XOR);
xsim_destroy_object(cross);
}

xtools_draw_range_image(x,y,th,image)
float x,y,th;
float image[NUM_SENSOR_PIXELS];
{
static xsim_object range_image;
static int range_image_inited = 0;
float range,alp;
float x1,x2;
float y1,y2;
float xx[2*NUM_SENSOR_PIXELS],yy[2*NUM_SENSOR_PIXELS];
int   ff[2*NUM_SENSOR_PIXELS];
int i;

if( !range_image_inited )
  {
  for(i=0 ; i<sensor_pixels; i++) ff[2*i+1] = 1;
  range_image = xsim_create_polyline(xx,yy,ff,2*sensor_pixels);
  range_image_inited = 1;
  }
else
  {
  xsim_render_object(world_window,range_image,0.0,0.0,0.0,xsim_magenta,XSIM_XOR);
  }

polar_to_global(0.0,0.0,&x1,&y1,x,y,th);

for(i=0 ; i<sensor_pixels; i++)
  {
  range = image[i];
  image_to_polar(i,&alp);
  polar_to_global(range,alp,&x2,&y2,x,y,th);

  xsim_polyline_x(range_image,2*i+0)=x1; 
  xsim_polyline_y(range_image,2*i+0)=y1;
  xsim_polyline_f(range_image,2*i+0)=0;
  xsim_polyline_x(range_image,2*i+1)=x2; 
  xsim_polyline_y(range_image,2*i+1)=y2;
  xsim_polyline_f(range_image,2*i+1)=1;
  }
/*for (i=0;i<10000;i++);*/
xsim_render_object(world_window,range_image,0.0,0.0,0.0,xsim_magenta,XSIM_XOR);
}

xtools_draw_beacon_scan(x,y,th,bl,num)
float x,y,th;
beacon_list bl[];
int num;
{
  static xsim_object beacon_image;
  static int beacon_image_inited = 0;
  static int old_n = 0;
  float range,alp;
  float x1,x2;
  float y1,y2;
  float xx[2*NUM_BEACONS],yy[2*NUM_BEACONS];
  int   ff[2*NUM_BEACONS];
  int i;

  if( !beacon_image_inited )  {
      for(i=0 ; i<num; i++) { ff[(2*i)+1] = 1; ff[2*i] = 0;}
      beacon_image = xsim_create_polyline(xx,yy,ff,2*beacon_n);
      beacon_image_inited = 1;
    } else {
      if (old_n > 0) 
	  xsim_render_object(world_window,beacon_image,0.0,0.0,0.0,
			     xsim_black,XSIM_XOR);
    }
  
  x1=x;y1=y;

  xsim_polyline_n(beacon_image)=2*num;
  for(i=0 ; i<num; i++) {
    float r,ray_th;
    x2= beacon_x[bl[i].id]; y2 = beacon_y[bl[i].id];
    r = sqrt(SQR(x2-x) + SQR(y2-y));
    ray_th = normalize(th+bl[i].ang);

    xsim_polyline_x(beacon_image,2*i+0)=x; 
    xsim_polyline_y(beacon_image,2*i+0)=y;
    xsim_polyline_f(beacon_image,2*i+0)=0;
    xsim_polyline_x(beacon_image,2*i+1)=x+(r*cos(ray_th)); 
    xsim_polyline_y(beacon_image,2*i+1)=y+(r*sin(ray_th));
    xsim_polyline_f(beacon_image,2*i+1)=1;
  }

  xsim_render_object(world_window,beacon_image,0.0,0.0,0.0,
		     xsim_black,XSIM_XOR);
  old_n = num;
}


xtools_render_grey_cell(row,col,grey,max,size)
int row,col,grey,max,size;
{
  int i,j;
  int scaledint;
  float scale;

  switch(size)
    {
    case 1: 
      dmat[0][0] = 0;
      break;
    case 2:
      dmat[0][0] = 0; dmat [0][1] = 2;
      dmat[1][0] = 3; dmat [1][1] = 1;
      break;
    case 3:
      dmat[0][0]=6;dmat[0][1]=8;dmat[0][2]=4;
      dmat[1][0]=1;dmat[1][1]=0;dmat[1][2]=3;
      dmat[2][0]=5;dmat[2][1]=2;dmat[2][2]=7;
      break;
    case 4:
      dmat[0][0] = 0;dmat[0][1] = 8;dmat[0][2] = 2;dmat[0][3] = 10;
      dmat[1][0] = 12;dmat[1][1] = 4;dmat[1][2] = 14;dmat[1][3] = 6;
      dmat[2][0] = 3;dmat[2][1] = 11;dmat[2][2] = 1;dmat[2][3] = 9;
      dmat[3][0] =15;dmat[3][1] = 7;dmat[3][2] = 13;dmat[3][3] = 5;
      break;
    case 5:
      printf("not valid cell size");
      return;
      break;
    case 6:
      dmat[0][0] = 24;dmat[0][1] = 32;dmat[0][2] = 16;dmat[0][3] = 26;
      dmat[0][4] = 34;dmat[0][5] = 18;
      dmat[1][0] = 4;dmat[1][1] = 0;dmat[1][2] = 12;dmat[1][3] = 6;
      dmat[1][4] = 2;dmat[1][5] = 14;
      dmat[2][0] = 20;dmat[2][1] = 8;dmat[2][2] = 28;dmat[2][3] = 22;
      dmat[2][4] = 10;dmat[2][5] = 30;
      dmat[3][0] = 27;dmat[3][1] = 35;dmat[3][2] =19;dmat[3][3] = 25;
      dmat[3][4]  = 33;dmat[3][5] = 17;
      dmat[4][0] = 7;dmat[4][1] = 3;dmat[4][2] = 15;dmat[4][3] = 5;
      dmat[4][4] = 1;dmat[4][5] = 13;
      dmat[5][0] = 23;dmat[5][1] = 11;dmat[5][2] = 31;dmat[5][3] = 21;
      dmat[5][4] = 9;dmat[5][5] = 29;
      break;
    }
  scale = (float)grey/(float)max;
  scaledint = (int)(scale*(size*size-1.0));
  for (i=0;i<size;i++)
    for (j=0;j<size;j++)
      xtools_render_grey_pixel(row*size+i,col*size+j,scaledint,size);
}

xtools_render_grey_pixel(row,col,intensity,size)
int row,col,intensity,size;
{
  int i,j,k;
  

  i = row % size;
  j = col % size;
  
  if (dmat[i][j] >= intensity)
    {
      XSetForeground(dpy,gc,WhitePixel(dpy,scn));
      XDrawPoint(dpy,world_window,gc,col,row);
    }
  else
    {
      XSetForeground(dpy,gc,BlackPixel(dpy,scn));
      XDrawPoint(dpy,world_window,gc,col,row);
    }
}

xtools_generate_range_image(image)
float image[NUM_SENSOR_PIXELS];
{
float x1,x2,x3,x4,x5;
float y1,y2,y3,y4,y5;
int   i,j;
float alp,range;
float sr,salp;

polar_to_global(0.0,0.0,&x1,&y1,robot_x,robot_y,robot_th);

for(i=0 ; i<sensor_pixels ; i++)
  {
  image[i] = sensor_maxr;
  image_to_polar(i,&alp);
  salp = K_alp * pow(alp,N_alp);
  alp += salp * gauss();
  range = sensor_maxr;
  polar_to_global(range,alp,&x2,&y2,robot_x,robot_y,robot_th);
  for(j=0; j<world_n; j++)
    {
    x3 = world_x1[j]; y3 = world_y1[j]; 
    x4 = world_x2[j]; y4 = world_y2[j]; 
    if( intersect(x1,y1,x2,y2,x3,y3,x4,y4,&x5,&y5) )
      {
      global_to_polar(x5,y5,&range,&alp,robot_x,robot_y,robot_th);
      if( range < sensor_maxr )
        {
        image[i] = (range < image[i]) ? range : image[i];
        }
      }
    } 
  }
for(i=0 ; i<sensor_pixels ; i++)
  {
  sr = K_r * pow(image[i],N_r);
  image[i] += sr * gauss();
  }
}

static int beacon_comp(a,b)
beacon_list *a,*b;
{
  if (a->ang < b->ang)
      return(-1);
  if (a->ang > b->ang)
      return(1);
  return(0);
}

void xtools_generate_beacon_scan(beacon_found,number)
beacon_list beacon_found[];
int *number;
{
  
  /* No Noise, range limits or walls, yet. */
  int i,j;
  int beacon_found_count = 0;
  float range;
  float x1,x2,x3,x4,y1,y2,y3,y4,x5,y5;

  /* Compute angle between robot heading and every beacon */
  for (i=0;i<beacon_n;i++) {
    int bad;
    bad = 0;
    beacon_found[beacon_found_count].ang = atan2(beacon_y[i]-robot_y,
				       beacon_x[i]-robot_x) - robot_th;
    beacon_found[beacon_found_count].ang = 
	normalize(beacon_found[beacon_found_count].ang);

    /* Too short is just as bad is too long */
    range = sqrt(SQR(beacon_x[i] - robot_x) + SQR(beacon_y[i] - robot_y));
    if ((range < beacon_minr) || (range > beacon_maxr))
	bad = 1;

    /* Now check for walls */
    x1 = robot_x; y1=robot_y;
    x2 = beacon_x[i]; y2 = beacon_y[i];
    x2 = robot_x + 0.995*(beacon_x[i]-robot_x);
    y2 = robot_y + 0.995*(beacon_y[i]-robot_y);
    for(j=0; j<world_n; j++) {
      x3 = world_x1[j]; y3 = world_y1[j]; 
      x4 = world_x2[j]; y4 = world_y2[j]; 
      if( intersect(x1,y1,x2,y2,x3,y3,x4,y4,&x5,&y5) )
	  bad = 1;
    }
        
    if (bad != 1) {
      beacon_found[beacon_found_count].id = i;
      beacon_found_count++;
    }
  }
  /* Sort in counter clockwise order */
  qsort((char *) beacon_found, beacon_found_count,
	sizeof(beacon_list),beacon_comp);

  /* Convert to output format */
  for (i=0;i<beacon_found_count;i++) {
    float sang;
    i = i;

    /* Eventually add noise here */
    if (BEACON_NOISE_ON) {
      sang = K_phi * pow(beacon_found[i].ang,N_phi);
      beacon_found[i].ang += sang * gauss();
      beacon_found[i].ang =  normalize(beacon_found[i].ang);
      
    }

  }
  /* Now, randomly pick 3 */
  if (beacon_found_count > 3) {
    int i,j,r,n,dummy;
    beacon_list new_b[3];
    dummy=1;
    n = beacon_found_count;
    for (i=0;i<3;i++) {      
      r = floor(((float) (n))*ran3(&dummy));
      new_b[i] = beacon_found[r];
      for (j=r;j<beacon_found_count-1;j++)
	  beacon_found[j] = beacon_found[j+1];
      n--;
    }
    for (i=0;i<3;i++) {      
      beacon_found[i] = new_b[i];
    }
    beacon_found_count = 3;
  }

    
  qsort((char *) beacon_found, beacon_found_count,
	sizeof(beacon_list),beacon_comp);
  *number = beacon_found_count;
}


static intersect(x1,y1,x2,y2,x3,y3,x4,y4,x5,y5)
float x1,x2,x3,x4,*x5;
float y1,y2,y3,y4,*y5;
{
float a,b,c,d,e,f;
float det,s,t;

a = x2-x1; b = -(x4-x3);
c = y2-y1; d = -(y4-y3);
e = x3-x1; f = y3-y1;

det = a*d - c*b;

if( det == 0.0 ) return(0);

s = (d*e-b*f)/det;
t = (-c*e+a*f)/det;

if( s < 0.0 || s > 1.0) return(0);
if( t < 0.0 || t > 1.0) return(0);

*x5 = x1 + s*(x2-x1);
*y5 = y1 + s*(y2-y1);

return(1);
}

static float gauss() /* from Numerical Recipes */
{
static int idum = 1; /* original version passed this in, its a seed */
static int iset = 0;
static double gset;
float fac,r,v1,v2;
   
if (iset == 0) 
  { 
  do 
    {
    v1 = 2.0*ran3(&idum) - 1.0;
    v2 = 2.0*ran3(&idum) - 1.0;
    r = v1 * v1 + v2 * v2;
    } while (r >= 1.0);
  fac = sqrt(-2.0*log(r)/r);
  gset = v1 * fac;
  iset = 1;
  return v2*fac;
  }
else 
  {
  iset = 0;
  return gset;
  }
}

static float ran3(idum)
int *idum;
{
static long state1[32] = { 3,
0x9a319039,0x32d9c024,0x9b663182,0x5da1f342,
0x7449e56b,0xbeb1dbb0,0xab5c5918,0x946554fd,
0x8c2e680f,0xeb3d799f,0xb11ee0b7,0x2d436b86,
0xda672e2a,0x1588ca88,0xe369735d,0x904f35f7,
0xd7158fd6,0x6fa6f051,0x616e6b96,0xac94efdc,
0xde3b81e0,0xdfa6fb5,0xf103bc02,0x48f340fb,
0x36413f93,0xc622c298,0xf5a42ab8,0x8a88d77b,
0xf5ad9d0e,0x8999220b,0x27fb47b9
};
static unsigned SEED = 1;
int n;
long rand;
float r;
static countr = 0;
n = 128;
if (*idum < 0)
  {
  setstate(SEED,state1,n);
  setstate(state1);
  *idum = 0;
  }
rand = random()&(0xfffffff);
r = ((float) rand)  / (float) ((0xfffffff) + 1);
return((float) r);
}

test_gauss()
{
int i,j;
int bucket[200];
float ran,cov;
float sx,sy,sxy;

for(i=0; i<200; i++)
  bucket[i] = 0;

for(i=0; i<1000000; i++)
  {
  ran = gauss();
  j = (int) ((ran+5.0)*20.0);
  bucket[j]++;
  }
for(i=0; i<200; i++)
  {
  int jstep;
  jstep = bucket[100]/79;
  for(j=0; j<bucket[i]; j+=jstep)
    printf("*");
  printf("\n");
  }

cov = 0.0;
sx = sy = sxy = 0.0;
for(i=0; i<1000000; i++)
  {
  float x,y;
  x = gauss(); y = gauss();
  sx += x*x;
  sy += y*y;
  sxy += x*y;
  }
printf("covariance:%f\n",sxy/sqrt(sx*sy));
}

void xtools_event_proc(dpy, event, win)
Display *dpy;
XEvent *event;
xsim_window win;
{
if( event->type == KeyPress )
  {
  char text[1];
  int count;
  KeySym sym;
  count = XLookupString(event,text,1,0,0);
  if(count >0) 
    xtools_keypress_handler(text[0],0);
  else
    {
    sym = XLookupKeysym(event,0);
    xtools_keypress_handler('\0',sym);
    }
  }
else if(event->type == ButtonPress )
 {
  switch(event->xbutton.button)
  {
  case 1: xtools_mouse_handler(event->xbutton.x,event->xbutton.y,DOWN_LEFT);break;
  case 2: xtools_mouse_handler(event->xbutton.x,event->xbutton.y,DOWN_MIDDLE);break;
  case 3: xtools_mouse_handler(event->xbutton.x,event->xbutton.y,DOWN_RIGHT);break;
  } 
 }
else if(event->type == ButtonRelease )
  {
  switch(event->xbutton.button)
    {
    case 1: xtools_mouse_handler(event->xbutton.x,event->xbutton.y,UP_LEFT);break;
    case 2: xtools_mouse_handler(event->xbutton.x,event->xbutton.y,UP_MIDDLE);break;
    case 3: xtools_mouse_handler(event->xbutton.x,event->xbutton.y,UP_RIGHT);break;
    } 
  }
else if(event->type == MotionNotify )
  {
  xtools_mouse_handler(event->xbutton.x,event->xbutton.y,DRAG);
  }
else if(event->type == Expose )
  {
  xsim_expose_handler(win,event->xany.display,event->xany.window);
  }
}

static void xtools_mouse_handler(mx, my, action)
int mx, my;
int action;
{
int r,c;

switch (action) 
  {
  case UP_LEFT: 
    {
    path_n = path_i;
    path_done = 1;
    for(path_i=0; path_i<path_n-1; path_i++)
      xtools_render_line(path_x[path_i+0],path_y[path_i+0], 
                         path_x[path_i+1],path_y[path_i+1],xsim_black,XSIM_COPY);
    path_i = 0;
    break;
    }
  case DOWN_LEFT: 
    {
    path_i = 0; 
    path_x[path_i] = (float)mx/(float)world_dc*(world_xf-world_x0);
    path_y[path_i] = (float)(world_dc-my)/(float)world_dr*(world_yf-world_y0);
    path_i++; 
    break;
    }
  case UP_MIDDLE: printf("xtools_mouse_handler: UP_MIDDLE\n"); break;
  case DOWN_MIDDLE: printf("xtools_mouse_handler: DOWN_MIDDLE\n"); break;
  case UP_RIGHT: printf("xtools_mouse_handler: UP_RIGHT\n"); break;
  case DOWN_RIGHT: printf("xtools_mouse_handler: DOWN_RIGHT\n"); break;
  case MOVE: printf("xtools_mouse_handler: MOVE\n"); break;
  case DRAG:
    {
    path_x[path_i] = (float)mx/(float)world_dc*(world_xf-world_x0);
    path_y[path_i] = (float)(world_dc-my)/(float)world_dr*(world_yf-world_y0);
    path_i++; 
    if(path_i>1) xtools_render_line(path_x[path_i-2],path_y[path_i-2], 
                                  path_x[path_i-1],path_y[path_i-1],xsim_black,XSIM_COPY);
    break;
    }
  }
}

static void xtools_keypress_handler(c,sym)
char c;
KeySym sym;
{
if( c != '\0')
  {
  if( c == 'q' )
    exit(0);
  else if( c == 'i' )
    xsim_zoom(world_window,0.9);
  else if( c == 'o' )
    xsim_zoom(world_window,1.1);
  else
    printf("xtools_keypress_handler: %c\n",c);
  }
else
  {
  if( sym == XK_Up )         xsim_pan(world_window, 0.0, 0.1);
  else if( sym == XK_Down )  xsim_pan(world_window, 0.0,-0.1);
  else if( sym == XK_Left )  xsim_pan(world_window,-0.1, 0.0);
  else if( sym == XK_Right ) xsim_pan(world_window, 0.1, 0.0);
  }
}

