#include <stdio.h>
#include "matrix.h"
/*****************************************************************************/
/*                                                                           */
/*                     THE FRIENDLY MATRIX LIBRARY                           */
/*                                                                           */
/* Written by Al Kelly in pinch. Concept is for elegance and not for speed.  */
/* All functions are smart about matrix dimensions relieving user from having*/
/* to pass all these around as arguments. Downside of this is run-time       */
/* indexing. Looks inefficient but there are usually much bigger fish to fry */
/* in the application code if you want efficiency.                           */
/*                                                                           */
/* Collision detection is automatic, that is something like matmul(A,A,B)    */
/* is automatically detected and dealt with, so you don't have to worry about*/
/* it. This makes for nice readable code.                                    */
/*                                                                           */
/*                                                                           */
/* 1) By convention, all functions free any memory they allocate EXCEPT any  */
/* structures which are passed back.                                         */
/*                                                                           */
/* In general, do not call a matrix function inside another call and you'll  */
/* be OK.                                                                    */
/*                                                                           */
/* 2) By convention, arguments are not destroyed                             */
/*                                                                           */
/* 4) Error checking is overdone in order to aid debugging.                  */
/*                                                                           */
/*                                                                           */
/*                                                                           */
/*****************************************************************************/
#define ABS(x) (((x) < 0.0) ?  (-(x)) : (x))

dvector newdvec(rows)
int rows;
{
dvector vec;
vec = (struct dvector_s *) malloc( sizeof (struct dvector_s) );
vec->rows = rows;
vec->cols = 1;
vec->buf = (int *) malloc( rows*sizeof(int) );
return(vec);
}

vector newvec(rows)
int rows;
{
vector vec;
vec = (struct vector_s *) malloc( sizeof (struct vector_s) );
vec->rows = rows;
vec->cols = 1;
vec->buf = (float *) malloc( rows*sizeof(float) );
return(vec);
}

matrix newmat(rows,cols)
int rows,cols;
{
matrix mat;
mat = (struct matrix_s *) malloc( sizeof (struct matrix_s) );
mat->rows = rows;
mat->cols = cols;
mat->buf = (float *) malloc( rows*cols*sizeof(float) );
return(mat);
}

freemat(mat)
matrix mat;
{
if( mat != NULL )
  {
  if(mat->buf != NULL) free(mat->buf);
  free(mat);
  }
}

matprint(mat,str,expr)
matrix mat;
char *str;
int expr;
{
int i,j;
if( expr )
  {
  printf("----------------------------%s--------------------------\n",str);
  for(i=0;i<mat->rows;i++)
    {
    for(j=0 ; j<mat->cols;j++) printf("%9.2lf ",m_el(mat,i,j));
    printf("\n");
    }
  printf("----------------------------------------------------------------\n");
  }
}

matident(mat)
matrix mat;
{
int i,j;

materr(mat->rows!=mat->cols,"matident: error - matrix not square");

for(i=0 ; i<mat->rows; i++) for(j=0 ; j<mat->cols; j++)
  {
  if(i == j) m_el(mat,i,i) = 1.0;
  else       m_el(mat,i,j) = 0.0;
  }
}

matdzero(mat)
dmatrix mat;
{
int i,j;

for(i=0 ; i<mat->rows; i++) for(j=0 ; j<mat->cols; j++) { m_el(mat,i,j) = 0; }
}

matzero(mat)
matrix mat;
{
int i,j;

for(i=0 ; i<mat->rows; i++) for(j=0 ; j<mat->cols; j++) { m_el(mat,i,j) = 0.0; }
}

matdiagel(mat,val)
matrix mat;
float val;
{
int i,j;

for(i=0 ; i<mat->rows; i++) for(j=0 ; j<mat->cols; j++)
  {
  if(i == j) m_el(mat,i,i) = val;
  else       m_el(mat,i,j) = 0.0;
  }
}

matdiag(mat,vec)
matrix mat;
vector vec;
{
int i,j;
materr(mat->rows!=vec->rows,"matdiag: error - size mismatch");
materr(mat->cols!=vec->rows,"matdiag: error - size mismatch");

for(i=0 ; i<mat->rows; i++) for(j=0 ; j<mat->cols; j++)
  {
  if(i == j) m_el(mat,i,i) = v_el(vec,i);
  else       m_el(mat,i,j) = 0.0;
  }
}

matcopy(A,B)
matrix A,B;
{
int i,j;

for(i=0 ; i<A->rows; i++) for(j=0 ; j<A->cols; j++) { m_el(A,i,j) = m_el(B,i,j); }
}

matadd(A,B,C)
matrix A,B,C;
{
int i,j,k;

materr(A->rows!=B->rows,"matmul: error - AB size mismatch");
materr(A->cols!=B->cols,"matmul: error - AB size mismatch");
materr(B->rows!=C->rows,"matmul: error - BC size mismatch");
materr(B->cols!=C->cols,"matmul: error - BC size mismatch");

for(i=0 ; i<A->rows; i++)
  for(j=0 ; j<A->cols ; j++)
    {
    m_el(A,i,j) = m_el(B,i,j)+m_el(C,i,j);
    }

}

matsub(A,B,C)
matrix A,B,C;
{
int i,j,k;

materr(A->rows!=B->rows,"matmul: error - AB size mismatch");
materr(A->cols!=B->cols,"matmul: error - AB size mismatch");
materr(B->rows!=C->rows,"matmul: error - BC size mismatch");
materr(B->cols!=C->cols,"matmul: error - BC size mismatch");

for(i=0 ; i<A->rows; i++)
  for(j=0 ; j<A->cols ; j++)
    {
    m_el(A,i,j) = m_el(B,i,j)-m_el(C,i,j);
    }

}

matmul(A,B,C)
matrix A,B,C;
{
int i,j,k;
matrix TB=NULL; /* absolutely must initialize to null */
matrix TC=NULL; /* absolutely must initialize to null */

materr(B->cols!=C->rows,"matmul: error - BC size mismatch");
materr(A->rows!=B->rows,"matmul: error - AB size mismatch");
materr(A->cols!=C->cols,"matmul: error - AC size mismatch");

if( A == B ) { TB = newmat(B->rows,B->cols); matcopy(TB,B); B = TB;}
if( A == C ) { TC = newmat(C->rows,C->cols); matcopy(TC,C); C = TC;}

for(i=0 ; i<B->rows; i++)
  for(j=0 ; j<C->cols ; j++)
    {
    m_el(A,i,j) = 0.0;
    for(k=0 ; k<B->cols ; k++)
      m_el(A,i,j) += m_el(B,i,k)*m_el(C,k,j);
    }

freemat(TB);
freemat(TC);
}

transpose(A,B)
matrix A,B;
{
int i,j;
matrix TB=NULL;

materr(A->rows!=B->cols,"transpose: error - size mismatch");
materr(A->cols!=B->rows,"transpose: error - size mismatch");

if( A == B ) { TB = newmat(B->rows,B->cols); matcopy(TB,B); B = TB; }

for(i=0 ; i<A->rows; i++)
  for(j=0 ; j<A->cols ; j++)
    {
    m_el(A,i,j) = m_el(B,j,i);
    }

freemat(TB);
}

materr(expr,str)
int expr;
char str[];
{
if( expr ) { printf(str); printf("\n"); exit(0); }
}

int matinv_3b3(m_out,m_in)
matrix m_out,m_in;
{
  double a,b,c,d,e,f,g,h,i;
  double DEN;
  
  materr((m_in->rows != 3) || (m_in->cols != 3) ||
      (m_out->rows != 3) || (m_out->cols != 3),
	 ("matinv_3b3: 3x3 matricies only\n"));
  a = m_el(m_in,0,0);
  b = m_el(m_in,0,1);
  c = m_el(m_in,0,2);
  d = m_el(m_in,1,0);
  e = m_el(m_in,1,1);
  f = m_el(m_in,1,2);
  g = m_el(m_in,2,0);
  h = m_el(m_in,2,1);
  i = m_el(m_in,2,2);

  DEN = (-(c*e*g) + b*f*g + c*d*h - a*f*h - b*d*i + a*e*i);
  if ((ABS(DEN)) < 0.000000000000001)
      printf( "matinv_3b3: Near Singular Maxtrix. \n");
  m_el(m_out,0,0) = (float) 	((-(f*h) + e*i)/DEN);
  m_el(m_out,0,1) = (float)      ((c*h - b*i)/DEN);
  m_el(m_out,0,2) = (float)      ((-(c*e) + b*f)/DEN);
  m_el(m_out,1,0) = (float)     ((f*g - d*i)/DEN);
  m_el(m_out,1,1) = (float)      ((-(c*g) + a*i)/DEN);
  m_el(m_out,1,2) = (float)      ((c*d - a*f)/DEN);
  m_el(m_out,2,0) = (float)     ((-(e*g) + d*h)/DEN);
  m_el(m_out,2,1) = (float)      ((b*g - a*h)/DEN);
  m_el(m_out,2,2) = (float)      ((-(b*d) + a*e)/DEN);
}


