
/* 
 *  Very simple Truth Maintenance System.
 * 
 *  Richard Goodwin, Oct 19, 1993.
 * 
 */

/* User level predicates 
 * 
 * tms-istrue(Fact,Value,Reason)
 * - attempts to unify fact with trueth value "Value" and a reason.
 * 
 * tms-assert(Fact,Value)
 * - attempts to assert fact with trueth value "Value".
 * 
 * tms-retract(Fact,Value)
 * - attempts to assert fact with trueth value "Value".
 * 
 * Facts in the tms are stored as :
 *
 * tms-node(Fact,Value,Reason)
 * - basic fact
 * 
 * tms-default([Fact|Conditions])
 * - Fact is true if conditions are true.
 * 
 */

/* 
 * tms-istrue(Fact,Response,Reason)
 * 
 * Returns the current value of the fact in the database and the
 * reason.  Caches values for speed and consistency.
 * 
 */

tms-istrue(Fact,Response,Reason) :-
	tms-istrue1([Fact],Response,Reason),
	!.

tms-istrue(Fact,Response,Reason) :-
	tms-node(Fact,Response,Reason).

tms-istrue1([],yes,[]).

tms-istrue1([Fact|Rest],yes,Reasons) :- 
	tms-node(Fact,yes,Reason),
	!,
	tms-istrue1(Rest,yes,RestReason),
	concat(Reason,RestReason,Reasons).

tms-istrue1([Fact|_],no,Reason) :- 
	tms-node(Fact,no,Reason),
	!.

tms-istrue1([Fact|Rest],yes,[[default,Fact,Reason]|RestReason]) :- 
	tms-find-default(Fact,Value,Reason),
	asserta(tms-node(Fact,Value,[[default,Fact,Reason]])),
	!,
	Value = yes,
	tms-istrue1(Rest,yes,RestReason).

tms-istrue1([Fact|Rest],N,[failure,Fact]) :- 
	asserta(tms-node(Fact,no,[[failure,Fact,[]]])),
	!,
	N=no.

/* 
 * tms-find-default(Fact,Value,Reason)
 * 
 * Finds any applicable default rules. Returns a reason the includes the list
 * of defaults and their values.
 * 
 */

tms-find-default(Fact,Value,[defaults,Reason,Defaults]) :- 
	setof([Fact|Clauses],tms-default([Fact|Clauses]),S),
	length(S,L),
	L > 0,
	!,
	tms-eval-defaults(S,Defaults),
	tms-select-default(Defaults,Fact,Value,Reason).

/* 
 * tms-select-default(Defaults,[],Fact,Reason)
 * 
 * Selects one of the possible defaults and supplies a reason.
 * Defaults - list of defaults including the reasons.
 * Fact - Selected default
 * Reason - atom giving reason.
 */

tms-select-default(Defaults,Fact,Value,longest) :-
	tms-select-defaults(Defaults,[],[],Facts),
	tms-vet-facts(Facts,Fact,Value).

/* 
 * tms-select-default(Defaults,[],Fact,Reason)
 * 
 * Selects one of the possible defaults and supplies a reason.
 * Defaults - list of defaults including the reasons.
 * Fact - Selected default
 * Reason - atom giving reason.
 */

tms-select-defaults([],Defaults,[],Defaults) :- !.

tms-select-defaults([],_,Facts,Facts).

tms-select-defaults([[default,Fact,yes,Reason]|Defaults],Rest,Done,RFacts) :-
	length(Reason,L),
	tms-select-default1(Defaults,L),
	tms-select-default1(Rest,L),
	tms-select-defaults(Defaults,[[default,Fact,yes,Reason]|Rest],
	    [[default,Fact,yes,Reason]|Done],RFacts).

tms-select-defaults([[default,Fact,Value,Reason]|Defaults],Rest,Done,RFacts) :-
	tms-select-defaults(Defaults,[[default,Fact,Value,Reason]|Rest],Done,RFacts).

/* 
 * tms-vet-facts(Facts)
 * 
 * Checks to see that all the acceptable defaults are unifiable.
 * Facts - list of Facts.
 * Fact - Selected default
 * Reason - atom giving reason.
 */

tms-vet-facts([], _, yes).

tms-vet-facts([[default,Fact,Value,Reason]|Rest], Fact,Value) :-
	tms-vet-facts(Rest,Fact,Value).

tms-vet-facts([[default,Fact,Value,_]|_], FactNew,inconsistent) :-
	functor(Fact,F,2),
	arg(1,Fact,X),
	functor(FactNew,F,2),
	arg(1,FactNew,X),
	arg(2,FactNew,inconsistent).
	
/* 
 * tms-select-default(Defaults,Length)
 * 
 * Succeeds if the list of defaults has a no clauses longer than length.
 */

tms-select-default1([],_).

tms-select-default1([[_,_,no|_]],_).

tms-select-default1([[default,Fact,yes,Reason]|Defaults],Len) :-
	length(Reason,L),
	Len > L,
	tms-select-default1(Defaults,L).

/* 
 * tms-eval-defaults(Defaults,EDefaults)
 * 
 * Evaluates each default.
 * Defaults - list of defaults.
 * EDefaults - list of evaluated defaults including the value and the reason.
 */

tms-eval-defaults([],[]).

tms-eval-defaults([[F|Clauses]|Rest],[[default,F,Value,Reason]|RestReason]) :-
	tms-istrue1(Clauses,Value,Reason),
	!,
	tms-eval-defaults(Rest,RestReason).

/* original tms-eval-defaults */
tms-eval-defaults([[F|Clauses]|Rest],[[default,F,no,Clauses]|RestReason]) :-
	!,
	tms-eval-defaults(Rest,RestReason).
/* modified tms-eval-defaults */
/* tms-eval-defaults([[F|Clauses]|Rest],RestReason) :-
	!,
	tms-eval-defaults(Rest,RestReason).
*/

/* 
* tms-assert(Fact,State)
* 
* Asserts the given fact with the given value.
* Fact - simple predicate of the form <predicate>(_item,_value).
* State - either "yes" or "no"
*/

tms-assert(Fact,State) :-
	tms-node(Fact,State,[[given,Fact,[]]]),
	!.

tms-assert(Fact,State) :-
	functor(Fact,F,2),
	arg(1,Fact,X),
	functor(FactOld,F,2),
	arg(1,FactOld,X),
	tms-node(FactOld,StateOld,Reason),
	tms-uncache(FactOld),
	asserta(tms-node(Fact,State,[[given,Fact,[]]])).

tms-assert(Fact,State) :-
	asserta(tms-node(Fact,State,[[given,Fact,[]]])).

/* 
 * tms-assert(Fact,State)
 * 
 * Asserts the given fact with the given value.
 * Fact - simple predicate of the form <predicate>(_item,_value).
 * State - either "yes" or "no"
 */

tms-retract(Fact,State) :-
	functor(Fact,F,2),
	arg(1,Fact,X),
	functor(FactOld,F,2),
	arg(1,FactOld,X),
	tms-node(FactOld,StateOld,Reason),
	tms-uncache(FactOld),
	tms-uncache(Fact).

/* 
 * tms-uncache(FactOld)
 * 
 * Removes all tms nodes that depend on the given assertion.
 * FactOld - the old fact to be removed.
 */

tms-uncache(Fact) :-
	retract(tms-node(Fact,_,_)),
	!,
	tms-uncache(Fact).

tms-uncache(Fact) :-
	setof([X,S,R],tms-node(X,S,R),Nodes),
	tms-removeNodes(Nodes,Fact),
	!.

/* 
 * tms-removeNodes(Nodes,Fact)
 * 
 * Retrace all node that depend on the given fact.
 * Nodes - list of tms nodes.
 * Fact - the fact to be retracted.
 */

tms-removeNodes([], _).

tms-removeNodes([[X,S,R]|Rest], Fact) :-
	tms-dependsOn(R,Fact),
	!,
	retract(tms-node(X,S,R)),
	tms-removeNodes(Rest, Fact).

tms-removeNodes([_|R], Fact) :-
	tms-removeNodes(R, Fact).

/* 
 * tms-dependsOn(Reason,Fact)
 * 
 * Succeeds if the reason depends on the given fact.
 * Reason - a justificaton for a fact.
 * Fact - the fact to be retracted.
 */
	
tms-dependsOn([Fact|_],Fact).

tms-dependsOn([H|_],Fact) :-
	tms-dependsOn(H,Fact).

tms-dependsOn([_|R],Fact) :-
	tms-dependsOn(R,Fact).

/* 
 * member(X,List)
 * 
 * Unifies X with items in the List.
 */

member(X,[X|_]).

member(X,[_|Y]) :- member(X,Y).

/* 
 * concat(ListA,ListB,List)
 * 
 * Concatinates ListA and ListB into List.
 */

concat([],L,L).

concat([X|Y],L,[X|List]) :- concat(Y,L,List).
