;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;                                                                       ;;
;;                                  TUT                                  ;;
;;                       Tree Unification Toolkit                        ;;
;;                                                                       ;;
;;                         Version 1.0 15-Aug-91                         ;;
;;                                                                       ;;
;;                          John R. R. Leavitt                           ;;
;;                              Alex Franz                               ;;
;;                    Center for Machine Translation                     ;;
;;                      Carnegie-Mellon University                       ;;
;;                                                                       ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;                                                                       ;;
;; Copyright (c) 1991                                                    ;;
;; Carnegie Mellon University. All Rights Reserved.                      ;;
;;                                                                       ;;
;;-----------------------------------------------------------------------;;
;;                                                                       ;;
;; File Created:   15-Aug-91 by JRRL                                     ;;
;; Last Edit Date: 27-Aug-91 by JRRL                                     ;;
;;                                                                       ;;
;;-----------------------------------------------------------------------;;
;; File Description                                                      ;;
;;                                                                       ;;
;; This file contains all of the code used solely for the unifier        ;;
;; component of TUT.                                                     ;;
;;                                                                       ;;
;;-----------------------------------------------------------------------;;



;;-----------------------------------------------------------------------;;
;; Change Log                                                            ;;
;;                                                                       ;;
;; 19-Aug-91 by JRRL: Separated out the unifier code into this file.     ;;
;;                                                                       ;;
;; 22-Aug-91 by amf: fixed :or by moving call to unify-symbol            ;;
;;                     within ntree-unify.                               ;;
;;                                                                       ;;
;; 26-Aug-91 by amf: fixed double :OR, :MULTIPLE                         ;;
;;                                                                       ;;
;; 27-Aug-91 by JRRL: made amf's :OR and :MULTIPLE fixes iterative...    ;;
;;                      minor gain achieved.                             ;;
;;                                                                       ;;
;;; 11-Jun-92 by amf: changed names of or-p, not-p to tut-or-p, tut-not-p
;;;                   because of clash with
;;;                   good ole' pseudo-unify.lisp
;;;                                                                  
;;; 12-Oct-92 by amf: changed more names to avoid name clashes
;;;
;;; 19-Oct-92 by jrrl: changed code to handle original style operators
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;;-----------------------------------------------------------------------;
;;; Package Statements:                                                   ;
;;;-----------------------------------------------------------------------;

(in-package 'user)


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;                                                                       ;;
;;                        Customization Variables                        ;;
;;                                                                       ;; 
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;-----------------------------------------------------------------------;;
;; Variable *TUT-STRING-UNIFY-FN*
;;
;; Specifes a user defined unification function for strings.

(defvar *tut-string-unify-fn* nil
  "Specifes a user defined unification function for strings.  If used, this
   variable should specify a function that will take two strings and attempt
   to unify them.  If the strings are unifiable, the unified value should be
   returned; if not, *tut-fail-value* should be returned.")


;;-----------------------------------------------------------------------;;
;; Variable *TUT-NUMBER-UNIFY-FN*
;;
;; Specifes a user defined unification function for numbers.

(defvar *tut-number-unify-fn* nil
  "Specifes a user defined unification function for numbers.  If used, this
   variable should specify a function that will take two numbers and attempt
   to unify them.  If the numbers are unifiable, the unified value should be
   returned; if not, *tut-fail-value* should be returned.")


;;-----------------------------------------------------------------------;;
;; Variable *TUT-SYMBOL-UNIFY-FN*
;;
;; Specifes a user defined unification function for symbols.

(defvar *tut-symbol-unify-fn* nil
  "Specifes a user defined unification function for symbols.  If used, this
   variable should specify a function that will take two symbols and attempt
   to unify them.  If the numbers are unifiable, the unified value should be
   returned; if not, *tut-fail-value* should be returned.")


;;-----------------------------------------------------------------------;;
;; Variable *TUT-OTHER-UNIFY-FN*
;;
;; Specifes a user defined unification function for miscellaneous data types.

(defvar *tut-other-unify-fn* nil
  "Specifes a user defined unification function for miscellaneous data types.
   If used, this variable should specify a function that will take two values,
   which may be of any type other than string, number, or symbol, and
   attempt to unify them.  If the values are unifiable, the unified value
   should be returned; if not, *tut-fail-value* should be returned.")



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;                                                                       ;;
;;                          Unification Functions                        ;;
;;                                                                       ;; 
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


#+(and :cmu :new-compiler)
(proclaim '(inline tree-unify-fs
		   tree-unify-or
		   tree-unify-multiple
		   tree-unify-not
		   tree-unify-set
		   tree-unify-multiset
		   tree-unify-op))

#-(and :cmu :new-compiler)
(proclaim '(notinline tree-unify-fs
		      tree-unify-or
		      tree-unify-multiple
		      tree-unify-not
		      tree-unify-set
		      tree-unify-multiset
		      tree-unify-op))

(proclaim '(inline tree-unify-symbol
		   tree-unify-string
		   tree-unify-number
		   tree-unify-other))

;;-----------------------------------------------------------------------;;
;; Function TREE-UNIFY-FS
;;
;; Unifies two FSs

(defun tree-unify-fs (fs1 fs2)
  (let (sub-fs failed assoc-result)
    (tut-trace-progn
     (format *standard-output* "~&-------------------------------------~%")
     (format *standard-output* "~&TUT [tree-unify-fs]: unifying f-structures~%")     
     (pprint fs1)
     (pprint fs2))
    (do ((slots fs1 (cdr slots)))
	((or failed (null slots))
	 (if failed
	     *tut-fail-value*
	     (progn (setf (cdr (last fs1)) fs2) fs1)))
      (setq assoc-result (assoc (caar slots) fs2 :test #'eq))
      (when assoc-result
	(setq sub-fs (ntree-unify (cadar slots) (second assoc-result)))
	(cond ((eq sub-fs *tut-fail-value*)
	       (setq failed t))
	      (t
	       (setf (cadar slots) sub-fs)
	       (setq fs2 (delete assoc-result fs2 :test #'eq))))))))


;;-----------------------------------------------------------------------;;
;; Function TREE-UNIFY-OR
;;
;; Function for :OR unification

;;; fs1 is (:OR ...)
;;; fs2 could be (:OR ...)

(defun tree-unify-or (fs1 fs2)
  (let (temp result)
    (tut-trace-progn
     (format *standard-output* "~&-------------------------------------~%")
     (format *standard-output* "~&TUT [tree-unify-or]: unifying f-structures~%")     
     (pprint fs1)
     (pprint fs2))
    (dolist (f1 (cdr fs1))
      (setq temp (tree-unify f1 fs2))
      (if (not (eq temp *tut-fail-value*))
	  (push temp result)))
    (cond ((null result)
	   *tut-fail-value*)
	  ((null (cdr result))
	   (car result))
	  ((tut-or-p fs2)
	   ;; get rid of nested :OR, e.g.
	   ;; ((:OR A A) (:OR A A))
	   ;; 26-Aug-91 by amf: fixed but slooow
	   ;; make this iterative
	   (let ((actual-result nil))
	     (dolist (x result (cons (tut-or-operator) (nreverse actual-result)))
	       (if (tut-or-p x)
		   (dolist (a (cdr x))
		     (push a actual-result))
		   (push x actual-result)))))
	  (t
	   (cons (tut-or-operator) (nreverse result))))))


;;-----------------------------------------------------------------------;;
;; Function TREE-UNIFY-MULTIPLE
;;
;; Function for :MULTIPLE unification

(defun tree-unify-multiple (fs1 fs2)
  (let (temp result)
    (tut-trace-progn
     (format *standard-output* "~&-------------------------------------------~%")
     (format *standard-output* "~&TUT [tree-unify-multiple]: unifying f-structures~%")    
     (pprint fs1)
     (pprint fs2))
    (dolist (f1 (cdr fs1))
      (setq temp (tree-unify f1 fs2))
      (if (eq temp *tut-fail-value*)
	  (return-from tree-unify-multiple *tut-fail-value*))
      (push temp result))
    (cond ((null (cdr result))
	   (car result))
	  ((tut-multiple-p fs2)
	   (setq result (nreverse result))
	   ;; get rid of nested :MULTIPLE
	   ;; 26-Aug-91 by amf: fixed but slow
	   (let ((actual-result nil))
	     (dolist (x result (cons (tut-multiple-operator) (nreverse actual-result)))
	       (if (tut-multiple-p x)
		   (dolist (a (cdr x))
		     (push a actual-result))
		   (push x actual-result)))))	   

	   ;;;(dolist (r (cdr result) (car result))
	     ;;;(setf (cdr (last (car result)))
		   ;;;(cdr r)))
	  (t
	   (cons (tut-multiple-operator) (nreverse result))))))


;;-----------------------------------------------------------------------;;
;; Function TREE-UNIFY-NOT
;;
;; Function for :NOT unification
;;
;; Comment: The top bit is way too meisterful, but is a tad more
;;          efficient than using if and nconc.

(defun tree-unify-not (fs1 fs2)
  (tut-trace-progn
   (format *standard-output* "~&--------------------------------------~%")   
   (format *standard-output* "~&TUT [tree-unify-not]: unifying f-structures~%")
   (pprint fs1)
   (pprint fs2))
  (cond ((tut-not-p fs2)
	 (list (tut-not-operator)
	       (cond ((tut-or-p (cadr fs1))
		      (cond ((tut-or-p (cadr fs2))
			     (setf (cdr (last (cadr fs1))) (cdadr fs2))
			     (cadr fs1))
			    (t
			     (setf (cdr (last (cadr fs1))) (cdr fs2))
			     (cadr fs1))))
		     (t
		      (cond ((tut-or-p (cadr fs2))
			     (setf (cdr (last (cadr fs2))) (cdr fs1))
			     (cadr fs2))
			    (t
			     (list (tut-or-operator) (cadr fs1) (cadr fs2))))))))
	((tree-unify-p (cadr fs1) fs2)
	 *tut-fail-value*)
	(t
	 fs2)))

;;-----------------------------------------------------------------------;;
;; Function TREE-UNIFY-SYMBOL
;;
;; Function for symbol unification

(defun tree-unify-symbol (fs1 fs2)
  (declare (symbol fs1))
  (tut-trace-progn
   (format *standard-output* "~&------------------------------------~%")
   (format *standard-output* "~&TUT [tree-unify-symbol]: unifying symbols~%")   
   (pprint fs1)
   (pprint fs2))
  (cond ((not (symbolp fs2))
	 *tut-fail-value*)
	(*tut-symbol-unify-fn*
	 (funcall *tut-symbol-unify-fn* fs1 fs2))
	((eq fs1 fs2)
	 fs1)
	(t
	 *tut-fail-value*)))


;;-----------------------------------------------------------------------;;
;; Function TREE-UNIFY-STRING
;;
;; Function for string unification

(defun tree-unify-string (fs1 fs2)
  (declare (string fs1))
  (tut-trace-progn
   (format *standard-output* "~&------------------------------------~%")
   (format *standard-output* "~&TUT [tree-unify-string]: unifying strings~%")   
   (pprint fs1)
   (pprint fs2))
  (cond ((not (stringp fs2))
	 *tut-fail-value*)
	(*tut-string-unify-fn*
	 (funcall *tut-string-unify-fn* fs1 fs2))
	((string-equal fs1 fs2)
	 fs1)
	(t
	 *tut-fail-value*)))


;;-----------------------------------------------------------------------;;
;; Function TREE-UNIFY-NUMBER
;;
;; Function for number unification

(defun tree-unify-number (fs1 fs2)
  (declare (number fs1))
  (tut-trace-progn
   (format *standard-output* "~&------------------------------------~%")
   (format *standard-output* "~&TUT [tree-unify-number]: unifying numbers~%")   
   (pprint fs1)
   (pprint fs2))
  (cond ((not (numberp fs2))
	 *tut-fail-value*)
	(*tut-number-unify-fn*
	 (funcall *tut-number-unify-fn* fs1 fs2))
	((= fs1 fs2)
	 fs1)
	(t
	 *tut-fail-value*)))


;;-----------------------------------------------------------------------;;
;; Function TREE-UNIFY-OTHER
;;
;; Function for unification of miscellaneous data types

(defun tree-unify-other (fs1 fs2)
  (tut-trace-progn
   (format *standard-output* "~&---------------------------~%")
   (format *standard-output* "~&TUT [tree-unify-other]: unifying~%")   
   (pprint fs1)
   (pprint fs2))
  (cond (*tut-other-unify-fn*
	 (funcall *tut-other-unify-fn* fs1 fs2))
	((equalp fs1 fs2)
	 fs1)
	(t
	 *tut-fail-value*)))


;;-----------------------------------------------------------------------;;
;; Function TREE-UNIFY-OP
;;
;; Function for unification of user defined operators

(defun tree-unify-op (fs1 fs2)
  (let ((op (car fs1)))
    (tut-trace-progn
     (format *standard-output* "~&-----------------------------------------------~%")
     (format *standard-output* "~&TUT [tree-unify-op]: unifying user-defined operators~%")    
     (pprint fs1)
     (pprint fs2))
    (cond ((fboundp (tut-op-fn-name op))
	   (funcall (tut-op-fn-name op) fs1 fs2))
	  (t
	   (error "~s has not been defined as a TUT operator.~%" op)))))



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;                                                                       ;;
;;                           Unifier Interface                           ;;
;;                                                                       ;; 
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;-----------------------------------------------------------------------;;
;; Function TREE-UNIFY
;;
;; Top-level non-destructive unification function

(defun tree-unify (fs1 fs2)
  "Unifies FS1 and FS2 non-destructively."
  (ntree-unify (copy-tree fs1) (copy-tree fs2)))


;;-----------------------------------------------------------------------;;
;; Function NTREE-UNIFY
;;
;; Top-level destructive unification function

(defun ntree-unify (fs1 fs2)
  (cond ((null fs1)        fs2)
	((null fs2)        fs1)
	((and (tut-fs-p fs1)
	      (tut-fs-p fs2)   (tree-unify-fs        fs1 fs2)))
	((tut-multiple-p fs1)  (tree-unify-multiple  fs1 fs2))
	((tut-multiple-p fs2)  (tree-unify-multiple  fs2 fs1))
	((tut-or-p fs1)        (tree-unify-or        fs1 fs2))
	((tut-or-p fs2)        (tree-unify-or        fs2 fs1))
	((tut-not-p fs1)       (tree-unify-not       fs1 fs2))
	((tut-not-p fs2)       (tree-unify-not       fs2 fs1))
	((tut-op-p fs1)        (tree-unify-op        fs1 fs2))
	((tut-op-p fs2)        (tree-unify-op        fs2 fs1))
	((symbolp fs1)     (tree-unify-symbol    fs1 fs2))
	((symbolp fs2)     *tut-fail-value*)
	((stringp fs1)     (tree-unify-string    fs1 fs2))
	((stringp fs2)     *tut-fail-value*)
	((numberp fs1)     (tree-unify-number    fs1 fs2))
	((numberp fs2)     *tut-fail-value*)
	(t                 (tree-unify-other     fs1 fs2))))


