;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;                                                                       ;;
;;                                  TUT                                  ;;
;;                       Tree Unification Toolkit                        ;;
;;                                                                       ;;
;;                         Version 1.0 08-Aug-91                         ;;
;;                                                                       ;;
;;                              Alex Franz                               ;;
;;                          John R. R. Leavitt                           ;;
;;                    Center for Machine Translation                     ;;
;;                      Carnegie-Mellon University                       ;;
;;                                                                       ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;                                                                       ;;
;; Copyright (c) 1991                                                    ;;
;; Carnegie Mellon University. All Rights Reserved.                      ;;
;;                                                                       ;;
;;-----------------------------------------------------------------------;;
;;                                                                       ;;
;; File Created:   08-Aug-91 by JRRL                                     ;;
;; Last Edit Date: 08-Aug-91 by JRRL                                     ;;
;;                                                                       ;;
;;-----------------------------------------------------------------------;;
;; File Description                                                      ;;
;;                                                                       ;;
;; This file contains all of the code for TUT, the Tree Unification      ;;
;; Toolkit.                                                              ;;
;;                                                                       ;;
;;-----------------------------------------------------------------------;;



;;-----------------------------------------------------------------------;;
;; Change Log                                                            ;;
;;                                                                       ;;
;; 08-Aug-91 by JRRL: Added the TUT Tracer, which consists of the        ;;
;;                      variable *TUT-TRACING* and the macros TUT-TRACE  ;;
;;                      and WITH-TUT-TRACING.                            ;;
;;; 11-Jun-92 by amf: changed names of or-p, not-p to tut-or-p, tut-not-p
;;;                   because of clash with
;;;                   good ole' pseudo-unify.lisp
;;;
;;; 12-Oct-92 by amf: changed names further to avoid clash with pseudo-unify
;;;
;;; 19-Oct-92 by jrrl: add the original style variant and adjusted code
;;;                      to do the right thing.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;;-----------------------------------------------------------------------;
;;; Package statements:                                                   ;
;;;-----------------------------------------------------------------------;

(in-package 'user)


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;                                                                       ;;
;;                             THE TUT TRACER                            ;;
;;                                                                       ;; 
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;-----------------------------------------------------------------------;;
;; Variable *TUT-TRACING*
;;
;; Tracing flag.

(defvar *tut-tracing* nil
  "this variable acts as a flag for tut tracing.  when not nil,
   tracing is on.")


;;-----------------------------------------------------------------------;;
;; Macro TUT-TRACE
;;
;; Conditionally print trace statements

(defmacro tut-trace (format-string &rest args)
  `(if *tut-tracing*
     (format *standard-output* ,format-string ,@args)))


;;-----------------------------------------------------------------------;;
;; Macro TUT-TRACE-PROGN
;;
;; Conditionally execute forms

(defmacro tut-trace-progn (&rest args)
  `(when *tut-tracing*
     ,@args))


;;-----------------------------------------------------------------------;;
;; Macro WITH-TUT-TRACING
;;
;; Wrapper macro for turning on tracing

(defmacro with-tut-tracing (&rest forms)
  `(let ((*tut-tracing* t))
     ,@forms))



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;                                                                       ;;
;;                        CUSTOMIZATION VARIABLES                        ;;
;;                                                                       ;; 
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;-----------------------------------------------------------------------;;
;; Variable *TUT-FAIL-VALUE*
;;
;; The value that tut should return in the case of a failure.

(defvar *tut-fail-value* :fail
  "")


;;-----------------------------------------------------------------------;;
;; Variable *TUT-OPERATOR-STYLE*
;;
;; This variable indicates whether original style or keyword style
;; operators should be used.

(defvar *tut-operator-style* :original
  "This variable indicates whether original style or keyword style
   operators should be used; value: :original or :keyword")




;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;                                                                       ;;
;;                         USER-DEFINED OPERATORS                        ;;
;;                                                                       ;; 
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(proclaim '(inline tut-op-fn-name tut-op-pred-name))

(defun tut-op-fn-name (op)
  (intern (concatenate 'string "tut-unify-" (symbol-name op))))


(defun tut-op-pred-name (op)
  (intern (concatenate 'string "tut-unify-p-" (symbol-name op))))


;;-----------------------------------------------------------------------;;
;; Macro DEFTUTOP
;;
;; Defines a new tut operator

(defmacro deftutop (op (fs1var fs2var) &rest forms)
  `(defun ,(tut-op-fn-name op) (,fs1var ,fs2var)
     ,@forms))


(defmacro deftutop-pred (op (fs1var fs2var) &rest forms)
  `(defun ,(tut-op-pred-name op) (,fs1var ,fs2var)
     ,@forms))



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;                                                                       ;;
;;                         STYLE HANDLING ROUTINES                       ;;
;;                                                                       ;; 
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(proclaim '(inline tut-or-operator
		   tut-multiple-operator
		   tut-not-operator))


;;-----------------------------------------------------------------------;;
;; Function TUT-OR-OPERATOR
;;
;; Returns the OR operator for the current style (errors on an invalid
;; style)

(defun tut-or-operator ()
  (case *tut-operator-style*
    (:original  '*or*)
    (:keyword   :or)
    (t          (error "Invalid operator style ~s was specified.~%"
		       *tut-operator-style*))))


;;-----------------------------------------------------------------------;;
;; Function TUT-MULTIPLE-OPERATOR
;;
;; Returns the MULTIPLE operator for the current style (errors on an 
;; invalid style)

(defun tut-multiple-operator ()
  (case *tut-operator-style*
    (:original  '*multiple*)
    (:keyword   :multiple)
    (t          (error "Invalid operator style ~s was specified.~%"
		       *tut-operator-style*))))



;;-----------------------------------------------------------------------;;
;; Function TUT-NOT-OPERATOR
;;
;; Returns the NOT operator for the current style (errors on an invalid
;; style)

(defun tut-not-operator ()
  (case *tut-operator-style*
    (:original  '*not*)
    (:keyword   :not)
    (t          (error "Invalid operator style ~s was specified.~%"
		       *tut-operator-style*))))




;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;                                                                       ;;
;;                               PREDICATES                              ;;
;;                                                                       ;; 
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(proclaim '(inline tut-or-p
		   tut-multiple-p
		   tut-not-p
		   tut-op-p))


;;-----------------------------------------------------------------------;;
;; Function TUT-OR-P
;;
;; Is fs an :or structure?

(defun tut-or-p (fs)
  (and (listp fs)
       (eq (car fs) (tut-or-operator))))


;;-----------------------------------------------------------------------;;
;; Function TUT-MULTIPLE-P
;;
;; Is fs an :multiple structure?

(defun tut-multiple-p (fs)
  (and (listp fs)
       (eq (car fs) (tut-multiple-operator))))


;;-----------------------------------------------------------------------;;
;; Function TUT-NOT-P
;;
;; Is fs a :not structure?

(defun tut-not-p (fs)
  (and (listp fs)
       (eq (car fs) (tut-not-operator))))


;;-----------------------------------------------------------------------;;
;; Function TUT-OP-P
;;
;; Is FS a user-defined operator structure?

(defun tut-op-p (fs)
  (and (listp fs)
       (atom (car fs))))


;;-----------------------------------------------------------------------;;
;; Function TUT-FS-P
;;
;; Is FS an FS?  (what a lame comment)

(defun tut-fs-p (fs)
  (and (listp fs)
       (listp (car fs))))
