;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;                                                                       ;;
;;                    Tools.Lisp: Tools Package for EHN                  ;;
;;                                                                       ;;
;;                          Eric H. Nyberg, 3rd                          ;;
;;                      Carnegie-Mellon University                       ;;
;;                                                                       ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;                                                                       ;;
;; Copyright (c) 1987, 1988, 1989 Carnegie-Mellon University.            ;;
;; All Rights Reserved.                                                  ;;
;;                                                                       ;;
;;-----------------------------------------------------------------------;;
;;                                                                       ;;
;; File Created:   25-Mar-87 by EHN                                      ;;
;; Last Edit Date: 03-Dec-89 by EHN                                      ;;
;;                                                                       ;;
;;-----------------------------------------------------------------------;;

(in-package 'user)

;;-----------------------------------------------------------------------;;
;; Stuff for Getting the Time
;;
;; Get individual parts of the universal time.
;;
;; (get-decoded-time) returns: second, minute, hour, date, month, year,
;;                             day-of-week, daylight-s-time-p, time-zone

(defmacro get-time-values () '(multiple-value-list (get-decoded-time)))

(defmacro sec      (&optional raw-time) (if raw-time `(nth 0 ,raw-time) '(nth 0 (get-time-values))))
(defmacro minute   (&optional raw-time) (if raw-time `(nth 1 ,raw-time) '(nth 1 (get-time-values))))
(defmacro hour     (&optional raw-time) (if raw-time `(nth 2 ,raw-time) '(nth 2 (get-time-values))))
(defmacro date     (&optional raw-time) (if raw-time `(nth 3 ,raw-time) '(nth 3 (get-time-values)))) 
(defmacro month    (&optional raw-time) (if raw-time `(nth 4 ,raw-time) '(nth 4 (get-time-values))))
(defmacro year     (&optional raw-time) (if raw-time `(nth 5 ,raw-time) '(nth 5 (get-time-values))))
(defmacro day      (&optional raw-time) (if raw-time `(nth 6 ,raw-time) '(nth 6 (get-time-values))))
(defmacro daylight (&optional raw-time) (if raw-time `(nth 7 ,raw-time) '(nth 7 (get-time-values))))
(defmacro tzone    (&optional raw-time) (if raw-time `(nth 8 ,raw-time) '(nth 8 (get-time-values))))

(defmacro month-string (num) 
  `(svref '#(0 "Jan" "Feb" "Mar" "Apr" "May" "Jun"
               "Jul" "Aug" "Sep" "Oct" "Nov" "Dec")
          ,num))

(defun time-string (&optional (raw-time (get-time-values)))
  (format nil "~2,'0d:~2,'0D:~2,'0D"
              (hour raw-time) (minute raw-time) (sec raw-time)))

(defun day-string (&optional (raw-time (get-time-values)))
  (format nil "~2,'0d-~a-~a"
              (date raw-time) (month-string (month raw-time)) (mod (year raw-time) 100)))

(defun sign-on-message (&optional (stream *standard-output*))
  (let (string1 string2)
    (setq string1
     (format nil 
       "~%;; ~a version ~a" (lisp-implementation-type) (lisp-implementation-version)))
    (setq string2  
     (format nil 
       "~%;; Started on (~a) ~a, ~a ~a" 
        (machine-type)(machine-instance)(day-string)(time-string)))
    (format stream   ";;---------------------------------------------------------------------------;;")
    (format stream "~%;;    -*- Center for Machine Translation / Carnegie Mellon University -*-    ;;")
    (format stream "~%;;---------------------------------------------------------------------------;;")
    (format stream "~78a;;" string1)
    (format stream "~78a;;" string2)
    (format stream "~%;;---------------------------------------------------------------------------;;")))


;;-----------------------------------------------------------------------;;
;; Macro RESETVAR
;;
;; Analogous to Interlisp-D function.

(defmacro resetvar (var tempval . forms)
  `(let ((oldval ,var)
         result)
     (unwind-protect
       (setf ,var ,tempval)
       (setf result (progn ,@forms))
       (setf ,var oldval))
     result))

;;-----------------------------------------------------------------------;;
;; Macro WITH-VAR-VALUES
;;
;; Like a multiple resetvar.

(defmacro with-var-values (binding-list &rest forms)
  (let (let-binding let-body-begin let-body-end newsym)
    (setq let-binding '(result))
    (setq binding-list (eval binding-list))
    (dolist (item binding-list)
      (setq newsym (gensym))
      (setq let-binding 
        (cons (list newsym (first item))
              let-binding))
      (setq let-body-begin
        (cons `(setf ,(first item) ,(second item))
               let-body-begin))
      (setq let-body-end
        (cons `(setf ,(first item) ,newsym)
               let-body-end)))
   `(let ,let-binding 
      (unwind-protect
        ,@let-body-begin
        (setf result (progn ,@forms))
        ,@let-body-end)
      result)))
      
;;-----------------------------------------------------------------------;;
;; Function WITH-READTABLE

(defmacro with-readtable (readtable . forms)
  `(resetvar *readtable* ,readtable ,@forms))

;;-----------------------------------------------------------------------;;
;; Macro DOFILE
;;
;; Analogous to DOLIST:
;;
;;  (dofile (<var> <filename> <returnform>)
;;    <bodyform>*)
;;
;; Opens the file given by <filename> for input, reads successive forms
;; from the file (setting the value of <var> to the form) and executes
;; the forms in <bodyform>* for each form. When end-of-file is reached,
;; the value of <returnform> (optional) is returned.

(defmacro dofile (args . forms)

 "Analogous to DOLIST: (dofile (<var> <filename> <returnform>)
                         <bodyform>*)
  <var> is set to each form in the file & <bodyform>* is evaluated."

  `(let (,(first args))
     (with-open-file (infile ,(second args) :direction :input)
       (loop 
         (setq ,(first args) (read infile nil infile))
         (if (eq ,(first args) infile) (return ,(third args)))
         ,@forms))))

;;-----------------------------------------------------------------------;;
;; Macro MAPFILE

(defmacro mapfile (args . forms)
  `(let (,(first args))
     (with-open-file (infile ,(second args) :direction :input)
       (with-open-file (outfile ,(third args) :direction :output)
         (loop
           (setq ,(first args) (read infile nil infile))
           (if (eq ,(first args) infile) (return))
           (pprint (progn ,@forms) outfile))))))

;;-----------------------------------------------------------------------;;
;; Macro COLLECT
;;
;; Like COLLECT i.s. opr in Interlisp-D. Maps a function across a list
;; returning all the non-NIL values.

(defmacro collect (fn list)
  `(let (val result)
     (dolist (element ,list result)
       (if (setq val (funcall ,fn element))
           (setq result (nconc result (list val)))))))

;;-----------------------------------------------------------------------;;
;; Macro SUBSET
;;
;; Like COLLECT, except it returns the list members that satisfy
;; the test, not the test result.

(defmacro subset (fn list)
  `(let (result)
     (dolist (element ,list result)
       (if (funcall ,fn element)
           (setq result (nconc result (list element)))))))

;;-----------------------------------------------------------------------;;
;; Macro TRACE-LOAD

(defmacro trace-load (filename &optional logfile)
  (if logfile
      `(progn (dribble ,logfile)
              (format *standard-output*
                "~%Loading forms from ~a ~a" ,filename (date))
              (dofile (form ,filename)
                (format *standard-output* "~%~%")
                (pprint form)
                (format *standard-output* "~%--> ~a" (eval form)))
              (dribble))
       `(progn
          (format *standard-output*
            "~%Loading forms from ~a ~a" ,filename (date))
          (dofile (form ,filename)
            (format *standard-output* "~%~%")
            (pprint form)
            (format *standard-output* "~%--> ~a" (eval form))))))

;;-----------------------------------------------------------------------;;
;; Macro GENERATE-LIST

(defmacro generate-list (number form)
  `(let (result)
     (dotimes (var ,number)
       (setq result (nconc result (list ,form))))
     result))
 
;;-----------------------------------------------------------------------;;
;; Function STRING-TO-LIST

(defvar *string-end-tag* (gensym))

(defun string-to-list (string)
  (let ((start 0)
        item result)
    (loop
      (multiple-value-setq (item start)
        (read-from-string string nil *string-end-tag* :start start))
      (if (eq *string-end-tag* item) (return result))
      (setq result (nconc result (list item))))))

;;-----------------------------------------------------------------------;;
;; Function STRING-TO-STRINGS

(defun string-to-strings (string &optional ignore)
  "Eliminate white space and return a list of strings."
  (let (result char charbag (offset 0) (length (length string)))
    (loop
      (cond ((= offset length)
	     (if charbag
		 (setf result
		       (nconc result (list (bag-to-string charbag)))))
	     (return result)))
      (setf char (char string offset))
      (cond ((or (eq char #\Space)
		 (eq char #\Newline))
	     (cond (charbag
		    (setf result
			  (nconc result (list (bag-to-string charbag))))
		    (setf charbag nil))))
	    ((member char ignore) nil)
	    (t (setf charbag (nconc charbag (list char)))))
      (incf offset))))

(defun bag-to-string (charbag)
  (let* ((newsize (length charbag))
	(new (make-string newsize)))
    (dotimes (i newsize new)
      (setf (char new i) (pop charbag)))))

;;-----------------------------------------------------------------------;;
;; Function UNPACK

#|(defun unpack (string)
  (let ((char 0)
        (end (1- (length string)))
        item result)
    (loop
      (setq item
        (schar string char))
      (setq result (nconc result (list item)))
      (if (eq end char) (return result))
      (incf char))))|#

;;-----------------------------------------------------------------------;;
;; Function PACK

#|(defun pack (list)
  (let* ((length (length list))
         (string (make-string length))
         (offset 0))
    (dolist (char list string)
      (setf (schar string offset) char)
      (incf offset))))|#

;;-----------------------------------------------------------------------;;
;; Function SUBSTRING
;;
;; Similar to the Interlisp-D function SUBSTRING, this function is
;; somewhat superior in function to SUBSEQ in that it can take negative
;; offsets, which are interpreted as positions from the right of the
;; string. No such bold claims are made about efficiency...

(defun substring (string start &optional (end (1- (length string))))

  "Returns a new string object containing a substring of its argument."
  "Start and End are treated as offsets beginning at 0. If they are
   negative, they are treated as offsets from the end of the string."

  (if (> 0 start) (setf start (+ start (length string))))
  (if (> 0 end) (setf end (+ end (length string))))
  (if (> start end) ""
      (let* ((oldsize (length string))
             (newsize (1+ (- end start)))
             (offset 0)
             (temp nil)
             (new (loop 
                    (if (>= start oldsize) (return nil))
                    (if (= offset newsize) (return temp))
                    (if (>= offset oldsize) (return temp))
                    (push (char string (+ start offset)) temp)
                    (setf offset (1+ offset)))))
        (cond (new
               (let* ((n (length new))
                      (s (make-string n)))
                 (loop
                   (decf n)
                   (if (null new) (return s))
                   (setf (char s n) (pop new))
                   )))
          (t "")))))

;;-----------------------------------------------------------------------;;
;; Function REPLACE-CHAR
;;
;; Replaces old with new in string.

(defun replace-char (string old new)
  (let ((offset 0)
        (length (length string)))
    (if (= 0 length)
        string
        (loop 
          (if (= offset length)
              (return string))
          (if (equal old (char string offset))
              (setf (char string offset) new))
          (setq offset (1+ offset))))))

;;-----------------------------------------------------------------------;;
;; Macro SPACES
;;
;; For indenting the output stream; useful for tracing.

(defun spaces (n &optional (stream *standard-output*))
  (dotimes (x n nil) (write-char #\Space stream)))

;;-----------------------------------------------------------------------;;
;; Macro MKLIST

(defmacro mklist (arg)
  `(cond
     ((listp ,arg) ,arg)
     ((null ,arg) ,arg)
     (t (list ,arg))))

;;-----------------------------------------------------------------------;;
;; Macro PP
;;
;; Calls grindef. If the function is compiled, prints a message.

(defmacro pp (symbol)
  `(grindef ,symbol))

;;-----------------------------------------------------------------------;;
;; Function PRETTY-F-STRUCTURE
;;
;; Good for f-structures with lots of embedding, probably useless for 
;; everything else.

(defun pretty-f-structure (list &optional 
                                (stream *standard-output*)
                                (indent 0)
                                (dont-indent nil))
  (unless dont-indent
    (terpri stream)
    (spaces indent stream))
  (format stream "(~a" (first list))
  (dolist (x (rest list))
    (terpri stream)
    (spaces (1+ indent) stream)
    (format stream "(~a" (first x))
    (if (small-enough? (second x))
        (format stream " ~a)" (second x))
        (pretty-f-structure (second x) stream (+ indent 3)))))

(defun pretty-col (list &optional 
                        (stream *standard-output*) 
                        (indent 0)
                        (dont-indent nil))
  (unless dont-indent
    (terpri stream)
    (spaces indent stream))
  (format stream "(~a" (first list))
  (incf indent)
  (dolist (x (rest list))
    (terpri stream)
    (spaces indent stream)
    (if (and (listp x) (listp (second x))(> (length (second x)) 2))
        (pretty-col x stream (1+ indent) t)
        (format stream "~a" x)))
  (format stream ")")
  (unless dont-indent
    (terpri stream))
  nil)

(defun small-enough? (list)
  (if (listp list)
      (if (null (rest list))
          t
          nil)
      t))
    
;;-----------------------------------------------------------------------;;
;; Function ABBREV-FILE

(defun abbrev-file (filename)
  (dofile (form filename)
    (if (listp form)
        (cond ((<= (length form) 3)(print form))
          (t (print (list (first form) (second form) (third form)))))
        (print form))))

;;-----------------------------------------------------------------------;;
;; Function ABBREV-LIST 

(defun abbrev-list (list &optional (num 2))
  (cond
    ((not (listp list)) list)
    ((= 0 num)
     (mapcar 
       #'(lambda (x) (if (symbolp x) x '(&)))
       list))
    (t
      (mapcar
        #'(lambda (l) (abbrev-list l (1- num)))
        list))))

;;-----------------------------------------------------------------------;;
;; Function EXTRACT-LIST

(defun extract-list (x y list)
  (setq list (nthcdr x list))
  (let (ans)
    (dotimes (var (1+ (- y x)))
      (setq ans (nconc ans (list (nth var list)))))
    ans))

;;-----------------------------------------------------------------------;;
;; Macro DONE-TRACE

(defmacro done-trace (string . forms)

  `(progn (format *standard-output* "~%~a..." ,string)
          (force-output *standard-output*)
          ,@forms
          (format *standard-output* "Done.")
          (force-output *standard-output*)))

;;-----------------------------------------------------------------------;;
;; Macro PRINTOUT

(defmacro printout (string &rest args)
  `(format *standard-output* ,string ,@args))

;;-----------------------------------------------------------------------;;
;; Macro MSETQ
;;
;; Save a little typing.

(defmacro msetq (&rest args)
  `(multiple-value-setq ,@args))

;;-----------------------------------------------------------------------;;
;; Macro WITH-OUTPUT-STREAM
;;
;; Uses RESETVAR on *standard-output*.

(defmacro with-output-stream (stream &rest forms)
  `(resetvar *standard-output* ,stream ,@forms))

;;-----------------------------------------------------------------------;;
;; Macro WITH-TRACE-STREAM
;;
;; Uses RESETVAR on *trace-output*.

(defmacro with-trace-stream (stream &rest forms)
  `(resetvar *trace-output* ,stream ,@forms))

;;-----------------------------------------------------------------------;;
;; Function MYEXPAND
;;
;; Continually expands an expression until no macro calls are present.
;; More thorough than MACROEXPAND, which stops when the top-level call
;; is no longer a macro call (although arguments may be embedded macro
;; calls).

(defun myexpand (expr)
  (cond((listp expr)
        (mapcar #'myexpand 
                (macroexpand expr)))
    (t expr)))

;;-----------------------------------------------------------------------;;
;; Function KEY-CONFIRM

(defun key-confirm (&optional (stream *standard-input*)
                              (message "[Press Enter to Continue]"))
  (format stream "~%~a " message)
  (read-line stream))

;;-----------------------------------------------------------------------;;
;; Macro ?

(defmacro ? (thing)
  `(describe ',thing))

;;-----------------------------------------------------------------------;;
;; Function TRACE-FILE
;;
;; Useful for debugging; when called on a filename, causes all functions
;; defined in that file to be traced. Intended to be called on the file
;; produced by COMPGEN, which contains lisp functions for generation
;; grammar rules.

(defun trace-file (filename)
  (let (fnlist)
    (dofile (form filename)
      (if (eq 'defun (first form))
          (setq fnlist (nconc fnlist (list (second form))))))
    (dolist (fn fnlist)
      (eval `(trace ,fn)))
    fnlist))

;;-----------------------------------------------------------------------;;
;; Function LOAD-FORMS

(defun load-forms (filename &rest form-names)
  (dofile (form filename nil)
    (if (memq (second form) form-names)
        (eval form))))

;;-----------------------------------------------------------------------;;
;; Function COMPILE-FILES

(defun compile-files (&rest files)
  (dolist (f files)
    (compile-file f)))

;;-----------------------------------------------------------------------;;
;; Macro FOR
;;
;; Use elegant syntax that compiles into tight iteration code. This means
;; I learned the infathomable syntax of DO once, in order to write this
;; macro; from now on, I can use the Interlisp-D CLISP syntax, which is
;; easier to remember.
;;
;; Syntax: (FOR <var> IN <list> {DO,COLLECT,LIST,SUM} <forms>)
;;
;; DO - generates a DOLIST; <var> is bound to each item in <list> and
;;      <forms> are evaluated. Returns NIL.
;;
;; COLLECT - generates a DO; <var> is bound to each item in <list> and
;;      <forms> are evaluated. The non-NIL results are returned in a list.
;;
;; LIST - generates a DO; <var> is bound to each item in <list> and
;;      <forms> are evaluated. All results are returned in a list.
;;
;; SUM - generates a DO; each result returned by <forms> should be a number, 
;;      which is added to the sum that is eventually returned.
;;
;; e.g.
;;
;;  * (for x in '(a b c) do (print x))
;;  A
;;  B
;;  C
;;  NIL
;;
;;  * (for x in '(a b c) list (list x))
;;  ((A)(B)(C))
;;
;;  * (for x in '(1 2 4 c y 6 n 7) collect (if (numberp x) x))
;;  (1 2 4 6 7)
;;
;;  * (for x in '(1 2 3 4) sum (* 2 x))
;;  20
;;

(defmacro for (local-var in list operator &rest forms)
  "(FOR <var> IN <list> {DO,LIST,COLLECT,SUM} <forms>)
   Like the Interlisp-D iteration macro. Result returned
   depends on operator. DO generates a DOLIST, and NIL
   is returned. LIST generates a DO, and a list containing
   all the results is returned (roughly like MAPCAR). COLLECT
   returns a list with all NILs filtered out. SUM returns a
   number - each intermediate result produced by <forms> should
   be a number to add to the sum."
  (declare (ignore in))
  (case operator
    (do `(dolist (,local-var ,list)
           ,@forms))
    (list
        `(do* ((value nil (nconc value 
                                 (list (progn ,@forms))))
               (list-var ,list (cdr list-var))
               (,local-var (first ,list) (first list-var)))
              ((null list-var) value)))
    (collect
        `(do* ((new nil (progn ,@forms))
               (value nil (if new 
                              (nconc value (list new))
                              value))
               (list-var ,list (cdr list-var))
               (,local-var (first ,list) (first list-var)))
              ((null list-var) value)))
    (sum 
        `(do* ((value 0 (+ value (progn ,@forms)))
               (list-var ,list (cdr list-var))
               (,local-var (first ,list) (first list-var)))
              ((null list-var) value)))
    (t (error "FOR: unknown operator: ~a" operator))))

;;-----------------------------------------------------------------------;;
;; Function OK

#|(defun ok (&optional (return-val '()))
  (debug::debug-return return-val))|#

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;                         End of File Tools.Lisp                        ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
