;      -*- Syntax: Common-Lisp; Package: USER; Base: 10; Mode: LISP -*-
;==============================================================================
;     File:  structure-tools.lisp
;	By:  Nicholas Brownlow	<ndb@clarit.com>
;     Path:  /afs/cs/project/cmt-29/catalyst/amt-ana/code/tools/release/
;  Started:  18 May 1995
; Modified:  03 December 1996	by <garof>
;
; Comments:  Functions for manipulating FS/IR trees.
;	     18-May-95 by ndb: Moved from INTERPRETER-TOOLS.LISP.
;
; Modified:  11 October 1995
;	By:  Nicholas Brownlow	<ndb@clarit.com>
;  Reasons:  Nicholas was the official "structure-tools" maintainer until then.
;
; Modified:  20 November 1996
;	By:  Joseph Giampapa	<garof@cs.cmu.edu>
;  Reasons:  To better document the functions of this file in order to maintain
;	     the Interpreter rules.
;
; Modified:  03 December 1996
;	By:  Joseph Giampapa	<garof@cs.cmu.edu>
;  Reasons:  To add *STRUCTURE-TOOLS-VS* version control constant.
;
; Modified:  <date>
;	By:  <name>		<full e-mail>
;  Reasons:  
;
;==============================================================================


;..............................................................................
;			Center for Machine Translation
;			  Carnegie Mellon University
;
;			   Copyright (c) 1995, 1996
;	       Carnegie Mellon University.  All Rights Reserved.
;..............................................................................


;..............................................................................
;			      Package Statements
;..............................................................................
(in-package :user)



;------------------------------------------------------------------------------
; For use by the maintainer's personal program.
; Added 03-Dec-96 by garof.
;------------------------------------------------------------------------------
(defconstant *STRUCTURE-TOOLS-VS* '5.0Analyzer)


;..............................................................................
;				   Functions
;..............................................................................


;------------------------------------------------------------------------------
;			Classify an FS/IR tree element
; 13-Nov-96-garof:
; Returns multiple value pairs, with the following meaning:
;
; Possible Values	Class	    Subclass		Meaning
; ---------------	-----	    --------		-------
; () ((...) ...)	:LIST	    :NOKEY    TREE is a list w/ no symbolic CAR
; :MULTIPLE *MULTIPLE*	:EXPRESSION :MULTIPLE TREE is a sequence expression
; :OR *OR*		:EXPRESSION :OR	      TREE is a disjunction expression
; :NOT *NOT*		:EXPRESSION :NOT      TREE is a negation expression
; (<all else>)		:LIST	    :KEY      TREE is a list w/ a symbolic CAR
; <all else>		:ATOM	    <TREE>    TREE is an atom
;
; Ex. (tree-classify '(number singular)) ---> :LIST :KEY
;
;------------------------------------------------------------------------------
(defun Tree-Classify (tree)
  "Classifies TREE, returning class and subclass."
  (if (listp tree)
      (let ((key (first tree)))
	(cond ((or (not tree)		; ()
		   (consp key))		; ((...) ...)
	       ;; TREE is a list with no symbolic CAR
	       (values :LIST :NOKEY))
	      ((or (eq key :MULTIPLE)
		   (eq key '*MULTIPLE*))
	       ;; TREE is a sequence expression
	       (values :EXPRESSION :MULTIPLE))
	      ((or (eq key :OR)
		   (eq key '*OR*))
	       ;; TREE is a disjunction expression
	       (values :EXPRESSION :OR))
	      ((or (eq key :NOT)
		   (eq key '*NOT*))
	       ;; TREE is a negation expression
	       (values :EXPRESSION :NOT))
	      (t
	       ;; TREE is a list with a symbolic CAR
	       (values :LIST :KEY))))
    ;; TREE is an atom
    (values :ATOM tree)))


;..............................................................................
;		      Simple FS/IR slot and filler access
;..............................................................................

;------------------------------------------------------------------------------
; 13-Nov-96-garof:
; Global proclamation that the code for these functions should be integrated
; into the calling routine, appearing "in line" in place of a procedure call.
;------------------------------------------------------------------------------
(proclaim
 '(inline Tree-Slot Tree-Filler Tree-Head List-Slot List-Filler List-Head))


;..............................................................................
;		       Tree Functions with Type Checking
;..............................................................................


;------------------------------------------------------------------------------
; 13-Nov-96-garof:
; Given:    "tree" = FS/IR  "key" = <FS/IR attribute name>
; Returns:  the FS/IR where the "key" is the head, NIL otherwise.
; Ex.:
; (Tree-Slot *fs-ir* 'number) --> (NUMBER SINGULAR)
; (Tree-Slot *fs-ir* 'quantity) --> (QUANTITY (*C-DECIMAL-NUMBER
;		(NUMBER-FORM NUMERIC) (NUMBER-TYPE CARDINAL) (INTEGER "3")))
;------------------------------------------------------------------------------
(defun Tree-Slot (tree key &optional (class nil) (subclass nil))
  "Given:    \"tree\" = FS/IR  \"key\" = <FS/IR attribute name>
Returns:  the FS/IR where the \"key\" is the head, NIL otherwise.
See also its defition for more documentation. [13-nov-96-garof]"

  (unless (and class subclass)
   (multiple-value-setq (class subclass) (Tree-Classify tree)))

  (case class
	(:LIST (assoc key (case subclass
				(:KEY (rest tree))
				(:NOKEY tree)
				(otherwise nil))
		      :test #'eq))
	(otherwise nil)))


;------------------------------------------------------------------------------
; 20-Nov-96-garof:
; Given:    "tree" = FS/IR  "key" = <FS/IR attribute name>
; Returns:  the FS/IR value of the slot where "key" is the head, NIL otherwise.
; Ex.:
; (Tree-Filler *fs-ir* 'number) --> SINGULAR
;------------------------------------------------------------------------------
(defun Tree-Filler (tree key &optional (class nil) (subclass nil))
  "Given:   \"tree\" = FS/IR  \"key\" = <FS/IR attribute name>
Returns: the FS/IR value of the slot where \"key\" is the head, NIL otherwise."
  (second (Tree-Slot tree key class subclass)))


;------------------------------------------------------------------------------
; If the "tree" is a "key list", return the first item.  In the case of a full
; IR, that will be the concept corresponding to the verb (of a sentence) or
; noun (of a noun phrase).
; Example
; (tag-interactively "drain the oil.")
; (tree-head (first (map-semantic-map *parse-value*))) --> *A-DRAIN
;------------------------------------------------------------------------------
(defun Tree-Head (tree &optional (class nil) (subclass nil))
  "If the \"tree\" is a \"key list\", return the first item.  In the case of a
full IR, that will be the concept corresponding to the verb (of a sentence) or
noun (of a noun phrase)."
  (unless (and class subclass)
	  (multiple-value-setq (class subclass)
			       (Tree-Classify tree)))
  (and (eq class :LIST)
       (eq subclass :KEY)
       (first tree)))



;..............................................................................
;			       With No Checking
;..............................................................................


;------------------------------------------------------------------------------
; 20-Nov-96-garof:
; Same as "Tree-Slot" but without type checking.  Use "Tree-Slot" instead.
;------------------------------------------------------------------------------
(defun List-Slot (list key)
  "Same as \"Tree-Slot\" but without type checking. Use \"Tree-Slot\" instead."
  (assoc key (if (symbolp (first list))
		 (rest list)
	       list)
	 :test #'eq))


;------------------------------------------------------------------------------
; 20-Nov-96-garof:
; Same as "Tree-Filler" but without type checking.  Use "Tree-Filler" instead.
;------------------------------------------------------------------------------
(defun List-Filler (list key)
  "Same as \"Tree-Filler\" but without type checking.  Use \"Tree-Filler\" instead."
  (second (list-slot list key)))


;------------------------------------------------------------------------------
; 20-Nov-96-garof:
; Same as "Tree-Head" but without type checking.  Use "Tree-Head" instead.
;------------------------------------------------------------------------------
(defun List-Head (list)
  "Same as \"Tree-Head\" but without type checking. Use \"Tree-Head\" instead."
  (first list))



;..............................................................................
;			   Mapping over FS/IR trees
;..............................................................................


;------------------------------------------------------------------------------
; Applies FUNCTION to each term in TREE expression.  If TREE is not an
; expression, applies FUNCTION to TREE.
;------------------------------------------------------------------------------
(defun Map-Expression (function tree)
  "Applies FUNCTION to each term in TREE expression.  If TREE is not an
expression, applies FUNCTION to TREE."
  (multiple-value-bind (class subclass)
		       (Tree-Classify tree)
		       (declare (ignore subclass))
		       (case class
			     (:EXPRESSION
			      (mapc function (rest tree)))
			     (otherwise
			      (funcall function tree)))
		       t))


;------------------------------------------------------------------------------
; Applies FUNCTION to each recursive level of TREE in a depth-first traversal.
; FUNCTION takes three arguments: tree object, class, and subclass.
;------------------------------------------------------------------------------
(defun Map-Tree (function tree)
  "Applies FUNCTION to each recursive level of TREE in a depth-first
traversal.  FUNCTION takes three arguments: tree object, class, and subclass."
  (multiple-value-bind
   (class subclass) (Tree-Classify tree)
   (funcall function tree class subclass)
   (case class
	 (:EXPRESSION (dolist (term (rest tree))
			      (Map-Tree function term)))
	 (:LIST (dolist (slot (case subclass
				    (:KEY (rest tree))
				    (:NOKEY tree)))
			(Map-Tree function (second slot)))))))



;..............................................................................
;			      Merging FS/IR Trees
;..............................................................................


;------------------------------------------------------------------------------
; Merges lists TREE0 and TREE1, with slots in TREE0 taking precedence of slots
; with the same keys in TREE1.  Destructively modifies TREE1.
;------------------------------------------------------------------------------
(defun List-Merge (tree0 tree1)
  "Merges lists TREE0 and TREE1, with slots in TREE0 taking precedence of slots
with the same keys in TREE1.  Destructively modifies TREE1."
  (let (class0
	subclass0
	class1
	subclass1
	slot1)
    (declare (ignore subclass0 subclass1))
    (multiple-value-setq (class0 subclass0)
			 (tree-classify tree0))
    (multiple-value-setq (class1 subclass1)
			 (tree-classify tree1))
    (if (and (eq class0 :LIST)
	     (eq class1 :LIST))
	(dolist (slot0 (case subclass0
			     (:KEY (rest tree0))
			     (:NOKEY tree0))
		       tree1)
		(if (setf slot1 (assoc (first slot0)
				       (case subclass1
					     (:KEY (rest tree1))
					     (:NOKEY tree1))
				       :test #'eq))
		    ;; Slot from TREE0 is also in TREE1: replace
		    (setf (second slot1) (second slot0))
		  ;; Slot from TREE0 is not in TREE1: add
		  (case subclass1
			(:KEY (push slot0 (rest tree1)))
			(:NOKEY (push slot0 tree1)))))
      tree0)))


;------------------------------------------------------------------------------
; Compare two FS/IR trees to see if they are functionally equal
;------------------------------------------------------------------------------
(defun Tree-Compare-Slots (slots0 slots1)
  "Compare two FS/IR trees to see if they are functionally equal"
  (and
   ;; Compare every slot value in SLOTS0 against the value for the same key in
   ;; SLOTS1
   (every #'(lambda (slot)
	      (tree-compare (second slot)
			    (second (assoc (first slot) slots1 :test #'eq))))
	  slots0)
   ;; Every slot in SLOTS1 must either a) have the same key as a slot in SLOTS0
   ;; (so we've already compared their values) or b) have a NIL value
   (every #'(lambda (slot)
	      (or (assoc (first slot) slots0 :test #'eq)
		  (not (second slot))))
	  slots1)))


;------------------------------------------------------------------------------
; Returns T iff TREE0 and TREE1 are functionally equal:
; - ignores ordering of slots;
; - (SLOT NIL) is equal to the absence of SLOT, since both cases unify with
;   (SLOT VALUE) and return NIL to a search for the value of SLOT.
;------------------------------------------------------------------------------
(defun Tree-Compare (tree0 tree1)
  "Returns T iff TREE0 and TREE1 are functionally equal:
 - ignores ordering of slots;
 - (SLOT NIL) is equal to the absence of SLOT, since both cases unify with
   (SLOT VALUE) and return NIL to a search for the value of SLOT."
  (let (class0
	subclass0
	class1
	subclass1)
    (multiple-value-setq (class0 subclass0)
			 (tree-classify tree0))
    (multiple-value-setq (class1 subclass1)
			 (tree-classify tree1))
    (cond ((and (eq class0 :LIST)
		(eq class1 :LIST))
	   ;; Both lists: both must have same key or both must lack key,
	   ;; must have comparable slot lists
	   (and (eq subclass0 subclass1)
		(case subclass0
		      (:KEY (and (eq (first tree0) (first tree1))
				 (Tree-Compare-Slots (rest tree0)
						     (rest tree1))))
		      (:NOKEY (tree-compare-slots tree0 tree1))
		      (otherwise nil))))
	  ;; Both expressions: must have same operator, comparable terms
	  ((and (eq class0 :EXPRESSION)
		(eq class1 :EXPRESSION))
	   (and (eq subclass0 subclass1)
		(= (length tree0) (length tree1))
		(every #'tree-compare (rest tree0) (rest tree1))))
	  ;; Both atoms: must be equal
	  ((and (eq class0 :ATOM)
		(eq class1 :ATOM))
	   (equal tree0 tree1))
	  (t
	   ;; Not same type: not comparable
	   nil))))


;------------------------------------------------------------------------------
; Test a constraint against an FS/IR tree
;------------------------------------------------------------------------------
(defmacro Tree-Test-Exp-Pred (exp-type)
  "Returns the predicate for an expression of type EXP-TYPE."
  `(case ,exp-type
	 (:MULTIPLE #'every)
	 (:OR #'some)
	 (:NOT #'notany)
	 (otherwise
	  (error "Bad expression operator ~S" ,exp-type))))


;------------------------------------------------------------------------------
; Test every slot value in SLOTS-TEST against the value for the same key in
; SLOTS-TREE
;------------------------------------------------------------------------------
(defmacro Tree-Test-Slots (slots-test slots-tree)
  "Test every slot value in SLOTS-TEST against the value for the same key in
SLOTS-TREE"
  `(every #'(lambda (slot)
	      (tree-test (second slot)
			 (second (assoc (first slot) ,slots-tree :test #'eq))))
	  ,slots-test))


;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
(defun Tree-Test (test tree)
  (cond ((or (eq test :DEFINED) (eq test '*DEFINED*))
	 ;; OK iff TREE is defined
	 tree)
	((or (eq test :UNDEFINED) (eq test '*UNDEFINED*))
	 ;; OK iff TREE is undefined
	 (not tree))
	(t
	 ;; Test TEST against TREE
	 (let (class-test subclass-test
			  class-tree subclass-tree)
	   (multiple-value-setq (class-test subclass-test)
				(tree-classify test))
	   (multiple-value-setq (class-tree subclass-tree)
				(tree-classify tree))
	   
	   (cond ((eq class-test :EXPRESSION)
		  ;; TEST is an expression: test terms with appropriate logic
		  (funcall (tree-test-exp-pred subclass-test)
			   #'(lambda (term)
			       (tree-test term tree))
			   (rest test)))

		 ((eq class-tree :EXPRESSION)
		  ;; TREE is an expression: test terms with appropriate logic
		  (funcall (tree-test-exp-pred subclass-tree)
			   #'(lambda (term)
			       (tree-test test term))
			   (rest tree)))

		 ((and (eq class-test :LIST)
		       (eq class-tree :LIST))
		  ;; Both lists: test if both keyed/unkeyed; if keyed, test if
		  ;; keys match (null TEST key matches any TREE key); test
		  ;; slot lists
		  (or (null test)	; Null test matches any tree
		      (and (eq subclass-test subclass-tree)
			   (case subclass-test
				 (:KEY
				  (and (or (not (first test))
					   (eq (first test) (first tree)))
				       (tree-test-slots (rest test)
							(rest tree))))
				 (:NOKEY (tree-test-slots test tree))
				 (otherwise nil)))))
		 
		 ((and (eq class-test :ATOM)
		       (eq class-tree :ATOM))
		  ;; Both atoms: must be equal
		  (equal test tree))
		 
		 (t
		  ;; Not same type: test fails
		  nil))))))



;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
(defparameter *Test-Tree-Test*
  `(;; Simple Atoms.
    (a a t)
    (a b nil)
    ;; Simple f-structures.
    (((root a)) ((root a) (root b)) t)
    (((root a)) ((root b)) nil)
    ;; :DEFINED, :UNDEFINED
    (((root *defined*)) ((root a)) t)
    (((root *defined*)) () nil)
    (((root *undefined*)) ((root a)) nil)
    (((root *undefined*)) () t)
    ;; LHS *OR*.
    ((*or* a b c) a t)
    ((*or* a b c) z nil)
    ((*or* ((root a)) ((root b))) ((root a)) t)
    ((*or* ((root a)) ((root b))) ((root c)) nil)
    ;; LHS *MULTIPLE*.
    ((*multiple* a a a) a t)
    ((*multiple* a a b) a nil)	
    ((*multiple* ((root a)) ((root a))) ((root a)) t)
    ((*multiple* ((root a)) ((root b))) ((root a)) nil)
    ;; LHS *NOT*.
    ((*not* a) b t)
    ((*not* a) a nil)
    ((*not* ((root a))) ((root b)) t)
    ((*not* ((root a))) ((root a)) nil)
    ;; RHS *OR*.
    (a (*or* a b c) t)
    (a (*or* b c) nil)
    (((root a)) (*or* ((root a)) ((root b))) t)
    (((root a)) (*or* ((root b)) ((root c))) nil)
    ))


;------------------------------------------------------------------------------
;------------------------------------------------------------------------------
(defun Test-Tree-Test (&optional (test-tree-test *test-tree-test*))
  (dolist (ttt test-tree-test)
	  (unless (eq (Tree-Test (first ttt) (second ttt)) (third ttt))
		  (format t "Failed: ~S~%" ttt))))


;..............................................................................
;		       FS/IR Tree Modification Routines
;..............................................................................

;------------------------------------------------------------------------------
; Applies FN-BEFORE and FN-AFTER to each slot-list level of TREE, destructively
; modifying TREE.
;------------------------------------------------------------------------------
(defun Tree-Modify-AUX (tree)
  "Applies FN-BEFORE and FN-AFTER to each slot-list level of TREE,
destructively modifying TREE."
  (declare (special fn-before fn-after))
  (let (class subclass)
    (multiple-value-setq (class subclass)
			 (Tree-Classify tree))
    (case class
	  (:EXPRESSION
	   (do ((terms (rest tree) (rest terms)))
	       ((endp terms))
	       (setf (car terms) (tree-modify-aux (car terms)))))
	  (:LIST
	   (when fn-before
		 (setf tree (funcall fn-before tree class subclass)))
	   (dolist (slot (case subclass
			       (:KEY (rest tree))
			       (:NOKEY tree)))
		   (setf (second slot) (tree-modify-aux (second slot))))
	   (when fn-after
		 (setf tree (funcall fn-after tree class subclass)))))
    tree))


;------------------------------------------------------------------------------
; Applies FN-BEFORE and FN-AFTER to each slot-list level of TREE, destructively
; modifying TREE.
;------------------------------------------------------------------------------
(defun Tree-Modify (tree &key (fn-before nil) (fn-after nil))
  "Applies FN-BEFORE and FN-AFTER to each slot-list level of TREE,
destructively modifying TREE."
  (declare (special fn-before fn-after))
  (Tree-Modify-AUX tree))


;;;==================================================================;
;;; Delete slots from an FS/IR tree
;;;  - destructively or nondestructively
;;;  - recursively or non-recursively

(defun delete-slots (tree slot-keys &optional (class nil) (subclass nil))
  (let (slots)
    (unless (and class subclass)
      (multiple-value-setq (class subclass)
	(tree-classify tree)))
    (case class
      ((:ATOM :EXPRESSION)
       tree)
      (:LIST
       (when (eq subclass :NOKEY)
	 ;; Make FS look like IR
	 (setf tree (cons nil tree)))

       (typecase slot-keys
	 (symbol
	  (setf slot-keys `(,slot-keys)))
	 (list)
	 (otherwise
	  (error "DELETE-SLOTS: invalid SLOT-KEYS type ~s" slot-keys)))

       (setf slots tree)
       (loop
	(unless (rest slots)
	  (return))
	(when (member (caadr slots) slot-keys :test #'eq)
	  (setf (rest slots) (cddr slots)))
	(setf slots (rest slots)))

       ;; Correct for FS/IR difference
       (if (eq subclass :KEY)
	   tree
	 (rest tree))))))
   
(defun remove-slots (tree slot-keys)
  (delete-slots (copy-list tree) slot-keys))


(defun tree-delete-slots (tree slot-keys)
  (tree-modify
   tree
   :fn-before #'(lambda (subtree class subclass)
		  (delete-slots subtree slot-keys class subclass))))

(defun tree-remove-slots (tree slot-keys)
  (tree-modify
   (copy-tree tree)
   :fn-before #'(lambda (subtree class subclass)
		  (delete-slots subtree slot-keys class subclass))))



;;;==================================================================;
;;; Manipulate expressions/results

(defun make-exp (op terms)
  "Makes an expression of OP and TERMS."
  (cons op terms)) 

(defun make-exp+ (op terms)
  "Makes an expression of OP and TERMS -- if the length of TERMS warrants it."
  (cond ((null terms) nil)		; No terms -- nil result
	((rest terms) (cons op terms))	; Many terms -- make expression
	(t (car terms))))		; One term -- return it


(defun failsafe (result &key (succeed result) (fail result))
  "Tests RESULT of tree operation, returning SUCCEED on success and FAIL on
failure.  SUCCEED and FAIL default to RESULT."
  (if (eq result :FAIL)
      fail
    succeed))


(defun failure-p (result)
  "Returns T for failed RESULT of tree operation."
  (eq result :FAIL))

(defun success-p (result)
  "Returns T for successful RESULT of tree operation."
  (not (eq result :FAIL)))


(defun fail-if (result)
  "Returns :FAIL for non-null RESULT, RESULT otherwise." 
  (if result
      :FAIL
    result))

(defun fail-if-not (result)
  "Returns :FAIL for null RESULT, RESULT otherwise." 
  (if result
      result
    :FAIL))


;;;==================================================================;
;;; Apply a function to the value at the end of a tree path

(defun tree-path-map-aux (tree path)
  (declare (special fn walk-exp))
  (multiple-value-bind (class subclass)
      (tree-classify tree)
    (cond
     ;; Expression: process each term.
     ((and (or path walk-exp)
	   (eq class :EXPRESSION))
      (mapc #'(lambda (term)
		(tree-path-map-aux term path))
	    (rest tree)))

     ;; End of path: apply function
     ((null path)
      (and fn
	   (funcall fn tree class subclass)))

     ;; List of slots: take ASSOC
     ((eq class :LIST)
      (let ((slot (assoc (first path) (case subclass
					(:KEY (rest tree))
					(:NOKEY tree)))))
	(and slot
	     (tree-path-map-aux (second slot) (rest path)))))
     
     ;; Anything else, do nothing
     (t
      nil))))


(defun tree-path-map (tree path &key
			   (fn nil)
			   (walk-exp nil) ; t
			   )
  (declare (special fn walk-exp))
  (tree-path-map-aux tree path))


(defun tree-read-first (tree path &key (walk-exp nil))
  "Returns the first value at the end of PATH in TREE."
  (let ((val nil))
    (block find
      (tree-path-map tree path
		     :fn #'(lambda (tree1 class1 subclass1)
			     (declare (ignore class1 subclass1))
			     (setf val tree1)
			     (return-from find))
		     :walk-exp walk-exp))
    val))


(defun tree-read-all (tree path &key (walk-exp nil))
  "Returns all values at the end of PATH in TREE."
  (let ((vals nil))
    (tree-path-map tree path
		   :fn #'(lambda (tree1 class1 subclass1)
			   (declare (ignore class1 subclass1))
			   (push tree1 vals))
		   :walk-exp walk-exp)
    (nreverse vals)))


;;;==================================================================;
;;; Test the value at the end of a tree path
;;;
;;; Apply expression logic to results.

(defun tree-path-test-aux (tree path)
  (declare (special val fn do walk-exp))
  (multiple-value-bind (class subclass)
      (tree-classify tree)
    (cond
     ;; For a disjunction, process each term.
     ;; Fail if all terms fail.
     ((and (or path walk-exp)
	   (eq subclass :OR))
      (or (some #'(lambda (term)
		    (success-p (tree-path-test-aux term path)))
		(rest tree))
	  :FAIL))

     ;; For a conjunction, process each term.
     ;; Fail if any term fails.
     ((and (or path walk-exp)
	   (eq subclass :MULTIPLE))
      (or (every #'(lambda (term)
		     (success-p (tree-path-test-aux term path)))
		 (rest tree))
	  :FAIL))

     ;; End of path: do whatever
     ((null path)
      (case do
	(:READ (if fn
		   (funcall fn tree class subclass)
		 tree))
	(otherwise :FAIL)))

     ;; else take ASSOC
     ((eq class :LIST)
      (let ((slot (assoc (first path) (case subclass
					(:KEY (rest tree))
					(:NOKEY tree)))))
	(if slot
	    (tree-path-test-aux (second slot) (rest path))
	  :FAIL)))
     
     ;; Anything else, we're hosed
     (t
      :FAIL))))


;------------------------------------------------------------------------------
; 27-Nov-96-garof:
; Searches the given FS or IR tree for an attribute (presented as a list item).
; Returns the attribute value if found, :FAIL otherwise.
; Calling "Tree-Path-Test" with "tree" instantiated to ":FAIL" returns ":FAIL".
; Values of ":DO" are ":READ" or ":UNIFY", with default ":READ".
; ":WALK-EXP" is a boolean key.  Default value "nil", may be "t".
;------------------------------------------------------------------------------
(defun Tree-Path-Test (tree path
			    &key (val nil) (fn nil) (do :READ) (walk-exp nil))

  "Searches the given FS/IR tree for an attribute (presented as a list item).
Returns the attribute value if found, \":FAIL\" otherwise.
Instantiating \"tree\" to \":FAIL\" returns \":FAIL\".
Values of \":DO\" are \":READ\" or \":UNIFY\", with default \":READ\".
\":WALK-EXP\" is a boolean key.  Default value \"nil\", may be \"t\"."

  (declare (special val fn do walk-exp))
  (Tree-Path-Test-Aux tree path)) 


;;;==================================================================;
;;; Change the value at the end of a tree path
(defun add-path (path val key &optional (add-key nil))
  (append (and add-key key)
	  (if (rest path)
	      (list (list (car path) (add-path (cdr path) val key t)))
	    (list (list (car path) val)))))

(defun flatten-terms (terms flatten-op)
  "Flattens any sub-expressions in TERMS having the operator FLATTEN-OP.
Destroys TERMS.  If FLATTEN-OP is nil, simply returns TERMS."
  (if flatten-op
      (let ((new-terms (cons nil terms))
	    subterms
	    last-subterms)
	(setf terms new-terms)
	;; Loop, inspecting the *next* element of TERMS and perhaps modifying
	;; the CDR of the *current* element.
	(loop
	 (unless (rest terms)
	   (return))
	 (if (and (listp (second terms))
		  (eq flatten-op (first (second terms))))
	     ;; The next term is an expression with FLATTEN-OP.
	     (if (setf subterms (copy-list (rest (second terms))))
		 ;; It's not empty: paste its SUBTERMS into TERMS at this point
		 (setf last-subterms (last subterms)
		       (rest last-subterms) (rest (rest terms))
		       (rest terms) subterms
		       terms last-subterms)
	       ;; It's empty: snip it out
	       (setf (rest terms) (rest (rest terms))))
	   (setf terms (rest terms))))
	(rest new-terms))
    terms))


(defun tree-path-set-aux (tree path &key (old-val nil))
  (declare (special val fn do op add-path walk-exp flatten-exp share path-cnt))
  (multiple-value-bind (class subclass)
      (tree-classify tree)
    (cond
     ;; For a disjunction, process each term.  Fail if all terms fail.  Delete
     ;; :FAILS and nulls, flatten any disjunction terms.  Make new result expression.
     ((and (or path walk-exp)
	   (eq subclass :OR))
      (let ((new-terms (mapcar #'(lambda (term) (tree-path-set-aux term path))
			       (rest tree))))
	(if (every #'(lambda (term) (eq term :FAIL)) new-terms)
	    :FAIL
	  (make-exp+ (first tree)
		     (delete-if #'(lambda (term) (or (null term) (eq term :FAIL)))
				(flatten-terms new-terms (and flatten-exp (first tree))))))))

     ;; For a conjunction, process each term.  Fail if any term fails.  Delete
     ;; nulls, flatten any conjunction terms.  Make new result expression.
     ((and (or path walk-exp)
	   (eq subclass :MULTIPLE))
      (let ((new-terms (mapcar #'(lambda (term) (tree-path-set-aux term path))
			       (rest tree))))
	(if (member :FAIL new-terms :test #'eq)
	    :FAIL
	  (make-exp+ (first tree)
		     (delete NIL (flatten-terms new-terms (and flatten-exp (first tree)))
			     :test #'eq)))))

     ;; End of path: do whatever.  Increment PATH-CNT.
     ((null path)
      (prog1
	  (case do
	    (:OVERWRITE (if fn
			    (funcall fn tree class subclass)
			  (if (or share (zerop path-cnt))
			      val
			    ;; For each insertion of VAL after the first, copy VAL
			    ;; to keep TREE a tree.
			    (copy-tree val))))
	    (:REMOVE nil)
	    (:PUSH
	     ;; copy val every time 
	     (make-exp+ '*MULTIPLE* (append old-val (list val))))
	    (otherwise :FAIL))
	(incf path-cnt)))

     ;; else take ASSOC
     ((eq class :LIST)
      (let (key slots slot)
	(case subclass
	  (:KEY (setf key (list (car tree))
		      slots (rest tree)))
	  (:NOKEY (setf key nil
			slots tree)))
	(setf slot (assoc (car path) slots)
	      slots (if slot
			;; Path exists
			(let ((new-filler (tree-path-set-aux
					   (second slot)
					   (rest path)
					   :old-val old-val
					   )))
			  (cond ((eq new-filler :FAIL) :FAIL)
				((null new-filler) (remove slot slots :test #'eq))
				(t
				 (cons (list (first slot) new-filler)
				       (remove slot slots :test #'eq)))))
		      ;; Path does not exist.  If paving, add it; else fail.
		      (if add-path
			  (nconc (add-path path (if fn
						    (funcall fn nil)
						  val)
					   (and key (list '*NO-HEAD)))
				 slots)
			:FAIL)))
	(failsafe slots :succeed (nconc key slots))))
     
     ;; Anything else, we're hosed
     (t
      :FAIL))))

;;; to implement :PUSH, look at setvalue function

;------------------------------------------------------------------------------
; 27-Nov-96-garof:
; For an example of how this works, see "Calculate-Semantic-Number" in
; "noun-rules.lisp".
; Values of key ":DO":
; [Default: :OVERWRITE], :PUSH, :PUSHEND, :POP, :POPEND, :UNIFY, :READ, :REMOVE
; Values of key ":OP":			 [Default: :MULTIPLE], :OR
; Values of boolean key ":ADD-PATH":	 [Default:   t], nil
; Values of boolean key ":WALK-EXP":	 [Default: nil], t
; Values of boolean key ":FLATTEN-EXP":	 [Default:   t], nil
; Values of boolean key ":SHARE":	 [Default: nil], t
;------------------------------------------------------------------------------
(defun Tree-Path-Set
  (tree path &key (val nil) (fn nil) (do :OVERWRITE) (op :MULTIPLE)
	(add-path t) (walk-exp nil) (flatten-exp t) (share nil))
  "Used for traversing an FS/IR tree and modifying its attributes and values.
For more information, see function documentation in program file."

  (declare (special val fn do op add-path walk-exp flatten-exp share))
  (let ((path-cnt 0)			; Count paths found
	old-val)
    (declare (special path-cnt))

    (when (member do '(:POP :POPEND :READ :REMOVE))
      ;; Don't want to create structure
      (setf add-path nil))
    ;; remember old value at path
    (when (eq do :PUSH)
      (let (val-class val-subclass)
	(setq old-val (tree-read-all tree path :walk-exp t))
	(multiple-value-setq (val-class val-subclass)
	  (tree-classify old-val))
	;; turn old-val into a list of f-structs
	(setq old-val 
	      (cond
	       ((eq val-class :LIST)
		old-val
		) 
	       ((eq val-class :EXPRESSION)
		(cdr old-val))))))
    (tree-path-set-aux tree path :old-val old-val)))


(defun tree-path-remove (tree path)
  (failsafe (tree-path-set tree path :do :remove) :fail tree)) 


;;;==================================================================;
;;; Combine repeated slots in FS/IR tree

(defun combine-trees (tree0 tree1 op)
  (let (class0
	subclass0
	class1
	subclass1)
    (multiple-value-setq (class0 subclass0)
      (tree-classify tree0))
    (multiple-value-setq (class1 subclass1)
      (tree-classify tree1))
    (cons op
	  (nconc (if (and (eq class0 :EXPRESSION)
			  (eq subclass0 op))
		     (rest tree0)
		   (list tree0))
		 (if (and (eq class1 :EXPRESSION)
			  (eq subclass1 op))
		     (rest tree1)
		   (list tree1))))))

(defun combine-duplicate-slots (tree op &optional (class nil) (subclass nil))
  (let (slots
	slot
	slots-d
	slot-d)
    (unless (and class subclass)
      (multiple-value-setq (class subclass)
	(tree-classify tree)))
    (case class
      (:LIST
       (when (eq subclass :NOKEY)
	 ;; Make FS look like IR
	 (setf tree (cons nil tree)))

       (setf slots tree)
       ;; In these loops, the loop variable points one cons behind the cons of
       ;; interest, so that we can delete the latter if need be.
       (loop
	(unless (rest slots)
	  (return))
	(setf slot (cadr slots)
	      slots-d (cdr slots))
	;; Check slots following SLOT for the same key as SLOT
	(loop
	 (unless (rest slots-d)
	   (return))
	 (setf slot-d (cadr slots-d))
	 (when (eq (first slot) (first slot-d))
	   ;; Same key: combine SLOT-D with SLOT and delete SLOT-D
	   (setf (second slot) (combine-trees (second slot) (second slot-d) op)
		 (cdr slots-d) (cddr slots-d)))
	 (setf slots-d (rest slots-d)))
	(setf slots (rest slots)))

       ;; Correct for FS/IR difference
       (if (eq subclass :KEY)
	   tree
	 (rest tree)))
      (otherwise
       tree))))


;;;==================================================================;
;;; Bundle :MULTIPLEs

(defun bundle-multiple (expression)
  "Bundles :MULTIPLE EXPRESSION."
  (list '*G-COORDINATION
	(list 'CONJUNCTS expression)
	(list 'CONJUNCTION 'NULL)))


(defun bundle-multiples (tree &key (function #'bundle-multiple)
			      (tree-test-not '(*G-COORDINATION))
			      (slots-not '(CONJUNCTS)))
  "Bundles all :MULTIPLE expressions in TREE which don't appear in SLOTS-NOT in
trees matching TREE-TEST-NOT.  FUNCTION is the bundler function, taking the
:MULTIPLE expression as its single argument."
  (declare (special function tree-test-not slots-not))
  (bundle-multiples-aux tree t))

(defun bundle-multiples-aux (tree bundle-ok)
  (declare (special function tree-test-not slots-not))
  (multiple-value-bind (class subclass)
      (tree-classify tree)
    (case class
      (:EXPRESSION
       ;; Recursively process expression TREE, passing on current BUNDLE-OK status
       (do ((terms (rest tree) (rest terms)))
	   ((endp terms))
	 (setf (car terms)
	       (bundle-multiples-aux (car terms) bundle-ok)))
       ;; Bundle as appropriate
       (if (and (eq subclass :MULTIPLE)
		bundle-ok)
	   (funcall function tree)
	 tree))
      (:LIST
       ;; Recursively process slots, setting BUNDLE-OK status as appropriate
       (setf bundle-ok (not (tree-test tree-test-not tree)))
       (dolist (slot (case subclass
		       (:KEY (rest tree))
		       (:NOKEY tree)))
	 (setf (second slot)
	       (bundle-multiples-aux
		(second slot)
		(or bundle-ok
		    (not (member (first slot) slots-not :test #'eq))))))
       tree)
      (otherwise
       tree))))


;;;==================================================================;
;;; Unpack an FS/IR tree into a list of trees with no disjunctions
(defmacro some-path (paths)
  "Returns T iff some path in PATHS is non-nil."
  `(some #'identity ,paths))

(defmacro match-paths (item paths)
  "Returns all paths in PATHS for which ITEM matches the first element."
  `(mapcan #'(lambda (path)
	       (if (eq (first path) ,item)
		   (list path)
		 nil))
	   ,paths))

(defmacro rest-paths (paths)
  "Pops the first element off each path in PATHS."
  `(mapcar #'rest ,paths))


(defun cross-multiply-ors (list key-p paths key-func value-func make-func)
  "Cross-multiplies disjunctions in LIST (a list of slots or an expression),
yielding a list of new trees which do not contain any disjunctions.  If KEY-P
is non-nil, LIST has an initial symbolic element which must be skipped.  PATHS
is either nil or a list of tree paths to explode disjunctions along.

The three functions apply to the elements of LIST, which are slots or
expression terms.  KEY-FUNC retrieves the key of an element (only applies to
slots); VALUE-FUNC retrieves the value; MAKE-FUNC takes an original element
and a new value and makes a new element of the same kind."
  (declare (special copy share))
  (let ((items (if key-p
		   (rest list)
		 list))
	match-paths
	(make-new-items nil)
	subresult
	(subresults nil))

    ;; Collect a list of recursive results for each item in ITEMS, building in
    ;; reverse.  Each result is a list of OR-less trees; if there were no ORs,
    ;; a singleton list.
    (dolist (item items)
      (push
       (if (and (or (not paths)
		    (not key-func)
		    (setf match-paths (match-paths (funcall key-func item) paths)))
		;; Recurse if a) no constraining paths were given, b) ITEMS
		;; are expression terms, not slots, or c) the current item
		;; matches some path element.  As we recurse, pop any matching paths.
		(or (rest (setf subresult (tree-explode-ors-1
					   (funcall value-func item)
					   (rest-paths match-paths))))
		    copy))
	   ;; If we recursed and a) the SUBRESULT was exploded into more than
	   ;; one tree or b) we're supposed to be copying ITEMS, make new
	   ;; items containing each tree and set the flag for making new items
	   ;; lists later.
	   (progn
	     (setf make-new-items t)
	     ;; Destructively modify the SUBRESULT list to save CONSes
	     (do ((sr subresult (rest sr)))
		 ((endp sr) subresult)
	       (setf (first sr) (funcall make-func item (first sr)))))
	 ;; Otherwise, just take the current item value or a copy
	 (list (if copy
		   (copy-tree item)
		 item)))
       subresults))

    (if make-new-items
	;; Now, if we're supposed to, make new items lists with cross-multiplied
	;; results, building in reverse (unreversing the procedure above)
	(let ((newresults (do ((srs subresults (rest srs))
			       (n 1 (* n (length (first srs)))))
			      ((endp srs) (make-list n)))))
	  ;; First, make a list big enough to hold all the new items lists
	  ;; Then, paste the subresults onto the new flat result lists; unless
	  ;; we're SHAREing structure, copy subresult members after the first paste
	  (dolist (subresult subresults)
	    (do ((nrs newresults (rest nrs))
		 (first t)
		 (sr subresult (or (rest sr)
				   (setf first nil) ; Pasted each member once
				   subresult)))
		((endp nrs))
	      (push (if (or first share)
			(first sr)
		      (copy-tree (first sr)))
		    (first nrs))))
	  ;; Add any key from original LIST
	  (when key-p
	    (do ((nrs newresults (rest nrs)))
		((endp nrs))
	      (push (first list) (first nrs))))
	  newresults)
      ;; Otherwise, return the original LIST
      (list list))))


(defun tree-explode-ors-1 (tree paths)
  "Working function for TREE-EXPLODE-ORS."
  (declare (special explode-atoms copy share))
  (if (or (not paths)
	  (some-path paths))
      ;; Either no PATHS specified or more elements on some path: proceed
      (flet ((make-term (item result) (declare (ignore item)) result)
	     (make-slot (item result) (list (first item) result)))
	(multiple-value-bind (class subclass)
	    (tree-classify tree)
	  (case class
	    (:EXPRESSION
	     (case subclass
	       (:OR (if (or explode-atoms
			    (notevery #'atom (rest tree)))
			;; Explode the disjunction
			(mapcan #'(lambda (term)
				    (tree-explode-ors-1 term paths))
				(rest tree))
		      ;; Don't explode a disjunction of atoms unless requested:
		      ;; just return TREE or a copy
		      (list (if copy
				(copy-tree tree)
			      tree))))
	       ;; Multiply out disjunctions in expression
	       (otherwise (cross-multiply-ors tree t paths
					      nil #'identity #'make-term))))
	    (:LIST
	     ;; Multiply out disjunctions in slot list
	     (cross-multiply-ors tree (eq subclass :KEY) paths
				 #'first #'second #'make-slot))
	    (otherwise (list tree)))))
    ;; Reached end of all PATHS: just return TREE or a copy
    (list (if copy
	      (copy-tree tree)
	    tree))))


(defun tree-explode-ors (tree &key (explode-atoms nil)
			      (paths nil)
			      (copy t)
			      (share nil))
  "Multiplies out disjunctions in FS or IR TREE, yielding a list of structures
which do not contain disjunctions.  Unless EXPLODE-ATOMS is non-nil,
disjunctions whose terms are all atoms, such as (*OR* + -), are left intact.
If PATHS is nil, all disjunctions are exploded; if PATHS is a list of tree
paths, only disjunctions along those paths are exploded.  If COPY is non-nil,
the resulting structures share no list structure with TREE; otherwise they
may.  If SHARE is non-nil, the resulting structures may share structure with
each other; otherwise they are disjunct.  To use as little space as possible,
therefore, call this function with (COPY NIL) and (SHARE T)."
  (declare (special explode-atoms copy share))
  (tree-explode-ors-1 tree paths))



;---eof structure-tools.lisp---
