;;;==================================================================;
;;; -*- Syntax: Common-Lisp; Package: USER; Base: 10; Mode: LISP -*-
;;;==================================================================;
;;;
;;;               Center for Machine Translation
;;;                 Carnegie-Mellon University
;;;                                                                       
;;;------------------------------------------------------------------;
;;;                                                                       
;;; Copyright (c) 1995
;;; Carnegie Mellon University. All Rights Reserved.                      
;;;                                                                       
;;;------------------------------------------------------------------;
;;;
;;;          File: info-tools.lisp
;;;  File created: 21-Jun-95 by ndb
;;;        Author: Alex Franz <amf@cs.cmu.edu>
;;;                Nicholas Brownlow <ndb@cs.cmu.edu>
;;; Last Modified: 21-Jul-95, 16:27-Jun-95 at 14:40
;;;
;;;------------------------------------------------------------------;
;;; Description                                                      
;;;
;;; Functions which return information about the Lisp environment.  Filing and
;;; feedback functions.
;;;
;;; Most of these are from "amf-tools.lisp".


;;;==================================================================;

;;; Package statements

(in-package :user)


;;;==================================================================;

;;; Date and time functions

(defun clock (&optional (universal-time (get-universal-time)))
  "Return a string that tells the current time."
  (multiple-value-bind (sec min hrs)
    (decode-universal-time universal-time)
    (format nil "~2D:~2,'0D:~2,'0D ~:[AM~;PM~]"
	    (if (> hrs 12) (- hrs 12) hrs) 
	    min
	    sec
	    (>= hrs 12))))

(defun clock-string (&optional (universal-time (get-universal-time)))
  "Return a string that tells the current time."
  (multiple-value-bind (sec min hrs)
    (decode-universal-time universal-time)
    (format nil "~D:~D:~D"
	    hrs
	    (floor min)
	    (floor sec))))

(defun date-string (&optional (universal-time (get-universal-time)))
  "Return a string that tells the current date."
  (multiple-value-bind (sec min hrs date month year day)
    (decode-universal-time universal-time)
    (declare (ignore sec min hrs))
    (format nil "~A-~D-~A-~2,'0D"
	    (svref #("Mon" "Tue" "Wed" "Thu" "Fri" "Sat" "Sun") day)
	    date
	    (svref #("???" "Jan" "Feb" "Mar" "Apr" "May" "Jun"
			   "Jul" "Aug" "Sep" "Oct" "Nov" "Dec") month)
	    (mod year 100))))

(defun date-string-short (&optional (universal-time (get-universal-time)))
  "Returns a short string indicating the current date."
  (multiple-value-bind (sec min hrs date month year day)
      (decode-universal-time universal-time)
    (declare (ignore sec min hrs day))
    (format nil "~D-~A-~2,'0D"
	    date
	    (svref #("???" "Jan" "Feb" "Mar" "Apr" "May" "Jun"
		     "Jul" "Aug" "Sep" "Oct" "Nov" "Dec") month)
	    (mod year 100))))


;;; The following function is biased for U.S.A. holidays, but can be
;;; edited to reflect any calendar.

(defun red-letter-day (&optional (universal-time (get-universal-time)))
  "Return a string that describes the current holiday, or NIL if none known"
  (multiple-value-bind (sec min hrs date month year day)
    (decode-universal-time universal-time)
    (declare (ignore sec min hrs year))
    (cond ((and (= month 1) (= date 1)) "New Year's Day")
	  ((and (= month 11) (<= 22 date 28) (= day 3)) "Thanksgiving")
	  ((and (= month 7) (= date 4)) "Independence Day")
	  ((and (= month 5) (= date 30)) "Memorial Day")
	  ((and (= month 4) (= date 1)) "April Fool's Day")
	  ((and (= month 10) (= date 31)) "Halloween")
	  ;; Add your favorite holidays here...
	  (t nil))))


;;;==================================================================;

;;; Lisp environment

(defun machine-instance-short ()
  "Returns a short string indicating the machine instance."
  (let ((name (machine-instance)))
    (subseq name 0 (position #\. name))))

(defun machine-type-short ()
  "Returns a short string indicating the machine type."

  ;modified 7/21/95 by gd to include "aix"
  #+:sun "sun"
  #+:ultrix "dec"
  #+:aix "aix"
  )

(defun user-name ()
  "Returns user name if environment variable queries supported."
  #+LUCID (environment-variable "USER")
  #-LUCID "unknown")

(defun lisp-id-string ()
  "Returns a string that indicates where and when this Lisp is running."
  (format nil
	  "~A-~A-~A-~A"
	  (machine-instance-short)
	  (user-name)
	  (date-string)
	  (clock-string)))


;;;==================================================================;

;;; Filing and feedback functions

(defmacro with-output-to-file (file &rest forms)
  `(with-open-file (outstream ,file :direction :output)
    (let ((*standard-output* outstream)
	  (*error-output* outstream))
      (format outstream ";;; Output-to-file started ~A~%~%"
	      (lisp-id-string))
      ,@forms
      (format outstream "~%~%;;; Output-to-file finished ~A~%"
	      (lisp-id-string)))))


;;; Rewritten 21-Jun-95 by ndb

(defun filepath (dirname filename)
  "Concatenates DIRNAME and FILENAME, making sure that exactly one slash
appears between them."
  (let ((dirname/ (and (plusp (length dirname))
		       (char= #\/ (schar dirname (1- (length dirname))))))
	(/filename (and (plusp (length filename))
			(char= #\/ (schar filename 0)))))
    
    (concatenate 'string
		 dirname
		 (if dirname/
		     (if /filename
			 (subseq filename 1)
		       filename)
		   (if /filename
		       filename
		     (concatenate 'string "/" filename))))))


(let ((dots 0))

  (defun reset-dots ()
    (setf dots 0))

  (defun print-dot (string &key (stream *standard-output*) (char #\.) (length 50))
    (if (zerop (mod (incf dots) length))
	(format stream " ~A ~A~%" dots string) ; then
	(write-char char stream)))

  (defun print-dot-10 (string &key (stream *standard-output*) (char #\.) (length 50))
    ;; only print one dot every ten
    (if (zerop (mod (incf dots) 10))
	;; then
	(if (zerop (mod dots (* 10 length)))
	    (format stream " ~A ~A~%" dots string) ; then
	    (write-char char stream) ; else
	    )))

  (defun print-last-dot (string &key (stream *standard-output*))
    (format stream " ~A ~A~%" dots string)))
