;;;==================================================================;
;;; -*- Syntax: Common-Lisp; Package: USER; Base: 10; Mode: LISP -*-
;;;==================================================================;
;;;
;;;               Center for Machine Translation
;;;                 Carnegie-Mellon University
;;;                                                                       
;;;------------------------------------------------------------------;
;;;                                                                       
;;; Copyright (c) 1995
;;; Carnegie Mellon University. All Rights Reserved.                      
;;;                                                                       
;;;------------------------------------------------------------------;
;;;
;;;          File: static-trie-machine.lisp
;;;  File created: Friday July 17, 1992
;;;        Author: Nicholas Brownlow <ndb@cs.cmu.edu>
;;; Last Modified:  1-May-95 at 16:33
;;;
;;;------------------------------------------------------------------;
;;; Description
;;; 
;;; A state-machine trie structure optimized for static run-time use.
;;;
;;; This code implements a trie as a deterministic finite-state automaton.
;;; Arc labels are stored in a hash table; they reference arrays of state
;;; transitions.

(in-package :user)

;;;------------------------------------------------------------------;

;;; These constants are intended as the interface to start/halt state values.

(defconstant *stm-start* 0 "Starting state")
(defconstant *stm-halt* -1 "Halting state")


;;;------------------------------------------------------------------;

;;; The STM structure holds the static trie machine and related values.

(defstruct stm				; Static trie machine
  (labels nil :type (or null hash-table)) ; Maps labels onto arcs arrays
  (test nil :type function)		; Test function for hash table
  (mode :DYNAMIC :type symbol)		; Operating mode: :STATIC or :DYNAMIC
  (start *stm-start* :type integer)	; Starting state
  (halt *stm-halt* :type integer)	; Halting state
  (max 0 :type integer)			; Highest state counter
  )


;;;------------------------------------------------------------------;

;;; The STM-ARC structure defines a transition from STATE to NEXT-STATE.
;;; VALUE is the value at STATE.

(defstruct stm-arc			; Static trie machine arc
  (state *stm-start* :type integer)	; State at beginning of arc
  (next-state *stm-halt* :type integer)	; State at end of arc
  value					; Value at state
  )


;;;------------------------------------------------------------------;

;;; These macros are intended as the interface to arc contents.

(defmacro stm-haltp (stm stm-arc)
  "Return T iff arc STM-ARC leads to the halt state for STM."
  `(eq (stm-arc-next-state ,stm-arc) (stm-halt ,stm)))

(defmacro stm-get-arc-value (stm-arc)
  "Returns value for STM-ARC."
  `(stm-arc-value ,stm-arc))

(defmacro stm-set-arc-value (stm-arc value)
  "Sets value for STM-ARC."
  `(setf (stm-arc-value ,stm-arc) ,value))

(defmacro stm-get-arc-next-state (stm-arc)
  "Returns next-state for STM-ARC."
  `(stm-arc-next-state ,stm-arc))


;;;==================================================================;

;;; Han-D binary search function

(defun binsearch (object array predicate &key
			(key #'identity)
			(test #'eql)
			(start 0)
			(end (length array)))
  (let (middle
	element)
    (psetq start (max 0 (min start end))
	   end (min (length array) (max start end)))
    (cond ((= start end)		; No elements
	   (return-from binsearch nil))
	  ((= (+ 1 start) end)		; Only one element
	   (if (funcall test object
			(funcall key (setq element (aref array start))))
	       (return-from binsearch element)
	       (return-from binsearch nil)))
	  (t				; Otherwise, adjust end
	   (decf end)))
    (loop
     (cond ((= (+ 1 start) end)		; Two elements
	    (if (funcall test object
			 (funcall key (setq element (aref array start))))
		(return-from binsearch element)
		(if (funcall test object
			     (funcall key (setq element (aref array end))))
		    (return-from binsearch element)
		    (return-from binsearch nil))))
	   (t				; More than two elements
	    (setq middle (+ start (floor (- end start) 2))
		  element (aref array middle))
	    (if (funcall test object (funcall key element))
		(return-from binsearch element)
		(if (funcall predicate object (funcall key element))
		    (setq end middle)
		    (setq start middle))))))))


;;;==================================================================;

;;; Low-level STM access functions and macros

(defun stm-new-state (stm)
  "Increment and return new state number."
  (incf (stm-max stm)))

(defun stm-arcs-search (stm arcs state)
  "Search for the arc for STATE in the arc array ARCS of STM.
If in :STATIC mode, use binary search; otherwise use sequential search."
  (if (eq (stm-mode stm) :STATIC)
      (binsearch state arcs #'< :test #'= :key #'stm-arc-state)
      (find state arcs :key #'stm-arc-state)))

(defun stm-arcs-add (stm arcs arc)
  "Add ARC to the arc array ARCS.  Change STM to :DYNAMIC mode. Return ARC." 
  (setf (stm-mode stm) :DYNAMIC)
  (vector-push-extend arc arcs)
  arc)

(defmacro stm-get-arcs (stm label)
  "Get the arcs array for LABEL in STM."
  `(gethash ,label (stm-labels ,stm)))

(defmacro stm-set-arcs (stm label arcs)
  "Set the arcs array for LABEL in STM to ARCS."
  `(setf (gethash ,label (stm-labels ,stm)), arcs))


;;;==================================================================;

;;; Overall STM creation and manipulation functions

(defconstant *stm-labels-init* 10000 "Initial number of arc labels")
(defconstant *stm-arcs-init* 10 "Initial number of arcs PER LABEL")

(defun make-static-trie-machine (&key (size *stm-labels-init*)
		      (test #'eql))
  "Create an stm with a hash table of size SIZE and test TEST to hold the arc labels."
  (make-stm :labels (make-hash-table :size size :test test)
	    :test test))

(defun make-stm-arcs (&key (size *stm-arcs-init*))
  "Create an adjustable, fill-pointer array of size SIZE to hold the arcs."
  (make-array size :element-type 'stm-arc
	      :adjustable t :fill-pointer 0))


(defun shrink-stm (stm)
  "Shrink all the arcs arrays in STM down to their fill-pointer values."
  (maphash #'(lambda (label arcs)
	       (stm-set-arcs stm label (shrink-stm-arcs arcs)))
	   (stm-labels stm)))

(defun shrink-stm-arcs (arcs &optional size)
  "Adjust the size of ARCS either to SIZE or to its fill-pointer value."
  (adjust-array arcs (or size (fill-pointer arcs)) :element-type 'stm-arc))


(defun sort-stm (stm)
  "Sort all the arcs arrays in STM in increasing order of arc states."
  (maphash #'(lambda (label arcs)
	       (stm-set-arcs stm label (sort-stm-arcs arcs)))
	   (stm-labels stm)))

(defun sort-stm-arcs (arcs)
  "Sort ARCS in increasing order of arc states."
  (sort arcs #'< :key #'stm-arc-state))


(defun stm-static (stm)
  "Switch STM into :STATIC mode, sorting all ARCS arrays so that
binary search can be used."
  (sort-stm stm)
  (setf (stm-mode stm) :STATIC))


(defun clear-stm (stm)
  "Clear the STM."
  (clrhash (stm-labels stm))
  (setf (stm-mode stm) :DYNAMIC)
  (setf (stm-max stm) 0))


(defun map-stm (function stm)
  "Applies FUNCTION to each arc in STM."
  (maphash #'(lambda (label arcs)
	       (declare (ignore label))
	       (map nil function arcs))
	   (stm-labels stm)))


;;;==================================================================;

;;; High-level STM access functions

(defun stm-get-arc (stm state label)
  "Get the arc for STATE and LABEL in STM; NIL if none."
  (if (= state (stm-halt stm))
      nil
      (stm-arcs-search stm (stm-get-arcs stm label) state)))

(defun stm-get-arc-new (stm state label &optional (endp nil))
  "Get the arc for STATE and LABEL in STM, creating a new one if necessary.
If ENDP is non-nil, this is the last step in the current machine computation."
  (let* (arcs				; Arcs array
	 arcsp				; Arcs array found?
	 arc)				; Arc object
    (if (= state (stm-halt stm))
	(return-from stm-get-arc-new nil))
    (setq arcs (stm-get-arcs stm label)
	  arcsp arcs)
    ;; If arcs array not found, make a new one
    (unless arcsp
      (setq arcs (stm-set-arcs stm label (make-stm-arcs))))
    ;; Search for arc only if arcs array not newly created
    (setq arc (and arcsp
		   (stm-arcs-search stm arcs state)))
    (if arc
	;; If arc exists, and next-state is the halting state,
	;; but the computation is not done, make a new next-state
	(if (and (= (stm-arc-next-state arc) (stm-halt stm))
		 (not endp))
	    (setf (stm-arc-next-state arc) (stm-new-state stm)))
	;; If arc does not exist, make a new one, with next-state
	;; set to the halting state if the computation is done
	;; and to a new statenumber otherwise
	(setq arc
	      (stm-arcs-add stm arcs (make-stm-arc :state state
					       :next-state (if endp
							       (stm-halt stm)
							       (stm-new-state stm))))))
    arc))


;;;==================================================================;

;;; Printout and statistics functions

(defun stm-print (stm)
  (maphash #'(lambda (label arcs)
	       (format t "~S:" label)
	       (map nil #'(lambda (arc)
			    (format t " ~D.~D" (stm-arc-state arc)
				    (stm-arc-next-state arc)))
		    arcs)
	       (terpri))
	   (stm-labels stm)))

(defun stm-statistics (stm)
  (let ((max-label "n/a")
	(max-arcs 0))
    (maphash #'(lambda (k v)
		 (when (> (length v) max-arcs)
		   (setq max-label k)
		   (setq max-arcs (length v))))
	     (stm-labels stm))
    (format t "~%Most arcs for a label: ~D arcs for ~S~%" max-arcs max-label)))

