;      -*- Syntax: Common-Lisp; Package: USER; Base: 10; Mode: LISP -*-
;==============================================================================
;     File:  interpreter.lisp
;	By:  Willy Walker	<wkw@pitt.edu>
;     Path:  ../cmt/catalyst/amt-ana/code/interpreter/<release-vs>
;  Started:  10 September 1993
; Modified:  03 December 1996	by <garof>
;
; Comments:  All of the central functions for the semantic interpreter.  Look
;	     at the bottom of the file for the main functions
;	     (Map-Semantic-Map, etc.).
;
; Modified:  10 September 193
;	By:  Willy Walker	<wkw@pitt.edu>
;  Reasons:  Functions here are adapted from the interpreter created for the
;	     catalyst project by Eric Nyberg
;
; Modified:  21 June 1995
;	By:  Nicholas Brownlow	<ndb@clarit.com>
;  Reasons:  Nicholas was the official "Interpreter" maintainer until then.
;
; Modified:  02 December 1996
;	By:  Joseph Giampapa	<garof@cs.cmu.edu>
;  Reasons:  To garof-ify this file, and to add version control markers to it.
;
; Modified:  <date>
;	By:  <name>		<full e-mail>
;  Reasons:  
;
;==============================================================================


;..............................................................................
;			Center for Machine Translation
;			  Carnegie Mellon University
;
;		     Copyright (c) 1993, 1994, 1995, 1996
;	       Carnegie Mellon University.  All Rights Reserved.
;..............................................................................


;..............................................................................
;			      Package Statements
;..............................................................................
(in-package :user)


;..............................................................................
; 14-Nov-96-garof:  For version control.
; 02-Dec-96-garof:  The directory should be correct if you use the defsystem.
;..............................................................................
;(load (compile-file "/afs/cs.cmu.edu/project/cmt/catalyst/amt-ana/code/interpreter/5.0Analyzer/ic_vers_ctrl.lisp"))


;------------------------------------------------------------------------------
; For use by the maintainer's personal program.
;------------------------------------------------------------------------------
(defconstant *INTERPRETER-VS* '5.0Analyzer)



;..............................................................................
;			       Global variables
;..............................................................................
(defvar *INTERPRETER-TRACE* nil)


;; 23-Feb-97 by EHN
(defvar *warn-unmapped-features* nil)

;..............................................................................
;			    F-Structure Predicates
;..............................................................................


;------------------------------------------------------------------------------; Callers of Atom-P: (Feature-Value-P)
;------------------------------------------------------------------------------
(defmacro Atom-P (value)
  `(or
    (symbolp ,value)
    (stringp ,value)
    (numberp ,value)))


;------------------------------------------------------------------------------
; Callers of Feature-Value-P:
; (Feature-Value-P Interpreter-Map-Consumes Interpreter-Map-Non-Semslot)
;
; NDB: T iff VALUE is the value for a feature  (i.e. an atom or expression)
;------------------------------------------------------------------------------
(defun Feature-Value-P (value)
  (or (atom-p value)
      (and
       (listp value)
       (Parser-Or-Or-Mult-Value-P value)
       (Feature-Value-P (second value)))))


;------------------------------------------------------------------------------
; Tells if the given symbol is a proper semantic head (i.e., it is a symbol
; and begins with an asterisk.
;
; Callers of SEM-HEAD-P:
; (Combine-Common-Slot-Fillers Interpreter-Map-Slot-Aux-2 Sem-Head-IR-P)
;------------------------------------------------------------------------------
(defun Sem-Head-P (slot-or-head)
  "Tells if the given symbol is a proper semantic head (i.e., it is a symbol
and begins with an asterisk."
  (and (symbolp slot-or-head)
       (char= (char (symbol-name slot-or-head) 0) #\*)))


;------------------------------------------------------------------------------
; Combines all the fillers in SLOTS-WITH-FILLERS in a single slot with the
; given semantic SLOT-KEY using the given operator MULT-OR-OR.  If combining
; under an :OR, remove identical fillers.  If only one filler remains, make
; that the single value of the slot.
;
; Callers of Combine-Common-Fillers: (Combine-Common-Slot-Fillers)
;------------------------------------------------------------------------------
(defun Combine-Common-Fillers (slot-key slots-with-fillers mult-or-or)
  "Combines all the fillers in SLOTS-WITH-FILLERS in a single slot with the
given semantic SLOT-KEY using the given operator MULT-OR-OR.  If combining
under an :OR, remove identical fillers.  If only one filler remains, make that
the single value of the slot."

  (let ((fillers (mapcar #'second slots-with-fillers)))
    (when (eq mult-or-or :OR)
	  (setq fillers (uniquify-terms fillers)))
    (and fillers
	 (list (list slot-key
		     (if (< (length fillers) 2)
			 (first fillers)
		       (cons mult-or-or fillers)))))))


;------------------------------------------------------------------------------
; Given a list of SLOTS-WITH-FILLERS (a headless IR), combine the fillers for
; slots with the same slot-key in a single slot using a :MULTIPLE or :OR
; operator as appropriate.
;
; Callers of Combine-Common-Slot-Fillers: (Invert-Mult-Or-Or)
;------------------------------------------------------------------------------
(defun Combine-Common-Slot-Fillers (slots-with-fillers mult-or-or)
  "Given a list of SLOTS-WITH-FILLERS (a headless IR), combine the fillers for
slots with the same slot-key in a single slot using a :MULTIPLE or :OR
operator as appropriate."
  
  ;; First make sure there are not :MULTIPLEs or :ORs that contain a
  ;; conjunction/disjunction of a single slot.
  (let ((processed-slots-with-fillers
	 (mapcar #'invert-mult-or-or-nested slots-with-fillers)))

    
    (cond ;;((< (length processed-slots-with-fillers) 2)
	   ;; Return a single slot-filler, if that is all that is left.
	   ;;
	   ;; NDB: How can this be right?  Surely we always want to return a
	   ;; LIST of slots.
	   ;;(first processed-slots-with-fillers))

     ((and (listp processed-slots-with-fillers)
	   (listp (first processed-slots-with-fillers))
	   (Sem-Head-P (first (first processed-slots-with-fillers))))
      processed-slots-with-fillers)

	   ;; Cannot combine slots if they simply contain proper IRs.  The
	   ;; test above tells that the first element (and, therefore, all
	   ;; elements) of PROCESSED-SLOTS-WITH-FILLERS contains a proper IR
	   ;; (has a semantic-head), and therefore there are no :MULTIPLEs or
	   ;; :ORs to be combined.  (I am not certain that this logic is
	   ;; correct.)
	   ;;
	   ;; NDB: This does occur.

	  (t	;; Loop through each slot in PROCESSED-SLOTS-WITH-FILLERS
	   (let ((new-slots-with-fillers nil)
		 slot-with-filler
		 slot-key
		 other-slots-with-fillers)

	     (loop
	      (unless processed-slots-with-fillers (return))
	      
	      (setf slot-with-filler (first processed-slots-with-fillers)
		    slot-key (first slot-with-filler))
		 
	      (if (interpreter-or-or-mult-symbol-p slot-key)

		  ;; A :MULTIPLE or :OR cannot be handled at this level.  Just
		  ;; copy it to NEW-SLOTS-WITH-FILLERS (SETQ below) and move on
		  ;; to the next element of PROCESSED-SLOTS-WITH-FILLERS
		  (setf other-slots-with-fillers nil
			processed-slots-with-fillers
			(rest processed-slots-with-fillers))

		;; Find all of the other elements of
		;; PROCESSED-SLOTS-WITH-FILLERS that have the same slot-key as
		;; the current one.
		(multiple-value-setq
		 (other-slots-with-fillers processed-slots-with-fillers)
		 (member-extract slot-key (rest processed-slots-with-fillers)
				 :key #'car)))

	      (setf new-slots-with-fillers
		    (if other-slots-with-fillers

			;; If the current slot-key has multiple occurrences,
			;; combine them together using COMBINE-COMMON-FILLERS
			(nconc
			 (Combine-Common-Fillers slot-key
			  (cons slot-with-filler other-slots-with-fillers)
			  mult-or-or)
			 new-slots-with-fillers)
		      
		      ;; Otherwise, just add the current slot to the list of
		      ;; slots to be returned by this function.
		      (cons slot-with-filler new-slots-with-fillers))))

	     new-slots-with-fillers)))))


;------------------------------------------------------------------------------
; Function Invert-Mult-Or-Or
;
; In the convoluted list of slots and fillers that is built up the interpreter
; rules, there may the inapprpriate :MULTIPLEs and :ORs.  "Invert-Mult-Or-Or"
; attempts to remove these inappropriate :MULTIPLEs and :ORs as shown in the
; example below.
; The optional "nested" argument is non-NIL for recursive and internal calls to
; "Invert-Mult-Or-Or" so that extra list structure is not built.
;
; For example (headless IRs):
;
; ((:MULTIPLE
;    (SLOT-X (*A-HEAD (A-SLOT A-FILLER)))
;    (SLOT-X (*A-DIFFERENT-HEAD (A-SLOT A-FILLER))))
;  (SLOT-Y (*ANOTHER-HEAD (A-SLOT A-FILLER))))
;
; Should become:
;
;  ((SLOT-X
;    (:MULTIPLE
;     (*A-HEAD (A-SLOT A-FILLER))
;     (*A-DIFFERENT-HEAD (A-SLOT A-FILLER))))
;   (SLOT-Y (*ANOTHER-HEAD (A-SLOT A-FILLER))))
;
; Callers of Invert-Mult-Or-Or-Nested: (Combine-Common-Slot-Fillers)
;------------------------------------------------------------------------------
(defun Invert-Mult-Or-Or-Nested (semstruct)
  "Simply calls INVERT-MULT-OR-OR with the NESTED argument set to T."
  (invert-mult-or-or semstruct t))


;------------------------------------------------------------------------------
; Callers of "Invert-Mult-Or-Or": (Interpreter-Map-Semslot-Multiple
; Interpreter-Map-Slot-Aux-2 Invert-Mult-Or-Or-Nested Map-PP-Role-Coordination)
;------------------------------------------------------------------------------
(defun invert-mult-or-or (semstruct &optional (nested nil))
  (let ((mult-or-or (interpreter-or-or-mult-p semstruct)))
    (cond (mult-or-or
	   (case mult-or-or
		 (:MULTIPLE
		  ;; promote slots, combine common fillers for slots
		  (setq semstruct
			(Combine-Common-Slot-Fillers
			 (rest semstruct) :MULTIPLE))
		  (if (or (< (length semstruct) 2)
			  (symbolp (first semstruct)))
		      
		      ;; If we have only one semantic structure, return it
		      ;; alone.  Let the NESTED argument tell whether to
		      ;; bundle the returned argument in a list.
		      (if nested
			  (if (symbolp (first semstruct))
			      semstruct
			    (first semstruct))
			(if (symbolp (first semstruct))
			    (list semstruct)
			  semstruct))

		    ;; We have a multiple slots and fillers, so place them
		    ;; under a :MULTIPLE
		    (if nested
			(cons :MULTIPLE semstruct)
		      (list (cons :MULTIPLE semstruct)))))
		 (:OR
		  
		  ;; try to combine all fillers under on slot; otherwise,
		  ;; should be a top-level disjunstion (at least higher-level).
		  (setq semstruct
			(combine-common-slot-fillers (rest semstruct) :OR))
		  (if (< (length semstruct) 2)
		      ;; If only one member of :OR, collapse.
		      (if nested
			  (first semstruct)
			semstruct)
		    (if nested
			(cons :OR semstruct)
		      (list (cons :OR semstruct)))))
		 (otherwise
		  (interpreter-warn "INVERT-MULT-OR-OR - unexpected conj: ~s"
				    mult-or-or)
		  nil)))
	  (t
	   ;; No :MULTIPLE or :OR, just return the semantic structure.
	   semstruct))))


;------------------------------------------------------------------------------
; Variable *Dont-Reverse-Multiple-Slots*
;
; Certain slots that are sometimes filled with *MULTIPLE*s should not have
; their elements reversed.
; Add them with the Add-Dont-Reverse-Multiple-Slotnames function.
;
; Readers of *Dont-Reverse-Multiple-Slots*:
; (Add-Dont-Reverse-Multiple-Slotnames Interpreter-Map-Slot-Aux-2)
;------------------------------------------------------------------------------
(defvar *DONT-REVERSE-MULTIPLE-SLOTS* '())


;------------------------------------------------------------------------------
(defun Add-Dont-Reverse-Multiple-Slotnames (&rest slots)
  (dolist (slot slots)
	  (pushnew slot *dont-reverse-multiple-slots*)))


;..............................................................................
; Interpreter-Map-Feature Interpreter-Map-Feature-Aux Change-Value-Or-Or-Mult
;
; When a feature is encountered in an f-structure, Interpreter-Map-Feature is
; called.  (A feature in an f-structure is a f-structure slot that is filled
; with a value, not an f-structure.)
;..............................................................................

;------------------------------------------------------------------------------
; When an f-structure feature is filled with a *OR* or *MULTIPLE* value, then
; map the *MULTIPLE* or *OR* to :MULTIPLE or :OR as appropriate.
;
; Callers of Change-Value-Or-Or-Mult:
;	    (Interpreter-Map-Consumes Interpreter-Map-Feature-Aux)
;------------------------------------------------------------------------------
(defun Change-Value-Or-Or-Mult (value)
  "When an f-structure feature is filled with a *OR* or *MULTIPLE* value, then
map the *MULTIPLE* or *OR* to :MULTIPLE or :OR as appropriate."
  (when (listp value)
	(let ((mult-or-or (parser-or-or-mult-value-p value)))
	  (when mult-or-or
		(setq value (cons mult-or-or (copy-list (rest value)))))))
  value)


;------------------------------------------------------------------------------
; Maps feature VALUE, returning result (always a list of IR slots, even if only
; one). The result depends on which of SEM-FEATURE and VALUE-FORM are provided:
; - SEM-FEATURE, VALUE-FORM -- result contains single IR slot:
;    - Feature = SEM-FEATURE
;    - Value = result of evaluating VALUE-FORM with $VALUE bound to VALUE
; - SEM-FEATURE, no VALUE-FORM
;    - Feature = SEM-FEATURE -- result contains single IR slot:
;    - Value = VALUE
; - No SEM-FEATURE, VALUE-FORM -- result is list of IR slots
;   returned by evaluating VALUE-FORM with $VALUE bound to VALUE
; - No SEM-FEATURE, No VALUE-FORM -- result is NIL"
;
; Callers of Interpreter-Map-Feature-Aux: (Interpreter-Map-Feature)
;------------------------------------------------------------------------------
(defun Interpreter-Map-Feature-Aux (sem-feature value-form value)
  "Maps feature VALUE, returning result (always a list of IR slots, even if
only one).  The result depends on which of SEM-FEATURE and VALUE-FORM are
provided:
 - SEM-FEATURE, VALUE-FORM -- result contains single IR slot:
    - Feature = SEM-FEATURE
    - Value = result of evaluating VALUE-FORM with $VALUE bound to VALUE
 - SEM-FEATURE, no VALUE-FORM
    - Feature = SEM-FEATURE -- result contains single IR slot:
    - Value = VALUE
 - No SEM-FEATURE, VALUE-FORM -- result is list of IR slots
   returned by evaluating VALUE-FORM with $VALUE bound to VALUE
 - No SEM-FEATURE, No VALUE-FORM -- result is NIL"
  
  (let (result)
    
    (setq value (change-value-or-or-mult value))

    ;; Evaluate the mapping lisp form or simply promote the
    ;; f-structure value into the semantic structure.
    (setq result (if value-form
		     (let (($value value))
		       (declare (special $value))
		       (declare (ignore $value))
		       (eval value-form))
		   value))

    ;; Return the appropriate structure.  Note that the semantic slot
    ;; (SEM-FEATURE) can be omitted in the rule.  In this case, the mapping
    ;; lisp form should return a list of semantic structure slots and
    ;; fillers.
    ;;
    ;; NDB: But if there's no Lisp form, return NIL.
    (if sem-feature
	(and result
	     (list (list sem-feature result)))
      (and value-form
	   result))))


;------------------------------------------------------------------------------
; Callers of Interpreter-Map-Feature: (Interpreter-Map-Non-Semslot)
;------------------------------------------------------------------------------
(defun interpreter-map-feature (mapforms value)
  "Applies mapping rule FORMS to feature VALUE by calling
INTERPRETER-MAP-FEATURE-AUX.  Returns appended results."
  (safe-mapcan-1 #'(lambda (mapform)
		     (interpreter-map-feature-aux (mapform-feature mapform)
						  (mapform-form mapform)
						  value))
		 mapforms))


;------------------------------------------------------------------------------
; Originally: If there's more than one result in the list, transforms:
;   ((a 1) (b 2) ...) --> ((:MULTIPLE (a 1) (b 2) ...))
; Surely this isn't necessary, since SEMANTIC-MAP-FEATURES is expecting
; a list of slots.  So I got rid of it. -- Thu Nov 17 1994 by ndb
;------------------------------------------------------------------------------

;;; Function INTERPRETER-MAP-SLOT INTERPRETER-MAP-SLOT-AUX MAP-FILLER

;;;;; Callers of MAP-FILLER: (INTERPRETER-MAP-SLOT-AUX)

;------------------------------------------------------------------------------
(defun map-filler (filler value-form head root nested)
  "Maps the FS FILLER (not an atom or expresssion) with HEAD and ROOT.
If VALUE-FORM is given, the result is produced by evaluating
VALUE-FORM with the following dynamic bindings:
 - $ROOT      = root of FILLER
 - $CAT       = cat of FILLER
 - $VALUE     = result of calling SEMANTIC-MAP! on FILLER
 - $HEAD      = HEAD (head of containing FS)
 - $HEAD-ROOT = ROOT (root of containing FS)
 - $FS        = FILLER
 - $NESTED    = NESTED
Otherwise, the result is produced by a recursive call to SEMANTIC-MAP!."

  ;; Note: $nested is not currently used by any mapping rules.
  (if value-form
      (let (($root (second (assoc 'ROOT filler)))
	    ($cat (second (assoc 'CAT filler)))
	    ($value (semantic-map! filler))
	    ($head head)
	    ($head-root root)
	    ($fs filler)
	    ($nested nested))
	(declare (special $root $cat $value $head $head-root $fs $nested))
	(declare (ignore $root $cat $value $head $head-root $fs $nested))
	(eval value-form))
    (semantic-map! filler)))


;;; Map the filler for a slot in the f-structure that is filled with
;;; an f-structure.  This gets quite involved when a *MULTIPLE* or
;;; *OR* is involved.  See the INTEPRETER-MAP-SLOT-AUX function
;;; to see how they are handled--that's the function that does all
;;; of the real work.

;;;;; Callers of INTERPRETER-MAP-SLOT-AUX: (INTERPRETER-MAP-SLOT INTERPRETER-MAP-SLOT-AUX)

;------------------------------------------------------------------------------
(defun interpreter-map-slot-aux (sem-feature
				 value-form
				 head
				 root
				 value
				 &key
				 (nested nil))
  "Maps slot VALUE, returning result (always a list of IR slots, even if
only one).  The result depends on which of SEM-FEATURE and VALUE-FORM
are provided."
  (let ((mult-or-or (parser-or-or-mult-value-p value)))

    ;; 10-Apr-97 by EHN -- but not here!
 #|   (when (and;; e.g. Q-MODIFIER
	   (member sem-feature *DONT-REVERSE-MULTIPLE-SLOTS*)
	   (eq mult-or-or :MULTIPLE))
      (warn "** Hey this should be reversed!: ~s = ~s" sem-feature value))|#

    (if mult-or-or

	;; We have a *MULTIPLE* or *OR*, collect the fillers.
	(let ((semstruct
	       (mapcar
		#'(lambda (filler)
		    (interpreter-map-slot-aux sem-feature value-form
					      head root filler :nested t))
		(rest value))))
	  ;; Did we get any semantic structures?
	  (when semstruct

	    ;; Remove any empty semantic structures
	    (setq semstruct (delete-if #'null semstruct))

	    ;; If we have more than 1 semantic-structure, perform the
	    ;; appropriate action for :OR or :MULTIPLE fillers.
	    (when (> (length semstruct) 1)
	      (case mult-or-or
		(:OR     
		 (setq semstruct (uniquify-terms semstruct)))
		(:MULTIPLE
		 (unless (member sem-feature *dont-reverse-multiple-slots*)
		   (setq semstruct (nreverse semstruct))))))

	    (cond ((< (length semstruct) 2)
		   ;; We have a single semantic structure, use it.
		   (setq semstruct (first semstruct)))
		  (t
		   (unless (symbolp (first semstruct))
		     ;; Unless the semantic structure already has a head
		     ;; or :MULTIPLE or :OR, then add the appropriate
		     ;; :MULTIPLE or :OR
		     ;;
		     ;; NDB: How could a list of structures possibly have a head???
		     (setq semstruct (cons mult-or-or semstruct))
		     (unless (or nested
				 (member-if #'sem-head-p (rest semstruct)
					    :key #'car))
		       ;; Fix the structure if this is not a nested
		       ;; call (only fix at the top level) and if
		       ;; all of the elements of the :MULTIPLE or
		       ;; :OR do not have semantic heads.
		       ;;
		       ;; NDB: INVERT-MULT-OR-OR works on (malformed)
		       ;; expressions whose terms are slots, not full IRs.
		       ;;
		       ;; NDB: This test is sometimes satisfied -- the
		       ;; recursive call to INTERPRETER-MAP-SLOT-AUX (and thus
		       ;; to MAP-FILLER) sometimes returns an expression or a
		       ;; slot.
		       (setq semstruct
			     (invert-mult-or-or semstruct))
		       ;; Collapse the semantic structure if there is
		       ;; only one.
		       (when (< (length semstruct) 2)
			 (setq semstruct (first semstruct)))))))

	    ;; Return the appropriate value.
	    ;;
	    ;; NOTE: Like INTERPETER-MAP-FEATURE, this should return a
	    ;; list of lists since f-structure mapping is performed using
	    ;; MAPCAN.
	    (cond (nested
		   ;; Nested call: just return the structure, it will be
		   ;; fixed, if necessary, by higher level calls.
		   semstruct)

		  ((null sem-feature)
		   ;; The mapping rule inserts the appropriate semantic
		   ;; slot(s) (no semantic slot given in the rule), so
		   ;; simply return the appropriate strucutre (see note
		   ;; above).
		   (if (symbolp (first semstruct))
		       (list semstruct)
		     semstruct))
		  (t
		   ;; Add the semantic slot to the returned semantic
		   ;; structure and return the appropriate structure.
		   ;; (See note above)
		   (list (list sem-feature semstruct))))))

      ;; No *OR* or *MULTIPLE*, simply map the slot
      (if (symbolp (first value))
	  (interpreter-warn "Unexpected symbol in f-structure for MAP-SLOT: ~s"
			    value)
	(let ((semfillers (map-filler value value-form head root nested)))
	  (when semfillers
	    (cond (nested
		   ;; Nested call: don't make any more structure
		   semfillers)

		  ((null sem-feature)
		   ;; The mapping rule inserts the appropriate semantic
		   ;; slot(s) (no semantic slot given in the rule), so
		   ;; simply return the appropriate structure (see note
		   ;; above).
		   (if (symbolp (first semfillers))
		       ;; Either a frame or a single slot
		       (list semfillers)
		     semfillers))
		       
		  (t
		   ;; Add the semantic slot to the returned
		   ;; semantic structure and return the
		   ;; appropriate structure.  (See note above)
		   (list (list sem-feature semfillers))))))))))


;;;;; Callers of INTERPRETER-MAP-SLOT: (INTERPRETER-MAP-NON-SEMSLOT)

;------------------------------------------------------------------------------
(defun interpreter-map-slot (mapforms head root value)
  "Applies mapping rule MAPFORMS to slot VALUE from FS with HEAD and ROOT
by calling INTERPRETER-MAP-SLOT-AUX.  Returns appended results."
  (safe-mapcan-1 #'(lambda (mapform)
		     (interpreter-map-slot-aux (mapform-feature mapform)
					       (mapform-form mapform)
					       head
					       root
					       value))
		 mapforms))

;;; Originally: If there's more than one result in the list, transforms:
;;;   ((a 1) (b 2) ...) --> ((:MULTIPLE (a 1) (b 2) ...))
;;; Surely this isn't necessary, since SEMANTIC-MAP-FEATURES is expecting
;;; a list of slots.  So I got rid of it.  -- Thu Nov 17 1994 by ndb

;;;------------------------------------------------------------------;

;;; Function INTERPRETER-MAP-CONSUMES
;;;
;;; This is the function that handles MULTIPLE-FEATURE-AND-SLOT rules.
;;; It is called "CONSUMES" because it consumes many elements of the
;;; f-structure and has to return a modified f-structure with those
;;; elements removed.  When this function is called, the first feature
;;; or slot covered by the multiple-feature-and-slot rule has been
;;; encountered.  We loop through the remaining features and slots
;;; performing SEMANTIC-MAP! on the slot fillers and simply preserving
;;; the values for the features (as if default feature and slot
;;; mapping had been done for each feature and slot covered by the
;;; rule).  This feature-value, slot-filler mapping is preserved in a
;;; list that is passed to the multiple-feature-and-slot rule lisp
;;; code.
;;;
;;; NOTE: By "feature" I mean an f-structure element that is filled
;;; with a single value and by "slot" I mean an f-structure element
;;; that is filled with another f-structure.

;;;;; Callers of INTERPRETER-MAP-CONSUMES: (INTERPRETER-MAP-NON-SEMSLOT)

;------------------------------------------------------------------------------
(defun interpreter-map-consumes (forms
				 consumes
				 head
				 head-root
				 feature
				 value
				 fs)
  (let ((slot-sems (list (list feature
			       (if (feature-value-p value)
				   (change-value-or-or-mult value)
				 (semantic-map! value)))))
	(slot-values (list (list feature value)))
	(new-fs (copy-list fs))
	result)

    ;; Loop through the remaining features covered by the
    ;; multiple-feature-and-slot rule performing default semantic or
    ;; feature mapping as appropriate.  Collect the mapping results
    ;; into SLOT-SEMS and preserve the original values in SLOT-VALUES.
    (do* ((features-to-consume (remove feature consumes)
			       (cdr features-to-consume))
	  (consume-feat (car features-to-consume)
			(car features-to-consume))
	  (other-feat-value (find consume-feat new-fs :test #'eq :key #'car)
			    (find consume-feat new-fs :test #'eq :key #'car))
	  (other-value (second other-feat-value)
		       (second other-feat-value))
	  (other-sem))
	((null consume-feat))
      (when other-feat-value
	(setq new-fs (delete other-feat-value new-fs :test #'eq))
	(setq other-sem
	      (if (feature-value-p other-value)
		  (list consume-feat (change-value-or-or-mult other-value))
		(list consume-feat (semantic-map! other-value))))
	(push other-feat-value slot-values)
	(push other-sem slot-sems)))

    ;; Call the lisp forms for the multiple-feature-and-slot rule.
    (setq result (safe-mapcan-1 #'(lambda (form)
				    (let (($head head)
					  ($head-root head-root)
					  ($fs fs)
					  ($slot-values slot-values)
					  ($slot-sems slot-sems))
				      (declare (special $head $head-root $fs $slot-values $slot-sems))
				      (declare (ignore $head $head-root $fs $slot-values $slot-sems))
				      (eval form)))
				forms))

    ;; Return the appropriate semantic structure and the modified f-structure.
    (values result
	    new-fs)))

;;; Originally: If there's more than one result in the list, transforms:
;;;   ((a 1) (b 2) ...) --> ((:MULTIPLE (a 1) (b 2) ...))
;;; Surely this isn't necessary, since SEMANTIC-MAP-FEATURES is expecting
;;; a list of slots.  So I got rid of it.  -- Tue Jan  3 1995 by ndb


;;;------------------------------------------------------------------;

;;; Function INTERPRETER-MAP-RULE

;;;;; Callers of INTERPRETER-MAP-RULE: (MAP-PP-ROLE-COORDINATION
;;;;; SEMANTIC-MAP-FEATURES)

;------------------------------------------------------------------------------
(defun interpreter-map-rule (slot cat head root rest-fs)
  "Maps FS SLOT:
 - If SLOT feature is skippable, does nothing.
 - Otherwise gets the mapping rule.
    - If it's a multiple rule, calls INTERPRETER-MAP-CONSUMES.
    - If the element is a feature (has symbolic value), calls INTERPRETER-MAP-FEATURE.
    - Otherwise calls INTERPRETER-MAP-SLOT."
  (let ((result nil))
    (cond ((not (listp slot))
	   (interpreter-warn "F-structure slot not a list: ~s" slot))
	  ((gethash (first slot) *features-to-skip*))
	  (t
	   (let* ((feature (first slot))
		  (value (second slot))
		  (mapinfo (getmapinfo cat feature))
		  (mapinfo-forms (and mapinfo
				      (fs-rule-mapforms mapinfo)))
		  (mapinfo-consumes (and mapinfo
					 (fs-rule-consumes mapinfo))))
	     (cond (mapinfo-consumes
		    ;; Rule is a multiple-feature-and-slot rule.  Set the
		    ;; result and get the new f-structure.
		    (multiple-value-setq (result rest-fs)
		      (interpreter-map-consumes mapinfo-forms
						mapinfo-consumes
						head
						root
						feature
						value
						rest-fs)))
		   ;; The slot-filler is a value (not an f-structure filler)
		   ;; Call INTERPRETER-MAP-FEATURE.
		   ((feature-value-p value)
		    (if mapinfo
			(setq result (interpreter-map-feature mapinfo-forms value))
		      (when *warn-unmapped-features*
			    (interpreter-warn "Feature not mapped: ~s Head: ~s, ~s, ~s"
					feature head cat root))))

		   ;; The slot-filler is filled by an f-structure.  Call
		   ;; INTERPRETER-MAP-SLOT.
		   (t
		    (if mapinfo
			(setq result (new-interpreter-map-slot slot mapinfo-forms head root value))
		      (interpreter-warn "Slot not mapped: ~s Head: ~s"
					feature head)))))))
    (values result rest-fs)))

(defvar *syn-multiples-to-reorder* '(modifier))

(defun new-interpreter-map-slot (slot mapinfo-forms head root value)
  (when (and (member (first slot) *syn-multiples-to-reorder*)
	     (eq (first value) '*multiple*))
    (interpreter-warn "*SYN-MULTIPLES-TO-REORDER*: ~s = ~s" (first slot) value)
    (setf (rest (second slot)) (nreverse (rest (second slot)))))
  (interpreter-map-slot mapinfo-forms head root value))
	
  
;;;==================================================================;

;;; Semantic Head Mapping

;------------------------------------------------------------------------------
(defun eval-head-func (root cat fs orig-cat form)
  "Establishes dynamic bindings for ROOT, CAT, FS, ORIG-CAT with $
prefix and evaluates head rule FORM.  Returns the result of evaluation."
  (let (($root root)
	($cat cat)
	($fs fs)
	($orig-cat orig-cat))
    (declare (special $root $cat $fs $orig-cat))
    (declare (ignore $root $cat $fs $orig-cat))
    (eval form)))


;------------------------------------------------------------------------------
(defun get-head-from-fs-aux (root cat fs orig-cat)
  "Loops through all the head rules for the given CAT.  For each rule
whose syn-constraint (form or FS) is satisfied by FS, gets the head (by
evaluating the sem-head form or retrieving the sem-head symbol).
Returns the first successful head result."
  (let (sem-head)
    (dolist (rule (gethash cat *interpreter-head-rule-hash*) nil)

      ;; Either eval constraint form or unify constraint FS
      (when (if (head-rule-syn-constraint-is-func rule)
		(eval-head-func root cat fs orig-cat
				(head-rule-syn-constraint rule))
	      (tree-test (head-rule-syn-constraint rule) fs))

	;; Either eval head form or get head symbol
	(setf sem-head (if (head-rule-sem-head-is-func rule)
			   (eval-head-func root cat fs orig-cat
					   (head-rule-sem-head rule))
			 (head-rule-sem-head rule)))
	(when sem-head

	  (when *interpreter-trace*
	    (format t "~&;;; [Interpreter Head] ~S (orig ~S) ~S >> ~S~%"
		    cat orig-cat (head-rule-comment rule) sem-head))
	  
	  (return-from get-head-from-fs-aux sem-head))))))


;;;;; Callers of GET-HEAD-FROM-FS: (SEMANTIC-MAP-LIST)
  
;------------------------------------------------------------------------------
(defun get-head-from-fs (root cat fs)
  (if (and (listp root)
	   (eq '*NOT* (first root)))
      nil
    (or (some-imcat-inherit cat
			    #'(lambda (the-cat)
				(get-head-from-fs-aux root the-cat fs cat)))
	(interpreter-warn "Found no head for root: ~s (cat ~s)" root cat)
	;; We just return NIL here, blocking further mapping of this FS
	;; '*no-head
	)))


;;;------------------------------------------------------------------;

;;; SEMANTIC-MAP-FEATURES

;;;;; Callers of SEMANTIC-MAP-FEATURES: (SEMANTIC-MAP-AUX)

;------------------------------------------------------------------------------
(defun semantic-map-features (fs cat head root)
  "Builds headless IR frame.  For each slot in FS, applies mapping rule
by calling INTERPRETER-MAP-RULE.  This is supposed to return a list of
IR slots (role-filler lists).  For each IR slot in this list:
 - if a :MULTIPLE expression, adds terms to IR frame
 - if an :OR expression, adds to IR frame (SEMANTIC-MAP-AUX handles these)
 - if an :OR expression whose second term is a symbol, adds term list
   to IR frame (I guess the term list is itself a slot!)
 - otherwise adds IR slot to IR frame"

  (let* (role-filler
	 feat-or-slot
	 feat-or-slot-sem
	 feat-or-slot-sems
	 mult-or-or
	 has-ors
	 (sem nil))

    ;; Loop through the features in the f-structure.
    (loop

     ;; Exit condition: quit when we're out of f-structure
     (when (null fs)
       (return (values sem has-ors)))

     ;; Get an element of the f-structure
     (setq feat-or-slot (first fs)
	   fs (rest fs))

     ;; Map the f-structure element.
     (multiple-value-setq (feat-or-slot-sems fs)
       (interpreter-map-rule feat-or-slot cat head root fs))

     ;; The result is a list of IR slots.
     (when feat-or-slot-sems

       (if (not (listp feat-or-slot-sems))
	   (interpreter-warn "non-list semantics: ~s from f-structure ~s"
			     feat-or-slot-sems fs)

	 ;; Loop through each element of the returned semantic structure.
	 (loop

	  ;; Exit condition
	  (unless feat-or-slot-sems
	    (return))

	  ;; Look at each element.  Isolate an individual cons cell so
	  ;; that it can be spliced into the new IR being built.  (I
	  ;; know that most of the code does not do a very good job of
	  ;; conserving memory, but I wanted to preserve it where I
	  ;; could.
	  ;; 
	  ;; NDB: Too damn risky!  Might modify list structure in the
	  ;; mapping rules themselves.
	  (setq ;; feat-or-slot-sem feat-or-slot-sems
		feat-or-slot-sem (list (first feat-or-slot-sems))
		role-filler (first feat-or-slot-sem)
		feat-or-slot-sems (rest feat-or-slot-sems))
	  (rplacd feat-or-slot-sem nil)

	  (when *interpreter-trace*
	    (format t "~&;;; [Interpreter Slot] ~S ~S >> ~S~%" cat (first feat-or-slot) role-filler))
       
	  (if (not (listp role-filler))
	      (interpreter-warn "non-list semantics: ~s from f-structure ~s"
				role-filler fs)
	    (progn
	      (setq mult-or-or (interpreter-or-or-mult-p role-filler))
	      ;; We have a rule that returned multiple features.
	      ;; We should not have a :MULTIPLE or :OR at this
	      ;; level.
	      (if mult-or-or
		  (if (not (list-of-lists-p (rest role-filler)))
		      (interpreter-warn "Semantic \"mult-or-or\" ~
                                                  is not a list-of-lists: ~s"
					role-filler)
		    (case mult-or-or
		      (:MULTIPLE
		       ;; If we have a :MULTIPLE, we just want to
		       ;; splice in the elements of the :MULTIPLE, but
		       ;; check to see if any of them have a :OR (that
		       ;; is handled in the function that calls this
		       ;; one (SEMANTIC-MAP-AUX).
		       (setf has-ors (or has-ors
					 (find :OR (rest role-filler) :key #'first))
			     sem (nconc sem (rest role-filler))))
		      (:OR
		       ;; We have an :OR.
		       ;; First, make sure that there are no
		       ;; duplicated elements in the role-filler.
		       (rplacd role-filler (uniquify-terms (rest role-filler)))

		       ;; Splice the :OR into the IR being
		       ;; built.  
		       (cond ((symbolp (first (rest role-filler)))
			      (setf sem (nconc sem (list (rest role-filler)))))
			     (t
			      (setf sem (nconc sem feat-or-slot-sem)
				    has-ors t))))
		      (otherwise
		       (interpreter-warn "SEMANTIC-MAP-FEATURES - unexpected conj: ~
                                          ~s in ~s"
					 mult-or-or role-filler))))

		;; Just splice the feature into the IR being built.
		(setf sem (nconc sem feat-or-slot-sem)))))))))))

;;;------------------------------------------------------------------;

;;; Function EXPAND-ORS
;;;
;;; Given a semantic structure (a headless IR).  Look for each element
;;; with a :OR.  Create a cross-product of all of the :OR fillers
;;; and create multiple headless IRs with the each :OR element cross-product
;;; and the non-:OR elements of the original headless IR.
;;;
;;; A simplified example (think of x, y, a, b, c, i, j, k as proper IR
;;; elements):
;;;
;;; (headless IR)
;;;
;;; ((:OR x y)
;;;  a
;;;  b
;;;  c
;;;  (:OR (:MULTIPLE i j) k))
;;;
;;; Would be converted to 4 headless IRs
;;;
;;; (x  <== element of the first :OR
;;;  a
;;;  b
;;;  c
;;;  i  <=\
;;;  j) <== collapsed :MULTIPLE element of the second :OR
;;;
;;; (y  <== other element of the first :OR
;;;  a
;;;  b
;;;  c
;;;  i  <=\
;;;  j) <== collapsed :MULTIPLE element of the second :OR
;;;
;;; (x
;;;  a
;;;  b
;;;  c
;;;  k)  <== other element of the second :OR
;;;
;;; (y
;;;  a
;;;  b
;;;  c
;;;  k)
;;;
;;; Does this help?

;;;;; Callers of EXPAND-ORS: (EXPAND-ORS SEMANTIC-MAP-AUX)

;------------------------------------------------------------------------------
(defun expand-ors (sem)
  (let (new-sem ors)

    ;; Loop through the headless IR collecting all of the elements
    ;; that have an :OR (splice in elements with :MULTIPLE).
    (do* ((last-cons-in-new-sem nil)
	  (old-sem sem)
	  (sem-element (first old-sem)
		       (first old-sem))
	  (mult-or-or (interpreter-or-or-mult-p sem-element)
		      (interpreter-or-or-mult-p sem-element)))
	((null old-sem))
      (cond (mult-or-or
	     (case mult-or-or
	       (:MULTIPLE;; this could get in from a :MULTIPLE inside an :OR
		(setq old-sem (nconc (rest sem-element) (rest old-sem)))
		(when last-cons-in-new-sem
		  (rplacd last-cons-in-new-sem old-sem)))
	       (:OR
		;; try to collapse alike elements
		(setq ors
		      (if ors
			  (cons (rest sem-element) ors)
			(list (rest sem-element))))
		(when last-cons-in-new-sem
		  (rplacd last-cons-in-new-sem (rest old-sem)))
		(setq old-sem (rest old-sem)))))
	    (t
	     (unless new-sem
	       (setq new-sem old-sem))
	     (setq last-cons-in-new-sem old-sem)
	     (setq old-sem (rest old-sem)))))

    (if ors
	;; If we have some :ORs, then produce a cross product of the
	;; individual elements of each of the :ORs and concatenate
	;; each cross product onto the non-:OR elements of the
	;; headless IR.  Make a recursive call to EXPAND-ORs to get any
	;; sub-elements of the :ORs that may have :MULTIPLEs or :ORs.
	(let ((all-or-elements (list-cross-product ors))
	      or-sem
	      or-sems)
	  (dolist (or-element all-or-elements)
	    (setq or-sem (expand-ors
			  (nconc or-element
				 (copy-tree new-sem))))
	    (setq or-sems
		  (if (eq (first or-sem) :OR)
		      (if or-sems
			  (nconc or-sems (rest or-sem))
			(rest or-sems))
		    (if or-sems
			(nconc or-sems (list or-sem))
		      (list or-sem)))))
	  ;; Make sure we don't have any resulting duplicate headless IRs.
	  (uniquify-terms or-sems))
      new-sem)))
	   

;;;==================================================================;

;;; SEMANTIC-MAP-AUX

;;;;; Callers of SEMANTIC-MAP-AUX: (SEMANTIC-MAP-LIST)

;------------------------------------------------------------------------------
(defun semantic-map-aux (fs root cat head)
  "Helper function for SEMANTIC-MAP!.  Maps all the features in the FS by
calling SEMANTIC-MAP-FEATURES.  Expands all top-level :OR expressions
in the resulting headless IR frame by calling EXPAND-ORS.  Adds head
to all resulting headless IRs.  If there's more than one, forms an :OR
expression of them."

  (let (sem has-ors)
    (multiple-value-setq (sem has-ors)
      (semantic-map-features fs cat head root))
    (if has-ors
	(let ((or-sems (expand-ors sem)))
	  (cond ((symbolp (first or-sems))
		 (list head or-sems))
		((< (length or-sems) 2)
		 (cons head (first or-sems)))
		(t
		 (cons :OR (mapcar #'(lambda (or-sem)
				       (cons head or-sem))
				   or-sems)))))
	(cons head sem))))



;;;==================================================================;

;;; Pre-Mapping

;------------------------------------------------------------------------------
(defun eval-pre-map-func (root cat fs orig-cat form)
  "Establishes dynamic bindings for ROOT, CAT, FS, and ORIG-CAT with $
prefix and evaluates FORM.  Returns the result of evaluation."
  (let (($root root)
	($cat cat)
	($fs fs)
	($orig-cat orig-cat))
    (declare (special $root $cat $fs $orig-cat))
    (declare (ignore $root $cat $fs $orig-cat))
    (eval form)))

;------------------------------------------------------------------------------
(defun pre-map-fs-aux (root cat fs orig-cat)
  "Applies each pre-map rule for CAT (part of speech) to FS, if FS
satisfies the syntactic constraints of the rule.  Returns modified FS."
  (dolist (rule (gethash cat *pre-map-rule-hash*) fs)
    ;; Either eval constraint form or unify constraint FS
    (when (if (pre-map-rule-constraint-is-func rule)
	      (eval-pre-map-func root cat fs orig-cat
				 (pre-map-rule-constraint rule))
	    (tree-test (pre-map-rule-constraint rule) fs))

      ;; Apply the lisp forms to modify FS
      (setf fs (eval-pre-map-func root
				  cat
				  fs
				  orig-cat
				  (pre-map-rule-map-func rule)))
      (when *interpreter-trace*
	(format t "~&;;; [Interpreter Pre-Map] ~S (orig ~S) ~S >> ~S~%"
		cat orig-cat (pre-map-rule-comment rule) fs))

      )))

;;;;; Callers of PRE-MAP-FS: (SEMANTIC-MAP-LIST)

;------------------------------------------------------------------------------
(defun pre-map-fs (root cat fs)
  "Apply :PRE-MAP rules to an f-structure beginning with the CAT (part
of speech) actually in the f-structure, then inheriting from that CAT
through all of the other CATs in the *INTERPRETER-CAT-HIER*."
  (some-imcat-inherit cat
		      #'(lambda (the-cat)
			  (setf fs (pre-map-fs-aux root the-cat fs cat))
			  nil))
  fs)


;;;==================================================================;

;;; Post-Mapping

;------------------------------------------------------------------------------
(defun eval-post-map-func (root cat fs ir orig-cat form)
  "Establishes dynamic bindings for ROOT, CAT, FS, IR, and ORIG-CAT with $
prefix and evaluates FORM.  Returns the result of evaluation."
  (let (($root root)
	($cat cat)
	($fs fs)
	($ir ir)
	($orig-cat orig-cat))
    (declare (special $root $cat $fs $ir $orig-cat))
    (declare (ignore $root $cat $fs $ir $orig-cat))
    (eval form)))

;------------------------------------------------------------------------------
(defun post-map-ir-aux (root cat fs ir orig-cat)
  "Apply each rule (a lisp form) for a given CAT (part of speech) to an
IR if the IR fits the semantic constraints (a lisp form) of the rule."
  (dolist (rule (gethash cat *post-map-rule-hash*) ir)
    (when (eval-post-map-func root cat fs ir orig-cat
			      (post-map-rule-constraint rule))

      (setf ir (eval-post-map-func root cat fs ir orig-cat
				   (post-map-rule-map-func rule)))

      (when *interpreter-trace*
	(format t "~&;;; [Interpreter Post-Map] ~S (orig ~S) ~S >> ~S~%"
		cat orig-cat (post-map-rule-comment rule) ir))

      )))


;;;;; Callers of POST-MAP-IR: (POST-MAP-IR SEMANTIC-MAP-LIST)

;------------------------------------------------------------------------------
(defun post-map-ir (root cat ir fs)
  "Apply :POST-MAP rules to an IR beginning with the CAT (part of
speech) actually in the source f-structure, then inheriting from that
CAT through all of the other CATs in the *INTERPRETER-CAT-HIER*.
Handle :OR and :MULTIPLE as appropriate."
  (let ((mult-or-or (interpreter-or-or-mult-p ir)))
    (if mult-or-or
	(case mult-or-or
	  (:OR
	   (let ((terms
		  (uniquify-terms
		   (delete nil (mapcar #'(lambda (one-ir)
					   (post-map-ir root cat one-ir fs))
				       (rest ir))))))
	     (if (< (length terms) 2)
		 (first terms)
	       (cons mult-or-or terms))))
	  (:MULTIPLE
	   (interpreter-warn "POST-MAP-IR inappropriate :MULTIPLE in IR: ~s"
			     ir))
	  (t
	   (interpreter-warn "POST-MAP-IR unexpected \"mult-or-or\": ~s in IR ~
                              ~s"
			     mult-or-or
			     ir)))
      (progn
	(some-imcat-inherit cat
			    #'(lambda (the-cat)
				(setf ir (post-map-ir-aux root the-cat fs ir cat))
				nil))
	ir))))


;;;==================================================================;


;;; SEMANTIC-MAP-LIST

;;;;; Callers of SEMANTIC-MAP-LIST: (SEMANTIC-MAP!)

;------------------------------------------------------------------------------
(defun get-category (fs)
  (or (list-filler fs 'CAT)
      *interpreter-cat-default*
      *interpreter-cat-root*))

;------------------------------------------------------------------------------
(defun semantic-map-list (fs)
  "Maps a non-expression FS.  First, gets root and cat and applies pre-map rules
by calling PRE-MAP-FS.  If new FS is an expression, calls SEMANTIC-MAP! on it.
Otherwise, gets new root and cat, gets head by calling GET-HEAD-FROM-FS.  If
head is a list, that's the IR.  Otherwise, maps new FS to IR by calling
SEMANTIC-MAP-AUX.  Finally, applies post-map rules to IR by calling
POST-MAP-IR."
  (let* ((root (list-filler fs 'ROOT))
	 (cat (get-category fs))
	 ;; Apply pre-mapping rules.
	 (new-fs (pre-map-fs root cat fs))
	 class subclass)
    (multiple-value-setq (class subclass)
      (tree-classify new-fs))
    (case class
      ;; If the pre-mapped FS is now an expression, call SEMANTIC-MAP! again.
      (:EXPRESSION (semantic-map! new-fs))
      (:LIST

       ;; Otherwise, get the semantic head for the f-structure using
       ;; GET-HEAD-FROM-FS.  Sometimes, the head-rule will return a full IR, in
       ;; that case don't call SEMANTIC-MAP-AUX on the f-structure.  If the
       ;; head-rule just returns the semantic head symbol, the map the
       ;; f-structure to a full IR using SEMANTIC-MAP-AUX.  Finally, do any
       ;; post semantic-mapping cleanup of the IR using the :POST-MAP rules by
       ;; calling POST-MAP-IR.
       
       (if (eq subclass :KEY)
	   (interpreter-warn "SEMANTIC-MAP-LIST: keyed pre-mapped FS ~S" fs)
	 (let* ((new-root (list-filler new-fs 'ROOT))
		(new-cat (get-category new-fs))
		(head (get-head-from-fs new-root new-cat new-fs))
		(ir (if (listp head)
			head
		      (semantic-map-aux new-fs new-root new-cat head))))

	   (when *interpreter-trace*
	     (format t "~&;;; [Interpreter IR] ~S >> ~S~%" new-cat ir))

	   (post-map-ir new-root new-cat ir new-fs))))
      (otherwise
       fs))))


;;;==================================================================;

;;; Function SEMANTIC-MAP! SAFE-SEMANTIC-MAP!
;;;
;;; Thu Jan 5 1995 by ndb -- Now not only removes duplicate terms from ORs,
;;; also reduces any resulting singleton ORs.

;;;;; Callers of SEMANTIC-MAP!: (INTERPRETER-MAP-CONSUMES MAP-FILLER
;;;;; MAP-NOMINALIZED-FORM MAP-QUALIFIER MAP-SEMANTIC-MAP
;;;;; MAP-TOP-LEVEL-PARTNAME SAFE-SEMANTIC-MAP!
;;;;; SEMANTIC-MAP-LIST)

;------------------------------------------------------------------------------
(defun semantic-map! (fs)
  "Recursively maps FS.  If FS is an expression, then calls
SEMANTIC-MAP-LIST on each term and forms an IR expression out
of the results.  If FS is not an expression, then calls
SEMANTIC-MAP-LIST on FS."
  (multiple-value-bind (class subclass)
      (tree-classify fs)
    (case class
      (:EXPRESSION
       (let ((terms (mapcar #'semantic-map-list (rest fs))))
	 (case subclass
	   (:MULTIPLE (make-exp+ :MULTIPLE (nreverse terms)))
	   (:OR (make-exp+ :OR (nreverse terms)))
	   (otherwise nil))))
      (:LIST (if (eq subclass :KEY)
		 (interpreter-warn "SEMANTIC-MAP!: keyed FS ~S" fs)
	       (semantic-map-list fs)))
      (otherwise fs))))


;;;;; Callers of SAFE-SEMANTIC-MAP!: (MAP-SEMANTIC-MAP)

;------------------------------------------------------------------------------
(defun safe-semantic-map! (fs)
  "This is an error trapping call to SEMANTIC-MAP!.  It will
a warning if an error is encountered.  (Doesn't trap
segmentation violations or bus errors.)"
  (let ((sem ;; (handler-case (semantic-map! fs)(error () :error))
	     (semantic-map! fs)))
    (if (eq sem :error)
	(interpreter-warn "HARD ERROR!!!")
      sem)))


;;;------------------------------------------------------------------;
;;; Top-Level SEMANTIC-MAP function
;;;
;;; OBSOLETE!!! Call MAP-SEMANTIC-MAP instead.  Ok for testing and when
;;; there is guaranteed to always be a single f-structure.  But, if you
;;; have a list of f-structures, call MAP-SEMANTIC-MAP over the list of
;;; f-structures instead so that identical IRs can be removed.

;------------------------------------------------------------------------------
(defun semantic-map-trace (fs)
  (semantic-map fs))

;------------------------------------------------------------------------------
(defun semantic-map (fs)
  (let ((results (map-semantic-map (list fs)
				   :log-errors nil
				   :return-partial-irs t
				   :trap-errors nil)))
    (if (= (length results) 1)
	(first results)
	results)))

;;;==================================================================;


;;; Function BUILD-SEMANTIC-MAP-RESULTS

;;;;; Callers of BUILD-SEMANTIC-MAP-RESULTS: (HANDLE-TOP-LEVEL-MULT-OR-OR MAP-SEMANTIC-MAP)

;------------------------------------------------------------------------------
(defun build-semantic-map-results (new-sems
				   fs
				   sems
				   sems-with-errors
				   log-errors
				   return-partial-irs
				   sentence
				   &key
				   (full-sem new-sems))
  "Formats the results to be returned from MAP-SEMANTIC-MAP.
MAP-SEMANTIC-MAP returns multiple values, the first value is the list
of IRs produced and the second is the f-structures, IRs, and error
messages produced for IRs that had errors during mapping.

If error:
 - Adds error log info to error messages list if LOG-ERRORS
 - Adds partial IR to IR list if RETURN-PARTIAL-IRS
Otherwise:
 - Adds IR to IR list"

  (cond (*interpreter-errors*
	 (if log-errors
	     (push `((:sentence ,sentence)
		     (:syntax ,fs)
		     (:semantics ,(copy-tree full-sem))
		     (:errors ,*interpreter-errors*))
		   sems-with-errors))
	 (if (and return-partial-irs new-sems)
	     (setf sems (nconc new-sems sems))))
	(t
	 (setf sems (nconc new-sems sems))))
  (values sems sems-with-errors))


;;;------------------------------------------------------------------;

;;; Function HANDLE-TOP-LEVEL-MULT-OR-OR

;;;;; Callers of HANDLE-TOP-LEVEL-MULT-OR-OR:
;;;;; (HANDLE-TOP-LEVEL-MULT-OR-OR MAP-SEMANTIC-MAP)

;------------------------------------------------------------------------------
(defun handle-top-level-mult-or-or (sem
				    fs
				    sems
				    sems-with-errors
				    log-errors
				    return-partial-irs
				    sentence
				    &key
				    (top-level t)) ; to handle recursive calls
  "Processes SEM, recursively splitting up (possibly nested) top-level
:OR and :MULTIPLE expressions into a flat list of IR frames.  NCONC's
this list to existing SEMS list.  At TOP-LEVEL, builds result
structures; at lower levels, just accumulates IR's.

For an :OR expression, processes all terms (after collapsing equal terms).
For a :MULTIPLE expression, processes just first term."
  (let ((full-sem sem)
	(mult-sems nil)
	(mult-or-or (interpreter-or-or-mult-p sem)))
    (cond (mult-or-or
	   (setq sem
		 (case mult-or-or
		   (:MULTIPLE
		    (interpreter-warn "Top-level :MULTIPLE ~S" sem)
		    (list (second sem)))
		   (:OR
		    ;; This shouldn't be necessary
		    (uniquify-terms (rest sem)))
		   (t
		    (interpreter-warn "Unknown top-level \"mult-or-or\" ~s: ~s"
				      mult-or-or sem)
		    (rest sem))))
	   (dolist (one-sem sem)
	     (setq mult-sems (handle-top-level-mult-or-or one-sem
							  nil
							  mult-sems
							  nil
							  nil
							  return-partial-irs
							  sentence
							  :top-level nil)))
	   (if top-level
	       (build-semantic-map-results mult-sems
					   fs
					   sems
					   sems-with-errors
					   log-errors
					   return-partial-irs
					   sentence
					   :full-sem full-sem)
	     (nconc mult-sems sems)))
	  (t
	   (if top-level
	       (build-semantic-map-results (list sem)
					   fs
					   sems
					   sems-with-errors
					   log-errors
					   return-partial-irs
					   sentence
					   :full-sem full-sem)
	     (cons sem sems))))))


;;;------------------------------------------------------------------;

;;; Function MAP-SEMANTIC-MAP
;;;
;;; This is the function that should be called from external programs.
;;; This function takes a list of f-structures (as is stored into
;;; *PARSE-VALUES* by the Generalized LR Parser.

;;;;; Callers of MAP-SEMANTIC-MAP: (SEMANTIC-MAP)

(defvar *interpreter-return-partial-irs* nil)
(defvar *interpreter-values* '())
(defvar *interpreter-error-values* '())
(defvar *interpreter-trap-errors* t)
(defvar *interpreter-msm-print-f-structures* nil)
(defvar *interpreter-msm-print-irs* nil)
(defvar *interpreter-msm-show-stats* nil)

;------------------------------------------------------------------------------
(defun map-semantic-map (fs-list
			 &key
			 (report-errors *interpreter-warn*)
			 (log-errors *interpreter-log-errors*)
			 (return-partial-irs *interpreter-return-partial-irs*)
			 (trap-errors *interpreter-trap-errors*)
			 (print-f-structures
			  *interpreter-msm-print-f-structures*)
			 (print-irs *interpreter-msm-print-irs*)
			 (show-stats *interpreter-msm-show-stats*)
			 (stream *standard-output*)
			 (sentence ""))
  
  "This function takes a list of f-structures (as is stored into
*PARSE-VALUES* by the Generalized LR Parser and returns two values.
The first value is the list of IRs produced from the input list of
f-structures and the second value is the list of error reporting
structures for errors encountered during the mapping of each
f-structure (if the keyword argument :LOG-ERRORS is non-NIL).  This
function allows a number of keyword arguments: :REPORT-ERRORS - if
non-NIL, tells if the interpreter should print errors encountered;
:LOG-ERRORS - if non-NIL, tells whether to collect error information;
:RETURN-PARTIAL-IRS - if non-NIL, tells the interpreter to return an
IR even if an error was encountered during mapping of that IR;
:TRAP-ERRORS - if non-NIL, then lisp ERRORS will be trapped (the user
will not be dropped into the lisp debugger; :print-f-structures - if
non-NIL, tells that each f-structure should be printed as it is
mapped; :PRINT-IRS - if non-NIL, tells that each IR should be printed
as it is produced; :SHOW-STATS - if non-NIL, tell that a simple count
of the number of input f-structures and output IRs should be printed
after all the input f-structures are mapped; :STREAM - (default
*STANDARD-OUTPUT*) the stream to which all messages, structures,
etc. are printed.; :SENTENCE - the input sentence (if available)

Maps all the given FS's, calling SAFE-SEMANTIC-MAP! if trapping errors
and SEMANTIC-MAP! otherwise.

For each result, calls HANDLE-TOP-LEVEL-MULT-OR-OR if successful and
BUILD-SEMANTIC-MAP-RESULTS if result NIL.

Calls UNIQUIFY-TERMS on result list."
  
  (let ((*interpreter-warn* report-errors)
	(*interpreter-log-errors* log-errors)
	sem
	(sems nil)
	(sems-with-errors nil))
    

    ;; Copy it once, here, and we can be destructive the rest of the way.
    (setf fs-list (copy-tree fs-list))

    (dolist (fs fs-list)
      (when (and print-f-structures stream)
	(format stream "~&~%;;; F-STRUCTURE --------------------------------------------------------------~%")
	(pns fs :stream stream))
      (setq *interpreter-errors* nil)
      (setq sem (if nil ;; trap-errors
		    (safe-semantic-map! fs)
		  (semantic-map! fs)))
      ;; SEM is one item: an IR frame or expression
      (when (and stream print-irs)
	(setq sem (first (interpreter-coda (list sem))))
	(format stream "~&~%;;; INTERLINGUA FRAME(S) -----------------------------------------------------~%")
	(if sem
	    (pns sem :stream stream)
	  (format stream "~%No Interlingua Generated.")))

      (multiple-value-setq (sems sems-with-errors)
	(cond (sem
	       (handle-top-level-mult-or-or sem
					    fs
					    sems
					    sems-with-errors
					    log-errors
					    return-partial-irs
					    sentence))
	      (log-errors
	       (build-semantic-map-results nil
					   fs
					   sems
					   sems-with-errors
					   log-errors
					   return-partial-irs
					   sentence)))))
    
    ;; reverse the result lists since they're built in reverse order
    (setq sems (uniquify-terms (nreverse (if (and stream print-irs)
						 sems
					       (interpreter-coda sems)))))
    (setq sems-with-errors (nreverse sems-with-errors))
    
    ;; set some globals so we can look at results, if necessary
    (setq *interpreter-values* sems)
    (setq *interpreter-error-values* sems-with-errors)

    (when (and show-stats stream)
      (format stream "~&~%;;; STATS: ~2d f-structure(s); ~2d interlingua frame(s) ------------------------~%~%" 
	      (length fs-list) (length sems)))

    (values sems sems-with-errors)))


;;;==================================================================;

;;; Testing functions

;------------------------------------------------------------------------------
(defun parse-and-map (sentence &key (verbose t))
  "Parses and maps SENTENCE.  If VERBOSE is non-nil, prints the sentence and
each f-structure and interlingua produced.  The list of interlinguas is left
in *INTERPRETER-VALUES*."
  (when verbose
    (format t "~&;;; SENTENCE: ~S~%" sentence))
  (if (quiet-parse sentence)
      (map-semantic-map *parse-value*
			:trap-errors nil
			:print-f-structures verbose
			:print-irs verbose
			:show-stats t)
    (when verbose
      (format t "~&;;; *** Does Not Parse ***~%")))
  t)

(setf (symbol-function 'pam) #'parse-and-map)

;------------------------------------------------------------------------------
(defun re-map (&key (verbose t))
  "Re-interprets the f-structures in *PARSE-VALUE*.  *PARSE-VALUE*
contains the list of f-structures from the last parsed sentence."
  (map-semantic-map *parse-value*
		    :trap-errors nil
		    :print-f-structures verbose
		    :print-irs verbose
		    :show-stats t)
  t)



;---eof interpreter.lisp---
