;      -*- Syntax: Common-Lisp; Package: USER; Base: 10; Mode: LISP -*-
;==============================================================================
;     File:  interpreter-tools.lisp
;	By:  Willy Walker	<wkw@pitt.edu>
;     Path:  ../cmt/catalyst/amt-ana/code/interpreter/<release-vs>
;  Started:  31 January 1994
; Modified:  03 December 1996	by <garof>
;
; Comments:  Contains general-use utility functions needed for the interpreter.
;
; Modified:  11 May 1994
;	By:  Willy Walker	<wkw@pitt.edu>
;  Reasons:  Moved many generic functions from other files into this one.
;
; Modified:  01 August 1995
;	By:  Nicholas Brownlow	<ndb@clarit.com>
;  Reasons:  Nicholas was the official "Interpreter" maintainer until then.
;
; Modified:  02 December 1996
;	By:  Joseph Giampapa	<garof@cs.cmu.edu>
;  Reasons:  To garof-ify this file, and to add version control markers to it.
;
; Modified:  <date>
;	By:  <name>		<full e-mail>
;  Reasons:  
;
;==============================================================================


;..............................................................................
;			Center for Machine Translation
;			  Carnegie Mellon University
;
;			Copyright (c) 1994, 1995, 1996
;	       Carnegie Mellon University.  All Rights Reserved.
;..............................................................................


;..............................................................................
;			      Package Statements
;..............................................................................
(in-package :user)


;..............................................................................
; 14-Nov-96-garof:  For version control.
; 02-Dec-96-garof:  The directory should be correct if you use the defsystem.
;..............................................................................
;(load (compile-file "/afs/cs.cmu.edu/project/cmt/catalyst/amt-ana/code/interpreter/5.0Analyzer/ic_vers_ctrl.lisp"))


;------------------------------------------------------------------------------
; For use by the maintainer's personal program.
;------------------------------------------------------------------------------
(defconstant *INTERPRETER-TOOLS-VS* '5.0Analyzer)


;------------------------------------------------------------------------------
; Macro Interpreter-Warn
;
; This macro provides a mechanism for reporting and recording errors in the
; interpreter.
;
; Reporting is controlled by *INTERPRETER-WARN* (if non-NIL, then an error
; message is printed to standard output), *INTERPRETER-LOG-ERRORS* (if non-NIL,
; then the error messages are saved in *INTERPRETER-ERRORS*; if NIL, then
; *INTERPRETER-ERRORS* is set to T when any error occurs)
;
; *INTERPRETER-ERRORS* should be reset after each input f-structure is mapped.
;
; Always returns NIL.
;..............................................................................
(defvar *INTERPRETER-WARN* nil)
(defvar *INTERPRETER-ERRORS* nil)
(defvar *INTERPRETER-LOG-ERRORS* nil)


;------------------------------------------------------------------------------
(defmacro Interpreter-Warn (string &rest args)
  (let ((error-string (gensym)))
    `(progn
       (if (or *INTERPRETER-WARN* *INTERPRETER-LOG-ERRORS*)
	   (let ((,error-string (format nil ,string ,@args)))
	     (when *INTERPRETER-WARN*
		   (format t "~&;;; [Interpreter] ~a~%" ,error-string))
	     (if *INTERPRETER-LOG-ERRORS*
		 (setf *INTERPRETER-ERRORS*
		       (nconc *INTERPRETER-ERRORS* (list ,error-string)))
	       (setf *INTERPRETER-ERRORS* t)))

	 (setf *INTERPRETER-ERRORS* t))
       nil)))


;------------------------------------------------------------------------------
; Macro List-Of-Lists-P
;
; Callers of List-Of-Lists-P: (Collapse-Disjuncts Define-Feature-Slot-Rule
;	Define-Multiple-Feature-Slot-Rule Define-Multiple-Feature-Slot-Rules
;	Semantic-Map-Features)
;------------------------------------------------------------------------------
(defmacro List-Of-Lists-P (list)
  "Returns non-NIL if a list constains only CONSes."
  `(and (listp ,list) (every #'consp ,list)))


;------------------------------------------------------------------------------
; Macro Safe-Mapcan-1
;
; Performs the same action as MAPCAN (when applied to a single list--thus the
; "1" in the name), but avoids the segmentation fault experienced when binary
; files are incorporated into an image.
;
; Callers of Safe-Mapcan-1: (Cross-Product-Aux Find-Subj-Slots
;	Get-Comp-Subj-Role-Info-List Interpreter-Map-Consumes
;	Interpreter-Map-Feature Interpreter-Map-Slot Map-PP-Without-Sem-Info
;	Map-Set-PP-Members Verb-Slot-Map-And-Gap) 
;------------------------------------------------------------------------------
(defmacro Safe-Mapcan-1 (func list)
  "MAPCAN a function over a single list.  Does more type checking than
traditional MAPCAN."
  
  (let ((result		(gensym))
	(accum-list	(gensym))
	(elem		(gensym)))
    `(let (,accum-list
	   ,result)
       (dolist (,elem ,list ,accum-list)
	       (setq ,result (funcall ,func ,elem))
	       (unless (listp ,result)
		       (Interpreter-Warn "SAFE-MAPCAN not-list result: ~s"
					 ,result)
		       (return ,accum-list))
	       (cond (,accum-list
		      (rplacd (last ,accum-list) ,result))
		     (t
		      (setq ,accum-list ,result)))))))



;..............................................................................
; Dealing with *MULTIPLE*, :MULTIPLE, *OR*, :OR, *NOT*, and :NOT
;
; Macros for finding and converting IR and f-structure conjunctions and
; disjunctions.
;..............................................................................
(defconstant *PARSER-MULT-MAPPING*
  '((*OR* . :OR) (*MULTIPLE* . :MULTIPLE) (*NOT* . :NOT)))


;------------------------------------------------------------------------------
(defmacro Parser-Or-Or-Mult-P (mult-or-or)
  `(cdr (assoc ,mult-or-or *PARSER-MULT-MAPPING*)))


;------------------------------------------------------------------------------
(defmacro Parser-Or-Or-Mult-Value-P (value)
  `(Parser-Or-Or-Mult-P (first ,value)))


;------------------------------------------------------------------------------
(defmacro Interpreter-Or-Or-Mult-Symbol-P (symbol)
  `(cdr (rassoc ,symbol *PARSER-MULT-MAPPING*)))


;------------------------------------------------------------------------------
(defmacro Interpreter-Or-Or-Mult-P (ir)
  `(Interpreter-Or-Or-Mult-Symbol-P (first ,ir)))



;..............................................................................
; Function Delete-Features
;
; Callers of Delete-Features: (Remove-Features)
;..............................................................................
(defun Delete-Features-Aux (feature fs) (delete feature fs :key #'car))


;------------------------------------------------------------------------------
; Removes the features given in FEATURES-TO-DELETE (a list of symbols or just
; a symbol) from the given f-structure (FS).
;------------------------------------------------------------------------------
(defun Delete-Features (features-to-delete fs)
  "Removes the features given in FEATURES-TO-DELETE (a list of symbols or just
a symbol) from the given f-structure (FS)."
  (typecase
   features-to-delete
   (symbol
    (Delete-Features-Aux features-to-delete fs))
   (list
    (dolist (feature features-to-delete fs)
	    (setq fs (Delete-Features-Aux feature fs))))
   (t
    (error "Invalid type for Features-To-Delete: ~s" features-to-delete))))


;------------------------------------------------------------------------------
; Function Remove-Features
;
; Non-destructive Delete-Features (copies the list first)
;
; Callers of Remove-Features: (Fix-As-Quant-As Insert-Feature-Aux
;	Insert-PP-Marker Map-Nominalized-Form Map-Set-PP-Members-Aux
;	Pre-Map-As-Quant-As-NP Pre-Map-Be-Verb Pre-Map-Complementizer
;	Pre-Map-Conditional Pre-Map-Date Pre-Map-Nontrans Pre-Map-Number-Range
;	Pre-Map-Partitive Pre-Map-Partname-NP Pre-Map-Rel-Cl Pre-Map-SMCScode
;	Pre-Map-SNRange Pre-Map-Such Pre-Map-TitleRef-Aux Re-Structure-Adj-Mod)
;------------------------------------------------------------------------------
(defun Remove-Features (features-to-delete fs)
  "Non-destructive version of DELETE-FEATURES."
  (Delete-Features features-to-delete (copy-tree fs)))


;------------------------------------------------------------------------------
; Macro Push-End
;
; Implements a PUSH-like macro that puts new elements onto the end of a list.
; Will not properly set a NIL "place" so either ensure the "place" is not NIL
; or do a (setf <place> (push-end <item> <place>))
;
; jrrl (John Leavitt) has an END-PUSH macro that performs the same function but
; it does not have the NIL restrictions of this macro.
;
; Callers of Push-End: (Extract-Event-And-Extent Extract-Qualifying-Events)
;------------------------------------------------------------------------------
(defmacro Push-End (item place)
  `(cond (,place (setf (cdr (last ,place)) (cons ,item nil)) ,place)
	 (t	 (cons ,item nil))))


;..............................................................................
; Function (N)Member-Extract
;
; Extracts members from a list that match the given criteria.  The functions
; that have an "N" prepended are destructive.  The keyword arguments work like
; keyword arguments to other list (and sequence) searching and extraction
; functions.
;
; Returns two values: the list of members that match the appropriate test and
; the list that remains after the members were removed.
;------------------------------------------------------------------------------
(defun %Member-Extract (list &key (item nil item-provided) test test-not key)
  (let (list-key
	member-list
	result-list
	rest-list)
    (loop		;; Loop though elements of the input list.
     (unless list (return (values member-list result-list)))
     
     ; get one cons cell at a time, to be spliced into appropriate list.
     (setq rest-list (rest list))
     (rplacd list nil)

     ; get the item for testing (if :KEY provided, use it as the extraction
     ; function
     (if key (setq list-key (funcall key (first list)))
       (setq list-key (first list)))

     ; Perform the test or test-not on the list.  If only a matching :ITEM is
     ; provided, use EQL to test the LIST-KEY against the :ITEM
     (if (cond (test (if item-provided (funcall test item list-key)
		       (funcall test list-key)))
	       (test-not (not (if item-provided
				  (funcall test-not item list-key)
				(funcall test-not list-key))))
	       (item-provided (eql item list-key)))
	 
	 ; The LIST-KEY passed the test, splice the cons cell into the member
	 ; list.
	 (setq member-list (if member-list (nconc member-list list) list))

       ; The LIST-KEY did not pass the test, splice the cons cell into the list
       ; that remains after member extraction
       (setq result-list (if result-list (nconc result-list list) list)))
     (setq list rest-list))))


;------------------------------------------------------------------------------
; Callers of NMember-Extract-If-Not: (Member-Extract-If-Not)
;------------------------------------------------------------------------------
(defun NMember-Extract-If-Not (test list &rest keyword-args)
  (apply #'%member-extract list :test-not test keyword-args))


;------------------------------------------------------------------------------
; Callers of Member-Extract-If-Not: NIL
;------------------------------------------------------------------------------
(defun Member-Extract-If-Not (test list &rest keyword-args)
  (apply #'nmember-extract-if-not test (copy-list list) keyword-args))
  

;------------------------------------------------------------------------------
; Callers of NMember-Extract-If: (Member-Extract-If)
;------------------------------------------------------------------------------
(defun NMember-Extract-If (test list &rest keyword-args)
  (apply #'%member-extract list :test test keyword-args))


;------------------------------------------------------------------------------
; Callers of Member-Extract-If: (Pre-Map-Partitive)
;------------------------------------------------------------------------------
(defun Member-Extract-If (test list &rest keyword-args)
  (apply #'nmember-extract-if test (copy-list list) keyword-args))


;------------------------------------------------------------------------------
; Callers of NMember-Extract: (Member-Extract)
;------------------------------------------------------------------------------
(defun NMember-Extract (item list &rest keyword-args)
  (apply #'%member-extract list :item item keyword-args))


;------------------------------------------------------------------------------
; Callers of Member-Extract: (Combine-Common-Slot-Fillers)
;------------------------------------------------------------------------------
(defun Member-Extract (item list &rest keyword-args)
  (apply #'nmember-extract item (copy-list list) keyword-args))



;..............................................................................
; Function Insert-Feature
;
; Inserts the &REST FEATURES into the given f-structure (FS).  Properly handles
; *OR*s and *MULTIPLE*s.
; 
; Example:
; 
; (Insert-Feature '((ROOT "truck") (CAT N))
;                   (NUMBER SG)
;                   (DET "the"))
;
; would return
;
; ((ROOT "truck") (CAT N) (NUMBER SG) (DET "the"))
;
; Callers: (Pre-Map-Complementizer Pre-Map-Such Insert-Feature)
;..............................................................................
(defun Insert-Feature-Aux (fs feature)
  (let* ((ins-feature (first feature))
	 (dup-feature (assoc ins-feature fs)))
    (if dup-feature
	(let ((dup-feature-filler (second dup-feature))) 
	  (cond ((and (listp dup-feature-filler)
		      (eq (first dup-feature-filler) '*MULTIPLE*))
		 (nconc (remove-features ins-feature fs)
			(list
			 (list ins-feature
			       (cons '*MULTIPLE*
				     (append
				      (rest dup-feature-filler)
				      (list (second feature))))))))
		(t
		 (nconc (remove-features ins-feature fs)
			(list
			 (list ins-feature
			       (cons '*MULTIPLE*
				     (list dup-feature-filler
					   (second feature)))))))))
      (cons feature fs))))


;------------------------------------------------------------------------------
; Inserts the &REST FEATURES into the given f-structure (FS).
;------------------------------------------------------------------------------
(defun Insert-Feature (fs &rest features)
  "Inserts the &REST FEATURES into the given f-structure (FS)."
  
  (if (Parser-Or-Or-Mult-P (first fs))
      (cons (first fs)
	    (mapcar #'(lambda (one-fs)
			(insert-feature one-fs features))
		    (rest fs)))
    (if (not (intersection (mapcar #'first fs)
			   (mapcar #'first features)))
	(append fs features)
      (dolist (feature features fs)
	      (setq fs (Insert-Feature-Aux fs feature))))))


;------------------------------------------------------------------------------
; Function Get-Feature
;
; From an f-structure, this function searches through *OR*s and *MULTIPLE*s for
; the first f-structure with the given feature.
;
; Callers of Get-Feature: (Gap-Events Get-Feature Is-Set-AdjP-FS-Without-Conj-P
; Pre-Map-Conditional-Aux)
;------------------------------------------------------------------------------
(defun Get-Feature (feature fs)

  "From an f-structure, this function searches through *OR*s and *MULTIPLE*s
for the first f-structure with the given feature."

  (if (Parser-Or-Or-Mult-P (first fs))
      (do* ((rest-fs (rest fs) (rest rest-fs))
	    (one-fs (first rest-fs) (first rest-fs))
	    (sought (get-feature feature one-fs) (get-feature feature one-fs)))
	   ((or sought (null (rest rest-fs))) sought))
    (second (assoc feature fs))))


;..............................................................................
; Function Search-Path-In-FS
;
; Given an f-structure (FS) and a feature list (FEATURE-LIST), this function
; searches through each feature in the feature list recursively in the given
; f-structure.
;
; The optional FIND-PREDICATE can be provided to extract the appropriate
; information at the end of the search.
;
; Also, the feature list may contain pairs in the form:
;
; (<feature> [:ALL | :LAST | :FIRST])
;
; This special feature format tells how to extract information when an *OR* or
; *MULTIPLE* is encountered at that element of the feature list. 
; :ALL	tells Search-Path-In-FS to search in all elements of *OR*s or
;	*MULTIPLE*s.
; :LAST	tells SEARCH-PATH-IN-FS to search the last element of *OR*s or
;	*MULTIPLE*s; and
; :FIRST tells Search-Path-In-FS to search in the first element of *OR*s or
;	*MULTIPLE*s.
;
; Callers of Search-Path-In-FS: (Interpreter-Map-Semslot-Aux)
;..............................................................................
(defun Search-Path-In-FS-Aux (fs feature rest-features find-predicate)
  (if (Parser-Or-Or-Mult-Value-P fs)
      (let ((key (or (and (symbolp feature) :ALL)
		     (second feature))))
	(some
	 #'(lambda (x)
	     (Search-Path-In-FS-Aux x feature rest-features find-predicate))
	 (case key
	       (:ALL	(rest fs))
	       (:LAST	(last fs))
	       (:FIRST	(list (first (rest fs))))
	       (t
		(Interpreter-Warn "Improper SEARCH-PATH key: ~s" key)))))
    (let ((sub-fs (second (assoc (or (and (symbolp feature) feature)
				     (first feature))
				 fs))))
      (when sub-fs
	    (if rest-features
		(Search-Path-In-FS-Aux sub-fs
				       (first rest-features)
				       (rest rest-features)
				       find-predicate)
	      (if find-predicate
		  (funcall find-predicate sub-fs)
		sub-fs))))))


;------------------------------------------------------------------------------
; Given an f-structure (FS) and a feature list (FEATURE-LIST), this function
; searches through each feature in the feature list recursively in the given
; f-structure.  The optional FIND-PREDICATE can be provided to extract the
; appropriate information at the end of the search.
;------------------------------------------------------------------------------
(defun Search-Path-In-FS (fs feature-list &optional find-predicate)

  "Given an f-structure (FS) and a feature list (FEATURE-LIST), this
function searches through each feature in the feature list recursively
in the given f-structure.  The optional FIND-PREDICATE can be provided
to extract the appropriate information at the end of the search."
  
  (cond ((listp feature-list)
	 (Search-Path-In-FS-Aux fs
				(first feature-list)
				(rest feature-list)
				find-predicate))
	(t
	 (Interpreter-Warn "Improper call to SEARCH-PATH-IN-FS! Argument: ~s"
			   feature-list)
	 nil)))


;------------------------------------------------------------------------------
; Function Delete-Nth
;
; Callers of Delete-Nth: (Remove-Nth)
;------------------------------------------------------------------------------
(defun Delete-Nth (n list)
  "Destructively removes the Nth member of LIST."
  (cond ((not (numberp n)) list)		 ;; error
	((= n 0)	   (cdr list))
	((< n 0)	   list)
	(t
	 (let ((l (- (length list) 1)))
	   (cond ((> n l)	list)
		 ((= n l)	(nbutlast list))
		 (t (rplacd	(nthcdr (- n 1) list)
				(nthcdr (+ n 1) list))
		    list))))))


;------------------------------------------------------------------------------
; Function Remove-Nth
;
; Callers of Remove-Nth: NIL
;------------------------------------------------------------------------------
(defun Remove-Nth (n list)
  "Non-destructively removes the Nth member of LIST."
  (delete-nth n (copy-list list)))


;..............................................................................
; Function Cross-Product
;
; Produces the cross product of the elements of two lists.
;
; Callers of CROSS-PRODUCT: NIL
;..............................................................................
(defun Cross-Product-Aux (list-of-lists list)
  (safe-mapcan-1 #'(lambda (x)
		     (mapcar #'(lambda (y)
				 (nconc (copy-list x) (list y)))
			     list))
		 list-of-lists))


;------------------------------------------------------------------------------
(defun Cross-Product (list1 list2)
  "Produces a list of every possible pairing of elements from LIST1 and LIST2"
  (Cross-Product-Aux (mapcar #'list list1) list2))


;------------------------------------------------------------------------------
; Function Cross-Product-Many
;
; Computes every possible n-tuple of elements from the n given LISTS.
;
; Callers of Cross-Product-Many: (List-Cross-Product)
;------------------------------------------------------------------------------
(defun Cross-Product-Many (&rest lists)
  (let ((new-list (mapcar #'list (first lists))))
    (dolist (List-To-Cross (rest lists) new-list)
	    (setq new-list (Cross-Product-Aux new-list list-to-cross)))))


;------------------------------------------------------------------------------
; Function List-Cross-Product
;
; Computes every possible n-tuple of elements from the given list of n LISTS.
;
; Callers of List-Cross-Product: (Expand-Ors)
;------------------------------------------------------------------------------
(defun List-Cross-Product (lists)
  (apply #'Cross-Product-Many lists))


;------------------------------------------------------------------------------
; Function FS-P
;------------------------------------------------------------------------------
(defun FS-P (fs)
  (and (listp fs)
       (listp (first fs))))


;------------------------------------------------------------------------------
; Function Any-FS-P
;
; Determine if a given structure is an f-structure.  Is more sophisticated than
; FS-P since it will look inside *MULTIPLE*s and *OR*s
;
; Callers of Any-FS-P: (Any-FS-P Define-Head-Rule Define-Pre-Mapping-Rules)
;------------------------------------------------------------------------------
(defun Any-FS-P (fs)
  (or (FS-P fs)
      (and
       (Parser-Or-Or-Mult-Value-P fs)
       (Any-Fs-P (second fs)))))


;------------------------------------------------------------------------------
; Function Find-All-Members
;
; Returns a list of all items that match the MEMBER test for the given ITEM and
; :KEY.
; Callers of Find-All-Members: (Find-All-Members Get-DMK-Entry)
;------------------------------------------------------------------------------
(defun Find-All-Members (item list &key (key #'identity))
  (when list
	(let ((first-member (member item list :key key)))
	  (when first-member
		(cons (first first-member)
		      (Find-All-Members item (rest first-member) :key key))))))



;..............................................................................
; Remove identical terms in list/disjunction; reduce unnecessary disjunction
;
; Callers of Uniquify-Terms: (Combine-Common-Fillers Expand-Ors
;	Handle-Top-Level-Mult-Or-Or Interpreter-Map-Semslot-Multiple
;	Interpreter-Map-Slot-Aux Map-Semantic-Map Post-Map-IR
;	Semantic-Map-Features Uniquify-Disjunction
;	Uniquify-Eliminate-Disjunction)
;
; Callers of Uniquify-Eliminate-Disjunction: (Semantic-Map!)
;..............................................................................

;------------------------------------------------------------------------------
(defparameter *COMPARISON-FUNCTION* 'Tree-Compare)


;------------------------------------------------------------------------------
; Removes identical terms from TERMS.  Returns result.
;------------------------------------------------------------------------------
(defun Uniquify-Terms (terms &key (comparison-function *comparison-function*))
  "Removes identical terms from TERMS.  Returns result."
  (Delete-Duplicates terms :test comparison-function))


;------------------------------------------------------------------------------
; EXP must be an :OR expression.  Removes identical terms.  Returns result.
;------------------------------------------------------------------------------
(defun Uniquify-Disjunction
  (exp &key (comparison-function *COMPARISON-FUNCTION*))
  "EXP must be an :OR expression.  Removes identical terms.  Returns result."

  (when
   (and (listp exp) (eq (first exp) :OR))
   (setf (rest exp)
	 (Uniquify-Terms (rest exp) :comparison-function comparison-function)))
  exp)


;------------------------------------------------------------------------------
; EXP must be an :OR expression.  Removes identical terms.  If only one is
; left, removes expression wrapper.  Returns result.
;------------------------------------------------------------------------------
(defun Uniquify-Eliminate-Disjunction
  (exp &key (comparison-function *COMPARISON-FUNCTION*))
  "EXP must be an :OR expression.  Removes identical terms.  If only one is
left, removes expression wrapper.  Returns result."
  (let (terms)
    (when (and (listp exp)
	       (eq (first exp) :OR))
      (setf terms (uniquify-terms (rest exp) :comparison-function comparison-function))
      (if (< (length terms) 2)
	  (setf exp (first terms))
	(setf (rest exp) terms)))
    exp))



;---eof interpreter-tools.lisp---
