;      -*- Syntax: Common-Lisp; Package: USER; Base: 10; Mode: LISP -*-
;==============================================================================
;     File:  number-rules.lisp
;	By:  Willy Walker	<wkw@pitt.edu>
;     Path:  ../cmt/catalyst/amt-ana/english/interpreter-rules/<release-vs>
;  Started:  19 November 1993
; Modified:  03 December 1996	by <garof>
;
; Comments:  Rules for numbers.  Creating the crystals and mapping numbers into
;	     appropriate modifier slots.
;
; Modified:  21 June 1995
;	By:  Nicholas Brownlow	<ndb@clarit.com>
;  Reasons:  Nicholas was the official "Interpreter" maintainer until then.
;
; Modified:  19 November 1996
;	By:  Joseph Giampapa	<garof@cs.cmu.edu>
;  Reasons:  To garof-ify this file, and to add version control markers to it.
;
; Modified:  03 December 1996
;	By:  Joseph Giampapa	<garof@cs.cmu.edu>
;  Reasons:  Removed references to "ir_vers_ctrl.lisp".
;
; Modified:  <date>
;	By:  <name>		<full e-mail>
;  Reasons:  
;
;==============================================================================


;..............................................................................
;			Center for Machine Translation
;			  Carnegie Mellon University
;
;		     Copyright (c) 1993, 1994, 1995, 1996
;	       Carnegie Mellon University.  All Rights Reserved.
;..............................................................................


;..............................................................................
;			      Package Statements
;..............................................................................
(in-package :user)


;..............................................................................
;			  Attempts at Version Control
;..............................................................................

;------------------------------------------------------------------------------
; For use by the maintainer's personal program.
;------------------------------------------------------------------------------
(defconstant *NUMBER-RULES-VS* '5.0Analyzer)



;..............................................................................
;		    Basic rules for mapping number crystals
;..............................................................................


;------------------------------------------------------------------------------
;				FIX-AS-QUANT-AS
;
; Fix the f-structure for "as many as 10" so that the structure comes out as
; described in the IR Spec.
;------------------------------------------------------------------------------
(defun Fix-As-Quant-As (fs)
  "Fix the f-structure for \"as many as 10\" so that the structure comes out as
described in the IR Spec."
  (let* ((quant (assoc 'QUANT fs))
	 (newquant (list 'QUANT
			 (cons (list 'AS-QUANT-AS '+)
			       (second quant))))
	 (newfs (remove-features '(AS-QUANT-AS QUANT) fs)))
    (cons newquant newfs)))


;..............................................................................
;				Number-Mapping
;
; For spelled-out forms of numbers, I simply create a hash table from which I
; can look up the IR structure for the given spelled-out form of the number.
;..............................................................................

;------------------------------------------------------------------------------
(defvar *Number-Mapping-Hash* (make-hash-table :size 20 :test #'equal)
  "Hash table from which the IR structure for a number's spelled-out form may
be found.")


;------------------------------------------------------------------------------
(defmacro Number-Mapping (&rest mappings)
  `(progn
     (clrhash *number-mapping-hash*)
     (dolist (one-mapping ',mappings)
	     (setf (gethash (first one-mapping) *number-mapping-hash*)
		   (second one-mapping)))))


;------------------------------------------------------------------------------
(Number-Mapping ("one-tenth"
		 ((CAT FRACTION)
		  (DENOM ((CAT CARD) (ROOT "ten")))
		  (NUMER ((CAT CARD) (ROOT "one")))))
		("one-eighth"
		 ((CAT FRACTION)
		  (DENOM ((CAT CARD) (ROOT "eight")))
		  (NUMER ((CAT CARD) (ROOT "one")))))
		("one-fifth"
		 ((CAT FRACTION)
		  (DENOM ((CAT CARD) (ROOT "five")))
		  (NUMER ((CAT CARD) (ROOT "one")))))
		("one-fourth"
		 ((CAT FRACTION)
		  (DENOM ((CAT CARD) (ROOT "four")))
		  (NUMER ((CAT CARD) (ROOT "one")))))
		("one-third"
		 ((CAT FRACTION)
		  (DENOM ((CAT CARD) (ROOT "three")))
		  (NUMER ((CAT CARD) (ROOT "one")))))
		("one-half"
		 ((CAT FRACTION)
		  (DENOM ((CAT CARD) (ROOT "two")))
		  (NUMER ((CAT CARD) (ROOT "one")))))
		("two-thirds"
		 ((CAT FRACTION)
		  (DENOM ((CAT CARD) (ROOT "three")))
		  (NUMER ((CAT CARD) (ROOT "two")))))
		("three-fourths"
		 ((CAT FRACTION)
		  (DENOM ((CAT CARD) (ROOT "four")))
		  (NUMER ((CAT CARD) (ROOT "three")))))
		("zero" "0")
		("one" "1")
		("two" "2")
		("three" "3")
		("four" "4")
		("five" "5")
		("six" "6")
		("seven" "7")
		("eight" "8")
		("nine" "9")
		("ten" "10")
		("eleven" "11")
		("twelve" "12")
		("thirteen" "13")
		("fourteen" "14")
		("fifteen" "15")
		("sixteen" "16")
		("seventeen" "17")
		("eighteen" "18")
		("nineteen" "19")
		("twenty" "20")
		("twenty-four" "24")
		("thirty" "30")
		("forty" "40")
		("fifty" "50")
		("sixty" "60")
		("seventy" "70")
		("eighty" "80")
		("ninety" "90")
		("hundred" "100")
		("thousand" "1000")
		("first" "1")
		("second" "2")
		("third" "3")
		("fourth" "4")
		("fifth" "5")
		("sixth" "6")
		("seventh" "7")
		("eighth" "8")
		("ninth" "9")
		("tenth" "10")
		("eleventh" "11")
		("twelfth" "12")
		("thirteenth" "13")
		("fourteenth" "14")
		("fifteenth" "15")
		("sixteenth" "16")
		("seventeenth" "17")
		("eighteenth" "18")
		("nineteenth" "19")
		("twentieth" "20")
		("hundredth" "100")
		("thousandth" "1000"))


;------------------------------------------------------------------------------
;				Pre-Map Number
;
; Create a number crystal from a number given its ROOT and CAT
;------------------------------------------------------------------------------
(defun Pre-Map-Number (root cat fs)
  "Create a number crystal from a number given its ROOT and CAT"
  (let ((alpha-mapping (gethash root *Number-Mapping-Hash*)))
    (if (consp alpha-mapping)

	;; Merge structure corresponding to alpha ROOT (may change CAT)
	(list-merge alpha-mapping fs)

      ;; Either no alpha mapping or string alpha mapping

      ;; 21-Feb-97 by EHN -- assume existing SEM.
#|
	  (setq fs
		(Tree-Path-Set
		 fs '(SEM)
		 :val (if (list-slot fs 'FRACTION)
			  '*C-MIXED-FRACTION
			'*C-DECIMAL-NUMBER))))
|#
      (setf fs
	    (Tree-Path-Set
	     fs '(NUMBER-TYPE)
	     :val (if (eq (second (assoc 'PERCENT fs)) '+)
		      'PERCENT
		    (case cat
			  ((NUMBER CARD) 'CARDINAL)
			  (ORD 'ORDINAL)
			  (t
			   (interpreter-warn "Unknown number cat: ~s (~s)"
					     cat root)))))
	    
	    fs (Tree-Path-Set
		fs '(NUMBER-FORM)
		:val (if alpha-mapping 'ALPHABETIC 'NUMERIC))
	    
	    fs (if alpha-mapping
		   (tree-path-set
		    fs '(INTEGER)
		    :val alpha-mapping)
		 fs)))))


;..............................................................................
;		      Numeric operator/separator keywords
;..............................................................................


;------------------------------------------------------------------------------
; Contains mappings from special numeric operator symbols (plus, &plusmn@,
; etc.) to their IR keywords.  Used by Get-Numeric-Operator-Keyword.
;------------------------------------------------------------------------------
(defvar *Numeric-Operator-Table* (make-hash-table :size 10 :test #'equal)
  "Contains mappings from special numeric operator symbols (plus, &plusmn@,
etc) to their IR keywords.  Used by Get-Numeric-Operator-Keyword.")


;------------------------------------------------------------------------------
(defmacro Fill-Numeric-Operator-Table (&rest pairs)
  (let ((pair (gensym)))
    `(dolist (,pair ',pairs)
	     (setf (gethash (first ,pair) *numeric-operator-table*)
		   (second ,pair)))))


;------------------------------------------------------------------------------
(Fill-Numeric-Operator-Table ("to" :TO)
			     ("&plusmn@" :PLUS-MINUS)
			     ("&mnplus@" :MINUS-PLUS)
			     ("-" :MINUS-SIGN)
			     ("&minus@" :MINUS-SIGN)
			     ("minus" :MINUS-SIGN)
			     ("+" :PLUS-SIGN)
			     ("&plus@" :PLUS-SIGN)
			     ("plus" :PLUS-SIGN))


;------------------------------------------------------------------------------
; Given a string of any of the following:
;   ("to" "&plusmn@" "&mnplus@" "-" "&minus@" "minus" "+" "&plus@," "plus")
; returns their corresponding IR keyword.
; Correspondences are maintained in the hash table, *Numeric-Operator-Table*.
;------------------------------------------------------------------------------
(defun Get-Numeric-Operator-Keyword (string)
  "Given a string of any of the following:
    (\"to\" \"&plusmn@\" \"&mnplus@\" \"-\" \"&minus@\" \"minus\" \"+\" \"&plus@,\" \"plus\")
  returns their corresponding IR keyword.
  Correspondences are maintained in the hash table, *Numeric-Operator-Table*."
  (let (keyword)
    (if (stringp string)
	(if (setq keyword (gethash string *Numeric-Operator-Table*))
	    keyword
	  (progn
	    (interpreter-warn "GET-NUMERIC-OPERATOR-KEYWORD unknown string: ~s"
			      string)
	    :UNKNOWN))
      (progn
	(interpreter-warn "GET-NUMERIC-OPERATOR-KEYWORD non-string value: ~s"
			  string)
	:UNKNOWN))))



;---eof number-rules.lisp---
