;      -*- Syntax: Common-Lisp; Package: USER; Base: 10; Mode: LISP -*-
;==============================================================================
;     File:  noun-rules.lisp
;	By:  William K. Walker	<wkw@pitt.edu>
;     Path:  ../cmt/catalyst/amt-ana/english/interpreter-rules/<release-vs>
;  Started:  17 November 1993
; Modified:  03 December 1996	by <garof>
;
; Comments:  Interpreter rules for mapping nouns, units, and props.
;
; Modified:  21 June 1995
;	By:  Nicholas Brownlow	<ndb@clarit.com>
;  Reasons:  Nicholas was the official "Interpreter" maintainer until then.
;
; Modified:  28 March 1996
;	By:  Joseph Giampapa	<garof@cs.cmu.edu>
;  Reasons:  cf. PR cat-tl/3179 "another" needs IR features to distinguish it
;	     from other determiners.
;	     The IR-Agenda Task Force Meeting of 29 January 1996 approved the
;	     addition of "another" to the IR Specification as a determiner,
;	     with the following features: 
;			reference:	indefinite
;			number:		singular
;			distance:	<no value>
;			successor:	+
;     N.B.:  IR for determiners modified:
;    Added:  SUCCESSOR	(+ -)
;
; Modified:  20 April 1996
;	By:  Joseph Giampapa	<garof@cs.cmu.edu>
;  Reasons:  cf. PR cat/3633 "GENERIC + feature missing from f-structure for
;	     COUNT - nouns."
;	     "GENERIC +" is equivalent to "REFERENCE" being absent from the IR,
;	     so I am removing all references to it from function
;	     "Map-Noun-Features".  In the PR there is a mini debate about
;	     whether to actually invent another feature value "(REFERENCE
;	     NO-REFERENCE)", or to just not list it in the IR.  This
;	     modification attempt to implement "(REFERENCE NO-REFERENCE)", with
;	     the assumption, that if it must be removed from the IR, it will be
;	     easier to modify than needing to put it in at a later date.
;     N.B.:  IR for determiners modified:
;  Removed:  GENERIC	(+ -)
;    Added:  REFERENCE	(DEFINITE, INDEFINITE, NO-REFERENCE)
;
;	     Function "Map-Noun-Features" revised, new function
;	     "Number-From-Count" and predicate "Multi-Count-P", created for
;	     this revision.  They affect "number".
;
; Modified:  09-27 November 1996
;	By:  Joseph Giampapa	<garof@cs.cmu.edu>
;  Reasons:  Fixes relative to PR 1827, in particular semantic number
;	     agreement.  Added two functions:  "Calculate-Semantic-Number" and
;	     "Calculate-Number-Agreement.
;
; Modified:  19 November 1996
;	By:  Joseph Giampapa	<garof@cs.cmu.edu>
;  Reasons:  To add version control markers.
;
; Modified:  03 December 1996
;	By:  Joseph Giampapa	<garof@cs.cmu.edu>
;  Reasons:  Removed references to "ir_vers_ctrl.lisp".
;
; Modified:  <date>
;	By:  <full name>	<full e-mail address>
;  Reasons:  
;
;==============================================================================


;..............................................................................
;			Center for Machine Translation
;			  Carnegie Mellon University
;
;			   Copyright (c) 1995, 1996
;	       Carnegie Mellon University.  All Rights Reserved.
;..............................................................................


;..............................................................................
;			      Package Statements
;..............................................................................
(in-package :user)


;..............................................................................
;			  Attempts at Version Control
;..............................................................................

;------------------------------------------------------------------------------
; For use by the maintainer's personal program.
;------------------------------------------------------------------------------
(defconstant *NOUN-RULES-VS* '5.0Analyzer)



;..............................................................................
;				  NOUN Rules
;		    The N cat also covers UNITs and PROPs.
;..............................................................................

			
;------------------------------------------------------------------------------
; Function PRE-MAP-SUCH
;
; "such" and "such a[n]" are mapped into the f-structure as determiners, re-map
; them into a *P-SUCH ATTRIBUTE and INDEFINITE reference (if "a[n]" is present)
; or (GENERIC +).
;
; 20-Apr-96-garof:
; This function is called from within "category-rules.lisp", only in the
; context of recognizing a determiner whose root string is either "such-a" or
; "such".
;
;			     Before Modifications
;
; Input Phrase:	"such a" AND "such an" (they are identical)
; F-S Segment:	(... (DET ((ROOT "such-a")(CAT DET))))
; IR Segment:	(... (ADJ ((ROOT "SUCH")(CAT ADJ)(SEM *P-SUCH)))
;		     (DET ((cat det)(root "a"))))
;
; Input Phrase:	"such"
; F-S Segment:	(... (DET ((ROOT "such")(CAT DET)(TOKEN 1))))
; IR Segment:	(... (GENERIC +)(ADJ ((ROOT "SUCH")(CAT ADJ)(SEM *P-SUCH))))
;
;			      After Modifications
;
; Input Phrase:	"such a" AND "such an" (they are identical)
; F-S Segment:	same as before modifications
; IR Segment:	same as before modifications
;
; Input Phrase:	"such"
; F-S Segment:	same as before modifications
; IR Segment:	(... (ADJ ((ROOT "SUCH")(CAT ADJ)(SEM *P-SUCH))))
;
; This function modifies the "F-Structure".  If "Map-Noun-Features" ignores
; "(GENERIC +)", then there should be no reason to reinsert it here.
; "(REFERENCE NO-REFERENCE)" certainly does not belong, as that is an feature
; of the interlingua.
;------------------------------------------------------------------------------
(defun Pre-Map-Such (fs)
  "4.10alpha_02:  \"such\" and \"such a[n]\" are mapped into the f-structure\
 as determiners, re-map them into a *P-SUCH ATTRIBUTE and INDEFINITE reference\
 (if \"a[n]\" is present)."

  (let ((such-root (second (assoc 'ROOT (second (assoc 'DET fs))))))
    (if (string-equal such-root "such-a")
	(insert-feature (remove-features 'DET fs)
			(list 'ADJ (copy-tree '((ROOT "such")
						(CAT ADJ)
						(SEM *P-SUCH))))
			(list 'DET (copy-tree '((CAT DET)
						(ROOT "a")))))
      (insert-feature (remove-features 'DET fs)
		      (list 'ADJ (copy-tree '((ROOT "such")
					      (CAT ADJ)
					      (SEM *P-SUCH))))))))


;------------------------------------------------------------------------------
; PRE-MAP-AS-QUANT-AS-NP
;
; "AS-QUANT-AS" structures (the f-structure feature) needs to be mapped into a
; special structure for handling by the INTERPRETER-CAT-RULE given below.
;------------------------------------------------------------------------------
(defun Pre-Map-As-Quant-As-NP (fs number-slot)
  "\"AS-QUANT-AS\" structures (the f-structure feature) needs to be mapped into
 a special structure for handling by the INTERPRETER-CAT-RULE given below."
  (let* ((quantity (assoc number-slot fs))
	 (quant (assoc 'QUANT (second quantity)))
	 (newquantity (list number-slot
			    (remove-features '(quant as-quant-as)
					     (second quantity))))
	 (newfs (remove quantity fs :test #'eq)))
    (list (list 'CAT 'COMPARATIVE)    
	  (list 'ROOT 'COMPARATIVE)    
	  (list 'DEGREE 'COMPARATIVE)    
	  (list 'COMPARISON 'EQUAL)    
	  (list 'COMPARISON-THEME (cons newquantity newfs))
	  quant)))


;------------------------------------------------------------------------------
; PRE-MAP-PARTITIVES
;
; Restructure a nominal FS with a PARTITIVE-QUANT, PARTITIVE-CARD, or
; PARTITIVE-PERCENT slot into a new FS which can be easily mapped into the
; require *G-PARTITIVE structure.
;------------------------------------------------------------------------------
(defun Pre-Map-Partitive (fs)
  "Restructure a nominal FS with a PARTITIVE-QUANT, PARTITIVE-CARD, or\
 PARTITIVE-PERCENT slot into a new FS which can be easily mapped into the\
 require *G-PARTITIVE structure."
  (let ((quantifier (tree-filler fs 'PARTITIVE-QUANT))
	;; Cardinal or percent but not both
	(quantity (or (tree-filler fs 'PARTITIVE-CARD)
		      (tree-filler fs 'PARTITIVE-PERCENT))))
    
    (setf fs (tree-path-remove fs '(PARTITIVE-QUANT))
	  fs (tree-path-remove fs '(PARTITIVE-PERCENT))
	  fs (tree-path-remove fs '(PARTITIVE-CARD)))
    
    (nconc (list (list 'CAT 'N)
		 (list 'SEM '*G-PARTITIVE)
		 (list 'SUBSTANCE fs))
	   (and quantifier
		(list (list 'QUANTIFIER quantifier)))
	   (and quantity
		(list (list 'QUANTITY quantity))))))


;------------------------------------------------------------------------------
; Function MAP-NUMBER-INFO
;
; 28-Mar-96-garof:  Reads F-Structure number features {SG PL (:OR SG PL)} and
; maps them to their corresponding IR symbols {SINGULAR PLURAL BOTH}.
;------------------------------------------------------------------------------
(defun map-number-info (value)
  "28-Mar-96-garof:  Reads F-Structure number features {SG PL (:OR SG PL)} and\
 maps them to their corresponding IR symbols {SINGULAR PLURAL BOTH}."
  (cond ((eq value 'SG) 'SINGULAR)
	((eq value 'PL) 'PLURAL)
	((and (consp value)
	      (eq (first value) :OR)
	      (null (set-difference (rest value) '(SG PL)))) 'BOTH)
	(t (interpreter-warn
	    "MAP-NUMBER-INFO: unknown value for NUMBER feature: ~a" value))))


;------------------------------------------------------------------------------
; Function MAP-DET-INFO
;
; 28-Mar-96-garof:  Maps the F-S syntactic determiner concept to the IR
; semantic feature list.  Given an F-Structure determiner head, returns
; multiple values which successively correspond to the slot names:
; (reference number distance successor).
;
; Implementation Note:  cf. PR cat-tl/3179  If we were ever to redesign the
; Interpreter, I think that it would be more appropriate to make the DMK
; maintain determiner semantic features for ease of maintenance.
;------------------------------------------------------------------------------
(defun Map-Det-Info (det-head)
  "28-Mar-96-garof:  Maps the F-S syntactic determiner concept to the IR
semantic feature list.  Given an F-Structure determiner head, returns multiple
values which successively correspond to the slot names: (reference number
distance successor)."
  (case det-head
	(*det-the	(values 'DEFINITE))
	(*det-a		(values 'INDEFINITE 'SINGULAR))
	(*det-an	(values 'INDEFINITE 'SINGULAR))
	(*det-another	(values 'INDEFINITE 'SINGULAR NIL '+))
	;; 24-Feb-97 by EHN -- for CNBC
	(*det-other	(values 'INDEFINITE 'PLURAL NIL '+))
	(*det-this	(values 'DEFINITE 'SINGULAR 'NEAR))
	(*det-that	(values 'DEFINITE 'SINGULAR 'FAR))
	(*det-these	(values 'DEFINITE 'PLURAL 'NEAR))
	(*det-those	(values 'DEFINITE 'PLURAL 'FAR))
	(otherwise	(interpreter-warn
			 "MAP-DETERMINER-INFO: unknown value for DET root: ~s"
			 det-head))))


;------------------------------------------------------------------------------
; 20-Apr-96-garof:
; Used by "Number-From-Count", below.  This predicate returns "T" if "COUNT" is
; a "list" which begins with ":OR", and that its elements are only "+" and "-".
;------------------------------------------------------------------------------
(defun Multi-Count-P (count)
  "Predicate which determines if COUNT can be singular or plural."
  (and (consp count)
       (eq (first count) :OR)
       (null (set-difference (rest count) '(+ -)))))


;------------------------------------------------------------------------------
; 19-Apr-96-garof:
; This section was yanked out of "Map-Noun-Features".  It was cumbersome to
; maintain, so I decided to try reworking it here.
;
; Given:  "count", "number", "lexical-number", and "orig-number", returns a
; revised value, destined to overwrite "number" in "Map-Noun-Features".
;
; If "count" is a list ("Multi-Count-P"), the following possibilities must be
; considered:
; (NUMBER SG)		==> (NUMBER (:OR SINGULAR MASS))
; (NUMBER PL)		==> (NUMBER PLURAL)
; (NUMBER (:OR PL SG))	==> (NUMBER (:OR SINGULAR PLURAL MASS))
;------------------------------------------------------------------------------
(defun Number-From-Count (number count lexical-number orig-number proper-noun)
  "Redefine the number of the noun given its NUMBER and COUNT features."

  (setq number (or number
		   (and lexical-number	(Map-Number-Info lexical-number))
		   (and orig-number	(Map-Number-Info orig-number))
		   'BOTH))

  ;; 23-Jul-97 by EHN -- added fix for (PROPER-NOUN +).

  (cond ((eq count '-) 'MASS)
	((eq proper-noun '+) number)
	((and (eq count '+)
	      (eq number 'BOTH))	(list :OR 'SINGULAR 'PLURAL))
	((and (eq count '+)
	      (not (eq number 'BOTH)))	number)
	((Multi-Count-P count)
	 (case number
	       (SINGULAR		(list :OR 'SINGULAR 'MASS))
	       (PLURAL			number)
	       (BOTH			(list :OR 'SINGULAR 'PLURAL 'MASS))))
	(t
	 (format t "Number-From-Count called with unexpected COUNT ")
	 (format t "[~s] value.~%" count)
	 NIL)))


;------------------------------------------------------------------------------
; Function MAP-NOUN-FEATURES
;
; NUMBER, REFERENCE, DISTANCE, etc. are inter-related NP features that must be
; mapped together into the IR.  Thus they have this :MULTIPLE-FEATURE-SLOT-RULE
; function.
;
; NOTE: One special thing done by this function is the insertion of
; IMPLIED-REFERENCE on NPs that do not seem to have a REFERENCE feature.  See
; the RESOLVE-IMPLIED-REFERENCE post mapping function.
;
; 28-Mar-96-garof:
; cf. PR cat-tl/3179
; Given a list of F-Structure noun features such as,
;		 ((COUNT +) (DET (*DET-ANOTHER)) (NUMBER SG))
; returns a list of IR noun features such as,
;	   ((NUMBER SINGULAR) (SUCCESSOR +) (REFERENCE INDEFINITE))
;
; 19-Apr-96-garof:
; cf. PR cat/3633
; "GENERIC +" is equivalent to "REFERENCE" being absent from the IR, so I am
; removing all references to it from this function.  In the PR there is a mini
; debate about whether to actually invent another feature value "(REFERENCE
; NO-REFERENCE)", or to just not list it in the IR.  This modification attempts
; to implement "(REFERENCE NO-REFERENCE)", with the assumption, that if it must
; be removed from the IR, it will be easier to modify than needing to put it in
; at a later date.
; N.B.:  IR for determiners modified:
; Removed:  GENERIC	(+ -)
;   Added:  REFERENCE	(DEFINITE, INDEFINITE, NO-REFERENCE)
;------------------------------------------------------------------------------
(defun Map-Noun-Features (slot-sems fs)
  "Maps nouns to their interlingual features.\
Last release for which it was modified:  4.10alpha_02."

  (let* ((count			(second (assoc 'COUNT slot-sems)))
	 (det			(second (assoc 'DET slot-sems)))
	 (genitive		(second (assoc 'GENITIVE slot-sems)))
	 (implied-reference	(second (assoc 'IMPLIED-REFERENCE slot-sems)))
	 (lexical-number	(second (assoc 'LEXICAL-NUMBER slot-sems)))
	 (nominalized		(second (assoc 'NOMINALIZED slot-sems)))
	 (orig-number		(second (assoc 'NUMBER slot-sems)))
	 ;; 23-Jul-97 by EHN -- added support for calling number-from-count
	 ;; with info about (proper-noun +)
	 (proper-noun		(second (assoc 'PROPER-NOUN fs)))
	 (reference		nil)
	 (number		nil)
	 (distance		nil)
	 (successor		nil))
    (declare (ignore genitive))

    (when det
	  (multiple-value-setq (reference number distance successor)
			       (Map-Det-Info (first det))))

    (when (or count proper-noun)
	  (setq number
		(Number-From-Count number count lexical-number orig-number proper-noun)))

    (when (and (null number)
	       orig-number)
	  (setq number (massage-orig-number orig-number)))

    (nconc (and number		(list (list 'NUMBER number)))
	   (and successor	(list (list 'SUCCESSOR successor)))
	   (or (and reference	(list (list 'REFERENCE reference)))
	       (and implied-reference
		    		(list (list 'IMPLIED-REFERENCE
					    implied-reference)))
	       (and (not reference)
		    (not implied-reference)
		    		(list (list 'REFERENCE 'NO-REFERENCE))))
	   (and distance	(list (list 'DISTANCE distance)))
	   (and (eq nominalized '+)
		(copy-tree '((NOMINAL +)))))))

(defun massage-orig-number (s)
  (case s
	(sg 'singular)
	(pl 'plural)
	(t s)))


;------------------------------------------------------------------------------
; Function RE-STRUCTURE-ADJ-MOD
;
; Put an ADJ-MOD f-structure feature with the number-value feature
; since that is what is should modify in the IR.
;------------------------------------------------------------------------------
(defun Re-Structure-Adj-Mod (fs)
  "Put an ADJ-MOD f-structure feature with the number-value feature since that\
 is what is should modify in the IR."
  (let* ((adj-mod	(assoc 'ADJ-MOD fs))
	 (number-value	(assoc 'NUMBER-VALUE fs)))
    (cond ((and adj-mod number-value)
	   (nconc (remove-features '(ADJ-MOD NUMBER-VALUE) fs)
		  (list (list 'NUMBER-VALUE
			      (append (second number-value) (list adj-mod))))))
	  (t fs))))


;------------------------------------------------------------------------------
; 13-Nov-96-garof:
; Per PR 1827, it was noticed that a noun modified by a QUANTITY should agree
; numerically with the value of the QUANTITY expression.  This
; "category-rules.lisp" callout is an attempt to rectify that problem.
; (tree-path-set *tsir1* '(number) :val 'plural :do :overwrite)
; There are three cases to handle:
; 1. A simple "quantity" under the noun head of the "Numbered-NP-Attribute":
;    Ex. (*O-GEAR-1
;	    (NUMBER SINGULAR)
;	    (NUMBERED-NP-ATTRIBUTE
;	       (*O-TOOTH
;	          (NUMBER PLURAL) <--- instead of SINGULAR
;	          (IMPLIED-REFERENCE +)
;	          (QUANTITY
;	             (*C-DECIMAL-NUMBER
;	                (NUMBER-FORM NUMERIC)
;	                (NUMBER-TYPE CARDINAL)
;	                (INTEGER "10"))))))

;------------------------------------------------------------------------------
; Numbers are composed of an integer, and optionally, decimal component.  Since
; these values are read from the IR subtree, the value of "int-val" will be a
; string representation of the number.  "dec-val" may not be valued in the IR
; subtree.  If this is the case, "Tree-Path-Test" would have returned a ":FAIL"
; (also the case for an error condition).
; Conditions to test:
; int-val    dec-val    agreement    return		reason
;  :FAIL      :FAIL         ?        :NO-CHANGE  This is an error condition.
;  :FAIL    <any-val>       ?        :NO-CHANGE  This is an error condition.
;    1        :FAIL	SING/MASS    :NO-CHANGE  FS is usually correct.
;    0          1  	SING/MASS    :NO-CHANGE  FS is usually correct.
;    0         >1         PLURAL     :PLURAL     non-singular decimal value
;    1          1         PLURAL     :PLURAL	 non-singular decimal value
;   >1        :FAIL       PLURAL     :PLURAL
;   >1      <any-val>     PLURAL     :PLURAL
;------------------------------------------------------------------------------
(defun Calculate-Number-Agreement (int-val dec-val)
  "Given the values of the IR attributes \"integer\" and \"decimal\", determine
if the modified noun must have its number agreement attribute modified.
Returns \":NO-CHANGE\" or \":PLURAL\" as mutually-exclusive values of this fn.
See program documentation for more explanation."

  (unless (equal int-val :FAIL) (setq int-val (parse-integer int-val)))
  (unless (equal dec-val :FAIL) (setq dec-val (parse-integer dec-val)))
  
  (cond ((equal int-val :FAIL)			    :NO-CHANGE)
	((and (= int-val 1) (equal dec-val :FAIL))  :NO-CHANGE)
	((and (zerop int-val) (= dec-val 1))	    :NO-CHANGE)
	((and (zerop int-val) (> dec-val 1))	    :PLURAL)
	((and (= int-val 1) (= dec-val 1))	    :PLURAL)
	((> int-val 1)				    :PLURAL)
	(t
	 (Interpreter-Warn
	  "[~A] unexpected combination: [~A ~S] [~A ~S]~%"
	  "Calculate-Number-Agreement" "int-val" int-val "dec-val" dec-val)
	 					    :NO-CHANGE)))


;------------------------------------------------------------------------------
; Given an IR subtree with a "quantity" attribute, return the IR with its
; "number" attribute recalculated to refect agreement with "quantity's" value.
;------------------------------------------------------------------------------
(defun Calculate-Semantic-Number (ir)
  "Given an IR subtree with a \"quantity\" attribute, return the IR with its
\"number\" attribute recalculated to refect agreement with \"quantity's\"
value.  See also \"Calculate-Number-Agreement\" and program code for further 
documentation."
  (let* ((quantity  (Tree-Path-Test ir '(quantity)))
	 (int-val   (Tree-Path-Test quantity '(integer)))
	 (dec-val   (Tree-Path-Test quantity '(DECIMAL)))
	 (agreement (Calculate-Number-Agreement int-val dec-val)))
    
    (case agreement
	  (:NO-CHANGE ir)
	  (:PLURAL    (Tree-Path-Set ir '(number) :val 'plural :do :overwrite))
	  (otherwise
	   (Interpreter-Warn "[~A] Bad agreement in CASE.  [argreement ~S]~%"
			     "Calculate-Semantic-Number" agreement)
	   ir))))



;---eof noun-rules.lisp---
