;;;==================================================================;
;;; -*- Syntax: Common-Lisp; Package: USER; Base: 10; Mode: LISP -*-
;;;==================================================================;
;;;
;;;               Center for Machine Translation
;;;                 Carnegie-Mellon University
;;;                                                                       
;;;------------------------------------------------------------------;
;;;                                                                       
;;; Copyright (c) 1995
;;; Carnegie Mellon University. All Rights Reserved.                      
;;;                                                                       
;;;------------------------------------------------------------------;
;;;
;;;          File: dmk-synlex-loader.lisp
;;;  File created: 13-May-92 by amf
;;;        Author: Alex Franz [amf@cs.cmu.edu]
;;; Last Modified:   3-Oct-95, 14:23-May-95 at 16:00
;;;
;;;------------------------------------------------------------------;
;;; Description
;;; 
;;; Loads syntactic lexicon from dmk files.
;;;
;;; See also: morph-analyzer.lisp


;;;==================================================================;
;;; Change Log
;;;
;;; 13-May-92 by amf: created
;;; 11-Nov-92 by amf: merged jrrl's efficiency changes


;;;==================================================================;
;;; TODO
;;;
;;; Handle :MORPH :PLURAL --> (Number Pl)
;;;  and other cases like this


;;;==================================================================;
;;; Documentation
;;;
;;; This code reads dmk files, and produces syntactic lexicon entries for
;;; single words.
;;;
;;; Procedure:
;;;
;;; 1. Iterate over all files indicated
;;; 2. Iterate over all dmk entries in file
;;; 3. Read in entry as list
;;; 4. If :CTE -, add syntactic feature
;;; 5. If any syntactic feature has multiple values, make *OR*
;;; 6. For any :MORPHOLOGY + value, ignore
;;; 7. For any :MORPHOLOGY - value, add to (FORM (:not list
;;; 8. For any irregular form on :MORPHOLOGY list, add to (FORM (:not ...)) list
;;; 9. Make full list of syntactic features, and store entry in raw lexicon
;;; 10. For each irregular form, make cached entry


;;;==================================================================;
;;; Package Statements

(in-package :USER)


;;;==================================================================;

;;; Global Variables

(defvar *dmk-directory*)

(defvar *shared-syn-feature-table*
  (make-hash-table :size 120 :test #'equal))

(defvar *shared-sem-feature-table*
  (make-hash-table :size 120 :test #'equal))


;;;==================================================================;

;;; Structure sharing

(defun shared-syn-features (features)
  (or (gethash features *shared-syn-feature-table*)
      (setf (gethash features *shared-syn-feature-table*)
	    features)))


(defun shared-sem-features (features)
  (or (gethash features *shared-sem-feature-table*)
      (setf (gethash features *shared-sem-feature-table*)
	    features)))
  

(defun share-syn-features (struct)
  (setf (dmk-syn-features struct)
        (shared-syn-features (dmk-syn-features struct))))

(defun share-sem-features (struct)
  (setf (dmk-sem-features struct)
        (shared-sem-features (dmk-sem-features struct))))


(defun set-struct-phrase-head-default (struct root)
  (unless (dmk-phrase-head struct)
    (case (dmk-pos struct)
      ((N PROP UNIT)          (setf (dmk-phrase-head struct) :LAST))
      ((ADJ ADV)              (setf (dmk-phrase-head struct) :NONE))      
      (V                      (setf (dmk-phrase-head struct) 1))      
      ((CONJ DET PREP QUANT)  (setf (dmk-phrase-head struct) :NONE))
      (OTHERWISE              (setf (dmk-phrase-head struct) :NONE))))
  ;; convert :LAST to number
  (if (eq (dmk-phrase-head struct) :LAST)
    (setf (dmk-phrase-head struct)
          (1+ (count #\space root)))))


;;;==================================================================;

;;; Filling in an entry

(defun set-syn-feat-defaults (struct)
  (case (dmk-pos struct)
    (N
     ;; count defaults to +
     (let ((count (assoc 'COUNT (dmk-syn-features struct))))
       (unless count
         (setf (dmk-syn-features struct)
               (cons (list 'count '+)
                     (dmk-syn-features struct))))))
    (V
     ;; valency defaults to TRANS
     (let ((valency (assoc 'valency (dmk-syn-features struct))))
       (unless valency
         (setf (dmk-syn-features struct)
               (cons (list 'valency 'trans)
                     (dmk-syn-features struct))))))))

(defun add-to-phrase-lex (struct root)
  (phrec-insert *eng-phrec*
                root
                root
                struct
                (dmk-phrase-head struct)))




(defun morph-list-lessp (list1 list2)
  "is list1 less than list2 on morph list?"
  (and (listp list1)
       (listp list2)
       (> (length list1) 1)
       (> (length list2) 1)
       (stringp (second list2))
       (not (stringp (second list1)))))

(defun handle-morph-slot (entry struct root &key (word-entry t))
  (let ((morph-neg nil)
        (number-neg nil)
	sorted-morph-list syn-feats num-val)
    ;; note defaults for words
    (if word-entry
      (cond ((or (eq (dmk-pos struct) 'ADV)
                 (eq (dmk-pos struct) 'ADJ))
             (push 'COMP morph-neg)
             (push 'SUPER morph-neg))))
    
    (setq sorted-morph-list (sort (cdr (assoc :MORPH entry))
				  #'morph-list-lessp))
    (dolist (morph sorted-morph-list)
      (let ((morph-symbol (intern (string-upcase (symbol-name (first morph)))))
            (morph-value (second morph)))
        (case morph-symbol
          (GENSG
           (cond ((eq morph-value '+)) ; do nothing
                 ((eq morph-value '-)
                  (setf (dmk-syn-features struct)
                        (append (dmk-syn-features struct)
                                (list (list 'GENSG '-)))))
                 
                 (t
                  (setf (dmk-syn-features struct)
                        (append (dmk-syn-features struct)
                                (list (list 'GENSG '-))))
                  (if word-entry
                    (make-irregular-fs morph-value
                                       struct
                                       root
                                       :syn-features '((NUMBER SG)(GENITIVE +)))))))
          (GENPL
           (cond ((eq morph-value '+)) ; do nothing
                 ((eq morph-value '-)
                  (setf (dmk-syn-features struct)
                        (append (dmk-syn-features struct)
                                (list (list 'GENPL '-)))))
                 (t
                  (setf (dmk-syn-features struct)
                        (append (dmk-syn-features struct)
                                (list (list 'GENPL '-))))	  
                  (if word-entry
                    (make-irregular-fs morph-value
                                       struct
                                       root
                                       :syn-features '((NUMBER PL) (GENITIVE +)))))))
          (PLURAL
           (cond ((eq morph-value '+)) ; do nothing
                 ((eq morph-value '-)
		  ;; add (NUMBER (:NOT PL)) unless theere already is
		  ;; a (NUMBER PL)
		  (setq syn-feats (dmk-syn-features struct))
		  (setq num-val (assoc 'NUMBER syn-feats))
		  (unless (and num-val
			       (listp num-val)
			       (second num-val)
			       (eq (second num-val) 'PL))
		    (push 'PL number-neg)))
                 (t ; plural form given
		  ;; if plural form is the same as singular, then just fix
		  ;; NUMBER feature
		  (cond ((string= morph-value root)
			 (setq syn-feats  (dmk-syn-features struct))
			 (setf syn-feats (remove (assoc 'NUMBER syn-feats)
						 syn-feats))
			 (setf (dmk-syn-features struct)			 
			       (append syn-feats
				       (list (list 'NUMBER (list ':OR 'SG 'PL))))))
			(t ; plural form given is no the same
			 (push 'PL number-neg)
			 (if word-entry
			     (make-irregular-fs morph-value
						struct
						root
						:syn-features '((NUMBER PL)))))))))
          
          (COMP
           (cond ((eq morph-value '-)) ; do nothing
                 ((eq morph-value '+)
                  (setq morph-neg (remove 'COMP morph-neg)))
                 ((stringp morph-value)
                  (if word-entry
                    (make-irregular-fs morph-value
                                       struct
                                       root
                                       :form-features '((FORM COMP)))))))
          
          (SUPER
           (cond ((eq morph-value '-)) ; do nothing
                 ((eq morph-value '+)
                  (setq morph-neg (remove 'SUPER morph-neg)))
                 ((stringp morph-value)
                  (if word-entry
                    (make-irregular-fs morph-value
                                       struct
                                       root
                                       :form-features '((FORM SUPER)))))))
          
          ((PRESPART PRESPARTPL PAST PASTPART PRESENT3SG PRESENT1SG  
                     PRESENTPL PAST13SG PASTPL)
           (cond ((eq morph-value '+)) ; do nothing
                 ((eq (dmk-pos struct) 'MOD)) ; do nothing -- this is spurious
                 ((eq morph-value '-)
                  (push  morph-symbol morph-neg))
                 ((stringp morph-value)
                  (push morph-symbol morph-neg)
                  (when word-entry
                    (make-irregular-fs morph-value
                                       struct
                                       root
                                       :form-features (list (list 'FORM morph-symbol)))
                    (when (member morph-symbol '(PAST PASTPART))
                      (compact-past-pastpart-if-necessary morph-value 'V)
                      (compact-past-pastpart-if-necessary morph-value 'AUX))))))
          
          (OTHERWISE
           (warn "~% [DMK syn reader]: unknown :morph keyword ~S~%"
                 morph-symbol) 
           (warn "~% [DMK syn reader]: errorful :morph ~S~%" (assoc :morph 
                                                                    entry)))))) ; end dolist
    
    ;; if any morph forms are ruled out, say so
    (add-morphological-syn-features morph-neg number-neg struct)))

(defun compact-past-pastpart-if-necessary (string cat)
  (let ((fs-list (eng-irregular-word-fs string))
        past-fs pastpart-fs other-fs)
    
    (dolist (fs fs-list)
      (cond ((and (eq (cadr (assoc 'FORM fs)) 'PAST) 
                  (eq (cadr (assoc 'CAT fs)) cat))
             (setq past-fs fs))
            ((and (eq (cadr (assoc 'FORM fs)) 'PASTPART) 
                  (eq (cadr (assoc 'CAT fs)) cat))
             (setq pastpart-fs fs))
            ;; all other fs get stored here
            (t (setq other-fs (cons fs other-fs)))))
    
    (unless (and past-fs pastpart-fs)
      (return-from compact-past-pastpart-if-necessary T))
    
    ;; adjust the pastpart entry to look like (FORM (*OR* PAST PASTPART))
    (setf (cdr (assoc 'FORM pastpart-fs))
          (list (list '*OR* 'PAST 'PASTPART)))
    
    ;; store the adjusted fs back in the irregular table
    (setf (eng-irregular-word-fs string)
          (cons pastpart-fs other-fs))
    
    ;; (format *standard-output* "~&~%>>> Kompacted string ~S cat ~S" string cat)
    ))

;;; if morphological forms are ruled out, include this information
;;; in the syntactic features

(defun add-morphological-syn-features (morph-neg number-neg struct)
  (cond ((and morph-neg
              (> (length morph-neg) 1))
         
         ;; build a (FORM (:NOT ...))
         (setf (dmk-syn-features struct)
               (append (dmk-syn-features struct)
                       (list (list 'FORM (cons :NOT (list (cons :OR morph-neg))))))))
        (morph-neg
         ;; only one negative form
         (setf (dmk-syn-features struct)
               (append (dmk-syn-features struct)
                       (list (list 'FORM (cons :NOT morph-neg)))))))
  (cond (number-neg
         (setf (dmk-syn-features struct)
               (append (dmk-syn-features struct)
                       (list (list 'NUMBER (cons :NOT number-neg))))))))


;;----------------------------------------------------------------------
;; 29-Jul-96 by EHN -- make this thing keep the SYL-DOUBLE from the 
;; DMK entry as a syn-feature in the f-structure.

(defun set-struct-syn-features (entry struct root orig-root)
  (let ((syn-features (cdr (assoc :syn-features entry))))
    ;; add :CTE - information to syn-features
    (if (eq (second (assoc :CTE entry)) '-)
      (push (list 'CTE '-) syn-features))
    ;; 29-Jul-96 by EHN add :SYL-DOUBLE + information to syn-features
    (when (eq (second (assoc :SYL-DOUBLE entry)) '+)
	  ;;(format t "~%ADDING SYL-DOUBLE: ~s ~s" root (assoc :pos entry))
	  (push (list 'SYL-DOUBLE '+) syn-features))

    (unless (string= root orig-root)
      (push (list 'ORIG-ROOT orig-root) syn-features))

    (setf (dmk-syn-features struct) 
          (add-or-to-syn-features syn-features))
    (unless (or (null (dmk-syn-features struct))
                (listp (dmk-syn-features struct)))
      (warn "~% [DMK syn reader]: errorful :syn-features ~S~%"
            (assoc :syn-features entry)))))
#|
(defun set-struct-syn-features (entry struct root orig-root)
  (let ((syn-features (cdr (assoc :syn-features entry))))
    ;; add :CTE - information to syn-features
    (if (eq (second (assoc :CTE entry)) '-)
      (push (list 'CTE '-) syn-features))

    (unless (string= root orig-root)
      (push (list 'ORIG-ROOT orig-root) syn-features))

    (setf (dmk-syn-features struct) 
          (add-or-to-syn-features syn-features))
    (unless (or (null (dmk-syn-features struct))
                (listp (dmk-syn-features struct)))
      (warn "~% [DMK syn reader]: errorful :syn-features ~S~%"
            (assoc :syn-features entry)))))
|#

(defun make-irregular-fs (surface-string struct root &key form-features syn-features)
  ;; make entry in root table
  (push root (eng-irreg-word-root surface-string))
  (push (append form-features
                syn-features
                (dmk-syn-features struct)
                (list 
		 (list 'SEM (dmk-concept struct))
		 (list 'ROOT root)
		 (list 'CAT (dmk-pos struct)))
                
                (if (not (string= root surface-string))
                  (list (list 'ORTHO surface-string)) ; then
                  NIL) ; else
                )
        (eng-irregular-word-fs surface-string)))

(defun add-or-to-syn-features (features)
  (mapcar #'(lambda (slot)
              (cond ((> (length slot) 2)
                     (list (first slot) (cons '*OR* (cdr slot))))
                    (t
                     slot)))
          features))

#|
(defun set-struct-caps (entry struct)
  ;; :CAPS {:FIRST :ALL :ANY} and :IRREG-CAPS is turned into this slot  
  
  (setf (dmk-capitalization-knowledge struct)
        (cond ((second (assoc :irreg-caps entry))) ; set it to this
              ((null (second (assoc :caps entry)))
               nil) 
              ((eq (second (assoc :caps entry))		   
                   :ANY)
               nil)
              ((eq (second (assoc :caps entry))		   
                   :ALL)
               (string-upcase (dmk-root struct)))
              ((eq (second (assoc :caps entry))		   
                   :FIRST)
               (string-capitalize (dmk-root struct)))))
  
  
  (unless (or (null (dmk-capitalization-knowledge struct))
              (stringp (dmk-capitalization-knowledge struct)))
    (warn "~% [DMK syn reader]: errorful :caps or :irreg-caps ~S~%~S~%" (assoc 
                                                                         :caps entry)
          (assoc :irreg-caps entry)))
  )
|#

;;; preprocess-root-string is defined in code/cte-checker/reader.lisp
;;; this keeps not preprocesing when it should... just call
;;; it always -- wastes time, but is safer.

(defun preprocess-root-string-maybe (string)
  "mirror parser preprocessor functions where necessary to derive proper root string"
  (string-downcase (preprocess-root-string string)))
  
;;;  (cond 
;;;    ((or (position #\< string)
;;;         (position #\; string)
;;;         (position #\, string)       
;;;         (position #\( string))
;;;     (string-downcase (preprocess-root-string string)))
;;;    (t string))

(defun set-struct-simple-slots (struct entry)
  (setf (dmk-concept struct) 
        (second (assoc :concept entry)))
  (unless (symbolp (dmk-concept struct))
    (warn "~% [DMK syn reader]: errorful :concept ~S~%"
          (assoc :concept entry)))
  
  (setf (dmk-pos struct)
        (second (assoc :pos entry)))
  (unless (symbolp (dmk-pos struct))
    (warn "~% [DMK syn reader]: errorful :pos ~S~%" (assoc :pos entry)))
  
  (setf (dmk-type struct) 
        (second (assoc :type entry)))
  (unless (or (null (dmk-type struct))
              (symbolp (dmk-type struct)))
    (warn "~% [DMK syn reader]: errorful :type ~S~%" (assoc :type entry)))
  
  ;; (setf (dmk-syl-double struct)
  ;; (second (assoc :syl-double entry)))
  ;; (unless (or (null (dmk-syl-double struct))
  ;; (symbolp (dmk-syl-double struct)))
  ;; (warn "~% [DMK syn reader]: errorful :syl-double ~S~%" (assoc :syl-double entry)))
  
  (setf (dmk-phrase-head struct)
        (second (assoc :phrase-head entry)))
  (unless (or (null (dmk-phrase-head struct))
              (numberp (dmk-phrase-head struct))
              (eq :NONE (dmk-phrase-head struct))
              (eq :LAST (dmk-phrase-head struct)))
    (warn "~% [DMK syn reader]: errorful :phrase-head ~S~%"
          (assoc :phrase-head entry)))
  
  ;; syn-features are set manually due to required *OR* frobbage
  
  (setf (dmk-sem-features struct) 
        (shared-sem-features (cdr (assoc :sem-features entry))))
  (unless (or (null (dmk-sem-features struct))
              (listp (dmk-sem-features struct)))
    (warn "~% [DMK syn reader]: errorful :sem-features ~S~%"
          (assoc :sem-features entry)))
  
  (setf (dmk-class struct)
        (second (assoc :class entry)))
  (unless (or (not (eq (dmk-pos struct) 'V))
              (symbolp (dmk-class struct)))
    (warn "~% [DMK syn reader]: errorful :class ~S~%" (assoc :class entry)))
  ) 

;;;==================================================================;

;;; Adding an entry

;(defun add-syn-lex-entry (entry)
;  (let* ((orig-root (string-downcase (second (assoc :ROOT entry))))
;         (root (preprocess-root-string-maybe orig-root))
;         (dmk (make-dmk))               ; New DMK struct
;         )
;    (unless (stringp root)
;      (warn "~% [DMK syn reader]: errorful :root ~S~%" (assoc :root entry)))
;
;    (set-struct-simple-slots dmk entry)
;    (set-struct-syn-features entry dmk root orig-root) 
;    (if (position #\SPACE root) 
;        (setf (dmk-type dmk) :PHRASE))
;    (set-syn-feat-defaults dmk)
;    ;; (set-struct-caps entry dmk)   ; use :caps and :irreg-caps slots
;        
;    (case (dmk-type dmk)
;      (:PHRASE
;       (set-struct-phrase-head-default dmk root)
;       (handle-morph-slot entry dmk root :word-entry nil) 
;       (share-syn-features dmk)
;       ;; If preference phrase: add <idiom> tags to root, insert irec
;       (when (assoc 'PREFER-PHRASE (dmk-syn-features dmk))
;         (phrec-insert *eng-irec* root root T)
;         ;; Move head one over to allow for {idiom}
;         (if (numberp (dmk-phrase-head dmk))
;             (incf (dmk-phrase-head dmk)))
;         (setq root (concatenate 'string "{idiom} " root " {/idiom}")))
;       ;; Ignore entry under following conditions
;       (unless (eq (dmk-pos dmk) 'GRAM)   
;         (add-to-phrase-lex dmk root)))
;
;      (otherwise                        ; DMK entry is a word     
;       (handle-morph-slot entry dmk root)
;       (share-syn-features dmk)
;       ;; Add dmk structure to dmk hash table
;
;       ;; 13-Sep-95 by amf: this is the place where the dmk structure gets added
;       ;; to the internal DMK hash table.
;       ;; to compact SEMs, do the following instead of just "push":
;       ;; 1. get previously stored entries for this root
;       ;; 2. if any match in everything except SEM feature, then just
;       ;;    merge the SEM feild of the new entry into the old entry
;       ;; 3. otherwise, push the new entry as before
;       ;;
;       ;; to compare, see dmk structure in file dmk-defs.lisp
;       ;; compare relevant slots: POS, TYPE, SYN-FEATURES, SEM-FEATURES, CLASS
;       
;       (push dmk (eng-dmk-word-entry root))))
;
;    (dmk-type dmk)))

;; ---------------------------------------------------------------------- 
;; 6. Patches to dmk-synlex-loader.lisp
;; 
;;  The function ADD-SYN-LEX-ENTRY has been patched:
;; 
;;    * When adding a new entry, it checks to see if the new entry is
;;      identical with an existing entry except for the CONCEPT; if so,
;;      the entries are merged via *OR* in the CONCEPT field.
;; 
;;  New functions:
;; 
;;    MERGE-OR-PUSH-ENTRY
;;     POS-MATCH-EXACT
;;     MERGE-OR-PUSH-SEM

(defun add-syn-lex-entry (entry)
  (let* ((orig-root (string-downcase (second (assoc :ROOT entry))))
	 (root (preprocess-root-string-maybe orig-root))
	 (dmk (make-dmk))		; New DMK struct
	 )
    (unless (stringp root)
      (warn "~% [DMK syn reader]: errorful :root ~S~%" (assoc :root entry)))

    (set-struct-simple-slots dmk entry)
    (set-struct-syn-features entry dmk root orig-root) 
    (if (position #\SPACE root) 
	(setf (dmk-type dmk) :PHRASE))
    (set-syn-feat-defaults dmk)
    ;; (set-struct-caps entry dmk)   ; use :caps and :irreg-caps slots
        
    (case (dmk-type dmk)
      (:PHRASE
       (set-struct-phrase-head-default dmk root)
       (handle-morph-slot entry dmk root :word-entry nil) 
       (share-syn-features dmk)
       ;; If preference phrase: add <idiom> tags to root, insert irec
       (when (assoc 'PREFER-PHRASE (dmk-syn-features dmk))
	 (phrec-insert *eng-irec* root root T)
	 ;; Move head one over to allow for {idiom}
	 (if (numberp (dmk-phrase-head dmk))
	     (incf (dmk-phrase-head dmk)))
	 (setq root (concatenate 'string "{idiom} " root " {/idiom}")))
       ;; Ignore entry under following conditions
       (unless (eq (dmk-pos dmk) 'GRAM)	  
	 (add-to-phrase-lex dmk root)))

      (otherwise			; DMK entry is a word	  
       (handle-morph-slot entry dmk root)
       (share-syn-features dmk)
       ;; Add dmk structure to dmk hash table

       ;; 13-Sep-95 by amf: this is the place where the dmk structure gets added
       ;; to the internal DMK hash table.
       ;; to compact SEMs, do the following instead of just "push":
       ;; 1. get previously stored entries for this root
       ;; 2. if any match in everything except SEM feature, then just
       ;;    merge the SEM feild of the new entry into the old entry
       ;; 3. otherwise, push the new entry as before
       ;;
       ;; to compare, see dmk structure in file dmk-defs.lisp
       ;; compare relevant slots: POS, TYPE, SYN-FEATURES, SEM-FEATURES, CLASS

       ;; 13-Sep-95 by EHN -- this used to be:
       ;; (push dmk (eng-dmk-word-entry root))
       
       (merge-or-push-entry dmk root)))

    (dmk-type dmk)))

(defun merge-or-push-entry (dmk root)
  (let* ((current (eng-dmk-word-entry root))
	 (merge-point (find dmk current :test #'pos-match-exact)))
    (if merge-point
	(setf (dmk-concept merge-point)
	      (merge-or-push-sem
	       (dmk-concept dmk) (dmk-concept merge-point)))
      (push dmk (eng-dmk-word-entry root)))))

(defun pos-match-exact (e1 e2)
  "Check to make sure the only difference is that the SEM field might differ."
  (and
   (eq (dmk-pos e1) (dmk-pos e2))
   (eq (dmk-type e1) (dmk-type e2))
   (equal (dmk-syn-features e1) (dmk-syn-features e2))
   (equal (dmk-sem-features e1) (dmk-sem-features e2))
   (equal (dmk-class e1) (dmk-class e2))))

(defun merge-or-push-sem (concept1 concept2)
  (cond ((and (listp concept2)
	      (eq '*or* (first concept2)))
	 ;; push.
	 (push concept1 (rest concept2))
	 concept2)
	((symbolp concept2)
	 ;; merge.
	 (list '*or* concept1 concept2))
	(t (error "MERGE-OR-PUSH-SEM: bad arguments: ~s ~s"
		  concept1 concept2))))



;;;==================================================================;

;;; Initialization and finalization

(defun init-lexicon () 
  (warn "~%Clearing syntactic root lexicon.~%")
  ;; clear root syn lex
  (clear-eng-dmk-lex) 
  (warn "~%Clearing cached irregular lexical f-structures.~%")
  (clear-eng-irregular-word-fs-table)
  (warn "~%Clearing irregular word root table.~%")
  (clear-eng-irregular-word-root-table)   
  ;; Clear cache of words that have been parsed into f-structures
  (warn "~%Clearing cached lexical f-structures.~%")
  (clear-word-fs-cache) 
  (warn "~%Setting up phrase recognizer and phrase preference module.~%")
  (setq *eng-phrec* (make-phrec))
  (setq *eng-irec* (make-phrec)))


(defun perform-lex-finalizations ()
  "things that have to be done after all DMK files have been read"
  (warn "~%Trimming phrase & idiom tries.~%")  
  (shrink-phrec *eng-phrec*)
  (shrink-phrec *eng-irec*)
  (warn "~%Switching phrase & idiom tries to static mode -- sorting arc arrays, going to binary search.~%")
  (phrec-static *eng-phrec*)
  (phrec-static *eng-irec*))


;;;==================================================================;

;;; Loading from the DMK files
;;; 
;;; The algorithm for finding real DMK directory:
;;;
;;; 1. If an argument is given to make-syn-lex-from-dmk-dir, then that's it.
;;; 2. Else, if the UNIV environment variable is set, use that.
;;; 3. Else, use *dmk-directory*
;;;
;;; Special case: :hydraulics just loads hydraulics supplementary DMK.

(defun add-final-slash-to-dir (dir)
  "adds trailing slash to directory name, if necessary"
  (cond ((and (stringp dir)
	      (char= (char dir (1- (length dir)))
		     #\/))
	 dir)
	((stringp dir)
	 (concatenate 'string dir "/"))))


(defun make-syn-lex-from-dmk-file (file)
  (let ((eof-marker (gensym)))
    (reset-dots)
    (with-open-file
     (instream file :direction :input)
     (do ((entry (read instream nil eof-marker nil)
		 (read instream nil eof-marker nil)))
	 ((eq entry eof-marker)
	  T)

       (case (add-syn-lex-entry entry)
	 (:PHRASE
	  (print-dot-10 "DMK entries read" :char #\P))
	 (otherwise
	  (print-dot-10 "DMK entries read")))))))


(defun make-syn-lex-from-dmk-dir (&key (directories nil)
				       (init-lexicon t)
				       (finalize t))
  "read all .dmk files in directories, and make syn lex"
  (if init-lexicon (init-lexicon))
  ;; get the real DMK directory, if necessary
  (if (null directories)
      (or #+lucid(setq directories (environment-variable "DMK_DIRECTORY"))
	  (and (boundp '*dmk-directory*)
	       (setq directories *dmk-directory*))
	  (warn "~% [DMK syn reader]: did not specify DMK directory")))

  (dolist (dir (if (listp directories)
		   directories		; then
		 (list directories)))	; else
	  (setq dir (add-final-slash-to-dir dir))
	  ;; 20-Jun-96 by EHN -- bind FILE-LIST and check for null, so
	  ;; we can signal an error if there aren't any DMK files. Old
	  ;; code was placing DIRECTORY call directly as list var in
	  ;; DOLIST, so fell through with no warning if files not found.	  
	  (let ((file-list (directory (concatenate 'string dir "*."
						   *dmk-file-extension*))))
	    (if (null file-list)
		(error "Found no DMK files in directory: ~s" dir))
	    (dolist (file file-list)
		    (format *standard-output* "~% [DMK syn reader]: Reading DMK file ~A~%" 
			    file)
		    (make-syn-lex-from-dmk-file file))))
  (if finalize (perform-lex-finalizations)))
