;;;==================================================================;
;;; -*- Syntax: Common-Lisp; Package: USER; Base: 10; Mode: LISP -*-
;;;==================================================================;
;;;
;;;               Center for Machine Translation
;;;                 Carnegie-Mellon University
;;;                                                                       
;;;------------------------------------------------------------------;
;;;                                                                       
;;; Copyright (c) 1995
;;; Carnegie Mellon University. All Rights Reserved.                      
;;;                                                                       
;;;------------------------------------------------------------------;
;;;
;;;          File: dmk-defs.lisp
;;;  File created: 14-May-92 by amf
;;;        Author: Alex Franz [amf@cs.cmu.edu]
;;; Last Modified:  14-Dec-95, 09:56-Aug-95 at 15:41
;;;
;;;------------------------------------------------------------------;
;;; Description
;;;
;;; Header file for Domain Model Kernel related action.
;;;
;;; See also: dmk-synlex-loader.lisp, morph-analyzer.lisp


;;;==================================================================;

;;; Change Log
;;;
;;;  6-Jun-95 by ndb: Added MAP-ENG-DMK-ENTRIES, MAP-PHRASAL-DMK-ENTRIES,
;;; MAP-IDIOM-DMK-ENTRIES
;;; 
;;;  6-Jun-95 by ndb: Added FIND-ALL-DMK-ENTRIES.


;;;==================================================================;

;;; Package Statements

(in-package :user)


;;;==================================================================;

;;; Structures

(defstruct dmk
  ;; root        ; string "foo"
  concept     ; symbol *O-FOO
  pos         ; symbol V
  type        ; keyword {:WORD :PHRASE}

  ;; :CTE information is included in the syn-features slot

  ;; :CAPS {:FIRST :ALL :ANY} and :IRREG-CAPS is turned into this slot
  ;; capitalization-knowledge  

  ;; syl-double   ; boolean symbol {+ -}

  ;; :MORPH information is included in the syn-features slot

  phrase-head  ; number 2
  syn-features ; list f-structure ((CTE -) (MEISTERFUL &VERY))
  sem-features ; list f-structure ((ASPECT POINT) (ACTIONFUL &NOT))
  class        ; symbol AGENT-THEME
  )


;;;==================================================================;

;;; DMK

(setq *empty-dmk-entry* (make-dmk)) ; for bomb-proofing

(defparameter *dmk-file-extension* "dmk")

;;;------------------------------------------------------------------;
;;; DMK table
;;; string --> dmk structure

(defvar *eng-dmk-table*    
   (make-hash-table :size 5000 :test #'equal))

(defmacro eng-dmk-word-entry (word)
  `(gethash ,word *eng-dmk-table*))

(defmacro eng-dmk-entry (word)
  `(gethash ,word *eng-dmk-table*))  

(defun clear-eng-dmk-lex ()
  (clrhash *eng-dmk-table*))

(defun map-eng-dmk-entries (function)
  "Applies FUNCTION to all single-word DMK entries.  FUNCTION takes two
arguments: the root string and the DMK entry structure."
  (maphash #'(lambda (root entries)
	       (dolist (dmk entries)
		 (funcall function root dmk)))
	   *eng-dmk-table*))


;;;------------------------------------------------------------------;
;;; DMK phrase tries

;;; phrase recognizer -- set up in dmk-synlex-loader.lisp
(defvar *eng-phrec*)

;;; idiom-recognizer -- set up in dmk-synlex-load.lisp
(defvar *eng-irec*)


;;;------------------------------------------------------------------;
;;; DMK phrases are now stored only in phrase trie

(defmacro eng-phrasal-dmk-entry (string)
  `(let ((query-result (phrec-query *eng-phrec* ,string)))
    (if query-result (arcval-dmk-entries query-result))))

(defmacro eng-idiom-dmk-entry (string)
  `(let ((query-result (phrec-query *eng-irec* ,string)))
    (if query-result (arcval-dmk-entries query-result))))


(defun map-phrasal-dmk-entries (function &optional (trie *eng-phrec*))
  "Applies FUNCTION to all phrasal DMK entries in TRIE.  FUNCTION takes two
arguments: the root string and the DMK entry structure."
  (map-stm #'(lambda (arc)
	       (let ((arcval (stm-arc-value arc)))
		 (when arcval
		   (dolist (dmk (arcval-dmk-entries arcval))
		     (funcall function (arcval-key arcval) dmk)))))
	   trie))

(defun map-idiom-dmk-entries (function &optional (trie *eng-irec*))
  "Applies FUNCTION to all idiom DMK entries in TRIE.  FUNCTION takes two
arguments: the root string and the DMK entry structure."
  (map-phrasal-dmk-entries function trie))


;;;------------------------------------------------------------------;

(defun find-all-dmk-entries (root &key (pos nil) (concept nil))
  "Returns a list of all single-word and phrasal DMK entries for ROOT which have
POS (if given) and/or CONCEPT (if given)."
  (and (stringp root)
       (let ((entries (nconc (copy-list (eng-dmk-entry root))
			     (copy-list (eng-phrasal-dmk-entry root)))))
	 (when pos
	   (setf entries (delete pos entries :key #'dmk-pos :test-not #'eq)))
	 ;; 14-Dec-95 by EHN -- This test must take into account that the
	 ;; CONCEPT field in the dmk entry might contain an *OR* -- this is
	 ;; the case for AMBIGTRUE entries. Changed :test-not from #'eq
	 (when concept
	   (setf entries
		 (delete concept entries
			 :key #'dmk-concept
			 :test-not #'(lambda (x y)
				       (or (eq x y)
					   (and (listp y)
						(member x y)))))))
	 entries)))


;;;==================================================================;

;;; Cache of f-structures for morphologically analyzed words

(defvar *use-word-fs-cache* nil)
(defvar *word-fs-cache-default-size* 100)

(defvar *word-fs-cache*
  (create-cache :size *word-fs-cache-default-size* :test #'equal)
  "Cache of f-structures for morphologically analyzed words")

(defun clear-word-fs-cache ()
  (clear-cache *word-fs-cache*))    


(defun use-word-fs-cache (&optional (arg nil arg-supplied))
  "Sets *USE-WORD-FS-CACHE* flag to ARG; if ARG is not supplied, toggles current
value."
  (setf *use-word-fs-cache*
	(if arg-supplied
	    arg
	  (not *use-word-fs-cache*))))

(defun set-word-fs-cache-size (size)
  "Sets the size parameter of the word FS cache to SIZE: the hash table is
allowed to grow to this size."
  (setf (cache-size *word-fs-cache*) size))


(defmacro word-fs-cache-get (word)
  `(cache-get *word-fs-cache* ,word))

(defmacro word-fs-cache-add (word parse)
  `(cache-add *word-fs-cache* ,word ,parse))

(defmacro word-fs-cache-remove (word)
  `(cache-remove *word-fs-cache* ,word))


;;;==================================================================;

;;; Cache of f-structures for irregular words

(defvar *eng-irregular-word-fs-table*
  (make-hash-table :size 300 :test #'equal))

(defmacro eng-irregular-word-fs (word)
  `(gethash ,word *eng-irregular-word-fs-table*))    

(defun clear-eng-irregular-word-fs-table ()
  (clrhash *eng-irregular-word-fs-table*))

;; table mapping irreg words to list of roots
;; e.g. "brought" -> ("bring")

(defvar *eng-irregular-word-root-table*
  (make-hash-table :size 300 :test #'equal))

(defmacro eng-irreg-word-root (word)
  `(gethash ,word *eng-irregular-word-root-table*))

(defun clear-eng-irregular-word-root-table ()
  (clrhash *eng-irregular-word-root-table*))


;;;==================================================================;

;;; Functions

;;; pjordan modified to return all dmk entries for string and cat

(defun eng-dmk-entry-cat (string cat)
  (let ((entries (if (position #\SPACE string)
		     (eng-phrasal-dmk-entry string)
		     (eng-dmk-entry string))))

    (remove-if-not #'(lambda (val)
		       (eq val cat))
		   entries :key #'(lambda (entry)
				    (dmk-pos entry)))))

    
;    (find cat entries :key #'(lambda (entry)
;			       (dmk-pos entry))
;	  :test #'eq)))
	  

