;;;==================================================================;
;;;  -*- Syntax: Common-Lisp; Package: USER; Base: 10; Mode: LISP -*-
;;;==================================================================;
;;;
;;;               Center for Machine Translation
;;;                 Carnegie-Mellon University
;;;                                                                       
;;;------------------------------------------------------------------;
;;;                                                                       
;;; Copyright (c) 1994
;;; Carnegie Mellon University. All Rights Reserved.                      
;;;                                                                       
;;;------------------------------------------------------------------;
;;;
;;;          File: domo.lisp
;;;  File created: Wed Dec 14 1994 by ndb
;;;        Author: Nicholas Brownlow <ndb@cs.cmu.edu>
;;; Last Modified: 22-Jun-95 at 19:08
;;;   
;;;------------------------------------------------------------------;
;;; File Description:
;;;
;;; Basic Domain Model code.


;;;==================================================================;
;;; Documentation

;;;==================================================================;
;;; Change Log
;;;

;;;==================================================================;
;;; To Do
;;;
;;; Storage should be: list or vector.  Forget fill vectors.

;;;==================================================================;

;;; Package Statements

(in-package :user)

;;;==================================================================;

;;; Feedback

(defun domo-error-record (target-bad role-bad filler-bad target role filler)
  "Returns a DoMo error record."
  (let ((error (nconc (and target-bad (list :TARGET))
		      (and role-bad (list :ROLE))
		      (and filler-bad (list :FILLER))))
	(triple (list target role filler)))
    (and error (list error triple triple))))


(defparameter *domo-message* (list :BUILD-ERROR :BUILD-WARNING
				   :ERROR :WARNING :MESSAGE)
  "List of DoMo message types toggled on")

(defun domo-message-toggle (&rest types)
  "Toggle DoMo message TYPES on/off."
  (dolist (type types)
    (setf *domo-message*
	  (if (member type *domo-message* :test #'eq)
	      (delete type *domo-message* :test #'eq)
	    (cons type *domo-message*))))
  *domo-message*)

(defun domo-message (type string &rest args)
  "Print DoMo message of TYPE, if TYPE is toggled on."
  (when (member type *domo-message* :test #'eq)
    (format t "~&;;; [DoMo-~A] " type)
    (apply #'format t string args)
    (terpri))
  nil)

(defun domo-quiet ()
  "Turn off all DoMo messages."
  (setf *domo-message* nil)) 


;;;==================================================================;

;;; DOMOD record structure

(defstruct (domod (:print-function domod-pretty-print))
  "DOmain MOdel Data:
SYMBOL - concept symbol
INDEX  - unique number (for fast searches)
SOURCE - where the symbol comes from: :DMK (and DM) or just :DM
IS-A   - inheritance information
LINKS  - links to other symbols (list of DOMOD-TO records)"
  (symbol nil :type symbol)
  (index 0 :type integer)
  (source nil :type symbol)
  (is-a nil :type list)
  (links nil :type (or list array)))


;;;==================================================================;

;;; DoMo source code authority precedence
;;;
;;; The various DoMo record source codes are ordered by authority.  When
;;; DOMO-ADD-SYMBOL is called on a symbol already existing in the DoMo, and
;;; the source code passed to the function is different from the source code
;;; in the existing record, the code with higher authority is put in the
;;; record.  (If UPDATE is T, the new code is put in the record regardless of
;;; authority.)

(defparameter *domo-source-precedence*
  '(:DMK :NON-DMK :DMK-PHRASE-HEAD :DMK-DISAMBIG :DM))

(defun domo-source-precedes (source0 source1)
  "Returns T iff SOURCE0 has higher precedence than SOURCE1."
  (member source1 (rest (member source0 *domo-source-precedence* :test #'eq))
	  :test #'eq))


;;;==================================================================;

;;; DOMOD-TO records: target DOMOD + linking roles

(defun make-domod-to (domod &rest roles)
  "Makes and returns a new DOMOD-TO record with filler DOMOD and link ROLES."
  (cons domod roles))

(defun domod-to-domod (domod-to)
  "Returns the filler domod from the DOMOD-TO record."
  (first domod-to))

(defmacro domod-to-roles (domod-to)
  "Returns the link roles from the DOMOD-TO record."
  `(rest ,domod-to))

(defmacro domod-to-role (domod-to)
  "Returns the first link role from the DOMOD-TO record."
  `(second ,domod-to))


(defun domod-find-domod-to (domod to-domod)
  "Searches DOMOD's links list for the DOMOD-TO record whose filler matches TO-DOMOD."
  (find to-domod (domod-links domod) :test #'eq :key #'domod-to-domod))


;;;------------------------------------------------------------------;

;;; Additional DOMOD record functions

(defun domod-is-a-list (domod)
  "Returns a list of the IS-A symbols for DOMOD."
  (map 'list #'domod-symbol (domod-is-a domod)))


(defun domod-has-links-p (domod)
  "Returns T iff DOMOD has links."
  (plusp (length (domod-links domod))))


(defun domod-links-list (domod)
  "Returns a list of the link data for DOMOD.  Each element is a list whose
first element is the filler symbol and whose tail contains the role symbols."
  (map 'list #'(lambda (domod-to)
		 (cons (domod-symbol (domod-to-domod domod-to))
		       (domo-roles-symbols (domod-to-roles domod-to))))
       (domod-links domod)))


(defun domod-links-frame (domod)
  "Returns a list of the link data for DOMOD.  Each element is a list whose
first element is the role symbol and whose tail contains the filler symbols."
  (let ((slots nil)
	slot
	filler
	role-symbol)
    (map nil #'(lambda (domod-to)
		 (setf filler (domod-symbol (domod-to-domod domod-to)))
		 (dolist (role (domod-to-roles domod-to))
		   (setf role-symbol (domo-role-symbol role)
			 slot (or (assoc role-symbol slots :test #'eq)
				  (first (push (list role-symbol) slots))))
		   (pushnew filler (rest slot) :test #'eq)))
	 (domod-links domod))
    slots))


(defun domod-pretty-print (domod &optional (stream *standard-output*) depth)
  "Pretty-prints DOMOD onto STREAM."
  (declare (ignore depth))
  (format stream "<~S [~D] from ~S"
	  (domod-symbol domod) (domod-index domod) (domod-source domod))
  (when (domod-is-a domod)
    (format stream "~%  IS-A ~S"
	    (domod-is-a-list domod)))
  (when (plusp (length (domod-links domod)))
    (format stream "~%  LINKS ~S"
	    (domod-links-list domod)))
  (format stream ">~%"))


;;;==================================================================;

;;; Primitive retrieval functions

(defmacro domo-get (symbol)
  "Gets the DOMOD for SYMBOL."
  `(get ,symbol :DOMO)) 

(defmacro domo-get-index (symbol)
  "Gets the index for SYMBOL."
  `(domod-index (domo-get ,symbol)))

(defmacro domo-get-source (symbol)
  "Gets the source code for SYMBOL."
  `(domod-source (domo-get ,symbol)))

(defmacro domo-get-is-a (symbol)
  "Gets the IS-A list for SYMBOL."
  `(domod-is-a (domo-get ,symbol)))

(defmacro domo-get-links (symbol)
  "Gets the links for SYMBOL."
  `(domod-links (domo-get ,symbol)))


;;;==================================================================;

;;; Map over DOMO

(defun domo-map (function)
  "Maps FUNCTION over all DOMO symbols.  FUNCTION takes one argument: the
symbol's DOMOD."
  (let (domod)
    ;; This is OK because DM symbols account for nearly 90% of user
    ;; package symbols
    (do-symbols (symbol :user)
      (when (setf domod (domo-get symbol))
	(funcall function domod)))))

(defun domo-pretty-print (&optional (stream *standard-output*))
  (domo-map #'(lambda (domod)
		(domod-pretty-print domod stream))))


;;;==================================================================;

;;; Add symbols to DOMO

(defvar *domo-n* 0)

(defun domo-add-symbol (symbol &key (source nil) (update nil))
  "Adds SYMBOL to DOMO from SOURCE (default :UNKNOWN), creating DOMOD for SYMBOL
if not already present, and updating source to SOURCE if UPDATE is non-nil or
if SOURCE has higher precedence than current source.  Returns DOMOD for
SYMBOL."
  (setf source (or source :UNKNOWN))
  (let ((domod (domo-get symbol)))
    (if domod
	(when (or update
		  (domo-source-precedes source (domod-source domod)))
	  (setf (domod-source domod) source))
      (progn
	(setf domod
	      (setf (domo-get symbol)
		    (make-domod :symbol symbol
				:index *domo-n*
				:source source)))
	(incf *domo-n*)))
    domod))

(defun domo-remove-symbol (symbol &optional domod)
  (declare (ignore domod))
  (remprop symbol :DOMO)) 


(defun domo-clear ()
  (domo-map #'(lambda (domod)
		(domo-remove-symbol (domod-symbol domod))))
  (setf *domo-n* 0))


;;;==================================================================;

;;; Add DoMo links

(defun domod-add-link (domod role to-domod)
  "Adds ROLE link from DOMOD to TO-DOMOD.  If TO-DOMOD is already on DOMOD's
links list, adds ROLE to the TO-DOMOD link; otherwise adds a new link."
  (let ((domod-to (domod-find-domod-to domod to-domod)))
    (if domod-to
	(pushnew role (domod-to-roles domod-to) :test #'eq)
      (push (make-domod-to to-domod role) (domod-links domod)))))

(defun domod-add-is-a (domod to-domod)
  "Adds TO-DOMOD to IS-A list for DOMOD.  If not already present, adds to end to
preserve ordering."
  (unless (member to-domod (domod-is-a domod) :test #'eq)
    (setf (domod-is-a domod) (nconc (domod-is-a domod) (list to-domod)))))

(defun domo-add-link (symbol role-symbol to-symbol &key (source nil))
  "If ROLE-SYMBOL is a valid role, adds ROLE-SYMBOL link from SYMBOL to
TO-SYMBOL.  Adds SYMBOL and TO-SYMBOL to DoMo, with SOURCE, if they aren't
already present.  Returns DOMOD for SYMBOL."
  (let ((domod (domo-add-symbol symbol :source source))
	(to-domod (domo-add-symbol to-symbol :source source))
	role)
    (case role-symbol
      (is-a (domod-add-is-a domod to-domod)) ; IS-A links stored separately
      (otherwise (when (setf role (domo-symbol-role role-symbol))
		   ;; ROLE-SYMBOL is associated with a role
		   (domod-add-link domod role to-domod))))
    domod)) 


;;;==================================================================;

;;; Remove DoMo links by target

(defun domod-remove-link (domod role to-domod)
  "Removes ROLE link from DOMOD to TO-DOMOD."
  (let ((domod-to (domod-find-domod-to domod to-domod)))
    (when domod-to
      ;; Delete the role
      (setf (domod-to-roles domod-to)
	    (delete role (domod-to-roles domod-to) :test #'eq))
      ;; Delete the DOMOD-TO record if empty of roles
      (unless (domod-to-roles domod-to)
	(setf (domod-links domod)
	      (delete to-domod (domod-links domod) :test #'eq :key #'domod-to-domod))))))

(defun domod-remove-is-a (domod to-domod)
  "Removes TO-DOMOD from the IS-A list for DOMOD." 
  (setf (domod-is-a domod)
	(delete to-domod (domod-is-a domod) :test #'eq)))

(defun domo-remove-link (symbol role-symbol to-symbol &key (source nil))
  "If ROLE-SYMBOL is a valid role, removes ROLE-SYMBOL link from SYMBOL to
TO-SYMBOL.  Adds SYMBOL and TO-SYMBOL to DoMo, with SOURCE, if they aren't
already present.  Returns DOMOD for SYMBOL."
  (let ((domod (domo-add-symbol symbol :source source))
	(to-domod (domo-add-symbol to-symbol :source source))
	role)
    (case role-symbol
      (is-a (domod-remove-is-a domod to-domod))	; IS-A links stored separately
      (otherwise (when (setf role (domo-symbol-role role-symbol))
		   ;; ROLE-SYMBOL is associated with a role
		   (domod-remove-link domod role to-domod))))
    domod)) 


;;;==================================================================;

;;; Remove DoMo links by role

(defun domod-remove-role (domod role)
  "Removes all ROLE links from DOMOD."
  (let* ((modify-links (cons nil (domod-links domod)))
	 (domod-tos modify-links)
	 domod-to)
    (loop
     (unless (rest domod-tos)
       (return))
     ;; Remove the role from the roles for this filler
     (setf domod-to (second domod-tos)
	   (domod-to-roles domod-to) (delete role (domod-to-roles domod-to) :test #'eq))
     (if (domod-to-roles domod-to)
	 (setf domod-tos (rest domod-tos))
       ;; If there are no roles for this filler, remove the filler
       (setf (rest domod-tos) (rest (rest domod-tos)))))
    (setf (domod-links domod) (rest modify-links)))) 

(defun domo-remove-role (symbol role-symbol &key (source nil))
  "If Removes all ROLE links from SYMBOL.  Adds SYMBOL to DoMo, with SOURCE, if not
already present.  Returns domod for SYMBOL."
  (let ((domod (domo-add-symbol symbol :source source))
	role)
    (case role-symbol
      (is-a (setf (domod-is-a domod) nil)) ; IS-A links stored separately
      (otherwise (when (setf role (domo-symbol-role role-symbol))
		   ;; ROLE-SYMBOL is associated with a role
		   (domod-remove-role domod role))))
    domod))


;;;==================================================================;

;;; Save and load DOMOD records
;;;
;;; <domod-save> ::= "(" <symbol> <source> <is-a> <link-to> ")"
;;; <is-a>       ::= "(" <parent-symbol>* ")"
;;; <link-to>    ::= "(" <to-symbol> <role-symbol>* ")"

(defun domod-save (domod &optional (stream nil))
  "Converts DOMOD to saveable format.  Prints it to STREAM if non-nil.
Returns saved DOMOD."
  (let ((domod-save (nconc (list (domod-symbol domod)
				 ;; Don't save the index
				 (domod-source domod)
				 (domod-is-a-list domod))
			   (domod-links-list domod))))
    (when stream
      (prin1 domod-save stream)
      (terpri stream))
    domod-save))


(defun domod-fill-is-a (domod is-a)
  "Fills IS-A list in DOMOD with data in saved IS-A."
  (setf (domod-is-a domod)
	(map 'list #'domo-add-symbol is-a)))

(defun domod-fill-links (domod links)
  "Fills links array in DOMOD with data in saved LINKS."
  (let (roles)
    (setf (domod-links domod)
	  (mapcan #'(lambda (link)
		      ;; Convert symbols to valid roles; don't add link if none
		      (and (setf roles (delete nil (domo-symbols-roles (rest link))))
			   (list (apply #'make-domod-to
					(domo-add-symbol (first link)) roles))))
		  links))))

(defun domod-load (domod-save)
  "Adds data in DOMOD-SAVE to DOMO.  Returns DOMOD for DOMOD-SAVE.  Overwrites
any existing data in DOMOD."
  (let ((domod (domo-add-symbol (first domod-save)
				;; Make sure source is updated; DOMOD may have
				;; been previously added with default source
				:source (second domod-save) :update t)))
    (domod-fill-is-a domod (third domod-save))
    (domod-fill-links domod (nthcdr 3 domod-save))
    domod))


;;;==================================================================;

;;; Save and load DOMOD records as frames.  Validate DOMO frames
;;;
;;; <frame> ::= "(" <symbol> <slot>* ")"
;;; <slot>  ::= "(" <role> <symbol>* ")"

(defun domod-save-frame (domod &optional (stream nil))
  "Converts DOMOD to frame format.  Prints it to STREAM if non-nil.  Returns
DOMOD frame."
  (let ((domod-frame (nconc (list (domod-symbol domod))
			    (and (domod-is-a domod)
				 (list (cons 'IS-A (domod-is-a-list domod))))
			    (domod-links-frame domod))))
    (when stream
      (prin1 domod-frame stream)
      (terpri stream))
    domod-frame))

(defun domod-load-frame (domod-frame &key (mode :ADD) (source nil))
  "Adds data in DOMOD-FRAME to DOMO.  Returns DOMOD for DOMOD-FRAME."
  (let ((domod (domo-add-symbol (first domod-frame) :source source)))
    (dolist (slot (rest domod-frame))
      (when (eq mode :REPLACE)
	;; Remove old role fillers before adding new
	(domo-remove-role (first domod-frame) (first slot)))
      (dolist (filler (rest slot))
	(domo-add-link (first domod-frame) (first slot) filler :source source)))
    domod))


;;;------------------------------------------------------------------;

(defun domo-validate-frame (domod-frame &key (sources '(:DMK :NON-DMK)))
  "Checks symbols and roles in DOMOD-FRAME.  Returns list of error records.
SOURCE is a list of valid source codes for concepts in the frame to be
checked."
  (let* (domod
	 target-bad role-bad filler-bad
	 (errors nil)
	 error)
    ;; Check target concept
    (setf domod (domo-get (first domod-frame))
	  target-bad (or (not domod)
			 (not (member (domod-source domod) sources :test #'eq))))
    (dolist (slot (rest domod-frame))
      ;; Check each slot role
      (setf role-bad (not (domo-symbol-role (first slot))))
      (dolist (filler (rest slot))
	;; Check each filler concept
	(setf domod (domo-get filler)
	      filler-bad (or (not domod)
			     (not (member (domod-source domod) sources :test #'eq))))
	(when (setf error (domo-error-record target-bad role-bad filler-bad
					     (first domod-frame) (first slot) filler))
					    
	  (push error errors))))
    errors))


;;;==================================================================;

;;; Save and load DOMOD IS-A lists
;;;
;;; <domod-is-a> ::= "(" <concept> <parent-concept>* [ "?" <note>* ] ")"

(defun domod-save-is-a (domod &optional (stream nil))
  "Converts DOMOD IS-A list to saveable format.  Prints it to STREAM if non-nil.
Returns formatted DOMOD IS-A list."
  (let ((domod-is-a (cons (domod-symbol domod)
			  ;; Don't save the index
			  (domod-is-a-list domod))))
    (when stream
      (prin1 domod-is-a stream)
      (terpri stream))
    domod-is-a))

(defun domod-load-is-a (domod-is-a &key (mode :ADD) (source nil))
  "Adds data in DOMOD-IS-A to DoMo.  Creates new concepts as needed with
SOURCE.  Returns DOMOD."
  (let ((domod (domo-add-symbol (first domod-is-a) :source source)))
    (when (eq mode :REPLACE)
      (domo-remove-role (first domod-is-a) 'IS-A))
    (dolist (parent (rest domod-is-a))
      (if (eq parent '?)
	  ;; Skip everything after a ?
	  (return)
	(domo-add-link (first domod-is-a) 'IS-A parent :source source)))
    domod))


(defun validate-is-a (form)
  (unless (and (listp form)
	       (symbolp (first form))
	       (every #'(lambda (item)
			  (or (stringp item)
			      (symbolp item)))
		      (rest form)))
    (format t "Ill-formed: ~S~%" form)))


(defun clean-is-a (form)
  (let ((clean-form nil))
    (dolist (term form)
      (cond ((eq term '?)
	     ;; Skip everything after a ?
	     (return))
	    ((symbolp term)
	     ;; Copy symbols, skipping strings
	     (push term clean-form))))
    (nreverse clean-form)))

(defun parse-is-a (form)
  "Parses IS-A FORM.  Returns two values: form with comments cleaned out;
comment list."
  (let ((clean-form nil)
	(comments nil)
	(state :START))
    (dolist (term form)
      (setf state
	    (case state
	      (:START
	       (cond ((eq term '?)
		      :COMMENT)
		     ((symbolp term)
		      (push term clean-form)
		      :START)
		     (t
		      (push term comments)
		      :START)))
	      (:COMMENT
	       (push term comments)
	       :COMMENT))))
    (values (nreverse clean-form)
	    (nreverse comments))))


;;;==================================================================;

;;; Save and load DOMO

(defun domo-save (file &key (format :DOMO))
  "Saves DOMO to FILE in FORMAT: :DOMO, :FRAME, or IS-A."
  (unless (member format '(:DOMO :FRAME :IS-A))
    (error "Bad DOMO format ~S" format))
  (with-open-file
   (stream file :direction :output :if-exists :rename-and-delete)
   (format t ";;; Saving DoMo to ~A file ~S~%" format file)
   (domo-map (case format
	       (:DOMO #'(lambda (domod)
			  (domod-save domod stream)))
	       (:FRAME #'(lambda (domod)
			   (domod-save-frame domod stream)))
	       (:IS-A #'(lambda (domod)
			  (domod-save-is-a domod stream)))))))

(defun domo-load (file &key (format :DOMO) (mode :ADD) (source nil))
  "Loads DOMO from FILE.
FORMAT -- :DOMO, :FRAME, or :IS-A;
MODE   -- :ADD or :REPLACE;
SOURCE is source for all new symbols in :FRAME format."
  (unless (member format '(:DOMO :FRAME :IS-A))
    (error "Bad DOMO format ~S" format))
  (unless (member mode '(:ADD :REPLACE))
    (error "Bad DOMO load mode ~S" format))
  (let (form)
    (with-open-file
     (stream file :direction :input)
     (format t ";;; Loading DoMo (mode ~S, source ~S)~%" mode source)
     (format t ";;; from ~A file ~S~%" format file)
     (loop
      (when (eq :EOF (setf form (read stream nil :EOF)))
	(return))
      (case format
	(:DOMO (domod-load form))
	(:FRAME (domod-load-frame form :mode mode :source source))
	(:IS-A (domod-load-is-a form :mode mode :source source)))))
    t)) 


;;;==================================================================;

;;; Validate DoMo file

(defvar *domo-errors* nil "List of DoMo error records")

(defun domo-validate-frames (file error-file &key
				  (sources '(:DMK :NON-DMK)))
  "Validates frames in FILE.  Sets *DOMO-ERRORS* to error list, returns number
of errors, outputs errors to ERROR-FILE if non-nil.  SOURCES is a list of valid
source codes for concepts in the frames to be checked."
  (let (form
	(errors nil))
    
    (with-open-file
     (stream file :direction :input)
     (loop
      (when (eq :EOF (setf form (read stream nil :EOF)))
	(return))
      (setf errors (nconc errors (domo-validate-frame form :sources sources)))))

    (when (and errors error-file)
      (with-open-file
       (stream error-file :direction :output :if-exists :rename-and-delete)
       (dolist (error errors)
	 (write error :stream stream :circle nil)
	 (terpri stream))))

    (setf *domo-errors* errors)
    (length errors))) 


;;;==================================================================;
