// CityTempService.java


package edu.cmu.aura.ps.examples;


import edu.cmu.aura.service.*;
import edu.cmu.aura.service.query.*;
import java.rmi.RemoteException;
import java.util.*;


/**
 * CityTempService is a sample "raw" service that receives
 * queries from clients and computes results only
 * when a query is received.  CityTempService only allows
 * the simplest of expressions roughly "city = x".
 * These are known as "key equality expressions".
 * Simple service provides a method than can sanity check
 * these types of queries and throw an exception if
 * there is an error in the query (e.g. if a more complicated 
 * expression is received, a non-existent attribute name is 
 * is specified, etc.)
 * 
 * Note that this class is compiled into aurasvc.jar so 
 * modifications will not take effect unless you take measures
 * such as changing the package or class name.
 * 
 * @author	Glenn Judd
 * @date	10/22/2001
 */
public class CityTempService extends SimpleService {
	protected String[]				cityNames;
	protected Attribute[]			nameAttribs;
	protected DynamicAttribute[]	tempAttribs;
	protected Random				rand;
	
	public static final String	SERVICE_NAME = "CityTemp";
	
	protected static final double	MIN_TEMP = 32.0;
	protected static final double	MAX_TEMP = 90.0;
	protected static final String	TEMP_ATTRIB_NAME = "temp";
	protected static final String	NAME_ATTRIB_NAME = "name";
	protected static final String	KEY_ATTRIB = NAME_ATTRIB_NAME;
	
	/**
	 * Creates the CityTempService.
	 * @param cityNames	city names to report temperatures for
	 * @param port	port on which to listen for requests
	 */
	public CityTempService(String[] cityNames, int port) throws Exception {
		super(SERVICE_NAME, port);
		this.cityNames = cityNames;
		System.out.println("CityTemp service listening on port: "+ port);
		
		// we now set the SimpleService attribNames that will 
		// be used for sanity checking incoming queries
		attribNames = new String[2];
		attribNames[0] = NAME_ATTRIB_NAME;
		attribNames[1] = TEMP_ATTRIB_NAME;
		
		// we now create attributes for the cities
		// names don't change so they are regular static attributes
		// while temperatures do change and may have
		// uncertainty associated with them so they are dynamic
		// attributes
		nameAttribs = new Attribute[cityNames.length];
		tempAttribs = new DynamicAttribute[cityNames.length];
		for (int i = 0; i < cityNames.length; i++) {
			System.out.println(cityNames[i]);
			nameAttribs[i] = new Attribute(NAME_ATTRIB_NAME, cityNames[i]);
			tempAttribs[i] = new DynamicAttribute(TEMP_ATTRIB_NAME, null);
		}
		
		// since we won't actually be measuring temparatures we
		// initialize an instance Random for returning random temperatures
		rand = new Random();
	}
	
	/**
	 * Handle incoming queries from clients.
	 * @param sAttribs	the attribs to select.
	 * @param serviceNames		names of services to retrieve 
	 * attributes from.
	 * @param expression	expression selecting which entities or
	 * relations in the service should be returned.
	 * @param attribReqs	requirements on dynamic attributes.
	 * @param timeLimit	this is used by clients to inform the service
	 * on when they need the results by.  This may also be viewed as
	 * a suggestion on how hard to work to retrieve the results.
	 * @return	the QueryResult of the query.
	 */
	public QueryResult query(ArrayList sAttribs, 
									ArrayList serviceNames,
									ExpressionNode expression,
									ArrayList attribReqs,
									long timeLimit)
									throws ServiceException {
		SimpleEqualityQuery	query;
		QueryResult			result;
		
		// the below line is a hack to cause this service to
		// ignore attribute requirements. Normally the 
		// processSimpleEqualityQuery() routine would throw an
		// error when attribute requirements are specified and
		// we don't support the ones specified (or any at all)
		// However, for the example cityTempClient2 to work
		// we simply ignore them attribReqs
		attribReqs = null; // Probably don't want do this in a real service
		
		// sanity check the query and return an object
		// that represents the query
		// this will also handle the case where all attributes
		// are selected by actually inserting the names of
		// all attributes into the list stored in query
		query = processSimpleEqualityQuery(sAttribs, serviceNames,
							expression, attribReqs,
							timeLimit, 
							null,			// attribute requirements are not 
											// allowed for any attributes
							ReqUtil.R_NONE, // no types of attribute
											// requirements may be specified
							KEY_ATTRIB,		// the key for this key equality 
											// expression
							java.lang.String.class); // the datatype for this
													 // key equality expression
		
		// check to see if they are using a wildcard to
		// select all cities
		// this check is actually not useful in this case
		// because simple equality queries disallow the
		// wildcard, but the code is here to show how
		// a wildcard could be handled
		if (expression != PrimitiveService.SELECT_ALL) {
			String		cityName;
			int			cityIndex;
			
			// find which city the expression is selecting
			cityName = (String)query.getEqExpressionVal();
			result = new QueryResult();
			result.setTime( System.currentTimeMillis() );
			result.setComplete(true);
			cityIndex = -1;
			for (int i = 0; i < cityNames.length; i++) {
				if ( cityNames[i].equals(cityName) ) {
					cityIndex = i;
					break;
				}
			}
			if (cityIndex < 0) {
				throw new RequestException(RequestException.ILLEGAL_VALUE, 
									"no such city: "+ cityName);
			}
			// note that below we do not access the sAttribs list passed
			// in, but refer to the value in the query since
			// it will have handled the wildcard case
			addCityToResult( result, cityIndex, query.getSAttribs() );
			return result;
		} else {
			result = new QueryResult();
			result.setTime( System.currentTimeMillis() );
			result.setComplete(true);
			for (int i = 0; i < cityNames.length; i++) {
				addCityToResult(result, i, sAttribs);
			}
			return result;
		}
	}

	/**
	 * Adds a city the the QueryResult.
	 * @param result	QueryResult to add the city to
	 * @param cityIndex	index of the city to add
	 * @param sAttribs	attribs to be returned
	 */
	protected void addCityToResult(QueryResult result, int cityIndex,
								   ArrayList sAttribs) {
		ArrayList	attribSet;
		
		// we add attributes to the result in the order
		// they are specified by the client
		attribSet = new ArrayList();
		for (int j = 0; j < sAttribs.size(); j++) {
			String	sAttrib;
							
			sAttrib = (String)sAttribs.get(j);
			if ( sAttrib.equals(NAME_ATTRIB_NAME) ) {
				attribSet.add(nameAttribs[cityIndex]);
			} else if ( sAttrib.equals(TEMP_ATTRIB_NAME) ) {
				double	randomTemp;
								
				randomTemp = Math.abs( rand.nextDouble() ) 
							 * (MAX_TEMP - MIN_TEMP) + MIN_TEMP;
				tempAttribs[cityIndex].setValue( new Double(randomTemp) );
				tempAttribs[cityIndex].setLastUpdate( 
								System.currentTimeMillis() );
				attribSet.add(tempAttribs[cityIndex]);
			} else {
				// this line should not be executed becuase
				// processKeyEqualityExpression() will check
				// to make sure they have passed in a valid
				// attribute
				throw new RuntimeException("panic");
			}
		}
		result.addAttribSet(attribSet);
	}
	
	/**
	 * Accept command line parameters and run the example.
	 * @param args	command line parameters
	 */
	public static void main(String[] args) {
		try {
			String[]		cityNames;
			int				port;
			CityTempService	cityTempService;
			
			if (args.length < 2) {
				System.out.println("args: <port> <cityName>...");
				return;
			}
			cityNames = new String[args.length - 1];
			for (int i = 0; i < cityNames.length; i++) {
				cityNames[i] = args[i + 1];
			}
			port = Integer.parseInt(args[0]);
			cityTempService = new CityTempService(cityNames, port);
		} catch (Exception e) {
			e.printStackTrace();
		}
	}
}
