// GenericQuery.java


package edu.cmu.aura.ps.examples;


import edu.cmu.aura.service.*;
import edu.cmu.aura.service.rpc.*;
import edu.cmu.aura.service.query.*;
import edu.cmu.aura.service.query.parser.*;
import java.awt.*;
import java.awt.event.*;
import java.util.ArrayList;
import javax.swing.*;


/**
 * GenericQuery is a client that can query any service.
 * This class uses the QParser to allow users to
 * input SQL-like commands to an arbitrary service.
 * 
 * Note that this class is compiled into aurasvc.jar so 
 * modifications will not take effect unless you take measures
 * such as changing the package or class name.
 * 
 * @author	Glenn Judd
 * @date	10/22/2000
 */
public class GenericQuery extends JPanel implements WindowListener, 
											ActionListener  {
	protected String				svcHostName;
	protected int					svcPort;
	protected QParser				qParser;
	protected JTabbedPane			tabbedPane;
	protected JScrollPane			scrollPane;
	protected ResultPanel			resultPanel;
	protected JButton				execButton;
	protected RemoteReference		pServiceRef;
	protected PrimitiveServiceStub	pServiceStub;
	protected QueryPanel[]			queryPanels;
	protected ArrayList				serviceNames;

	/**
	 * Creates a GenericQuery instance.
	 * @param hostName	the hostName on which the service resides
	 * @param port	the port on which the service is listening
	 */
	public GenericQuery(String svcHostName, int svcPort) {
		this.svcHostName = svcHostName;
		this.svcPort = svcPort;
		qParser = new QParser();
		
		pServiceRef = new RemoteReference(svcHostName, svcPort);
		pServiceStub = new PrimitiveServiceStub(pServiceRef);		
		init();
	}
	
	/**
	 * Initialize gui.
	 */
	protected void init() {
		GridBagLayout		gridBag;
		GridBagConstraints	c;
		
		gridBag = new GridBagLayout();
		c = new GridBagConstraints();
		setLayout(gridBag);		

		initQueryPanels();
		
		c.gridx = 0;
		c.gridy = 0;
		c.gridwidth = 10;
		c.gridheight = 2;
		c.weightx = 10.0;
		c.weighty = 2.0;
		c.fill = GridBagConstraints.BOTH;
		gridBag.setConstraints(tabbedPane, c);
		add(tabbedPane);
		
		execButton = new JButton("Execute Query");
		execButton.addActionListener(this);
		execButton.setActionCommand("execQuery");
		c.gridx = 0;
		c.gridy = 2;
		c.gridwidth = 4;
		c.gridheight = 1;
		c.weightx = 4.0;
		c.weighty = 1.0;
		c.fill = GridBagConstraints.NONE;
		gridBag.setConstraints(execButton, c);
		add(execButton);
		
		resultPanel = new ResultPanel();
		scrollPane = new JScrollPane(resultPanel);
		c.gridx = 0;
		c.gridy = 3;
		c.gridwidth = 10;
		c.gridheight = 7;
		c.weightx = 10.0;
		c.weighty = 10.0;
		c.fill = GridBagConstraints.BOTH;
		gridBag.setConstraints(scrollPane, c);
		add(scrollPane);
	}
	
	/**
	 * GUI QueryPanel initialization.
	 */
	protected void initQueryPanels() {		
		tabbedPane = new JTabbedPane();
		
		queryPanels = new QueryPanel[1];
		queryPanels[0] = new GenericQueryPanel(qParser);
		tabbedPane.addTab("Generic Query", queryPanels[0]);
	}
	
	/**
	 * Receives an event when the user clicks on the
	 * execQuery button.
	 */
	public void actionPerformed(ActionEvent e) {
		if ( e.getActionCommand().equals("execQuery") ) {
			execQuery();
		}
	}
	
	/**
	 * Executes a query and displays the result.
	 */
	protected void execQuery() {
		Query		query;
		QueryResult	result;
		QueryPanel	sQueryPanel;
		
		//resultPanel.test();
		sQueryPanel = getSelectedQueryPanel();
		if ( sQueryPanel.isSimpleQuery() ) {
			query = sQueryPanel.getQuery();
			System.out.println(query);
			try {
				result = pServiceStub.query( 
								query.getSAttribs(), 
								 query.getServiceNames(), 
								 query.getExpression(),
								 query.getAttribReqs(),
								 query.getTimeLimit() );
				System.out.println(result);
				resultPanel.clear();
				resultPanel.displayResult(result);
			} catch (Exception e) {
				e.printStackTrace();
			}
		} else {
			sQueryPanel.execQuery(pServiceStub, resultPanel);
		}
	}
	
	/**
	 * Determine the currently selected query panel.
	 */
	protected QueryPanel getSelectedQueryPanel() {
		int	index;
		
		index = tabbedPane.getSelectedIndex();
		return queryPanels[index];
	}
 
	public void windowActivated(WindowEvent e) {
	}
	
	public void windowClosed(WindowEvent e) {
	}
	
	public void windowClosing(WindowEvent e) {
		System.exit(0);
	}
	
	public void windowDeactivated(WindowEvent e) {
	}
	
	public void windowDeiconified(WindowEvent e) {
	}
	
	public void windowIconified(WindowEvent e) {
	}
	
	public void windowOpened(WindowEvent e) {
	}
	
	/**
	 * Accept command line parameters from the user and run the example.
	 * @param args	command line paramters
	 */
	public static void main(String[] args) {
		try {
			GenericQuery	genericQuery;
			JFrame		frame;
			Dimension	frameSize;
			String		svcHostName;
			int			svcPort;
			
			if (args.length != 2) {
				System.out.println("args: <svcHostName> <svcPort>");
				return;
			}
			svcHostName = args[0];
			svcPort = Integer.parseInt(args[1]);
			genericQuery = new GenericQuery(svcHostName, svcPort);
			frame = new JFrame("GenericQuery");
			frameSize = new Dimension(400, 400);
			frame.setContentPane(genericQuery);
			frame.pack();
			frame.setSize(frameSize);
			frame.show();
			frame.addWindowListener(genericQuery);
		} catch (Exception e) {
			e.printStackTrace();
		}
	}
}
