// GenericClient.java


package edu.cmu.aura.ps.examples.genericClient;


import edu.cmu.aura.service.*;
import edu.cmu.aura.service.rpc.*;
import edu.cmu.aura.service.query.*;
import edu.cmu.aura.service.query.parser.*;
import edu.cmu.aura.service.xml.xmlschema.TimeInstant;
import java.awt.*;
import java.awt.event.*;
import java.io.IOException;
import java.rmi.RemoteException;
import java.util.*;
import javax.swing.*;


/**
 * GenericClient is a client that can query any service.
 * This class uses the QParser to allow users to
 * input SQL-like commands to an arbitrary service.
 * 
 * Note that this class is compiled into aurasvc.jar so 
 * modifications will not take effect unless you take measures
 * such as changing the package or class name.
 * 
 * @author	Glenn Judd
 * @date	8/02/2002
 */
public class GenericClient extends JPanel implements WindowListener, 
											ActionListener, AdjustmentListener {
	protected String				svcHostName;
	protected int					svcPort;
	protected QParser				qParser;
	protected RemoteReference		pServiceRef;
	protected PrimitiveServiceStub	pServiceStub;
	
	protected JScrollPane			scrollPane;
	protected ResultPanel			resultPanel;
	protected JButton				execButton;
	protected JButton				stopButton;
	protected JComboBox				queryBox;
	protected JLabel				hostNameLabel;
	protected JComboBox				hostNameBox;
	protected JLabel				portLabel;
	protected JComboBox				portBox;
	protected JScrollBar			rpVertScrollBar;
	protected boolean				scrollAtBottom;
	protected boolean				waitingForAdjust;
	
	protected HashMap				queryHostMap;
	protected HashMap				queryPortMap;
	
	protected ResultListener		resultListener;
	protected int					curPQID;
	protected String				curPQServiceName;

	/**
	 * Creates a GenericClient instance with a specified host
	 * and port.
	 */
	public GenericClient() {
		init();
	}	
	
	/**
	 * Creates a GenericClient instance with a specified host
	 * and port.
	 * @param hostName	the hostName on which the service resides
	 * @param port	the port on which the service is listening
	 */
	public GenericClient(String svcHostName, int svcPort) {
		init();
		setServiceLoc(svcHostName, svcPort);
		hostNameBox.addItem(svcHostName);
		portBox.addItem( new Integer(svcPort).toString() );
	}
	
	/**
	 * initialization
	 */
	protected void init() {
		curPQID = -1;
		try {
			resultListener = new ResultListener(this);
		} catch (IOException ioe) {
			ioe.printStackTrace();
			throw new RuntimeException("Unexpected ioe");
		}
		qParser = new QParser();
		queryHostMap = new HashMap();
		queryPortMap = new HashMap();
		initGui();
		stopButton.setEnabled(false);
	}
		
	/**
	 * Initialize gui.
	 */
	protected void initGui() {
		GridBagLayout		gridBag;
		GridBagConstraints	c;
		
		gridBag = new GridBagLayout();
		c = new GridBagConstraints();
		setLayout(gridBag);		

		hostNameLabel = new JLabel("Host");
		c.gridx = 0;
		c.gridy = 0;
		c.gridwidth = 2;
		c.gridheight = 1;
		c.weightx = 2.0;
		c.weighty = 1.0;
		c.fill = GridBagConstraints.NONE;
		gridBag.setConstraints(hostNameLabel, c);
		add(hostNameLabel);

		hostNameBox = new JComboBox();
		hostNameBox.setEditable(true);
		hostNameBox.addActionListener(this);
		hostNameBox.setActionCommand("hostNameBox");
		c.gridx = 2;
		c.gridy = 0;
		c.gridwidth = 5;
		c.gridheight = 1;
		c.weightx = 4.0;
		c.weighty = 1.0;
		c.fill = GridBagConstraints.HORIZONTAL;
		gridBag.setConstraints(hostNameBox, c);
		add(hostNameBox);
		
		portLabel = new JLabel("Port");
		c.gridx = 7;
		c.gridy = 0;
		c.gridwidth = 2;
		c.gridheight = 1;
		c.weightx = 0.0;
		c.weighty = 1.0;
		c.fill = GridBagConstraints.NONE;
		gridBag.setConstraints(portLabel, c);
		add(portLabel);		

		portBox = new JComboBox();
		portBox.setEditable(true);
		portBox.addActionListener(this);
		portBox.setActionCommand("portBox");
		c.gridx = 9;
		c.gridy = 0;
		c.gridwidth = 2;
		c.gridheight = 1;
		c.weightx = 4.0;
		c.weighty = 1.0;
		c.fill = GridBagConstraints.HORIZONTAL;
		gridBag.setConstraints(portBox, c);
		add(portBox);
		
		queryBox = new JComboBox();
		queryBox.addActionListener(this);
		queryBox.setActionCommand("queryBox");
		queryBox.setEditable(true);
		c.gridx = 0;
		c.gridy = 1;
		c.gridwidth = 11;
		c.gridheight = 1;
		c.weightx = 11.0;
		c.weighty = 1.0;
		c.fill = GridBagConstraints.HORIZONTAL;
		gridBag.setConstraints(queryBox, c);
		add(queryBox);		
		
		execButton = new JButton("Execute Query");
		execButton.addActionListener(this);
		execButton.setActionCommand("execQuery");
		c.gridx = 2;
		c.gridy = 2;
		c.gridwidth = 4;
		c.gridheight = 1;
		c.weightx = 4.0;
		c.weighty = 1.0;
		c.fill = GridBagConstraints.NONE;
		gridBag.setConstraints(execButton, c);
		add(execButton);

		stopButton = new JButton("Stop Query");
		stopButton.addActionListener(this);
		stopButton.setActionCommand("stopQuery");
		c.gridx = 7;
		c.gridy = 2;
		c.gridwidth = 4;
		c.gridheight = 1;
		c.weightx = 4.0;
		c.weighty = 1.0;
		c.fill = GridBagConstraints.NONE;
		gridBag.setConstraints(stopButton, c);
		add(stopButton);		
		
		resultPanel = new ResultPanel();
		scrollPane = new JScrollPane(resultPanel);
		rpVertScrollBar = scrollPane.getVerticalScrollBar();
		rpVertScrollBar.addAdjustmentListener(this);
		c.gridx = 0;
		c.gridy = 3;
		c.gridwidth = 11;
		c.gridheight = 7;
		c.weightx = 11.0;
		c.weighty = 10.0;
		c.fill = GridBagConstraints.BOTH;
		gridBag.setConstraints(scrollPane, c);
		add(scrollPane);
	}
	
	protected void checkHostAndPort() {
		String	newHost;
		String	newPortDef;
		int		newPort;
		
		newHost = (String)hostNameBox.getSelectedItem();
		if (newHost == null || newHost.trim().length() == 0) {
			return;
		}
		newPortDef = (String)portBox.getSelectedItem();
		if (newPortDef == null || newPortDef.trim().length() == 0) {
			return;
		}
		newPort = Integer.parseInt(newPortDef);
		if ( !newHost.equals(svcHostName) 
				|| newPort != svcPort) {
			setServiceLoc(newHost, newPort);
		}
	}
	
	protected void setServiceLoc(String hostName, int port) {
		if (curPQID != -1) {
			stopQuery();
		}
		this.svcHostName = hostName;
		this.svcPort = port;
		pServiceRef = new RemoteReference(svcHostName, svcPort);
		pServiceStub = new PrimitiveServiceStub(pServiceRef);		
	}
	
	/**
	 * Receives an event when the user clicks on the
	 * execQuery or stopQuery buttons.
	 */
	public void actionPerformed(ActionEvent e) {
		if ( e.getActionCommand().equals("execQuery") ) {
			execQuery();
		} else if ( e.getActionCommand().equals("stopQuery") ) {
			stopQuery();
		} else if ( e.getActionCommand().equals("hostNameBox") ) {
			String	selectedHost;
			String	query;
			
			selectedHost = (String)hostNameBox.getSelectedItem();
			if (((DefaultComboBoxModel)hostNameBox.getModel()).getIndexOf(selectedHost) == -1) {				
				hostNameBox.addItem(selectedHost);
			}
			checkHostAndPort();
			query = (String)queryBox.getSelectedItem();
			queryHostMap.put( query, hostNameBox.getSelectedItem() );
		} else if ( e.getActionCommand().equals("portBox") ) {
			String	portDef;
			String	query;
			
			portDef = (String)portBox.getSelectedItem();
			if (((DefaultComboBoxModel)portBox.getModel()).getIndexOf(portDef) == -1) {				
				portBox.addItem(portDef);
			}
			checkHostAndPort();
			query = (String)queryBox.getSelectedItem();
			queryPortMap.put( query, portBox.getSelectedItem() );
		} else if ( e.getActionCommand().equals("queryBox") ) {
			String	query;
			
			validate();
			query = (String)queryBox.getSelectedItem();
			if (((DefaultComboBoxModel)queryBox.getModel()).getIndexOf(query) == -1) {				
				queryBox.addItem(query);
				queryPortMap.put( query, portBox.getSelectedItem() );
				queryHostMap.put( query, hostNameBox.getSelectedItem() );
			}
			portBox.setSelectedItem( queryPortMap.get(query) );
			hostNameBox.setSelectedItem( queryHostMap.get(query) );
		}
	}
		
	/**
	 * Executes a query and displays the result.
	 */
	protected void execQuery() {
		String		queryDef;
		Query		query;
		QueryResult	result;
		
		queryDef = (String)queryBox.getSelectedItem();
		if (queryDef == null || queryDef.trim().length() == 0) {
			return;
		}
		try {				
			query = qParser.parseQuery(queryDef);
			System.out.println(query);
			resultPanel.clear();
			
			if ( !(query instanceof PostedQuery) ) {
				/*
				result = pServiceStub.query( 
								query.getSAttribs(), 
								 query.getServiceNames(), 
								 query.getExpression(),
								 query.getAttribReqs(),
								 query.getTimeLimit() );
				*/
				result = pServiceStub.query(query);
				System.out.println(result);
				resultPanel.displayResult(result);
			} else {
				PostedQuery	pQuery;
				ArrayList	serviceNames;
				
				if (curPQID != -1) {
					throw new RuntimeException("panic");
				}
				// FIXME - WE ONLY SUPPORT THIS FOR SINGLE SERVICE QUERIES
				pQuery = (PostedQuery)query;
				pQuery.setQueryClient(resultListener);
				curPQID = pServiceStub.postQuery(pQuery);
				stopButton.setEnabled(true);
				serviceNames = pQuery.getServiceNames();
				if (serviceNames == null || serviceNames.size() == 0) {
					curPQServiceName = null;
				} else {
					curPQServiceName = (String)serviceNames.get(0);
				}
				resultPanel.append("Posted Query: "+ curPQID +"\n");
				resultPanel.append( "Local Time: "
					+(new TimeInstant(System.currentTimeMillis()).toString()) 
					+"\n");
			}
		} catch (Exception e) {
			e.printStackTrace();
		}
	}
	
	protected void stopQuery() {
		if (curPQID == -1) {
			throw new RuntimeException("panic");
		}
		try {
			pServiceStub.removeQuery(curPQServiceName, curPQID);
		} catch (IOException ioe) {
			ioe.printStackTrace();
		}
		curPQID = -1;
		curPQServiceName = null;
		stopButton.setEnabled(false);
	}
	
	public void windowActivated(WindowEvent e) {
	}
	
	public void windowClosed(WindowEvent e) {
	}
	
	public void windowClosing(WindowEvent e) {
		System.exit(0);
	}
	
	public void windowDeactivated(WindowEvent e) {
	}
	
	public void windowDeiconified(WindowEvent e) {
	}
	
	public void windowIconified(WindowEvent e) {
	}
	
	public void windowOpened(WindowEvent e) {
	}

	public void adjustmentValueChanged(AdjustmentEvent e) {
		if (waitingForAdjust && scrollAtBottom && e.getValue() > 0) {
			rpVertScrollBar.setValue( rpVertScrollBar.getMaximum() );
			waitingForAdjust = false;
		}
	}
	
	public void queryExecuted(int queryID, QueryResult queryResult) throws RemoteException {
		scrollAtBottom = rpVertScrollBar.getValue() + rpVertScrollBar.getVisibleAmount()
						 >= rpVertScrollBar.getMaximum();
		if (scrollAtBottom) {
			waitingForAdjust = true;
		}
		resultPanel.append("queryID: "+ queryID +"\n");
		resultPanel.displayResult(queryResult);
	}		
	
	/**
	 * Accept command line parameters from the user and run the example.
	 * @param args	command line paramters
	 */
	public static void main(String[] args) {
		try {
			GenericClient	genericClient;
			JFrame			frame;
			Dimension		frameSize;
			String			svcHostName;
			int				svcPort;
			
			if (args.length != 2 && args.length != 0) {
				System.out.println("args: none");
				System.out.println("or");
				System.out.println("args: <svcHostName> <svcPort>");
				return;
			}
			if (args.length == 2)  {
				svcHostName = args[0];
				svcPort = Integer.parseInt(args[1]);
				genericClient = new GenericClient(svcHostName, svcPort);
			} else {
				genericClient = new GenericClient();
			}
			frame = new JFrame("GenericClient");
			frameSize = new Dimension(400, 400);
			frame.setContentPane(genericClient);
			frame.pack();
			frame.setSize(frameSize);
			frame.show();
			frame.addWindowListener(genericClient);
		} catch (Exception e) {
			e.printStackTrace();
		}
	}
}
