ATK_INTER
#ifndef _tlex_H_
#define _tlex_H_
/* tlex.H - lexical analyzer which reads from ATK text  */
/*
	Copyright Carnegie Mellon University 1992, 1994 - All rights reserved
	$Disclaimer: Andrew User Interface System - Binary Distribution 7.5
// 
// Permission to use, copy, modify, and distribute this software for any 
// purpose is hereby granted, provided (a) that no fee is charged for the 
// software, for the medium on which it is distributed, for the 
// distribution process, or for effort involved in making the distribution;  
// (b) that all copyright notices, this permission notice, and the 
// following disclaimer remain in these files and appear in supporting 
// documentation;  (c) that you do not translate, reverse engineer, 
// decompile, or disassemble the software; and (d) that the names of 
// IBM, Carnegie Mellon University, and other copyright holders not 
// be used in advertising or publicity pertaining to distribution of the 
// software.
// 
// CARNEGIE MELLON UNIVERSITY, IBM, AND THE OTHER 
// COPYRIGHT HOLDERS DISCLAIM ALL WARRANTIES 
// WITH REGARD TO THIS SOFTWARE, INCLUDING ALL 
// IMPLIED WARRANTIES OF MERCHANTABILITY AND 
// FITNESS.  IN NO EVENT SHALL CARNEGIE MELLON 
// UNIVERSITY, IBM, OR ANY OTHER COPYRIGHT HOLDER 
// BE LIABLE FOR ANY SPECIAL, INDIRECT OR 
// CONSEQUENTIAL DAMAGES OR ANY DAMAGES 
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR 
// PROFITS, WHETHER IN AN ACTION OF CONTRACT, 
// NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING 
// OUT OF OR IN CONNECTION WITH THE USE OR 
// PERFORMANCE OF THIS SOFTWARE.
//  $
*/

/*
The routines in tlex.c are suited for recognizing

	identifier  (can include reserved words)
	number: float or double real, decimal, hex, or octal integer
	string
	comments
	whitespace
	special characters (includes thongs)

    tlex_tables values are generated by gentlex.

*/

#include <gentlex.h>
#include <ATK.H>
#define tlex_RECENTSIZE 10

class tlex : public ATK {
public:
	virtual ATKregistryEntry *ATKregistry();

	static int LexFunc(void *lexrock, void *yylval);
		/* Suitable for use as the lexer function required by 
		the parser object.  The lexrock value should be a
		tlex object. */

	virtual void SetText(struct text *text, long pos, long len);
		/* sets the source text for the lexeme stream */

	virtual long RecentPosition(int index, long *len);
		/* for token 'index', set len to length and return position.
		index = 0 is the most recent token,
		its predecessors are indexed with negative numbers:
			-1 -2 ... -tlex_RECENTSIZE+1*/

	virtual long RecentIndent(int index);
		/* report the indentation of the 'index'th most recent token,
		    where index is as for RecentPosition .
		    A token preceded by anything other than white space
		    is reported as having indentation 999. */

	virtual void Repeat(int index);
		/* backup and repeat tokens starting with the index'th
		    most recent token, where index is as for RecentPosition */

	virtual void Error(char *msg);
		/* a lexical error is reported by calling the
			error handler after setting up a dummy token
			for the error text
			The msg is expected to be in static storage.  */

	/* the following methods are for calls generated by macros and
		gentlex code.  They should not be called by clients. */

	virtual int FetchChar();
		/* fetches character at this->pos; sets charx/lastcharx
			may return EOF */
	virtual int PutTokChar(char c);
		/* stores the character c in the token buffer; usually
			called only when token buffer size is too small */

	/* the rock is an argument to tlex_Create.  It is accessible
		to any routine having a pointer to the resulting tlex. */
	inline void  * GetRock()		{ return (this->rock); }
	inline void SetRock(void *r)	{ (this->rock = (r)); }

	/* the Global block is established in the xxx.tlc file in reaction
		to the -global- tokenclass in the xxx.tlx file.  Clients
		of the tlex can access the global block via Global() */
	inline struct tlex_Recparm *Global()		{ return (this->lextab->global); }

	/* recognizers and handlers modify the results from NextToken
	by calling macro methods to adjust these attributes:
		token number
		TokenValue (yylval)
		current character and scan position in the source text
		position and length of source for current token
		characters in current token
	or report an error
	*/

	/* the TokenNumber is the number to be reported to the parser 
	 This is usually set by default based on the argument to the 
	 tokenclass line in the xxx.tlx file. It may be reset to a 
	 value created by a tokennumber line within a tokenclass block.
	*/
	inline void SetTokenNumber(int n)	{ (this->tokennumber = (n)); }
	inline int   GetTokenNumber()	{ return (this->tokennumber); }

	/* the TokenValue is the value for yylval.  These values serve
		as the initial values in the value stack maintained
		by the parser in parallel with the state stack */
	inline void SetTokenValue(void *v)	{ (this->tokenvalue = (v)); }
	inline void  * GetTokenValue() 	{ return (this->tokenvalue); }

	/* the current position in the input is CurrPos where the
	 character is as given by CurrChar.  By convention each
	 lexical analysis routine leaves CurrPos/CurrChar referring
	 to the first character to be considered for the next token.  
	 NextChar moves CurrPos ahead one character, fetches the
	 next character, and returns it.
	 CurrChar is not declared unsigned, and EOF is negative;
	 however, the character value does NOT have its sign extended.
	 BackUp moves backward by n characters, resetting CurrPos/CurrChar
	 (a negative n value is acceptable and moves the position forward)
	 See also Advance, below, which combines NextChar with storing
		the prior character in the tokentext.
	*/
	inline long   CurrPos()		{ return (this->currpos); }
	inline int   CurrChar()		{ return (this->currchar); }
	inline int NextChar()		{ return ((this->currpos++, 
			((this->charx++ < this->lastcharx) 
			   ? (this->currchar 
				= (int)(*((unsigned char *)(this->charx)))) 
			   : this->FetchChar()))); }
	inline int BackUp(int n)		{ return (this->currpos -= (n), this->FetchChar()); }

					
	/* The position of the token text in the input source is
	 recorded and is available via 
		GetTokPos - the position of the first character
		GetTokEnd - the position of the last character
	 StartToken records CurrPos as the position at which the token begins.
	 EndToken records the token as ending one character before Currpos.
	 There is no harm in calling StartToken or EndToken more than once,
	 although these functions also affect the token text buffer,
	 as noted below.
	*/
	inline long   GetTokPos()		{ return (this->tokpos); }
	inline long   GetTokEnd()		{ return (this->tokend); }
	inline void StartToken()		{ (this->tokbufx = this->tokenbuffer, 
					this->tokpos = this->currpos); }
	inline void EndToken()		{ (this->tokend = this->currpos-1,  
					*(this->tokbufx) = '\0'); }

	/* some tokens are recorded by the lexer as
	 a character string which can be retrieved by GetTokenText.
	 In particular, when C code is called from a tokenclass block,
	 the text is the sequence of characters from the source that 
	 caused this tokenclass to be activated.  
	 TokenText is stored for ScanId, ScanToken, ScanWhitspace.
	 A canonical form of the number is stored for ScanNumber.

	 StartToken and EndToken (above) have the additional functionality,
	 respectively, of clearing the token buffer and finishing it with
         a null character.

	 GetTokenText returns a pointer to the token text string.
	 PrevTokenText returns a pointer to the text of the 
		previous token.
	 ClearTokenText clears the text to an empty string.
	 AppendToTokenText appends a character to the text.
	 TruncateTokenText removes n characters from its end.
	 Advance appends the current character to the token text and
		then calls NextChar
	*/
	inline char  * GetTokenText()		{ return (this->tokenbuffer); }
	inline char  * PrevTokenText()		{ return (this->alttokenbuf); }
	inline void ClearTokenText()	{(this->tokbufx = this->tokenbuffer);}
	inline int AppendToTokenText(int c)  { 
			return ((this->tokbufx <= this->tokbuflastx) 
					? (int)(*this->tokbufx++ = (c)) 
					: (int)this->PutTokChar((c))); }
	inline void TruncateTokenText(int n) {
	    if((n) >= 0 && (n) <= this->tokbufx - this->tokenbuffer) {
		(void)(this->tokbufx -= n );
	    } else {
		this->ClearTokenText();
	    }
	}
	inline int Advance()	{ return ((this->AppendToTokenText(this->currchar)), 
					this->NextChar()); }



	static struct tlex * Create(struct tlex_tables *description, void *rock,
			struct text *text, long pos, long len)
			 ;
		/* the rock is available to any function passed this tlex
		The text, pos, and len specify a portion of a text to be processed */

	
	static boolean InitializeClass();

	
	tlex();
	
	virtual ~tlex();


	struct tlex_tables *lextab;

	void *rock;		/* available to action routines */

	struct text *text;	/* the text we are scanning */
	long startpos;		/* beginning bound of text */
	long currpos;		/* where currchar is from */
	long lastpos;		/* last position to use */

	char *charx;		/* pointer to currpos'th character in text */
	char *lastcharx;	/* last character fetchable via *charx++
				(either by reaching gap or by EOF) */

	int currchar;		/* cache copy of character at currpos
				   (may be EOF) */

	long tokpos;		/* first pos for text of current token */
	long tokend;		/* last pos of text for current token */

	int tokennumber;	/* token number to return from NextToken */
	void *tokenvalue;	/* value to assign to *pyylval */

	/* tokenbuffer and alttokenbuf are interchanged each time
		to cheaply implement PrevTokenText */
	char *tokenbuffer;	/* characters of token */
	char *tokbufx;		/* where to store next char in tokenbuffer */
	char *tokbuflastx;	/* where to store last char in tokenbuffer
				  tokenbuffer+(2 less than size of tokenbuffer) */
	char *alttokenbuf;	/* token buffer for previous token */
	char *altbuflastx;	/* end of alttokenbuffer */

	/* arrays to store text positions of recent tokens */
	long RecentPos[tlex_RECENTSIZE];
	long RecentLen[tlex_RECENTSIZE];
	int RecentIndex;
};

#endif /* _tlex_H_ */
