//**********************************************************************
//  samp_deer_fear.h  -  sample program using aipDecision.h, aipEmotion
//
//  The decision is whether a deer, eating grass in a meadow, 
//  should run away.
//
//  In these classes, a deer-decision has two options:
//    Stay - which has a fear which is affected by messages
//    Run_Away  - which has a goodness
//  A series of messages is passed to the decision, which passes
//  them to each option.  A message can contain information about
//  what the deer has just heard or smelled.
//
//  The idea of using the emotion of fear is that it is easier to
//  define when the deer should become more afraid, than it is to
//  define when the deer should run away.
//
//  In this trivial example, messages are provided for only a short
//  period of time, so the goodness associated with Run_Away is
//  constant.  
//
//  Run_Away goodness is quite negative - the deer needs to eat -
//  it can't run away every time the slightest sound or smell is 
//  detected.  Running away is also negative because it increases
//  the chance of getting separated from the other deer.
//
//  Run_Away will only be chosen if the Stay fear goodness is more 
//  negative than the Run_Away goodness.
//
//  Developers/Contributers:
//    [BRM] Brian Marshall - Calgary - bmarshal@agt.net
//
//  05/09/17  [BRM] development begins
//
//----------------------------------------------------------------------

#ifndef samp_deer_h_guard
#define samp_deer_h_guard

#include "aipDecision.h"

//----------------------------------------------------------------------
//  Forward Declaration of Classes - subclassing shown by indentation

// class aipBase;                        ( in aipBase.h )
//   class aipDemon;                     ( in aipPandemonium.h )
//     class aipDecision;                ( in aipDecision.h )
         class DeerDecision;    // decision to run or stay
//     class aipOption;                  ( in aipDecision.h )
         class DeerOption;      // option in the decision
           class RunOption;     // option to run
           class StayOption;    // option stay
//     class aipAspect;         // aspect in the fear emotion
         class FearSmellWolf;   // fear of staying: smell-wolf
         class FearHearNoise;   // fear of staying: hear-noise
// class aipMsg;                         ( in aipBase.h )
     class DeerMsg;             // info about hearing, smelling


//----------------------------------------------------------------------
//  Constants

                     // run away if staying is worse than...
const aipG  g_Run_Away = aipPrettyBad;

                                   // message typ
const short Info_Seems_Safe  = 1;
const short Info_Smell_Wolf  = 2;
const short Info_Hear_Noise  = 3;
                                   // message subtyp
const short Info_Weak   = 1;
const short Info_Medium = 2;
const short Info_Strong = 3;

//----------------------------------------------------------------------
//  DeerDecision

class DeerDecision : public aipDecision {

  aipFear * m_stay_fear_ptr;     // for reporting (if not zero)

public:

  DeerDecision () { m_stay_fear_ptr = 0; }
  virtual ~DeerDecision () {}

  void add_deer_option (DeerOption *x);    // sets m_stay_fear_ptr

  DeerOption * deer_decide () { return (DeerOption *) decide(); }

  aipFear * stay_fear () const { return m_stay_fear_ptr; }

};

//----------------------------------------------------------------------
//  DeerOption  

class DeerOption : public aipOption {    // a base class
public:
  DeerOption () {}
  virtual ~DeerOption () {}
  virtual aipG g() { return aipNeutral; }
  virtual aipFear * option_fear () const { return 0; }
  virtual const char * description () const =0;
};

// - - - - - - - - - - - - - - - - - - - - - - - - - - -
//  Option: Run_Away

class RunOption : public DeerOption {
public:
  RunOption () {}
  virtual ~RunOption () {}
  virtual aipG g_opt() { return g_Run_Away; }
  const char * description () const { return "Run_Away"; }
};

// - - - - - - - - - - - - - - - - - - - - - - - - - - -
//  Option: Stay

class StayOption : public DeerOption {
  aipFear   * m_fear;
public:
  StayOption () { m_fear = new aipFear; }
  virtual ~StayOption () { if (m_fear) delete m_fear; }
  virtual void take_msg (aipMsg *m) { m_fear->take_msg(m); }
  aipFear * fear () { return m_fear; }
  virtual aipG g_opt();
  virtual aipFear * option_fear () const { return m_fear; }
  const char * description () const { return "Stay"; }
};


//----------------------------------------------------------------------
//  Fear aspects associated with the Stay Option
//
//  A stay-option has a fear, and this fear has aspects.

// - - - - - - - - - - - - - - - - - - - - - - - - - - -

class FearSmellWolf : public aipAspect {
public:
  FearSmellWolf () {}
  virtual ~FearSmellWolf () {}
  virtual void take_msg (aipMsg *m);
};

// - - - - - - - - - - - - - - - - - - - - - - - - - - -

class FearHearNoise : public aipAspect {
public:
  FearHearNoise () {}
  virtual ~FearHearNoise () {}
  virtual void take_msg (aipMsg *m);
};

// - - - - - - - - - - - - - - - - - - - - - - - - - - -

class FearSeemsSafe : public aipAspect {
public:
  FearSeemsSafe () {}
  virtual ~FearSeemsSafe () {}
  virtual void take_msg (aipMsg *m);
};

//======================================================================
//  DeerMsg  -  message of/in a deer that is passed to the decision
//
//  In this subclass, the message typ and subtyp carry information.

class DeerMsg : public aipMsg {
public:
  DeerMsg (short condition_type, short condition_strength)
                    : aipMsg(condition_type,condition_strength) {}
  virtual ~DeerMsg () {}
  const char * typ_description ();
  const char * subtyp_description ();
};

//======================================================================

#endif

//**********************************************************************
