//======================================================================
//  samp_decision.cpp  - trivial sample program using aipDecision.h
//
//  The decision is whether a deer, eating grass in a meadow, 
//  should run away.
//
//  In this program, a deer-decision has two options:
//    Stay - which has a fear which is affected by messages
//    Run  - which has a goodness
//  A series of messages is passed to the decision, which passes
//  them to each option.  A message can contain information about
//  what the deer has just heard or smelled.
//
//  The idea of using the emotion of fear is that it is easier to
//  define when the deer should become more afraid, than it is to
//  define when the deer should run away.
//
//  In this trivial example, messages are provided for only a short
//  period of time, so the goodness associated with Run_Away is
//  constant.  
//
//  Run_Away goodness is quite negative - the deer needs to eat -
//  it can't run away every time the slightest sound or smell is 
//  detected.  Running away is also negative because it increases
//  the chance of getting separated from the other deer.
//
//  Run_Away will only be chosen if the fear of staying becomes 
//  worse than the Run goodness.
//
//  In other applications, options and/or the decision might have
//  emotions, or there may be no emotions used at all.
//
//  This program does not use aipHopeOption or aipHopeDecision.
//  The program samp_problem.cpp does use these classes to do
//  problem-solving.
//
//  Developers/Contributers:
//    [BRM] Brian Marshall - Calgary - bmarshal@agt.net
//
//  05/11/09  [BRM] improved sample problem series of messages
//  05/09/17  [BRM] development begins
//
//----------------------------------------------------------------------
//  Building the executable
//
//  On Linux, you can call the compiler gcc or g++ (although on
//  Fedora, use g++ because gcc finds the wrong libraries)...
//
//    gcc -o samp_decision samp_decision.cpp samp_deer_fear.cpp \ 
//                         aipDecision.cpp aipEmotion.cpp \ 
//                         aipPandemonium.cpp aipBase.cpp aipGood.cpp
//
//  In the example, above, spaces have been added after the 
//  backshlashs - the Borland compiler does not like them at the ends
//  of lines, even in comments.  
//
//
//----------------------------------------------------------------------

#include "samp_deer_fear.h"
#include <iostream>
using namespace std;

//----------------------------------------------------------------------
//  Forward Declarations of Functions

DeerDecision * assemble_new_decision();

DeerMsg * get_msg ();

//======================================================================
//  Program main

int main () {

  // We know about argc and argv, but the Borland compiler
  // does not like them if they are not used.

  DeerDecision *decision = assemble_new_decision();
  if (!decision) {
     cout << "ABORT - Could not create decision\n\n";
     return 1;
  }

     // This next pointer is just used for reporting
  aipFear *stay_fear = decision->stay_fear();
  if (!stay_fear) {
    cout << "ABORT - decision stay-fear not set\n\n";
    return 1;
  }

  cout << "\nA deer is eating grass.  The deer repeatedly\n"
       << "has to decide whether to stay or run away.\n"
       << "We create a decsion-object and pass it a series\n"
       << "of messages that affect this decision.\n\n";

  for (DeerMsg *msg = get_msg(); msg; msg = get_msg() ) {

    cout << "msg: " << msg->typ_description();
    cout << "  "    << msg->subtyp_description();

    decision->take_msg(msg);

    DeerOption *choice = decision->deer_decide();

    if (!choice) {
      cout << "ABORT - failure to make choice\n\n";
      break;
    }

    cout << "  choice: " << choice->description()
         << "  stay-fear: ("
         << stay_fear->g().numeric_value() << ")   "
         << stay_fear->g().description() << "\n";
  }

  cout << "\n";

  delete decision;

  return 0;

}


//----------------------------------------------------------------------
//  assemble_new_decision

DeerDecision * assemble_new_decision() {

  DeerDecision *decision = new DeerDecision;
  if (!decision) { 
         cout << "ABORT - no new decision\n\n"; return 0; }

  RunOption *run_option = new RunOption;  
  if (!run_option) { 
         cout << "ABORT - no new run_option\n\n"; return 0; }
  decision->add_deer_option(run_option);

  StayOption *stay_option = new StayOption;
  if (!stay_option) { 
         cout << "ABORT - no new stay_option\n\n"; return 0; }
  decision->add_deer_option(stay_option);

  FearSmellWolf *fear_smell_wolf = new FearSmellWolf;
  if (!fear_smell_wolf) { 
         cout << "ABORT - no new fear_smell_wolf\n\n"; return 0; }
  stay_option->fear()->add_aspect(fear_smell_wolf);

  FearHearNoise *fear_hear_noise = new FearHearNoise;
  if (!fear_hear_noise) { 
         cout << "ABORT - no new fear_hear_noise\n\n"; return 0; }
  stay_option->fear()->add_aspect(fear_hear_noise);

  FearSeemsSafe *fear_seems_safe = new FearSeemsSafe;
  if (!fear_seems_safe) { 
         cout << "ABORT - no new fear_seems_safe\n\n"; return 0; }
  stay_option->fear()->add_aspect(fear_seems_safe);

  return decision;

}

//----------------------------------------------------------------------
//  get_msg  -  get the next message to be passed to the decision

DeerMsg * get_msg () {

  static int num_msg = 21;
  static int next_msg = 0;

  static DeerMsg msg[] = {
        DeerMsg (Info_Seems_Safe,  0),
        DeerMsg (Info_Hear_Noise,  Info_Weak),
        DeerMsg (Info_Seems_Safe,  0),
        DeerMsg (Info_Smell_Wolf,  Info_Weak),
        DeerMsg (Info_Hear_Noise,  Info_Weak),
        DeerMsg (Info_Seems_Safe,  0),
        DeerMsg (Info_Hear_Noise,  Info_Weak),
        DeerMsg (Info_Hear_Noise,  Info_Strong),
        DeerMsg (Info_Seems_Safe,  0),
        DeerMsg (Info_Seems_Safe,  0),
        DeerMsg (Info_Hear_Noise,  Info_Weak),
        DeerMsg (Info_Smell_Wolf,  Info_Medium),
        DeerMsg (Info_Hear_Noise,  Info_Weak),
        DeerMsg (Info_Hear_Noise,  Info_Medium),
        DeerMsg (Info_Smell_Wolf,  Info_Weak),
        DeerMsg (Info_Hear_Noise,  Info_Weak),
        DeerMsg (Info_Seems_Safe,  0),
        DeerMsg (Info_Seems_Safe,  0),
        DeerMsg (Info_Seems_Safe,  0),
        DeerMsg (Info_Seems_Safe,  0),
        DeerMsg (Info_Seems_Safe,  0),
        DeerMsg (Info_Seems_Safe,  0)
  };

  if (next_msg == num_msg) return 0;

  return msg + next_msg++;

}
//======================================================================
