//**********************************************************************
//  rr_solve.h  -   Assign Resources to Requirements 
//
//  An rrsProblem is a subclass of aipHHProblem.
//  rrsProblem::solve_sw_problem()  uses the High-Hope technique.  
//  See aipHighHope.h
//
//  A rrsProblem has a set of decisions, each of which has a set 
//  of options.  
//
//  In this version, a Decision is associated with a Requirement-Day;
//  an Option refers to a Resource that can satisfy it.
//
//  Everything to do with a decision being trivial is commented out.
//  It will be implemented in a later release.
//
//  Copyright (c)  2008  Brian Marshall
//
//  Developers/Contributers:
//    [BRM] Brian Marshall - Calgary - bmarshal@agt.net
//
//  08/01/20  [BRM] began development: split out layer from staffwhen
//
//----------------------------------------------------------------------
//  The Data Model
//
//  In this version, only Requirements have Decisions, which means
//  that Resources always have an Option.
//
//                            [Problem]
//                                +           [Requirements]
//                                |              +
//                                v              v
//                                v              v 
//                          [Decisions]<---+[Req-Days]
//                       v----o   +
//                       v        |
//              [chosen_opts]     |           [Resources]
//                       ^        |              +
//                       |        v              v
//                       |        v              v
//                       ----o[Options]<<--+[Res-Days]
//
//  Notes:
//   - Problems have Decisions, which have Options
//   - Requirements and Resources may have a Decision
//   - Decisions may be decided multiple times, so they have a list
//        of chosen Options.
//
//----------------------------------------------------------------------
//  Important Note
//
//  In this file, the functions in rrsOption:
// virtual aipG g_opt_cmp() { return g_user_constant() + m_g_dynamic; }
// virtual aipG g_opt_usr() { return g_user_constant() + m_g_dynamic; }
//  override functions in the HighHope classes - g_dynamic affects 
//  making decisions, comparing solutions and the goodness that can
//  be reported to the user.
//
//  To solve a particular type of Requirement-Resource problem,
//  a set of sub-classes of rr_solve classes are written, and 
//  they call option.set_g_dynamic(aipG x) based on aspects
//  of the particular problem.
//
//----------------------------------------------------------------------

#ifndef rr_solve_h_guard
#define rr_solve_h_guard

#include "aipHighHope.h"
#include "aipTime.h"

//----------------------------------------------------------------------
//  Classes.  Sub-classing is shown by indentation.

//  class  aipG  is a typedef for  aipGoodness

// class aipBase;                        ( in aipBase.h )
//   class aipImportance;                ( in aipImportance.h )
//     class aipHHImportance;            ( in aipHighHope.h )
//       class rrsReqImp;          // importance of a requirement
//   class aipProblem;                   ( in aipProblem.h )
//     class aipHHProblem;               ( in aipHighHope.h )
         class rrsProblem;         // rr-solve problem
//   class aipDemon;                     ( in aipPandemonium.h )
//     class aipDecision;                ( in aipDecision.h )
//       class aipHHDecision;            ( in aipHighHope.h )
           class rrsDecision;      // decision: for a requirement
//     class aipOption;                  ( in aipDecision.h )
//       class aipHHOption;              ( in aipHighHope.h )
           class rrsOption;        // option: a resource
       class rrsRequirement;       // requirement to be satisfied
       class rrsReqDay;            // requirement on a day
       class rrsResource;          // resource to satisfy requirement
       class rrsResDay;            // resource on a day
// class aipDemonItr;                    ( in aipPandemonium.h )
     class rrsRequirementItr;      // decision iterator
     class rrsReqDayItr;           // decision iterator
     class rrsResourceItr;         // decision iterator
     class rrsResDayItr;           // decision iterator
     class rrsDecisionItr;         // decision iterator
// class aipMsg                          ( in aipBase.h )
     class rrsMsg;                 // message for rr-solve solving

//======================================================================
//  rrsReqImp  -  Importance of a position

class rrsReqImp : public aipHHImportance {

  long    m_left_bhnd;  // being left behind

public:

  rrsReqImp () { m_left_bhnd = 0; }
  virtual ~rrsReqImp () {}

  virtual long val () const {
    return aipHHImportance::val() + m_left_bhnd;
  }

  virtual void take_msg (aipMsg *m);
  virtual void note_rrs_decide (rrsDecision *d, rrsOption *o);

};


//======================================================================
//  rrsProblem  -  rrSolve problem - assigning resources to requirements
//
//  solve_rrs_problem () returns 1 (true) if a solution is found,
//  zero otherwise.
//
//  log_this_try() works best if decisions are stored in Req,Day order

class rrsProblem : public aipHHProblem {

  aipPandemonium  * m_requirements;
  aipPandemonium  * m_resources;

  int  m_should_log_options;  // 1 (true) = log to cout

protected:

  virtual aipHHDecision * next_decision();

  virtual aipG too_bad_to_go_on() const;

  virtual void log_this_try ();  // to cout

  aipPandemonium  * requirement_pandemonium () const { // for iterators
    return m_requirements; 
  }

  aipPandemonium  * resource_pandemonium () const {    // for iterators
    return m_resources;
  }

public:

  rrsProblem ();
  virtual ~rrsProblem ();

  void set_num_try (long x) { aipHHProblem::set_num_try(x); }

  void add_requirement (rrsRequirement *x);
  void add_resource    (rrsResource *x);
  void add_decision    (rrsDecision *x);

  void enable_log_options  () { m_should_log_options = 1; }
  void disable_log_options () { m_should_log_options = 0; }

  int should_log_options () const { return m_should_log_options; }

  rrsRequirementItr     requirement_iterator() const;
  rrsResourceItr        resource_iterator() const;
  rrsDecisionItr        decision_iterator() const;

  virtual void take_msg (aipMsg *m);
  virtual void note_decide (aipHHDecision *d, aipHHOption *o);

//  rrsReqDay * trivial_reqday () const;

  virtual int solve_rr_problem ();       // use this, not solve()

};


//======================================================================
//  rrsDecision  -  choosing a res for a reqday
//
//  inherits: importance, a set of options, more

class rrsDecision : public aipHHDecision {

  rrsReqDay   * m_reqday;        // reqday that needs resource-days

public:

  rrsDecision (rrsReqDay *a_reqday, long num_to_decide =1);
  virtual ~rrsDecision();

  void add_rrs_option (rrsOption *x);  // use this, not add_hh_option

  virtual void take_msg (aipMsg *m);

  rrsProblem * prob () const { return (rrsProblem*)owner(); }

  rrsReqDay * reqday () const { return m_reqday; }

  virtual long imp () const;

  rrsOption * rrs_opt_cur  () const { return (rrsOption*)opt_cur();  }
  rrsOption * rrs_opt_bsf  () const { return (rrsOption*)opt_bsf();  }

};

//======================================================================
//  rrsOption  - a resource assigned to a requirement on a day

class rrsOption : public aipHHOption {

  rrsResDay   * m_resday;         // the subject of this option

  aipG          m_g_dynamic;      // g of situational aspects

public:

  rrsOption(rrsResDay *a_resday, aipG g_constant);
  virtual ~rrsOption ();

  virtual void take_msg (aipMsg *m);

  void set_g_dynamic (aipG x);
  void add_g_dynamic (aipG x);

  rrsDecision * rrs_chooser () const { 
    return (rrsDecision*)hh_opt_chooser(); 
  }

  rrsResDay * resday () const { return m_resday; }

  virtual aipG g_opt();
 
  virtual aipG g_opt_cmp();
  virtual aipG g_opt_usr();

};

//======================================================================
//  rrsRequirement  -  a requirement requiring resources
//
//  A requirement is owned by a problem; it owns requirement-days.

class rrsRequirement : public aipDemon {

  long              m_id;

  rrsProblem      * m_prob;

  aipPandemonium  * m_reqdays;

  aipHHHope       * m_hope;         // how well this requirement is doing
  rrsReqImp         m_importance;   // for choosing next decision

protected:

  aipPandemonium  * reqday_pandemonium () const {    // for iterators
    return m_reqdays; 
  }

public:

  rrsRequirement(long a_id, rrsProblem *a_prob);
  virtual ~rrsRequirement();

  void add_reqday  (rrsReqDay *x);

  long imp () const { return m_importance.val(); }

  virtual void take_msg (aipMsg *m);
  virtual void note_rrs_decide (rrsDecision *d, rrsOption *o);

  int is_satisfied () const;

  rrsReqDay * first_unsatisfied_reqday () const;

//  rrsReqDay * trivial_reqday () const;

  rrsProblem * prob () const { return m_prob; }
  long         id   () const { return m_id;   }

  rrsReqDayItr  reqday_iterator() const;

};


//======================================================================
//  rrsReqDay  -  a requirement on a day (requiring one or more resources)
//
//  set_dcsn() must be called after the decision is created.

class rrsReqDay : public aipDemon {

  rrsRequirement  * m_req;
  aipTime           m_day;

  rrsDecision     * m_dcsn;

  aipHHHope       * m_hope;         // how well this reqday is doing
  aipHHImportance   m_importance;

public:

  rrsReqDay(rrsRequirement *a_req, long a_yyyymmdd);
  virtual ~rrsReqDay();
  void set_dcsn (rrsDecision *a_dcsn) { m_dcsn = a_dcsn; }

  long imp () const { return m_importance.val(); }

  virtual void take_msg (aipMsg *m);
  virtual void note_rrs_decide (rrsDecision *d, rrsOption *o);

  aipTime          day  () const { return m_day;  }
  rrsRequirement * req  () const { return m_req;  }
  rrsDecision    * dcsn () const { return m_dcsn; }

  long num_decided () const { return m_dcsn->num_decided(); }

  int is_satisfied     () const;
  int has_surplus_opts () const;

};


//======================================================================
//  rrsResource  -  a resource that can satisfy requirements

class rrsResource : public aipDemon {

  long              m_id;

  rrsProblem      * m_prob;

  aipPandemonium  * m_resdays;

  aipHHHope       * m_hope;        // how well this resource is doing
  int               m_is_in_cur_solution;

protected:

  aipPandemonium  * resday_pandemonium () const {   // for iterators
    return m_resdays;
   }

public:

  rrsResource(long a_id, rrsProblem *a_prob);
  virtual ~rrsResource();

  void add_resday  (rrsResDay *x);

  virtual void take_msg (aipMsg *m);
  virtual void note_rrs_decide (rrsDecision *d, rrsOption *o);

  rrsProblem * prob () const { return m_prob; }
  long         id   () const { return m_id;   }

  rrsResDayItr  resday_iterator() const;

};


//======================================================================
//  rrsResDay  -  a resource on a day

class rrsResDay : public aipDemon {

  aipTime          m_day;

  rrsResource    * m_res;

  aipPandemonium * m_opts;    // opts where this resday is the subject

  aipHHHope      * m_hope;         // how well this resday is doing
  int              m_is_in_cur_solution;

protected:

  aipPandemonium  * opt_pandemonium () const {   // for iterators
    return m_opts;
  }

public:

  rrsResDay(rrsResource *a_res, long a_yyyymmdd);
  virtual ~rrsResDay();

  void add_opt (rrsOption *o) { m_opts->add(o); }

  virtual void take_msg (aipMsg *m);
  virtual void note_rrs_decide (rrsDecision *d, rrsOption *o);

  aipTime       day () const { return m_day; }
  rrsResource * res () const { return m_res; }

};


//======================================================================
//  rrsRequirementItr  -  Iterator for Requirements in a Problem

class rrsRequirementItr : public aipDemonItr {

public:

  rrsRequirementItr (aipPandemonium *p) {
    set_demon_itr(p,0);
  }

  rrsRequirementItr (const rrsRequirementItr& x) {
    set_demon_itr (x.pandemonium(), x.current());
  }

  virtual ~rrsRequirementItr () {}

  rrsRequirementItr& operator = (const rrsRequirementItr& x) {
    set_demon_itr(x.pandemonium(), x.current());
    return *this;
  }

  rrsRequirement * first () {
    return (rrsRequirement*)aipDemonItr::first();
  }

  rrsRequirement * next () {
    return (rrsRequirement*)aipDemonItr::next();
  }

};

//======================================================================
//  rrsReqDayItr  -  Iterator for ReqDays of a Requirement

class rrsReqDayItr : public aipDemonItr {

public:

  rrsReqDayItr (aipPandemonium *p) {
    set_demon_itr(p,0);
  }

  rrsReqDayItr (const rrsReqDayItr& x) {
    set_demon_itr (x.pandemonium(), x.current());
  }

  virtual ~rrsReqDayItr () {}

  rrsReqDayItr& operator = (const rrsReqDayItr& x) {
    set_demon_itr(x.pandemonium(), x.current());
    return *this;
  }

  rrsReqDay * first () {
    return (rrsReqDay*)aipDemonItr::first();
  }

  rrsReqDay * next () {
    return (rrsReqDay*)aipDemonItr::next();
  }

};

//======================================================================
//  rrsResourceItr  -  Iterator for Resources in a Problem

class rrsResourceItr : public aipDemonItr {

public:

  rrsResourceItr (aipPandemonium *p) {
    set_demon_itr(p,0);
  }

  rrsResourceItr (const rrsResourceItr& x) {
    set_demon_itr (x.pandemonium(), x.current());
  }

  virtual ~rrsResourceItr () {}

  rrsResourceItr& operator = (const rrsResourceItr& x) {
    set_demon_itr(x.pandemonium(), x.current());
    return *this;
  }

  rrsResource * first () {
    return (rrsResource*)aipDemonItr::first();
  }

  rrsResource * next () {
    return (rrsResource*)aipDemonItr::next();
  }

};

//======================================================================
//  rrsResDayItr  -  Iterator for ResDays of a Resource

class rrsResDayItr : public aipDemonItr {

public:

  rrsResDayItr (aipPandemonium *p) {
    set_demon_itr(p,0);
  }

  rrsResDayItr (const rrsResDayItr& x) {
    set_demon_itr (x.pandemonium(), x.current());
  }

  virtual ~rrsResDayItr () {}

  rrsResDayItr& operator = (const rrsResDayItr& x) {
    set_demon_itr(x.pandemonium(), x.current());
    return *this;
  }

  rrsResDay * first () {
    return (rrsResDay*)aipDemonItr::first();
  }

  rrsResDay * next () {
    return (rrsResDay*)aipDemonItr::next();
  }

};

//======================================================================
//  rrsDecisionItr  -  Iterator for Decisions in a Problem

class rrsDecisionItr : public aipDemonItr {

public:

  rrsDecisionItr (aipPandemonium *p) {
    set_demon_itr(p,0);
  }

  rrsDecisionItr (const rrsDecisionItr& x) {
    set_demon_itr (x.pandemonium(), x.current());
  }

  virtual ~rrsDecisionItr () {}

  rrsDecisionItr& operator = (const rrsDecisionItr& x) {
    set_demon_itr(x.pandemonium(), x.current());
    return *this;
  }

  rrsDecision * first () {
    return (rrsDecision*)aipDemonItr::first();
  }

  rrsDecision * next () {
    return (rrsDecision*)aipDemonItr::next();
  }

};

//======================================================================
//  rrsMsg  -  message for StaffWhen problem-solving

class rrsMsg : public aipHHMsg {

public:

  rrsMsg (rrsProblem *p, short typ) : aipHHMsg(p,typ) {}
  ~rrsMsg() {}

};

//======================================================================

#endif

//======================================================================

