//**********************************************************************
//  aipTime.h  -  times and dates
//
//  Copyright (c)  2008  Brian Marshall
//
//  See the license at end of this file.
//
//  With some compilers, when an instance of aipTime is created, the
//  hour will be off if the date is not in Daylight Savings Time (DST)
//  and the current time is in DST or vice versa.
//
//  Developers/Contributers:
//    [BRM] Brian Marshall - Calgary - bmarshal@agt.net
//
//  08/08/02  [BRM] added day-abbreviation to aipTime.get()
//  08/07/09  [BRM] fixed a few bugs, improved the structure
//  08/02/03  [BRM] began development
//
//----------------------------------------------------------------------

#ifndef aipTime_h_guard
#define aipTime_h_guard

#include <time.h>

           // just for the functions and constants...
#include "aipBase.h"

//----------------------------------------------------------------------
//  Classes

class aipTime;          // time and date
class aipTimeOfDay;     // hours and minutes

//----------------------------------------------------------------------
//  Constants

static const char * aip_Day_Abbrv[7] = { 
          "Sun", "Mon", "Tue", "Wed", "Thu", "Fri", "Sat" };

static const char * aip_Month_Abbrv[12] = { 
          "Jan", "Feb", "Mar", "Apr", "May", "Jun", 
          "Jul", "Aug", "Sep", "Oct", "Nov", "Dec" };

static const int aip_Days_In_Month[12] = {
          31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31};
      //  Jan Feb Mar Apr May Jun Jul Aug Sep Oct Nov Dec

//----------------------------------------------------------------------
//  Global Functions

const char * aip_str_day (long iday);          // return "Sun" from 0

long         aip_i_month   (const char* x);    // return 1 from jan
const char * aip_str_month (long imon);        // return "Jan" from 1

int  aip_is_leap_year (long yr);

long  aip_days_in_month (long yr, long mo);


//======================================================================
//  aipTime  -  date and time
//
//  aipTime objects are small and can be copied around.
//  Comparing and determining differences is fast.
//  The following are more involved, but are generally used
//  only in setting-up the problem:
//    - creating and setting instances
//    - adding seconds or days to an instance

class aipTime {

  time_t   m_time;

protected:

  void set  (long year, long mon, long day, 
                        long hour =0, long min =0, long sec =0);
  void set  (long yyyymmdd, long hhmmss =0);
  void set  (const aipTime& x);
            // delim is any non-alphanum; hh, mm, ss are optional
  void set (const char* x);   // "17 Feb 1998" or "17/02/2008 16:30"
                              // "2008-02-17" ie. insert format

  int eq (const aipTime& x) const;
  int ne (const aipTime& x) const;
  int lt (const aipTime& x) const;
  int gt (const aipTime& x) const;
  int le (const aipTime& x) const;
  int ge (const aipTime& x) const;

public:

  aipTime () {}
  aipTime (long year, long mon, long day, 
                      long hour =0, long min =0, long sec =0);
  aipTime (long yyyymmdd, long hhmmss =0);
  aipTime (const aipTime& x) { set(x);  }
  aipTime (const char*    x) { set(x); }  // ex: "12 Feb 2008 16:30"
  ~aipTime () {}

  aipTime& operator = (const aipTime& x) {
    set(x);
    return *this;
  }

  int is_valid () const { return m_time != (time_t)(-1); }

  void set_daylight_savings_time (struct tm *ts) const;

  void add_seconds (long num);
  void add_hours   (long num);
  void add_days    (long num);

  time_t tt_time () const { return m_time; }

  long days_since_sunday () const;
  long days_since_jan1   () const;

  long seconds_since (const aipTime& x) const;
  long minutes_since (const aipTime& x) const;
  long hours_since   (const aipTime& x) const;
  long days_since    (const aipTime& x) const;

  void time_since (const aipTime& x,
                   long *days, long *hrs, 
                   long *mins, long *secs) const;

  long hours_after (const aipTime& x) const;  // num different
  long days_after  (const aipTime& x) const;  // num different

                    // ex: 2008, 7, 12, 16, 30, 0, "Jul", "Sat"
  int get (long *year,    long *mon =0, long *day =0, 
           long *hour =0, long *min =0, long *sec =0,
           char *smon =0, char *sweekday =0) const;

                              // write date string to a pointer
  void dd_mon_yyyy       (char *dstr) const;  // "17/Jan/2008"
  void dd_mon_yyyy_hh_mm (char *dstr) const;  // "17/Jan/2008_16:30"
  void dd_mm_yy          (char *dstr) const;  // "17/01/08"
  void dd_mm_yyyy        (char *dstr) const;  // "17/01/2008"
  void yyyy_mm_dd        (char *dstr) const;  // "2008-01-17"

  long yyyymmdd () const;     // return date as long 
  long hhmm     () const;     // return hhmm as long 

  int operator == (const aipTime& x) const { return eq(x); }
  int operator != (const aipTime& x) const { return ne(x); }
  int operator <  (const aipTime& x) const { return lt(x); }
  int operator >  (const aipTime& x) const { return gt(x); }
  int operator <= (const aipTime& x) const { return le(x); }
  int operator >= (const aipTime& x) const { return ge(x); }

};

//======================================================================
//  aipTimeOfDay - hours and minutes

class aipTimeOfDay {

  char   m_hour;        // 0 to 23
  char   m_minute;      // 0 to 59

protected:

  void reset () { m_hour = m_minute = -1; }
  void set (long hour, long minute);
  void set (const aipTimeOfDay& x);
  void set (const char *x);  // ex: "16:30" or "4:30 pm"

  int eq (const aipTimeOfDay& x) const;
  int ne (const aipTimeOfDay& x) const;
  int lt (const aipTimeOfDay& x) const;
  int gt (const aipTimeOfDay& x) const;
  int le (const aipTimeOfDay& x) const;
  int ge (const aipTimeOfDay& x) const;

public:

  aipTimeOfDay (long hr, long min)     { set(hr,min); }
  aipTimeOfDay (const aipTimeOfDay& x) { set(x);      }
  aipTimeOfDay (const char* x)         { set(x);      }
  ~aipTimeOfDay () {} // ex: "16:30" or "4:30 pm"

  aipTimeOfDay& operator = (const aipTimeOfDay& x) {
    set(x);
    return *this;
  }

  void add_hours   (long num);
  void add_minutes (long num);

  long hours_since   (const aipTimeOfDay& x);
  long minutes_since (const aipTimeOfDay& x);

  int is_valid () const { return m_hour >= 0; }

  long hour   () const { return (long)m_hour;   }
  long minute () const { return (long)m_minute; }

  void hh_mm    (char *dstr) const;  // writes to dstr - "16:30"
  void hh_mm_ss (char *dstr) const;  // writes to dstr - "16:30:00"
  void hh_mm_12 (char *dstr) const;  // writes to dstr - "04:30 pm"

  int operator == (const aipTimeOfDay& x) const { return eq(x); }
  int operator != (const aipTimeOfDay& x) const { return ne(x); }
  int operator <  (const aipTimeOfDay& x) const { return lt(x); }
  int operator >  (const aipTimeOfDay& x) const { return gt(x); }
  int operator <= (const aipTimeOfDay& x) const { return le(x); }
  int operator >= (const aipTimeOfDay& x) const { return ge(x); }

};

//======================================================================

#endif

//======================================================================
//                           License
//
//   Permission is hereby granted, free of charge, to any 
//   person obtaining a copy of this software and associated 
//   documentation files (the "Software"), to deal in the Software 
//   without restriction, including without limitation the rights 
//   to use, copy, modify, merge, publish, distribute, sublicense, 
//   and/or sell copies of the Software, and to permit persons to 
//   whom the Software is furnished to do so, subject to the 
//   following conditions:
//
//   The copyright notice and this license shall be included in all 
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, 
//   EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES 
//   OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND 
//   NON-INFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT 
//   HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, 
//   WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
//   FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR 
//   OTHER DEALINGS IN THE SOFTWARE.
//
//**********************************************************************
