//**********************************************************************
//  aipGood.h   -   Goodness
//
//  Provides:  aipGoodness (plus aipGBase)
//
//  This header file declares classes, constants and functions
//  to provide a "goodness" type.
//
//  See Design and Usage Notes near the end of this file.
//  The short story is:  Goodnesses are combined by adding.
//
//  Copyright (c)  1999, 2005  Brian Marshall
//
//  See the license at end of this file.
//
//  Developers/Contributers:
//    [BRM] Brian Marshall - Calgary - bmarshal@agt.net
//
//  05/10/28  [BRM] added constants aipPrettyGood and aipPrettyBad
//  05/09/07  [BRM] removing importance and likelihood
//  05/07/31  [BRM] reworking began.
//  99/05/24  [BRM] added  goodness * and / longs.
//  99/05/01  [BRM] naming conventions changed.
//  99/03/07  [BRM] development begun.
//
//----------------------------------------------------------------------

#ifndef aipGood_h_guard
#define aipGood_h_guard

//----------------------------------------------------------------------
//  Classes and typedefs.  Sub-classing is shown by indentation.

class aipGBase;
   class  aipGoodness;

typedef aipGoodness    aipG;

//======================================================================
//  aipGBase   -   Base class.
//
//  This is a base class - the constructors are protected.

class aipGBase {

  long  m_val;

protected:

  long val () const { return m_val; }

  void set (const aipGBase& x) { m_val = x.val(); }
  void set (long x) { m_val = x; }

  bool eq (const aipGBase& x) const { return m_val == x.val(); }
  bool ne (const aipGBase& x) const { return m_val != x.val(); }
  bool lt (const aipGBase& x) const { return m_val <  x.val(); }
  bool gt (const aipGBase& x) const { return m_val >  x.val(); }
  bool le (const aipGBase& x) const { return m_val <= x.val(); }
  bool ge (const aipGBase& x) const { return m_val >= x.val(); }

  void combine (const aipGBase& x) {
    set ( val() + x.val() ); 
  }

  void uncombine (const aipGBase& x) {
    set ( val() - x.val() ); 
  }

  aipGBase ()                   { m_val = 0;       }
  aipGBase (const aipGBase& x)  { m_val = x.val(); }
  aipGBase (long x)             { m_val = x;       }

public:

  ~aipGBase () {}

  long sign    () const { return ( (m_val<0) ? -1 : 1 ); }
  long is_neg  () const { return (m_val  < 0); }
  long is_pos  () const { return (m_val  > 0); }
  long is_zero () const { return (m_val == 0); }

  long is_manditory () const { return (m_val >  900000); }
  long is_forbidden () const { return (m_val < -900000); }

  const char * description () const { return ""; }

};


//======================================================================
//  aipGoodness  -  How good or bad something is.

class aipGoodness : public aipGBase {

public:

  aipGoodness ()                      : aipGBase()   {}
  aipGoodness (const aipGoodness& x)  : aipGBase(x)  {}
  aipGoodness (long x)                : aipGBase(x)  {}

  ~aipGoodness () {}

  aipGoodness& operator = (const aipGoodness& x) {
    set(x);
    return *this;
  }

  bool operator == (const aipGoodness& x) const { return eq(x); }
  bool operator != (const aipGoodness& x) const { return ne(x); }
  bool operator <  (const aipGoodness& x) const { return lt(x); }
  bool operator >  (const aipGoodness& x) const { return gt(x); }
  bool operator <= (const aipGoodness& x) const { return le(x); }
  bool operator >= (const aipGoodness& x) const { return ge(x); }

  aipGoodness& operator += (const aipGoodness& x) {
    combine(x);
    return *this;
  }

  aipGoodness& operator -= (const aipGoodness& x) {
    uncombine(x);
    return *this;
  }

  void set_numeric_value (long x) { set(x); }

  aipGoodness calc_fraction (long numerator, long denominator);

  void apply_factor_1_to_5 (long f) {
    if (f <= 1) {
      set ( val() / 4 );
    } else if (f == 2) {
      set ( val() / 2 );
    } else if (f == 4) {
      set ( val() * 2 );
    } else if (f >= 5) {
      set ( val() * 4 );
    }
  }

  long numeric_value () const { return val(); }

  long absolute_value () const { 
    return (val() >= 0) ? val() : (0 - val()); 
  }

  const char * description () const;

};


//======================================================================
//  Global functions
//
//----------------------------------------------------------------------
//      Goodness plus a Goodness yields a Goodness

aipGoodness operator + (const aipGoodness& x, 
                        const aipGoodness& y);

aipGoodness operator - (const aipGoodness& x, 
                        const aipGoodness& y);


//======================================================================
//  Constants
//
//----------------------------------------------------------------------
//  aipGoodness constants.

const aipGoodness aipForbidden         = aipGoodness( -999999 );
const aipGoodness aipVeryVeryBad       = aipGoodness(   -1024 );
const aipGoodness aipVeryBad           = aipGoodness(    -512 );
const aipGoodness aipBad               = aipGoodness(    -256 );
const aipGoodness aipQuiteBad          = aipGoodness(     -64 );
const aipGoodness aipPrettyBad         = aipGoodness(     -32 );
const aipGoodness aipFairlyBad         = aipGoodness(     -16 );
const aipGoodness aipSomewhatBad       = aipGoodness(      -8 );
const aipGoodness aipLittleBitBad      = aipGoodness(      -4 );
const aipGoodness aipSlightlyBad       = aipGoodness(      -2 );
const aipGoodness aipVerySlightlyBad   = aipGoodness(      -1 );
const aipGoodness aipNeutral           = aipGoodness(       0 );
const aipGoodness aipVerySlightlyGood  = aipGoodness(       1 );
const aipGoodness aipSlightlyGood      = aipGoodness(       2 );
const aipGoodness aipLittleBitGood     = aipGoodness(       3 );
const aipGoodness aipSomewhatGood      = aipGoodness(       4 );
const aipGoodness aipFairlyGood        = aipGoodness(       6 );
const aipGoodness aipPrettyGood        = aipGoodness(       8 );
const aipGoodness aipQuiteGood         = aipGoodness(      10 );
const aipGoodness aipGood              = aipGoodness(      12 );
const aipGoodness aipVeryGood          = aipGoodness(      16 );
const aipGoodness aipVeryVeryGood      = aipGoodness(      24 );
const aipGoodness aipManditory         = aipGoodness(  999999 );

//----------------------------------------------------------------------
//======================================================================

#endif

//======================================================================
//    Design and Usage Notes
//
//----------------------------------------------------------------------
//  Fast 4-byte objects - No Polymorphism
//
//  aipGoodness is instantiated as a 4-byte objects that are can be 
//  passed, copied and manipulated very quickly.  Most class function 
//  bodies are in this header file, to encourage the compiler to 
//  implement them inline.
//
//  The aipGoodness class is not polymorphic - the programmer is 
//  expected to know the type of the object.  The class contains no 
//  virtual functions (and the destructors do nothing).
//
//----------------------------------------------------------------------
//  Goodness
//
//  Goodness is a measure of how good or bad something is. A goodness
//  is specified using constants (defined below). Goodnesses may be:
//    - compared as if are numeric values
//    - combined by adding (separated by subtracting)
//
//  Application of Goodness:
//       In a decision, one option may be chosen over another 
//       by comparing the goodness associated with each option.
//
//  If the goodness of a situation is calculated as the sum of a large
//  number of goodness values, the value of this goodness, on its own,
//  might not have a lot of meaning.  On the other hand, comparing the
//  goodness of two situations is meaningful.
//    
//======================================================================
//                           License
//
//   Permission is hereby granted, free of charge, to any 
//   person obtaining a copy of this software and associated 
//   documentation files (the "Software"), to deal in the Software 
//   without restriction, including without limitation the rights 
//   to use, copy, modify, merge, publish, distribute, sublicense, 
//   and/or sell copies of the Software, and to permit persons to 
//   whom the Software is furnished to do so, subject to the 
//   following conditions:
//
//   The copyright notice and this license shall be included in all 
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, 
//   EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES 
//   OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND 
//   NON-INFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT 
//   HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, 
//   WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
//   FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR 
//   OTHER DEALINGS IN THE SOFTWARE.
//
//**********************************************************************

