//**********************************************************************
//  aipDecision.h   -   Decision -  a set of options, 
//                                  one of which is to be chosen.
//
//  Provides: aipDecision  aipOption  aipOptionItr
//
//  An aipDecision decision:
//   - has a set of options
//   - takes a series of messages, which it passes to its options  
//   - has a decide().
//
//  Copyright (c)  2005, 2008  Brian Marshall
//
//  See the license at end of this file.
//
//  Developers/Contributers:
//    [BRM] Brian Marshall - Calgary - bmarshal@agt.net
//
//  08/01/01  [BRM] removed delete_option(); untested trivial dcsns
//  05/11/07  [BRM] changed following comment line
//                  support for decisions sharing a set of options
//  05/10/28  [BRM] can choose randomly from multiple best options
//  05/09/17  [BRM] development begins
//
//----------------------------------------------------------------------

#ifndef aipDecision_h_guard
#define aipDecision_h_guard

#include "aipEmotion.h"

//----------------------------------------------------------------------
//  Classes.  Sub-classing is shown by indentation.

//  class  aipG  is a typedef for  aipGoodness

//  Hope is a compound emotion composed of Fear, Greed and Curiosity

// class aipBase;                      ( in aipBase.h )
//   class aipDemon;                   ( in aipPandemonium.h )
       class aipDecision;         // Set of options and decide() 
       class aipOption;           // Option in a decision.
// class aipDemonItr;                  ( in aipPandemonium.h )
     class aipOptionItr;          // option iterator


//======================================================================
//  aipDecision
//
//  A decision has:
//    - a set of options
//    - a decide() function to pick one
//
//  A decision may be set to trivial, temporarily or permanently, and:
//  decide() will choose the first non-forbidden option it finds.
//  A decision would be trivial if it had only one (remaining) 
//  non-forbidden option.  A set of decisions would each be trivial 
//  if they share a set of options and there are just enough.
//
//  A set of options may be shared by more than one decision,
//  so long as an option can only be chosen by one decision.
//  The first decision to which an option is added becomes its
//  owner for destruction purposes.  
//
//  Use of m_owner, here, for decisions, is not required.

class aipDecision : public aipDemon {

  aipBase        * m_owner;     // Owner of this decision or zero.

  aipPandemonium * m_options;

  int              m_is_trivial;

protected:

  aipPandemonium * option_pandemonium() const {  // to make iterators
    return m_options;
  }

public:

  aipDecision ();
  virtual ~aipDecision ();

  void set_owner (aipBase *x) { m_owner = x; }

  void add_option (aipOption *x);

  void   set_is_trivial () { m_is_trivial = 1; }
  void reset_is_trivial () { m_is_trivial = 0; }

  virtual void take_msg (aipMsg *m);

  aipOptionItr option_iterator() const;

  aipBase * owner () const { return m_owner; }

  virtual aipG g() { return aipNeutral; }

  virtual aipOption * decide();                     // decide

};


//======================================================================
//  aipOption  -  an option owned by a decision
//
//  This is a base class - subclasses can provide:
//    g_opt() to return a goodness used for deciding.
//    g_opt_usr() to return goodness to report to users.
//
//  The owner of an option is a decision.  However, options can
//  be shared, and if this is happening, m_owner will not necessarily
//  be the decision that picked the option.
//
//  The owner of an option is automatically set 
//  in aipDecision::add_option().

class aipOption : public aipDemon {

  aipDecision  * m_owner;        // owner for destruction purposes
  aipDecision  * m_chooser;      // decision that chose this option

  int            m_is_shared;    // option shared between decisions

public:

  aipOption () { m_owner = m_chooser = 0; m_is_shared = 0; }
  virtual ~aipOption () {}

  void set_owner     (aipDecision *x) { m_owner = x;     }
  void set_chooser   (aipDecision *x) { m_chooser = x;   }
  void set_is_shared ()               { m_is_shared = 1; }

  virtual void take_msg (aipMsg *m) {}

  virtual aipG g_opt() { return aipNeutral; }

  virtual aipG g_opt_usr() { return aipNeutral; }

  void reset_chooser () { m_chooser = 0; }

  aipDecision * owner     () const { return m_owner;     }
  aipDecision * chooser   () const { return m_chooser;   }
  int           is_shared () const { return m_is_shared; }

};


//======================================================================
//  aipOptionItr  -  Iterator for options in a decision

class aipOptionItr : public aipDemonItr {

public:

  aipOptionItr () : aipDemonItr() {}

  aipOptionItr (aipPandemonium *p) {
    set_demon_itr (p,0);
  }

  aipOptionItr (const aipOptionItr& x) {
    set_demon_itr (x.pandemonium(), x.current());
  }

  virtual ~aipOptionItr () {}

  aipOptionItr& operator = (const aipOptionItr& x) {
    set_demon_itr (x.pandemonium(), x.current());
    return *this;
  }

  aipOption * first() { return (aipOption*)aipDemonItr::first(); }
  aipOption * next()  { return (aipOption*)aipDemonItr::next();  }

};


//======================================================================

#endif

//======================================================================
//                           License
//
//   Permission is hereby granted, free of charge, to any 
//   person obtaining a copy of this software and associated 
//   documentation files (the "Software"), to deal in the Software 
//   without restriction, including without limitation the rights 
//   to use, copy, modify, merge, publish, distribute, sublicense, 
//   and/or sell copies of the Software, and to permit persons to 
//   whom the Software is furnished to do so, subject to the 
//   following conditions:
//
//   The copyright notice and this license shall be included in all 
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, 
//   EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES 
//   OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND 
//   NON-INFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT 
//   HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, 
//   WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
//   FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR 
//   OTHER DEALINGS IN THE SOFTWARE.
//
//**********************************************************************

