;;; -*-Scheme-*-
;;;
;;; $Id: cache.sc,v 1.4 1993/02/25 17:13:49 cph Exp $
;;;
;;; Copyright (c) 1993 Massachusetts Institute of Technology
;;;
;;; This material was developed by the Scheme project at the
;;; Massachusetts Institute of Technology, Department of Electrical
;;; Engineering and Computer Science.  Permission to copy this
;;; software, to redistribute it, and to use it for any purpose is
;;; granted, subject to the following restrictions and understandings.
;;;
;;; 1. Any copy made of this software must include this copyright
;;; notice in full.
;;;
;;; 2. Users of this software agree to make their best efforts (a) to
;;; return to the MIT Scheme project any improvements or extensions
;;; that they make, so that these may be included in future releases;
;;; and (b) to inform MIT of noteworthy uses of this software.
;;;
;;; 3. All materials developed as a consequence of the use of this
;;; software shall duly acknowledge such use, in accordance with the
;;; usual standards of acknowledging credit in academic research.
;;;
;;; 4. MIT has made no warrantee or representation that the operation
;;; of this software will be error-free, and MIT is under no
;;; obligation to provide any services, by way of maintenance, update,
;;; or otherwise.
;;;
;;; 5. In conjunction with products arising from the use of this
;;; material, there shall be no use of the name of the Massachusetts
;;; Institute of Technology nor of any adaptation thereof in any
;;; advertising, promotional, or sales literature without prior
;;; written consent from MIT in each case.

;;;; Method Caches for Scheme Object System

;;; From "Efficient Method Dispatch in PCL", Gregor Kiczales and Luis
;;; Rodriguez, Proceedings of the 1990 ACM Conference on Lisp and
;;; Functional Programming.  Parts of this code are based on the
;;; September 16, 1992 PCL implementation.

(module soscache (top-level))
(include "mitutil.sch")
(include "random.sch")
(include "cache.sch")

(define-in-line (bit-and x y)
  ((lap (x y)
	(S2CUINT_TSCP (BITAND32 (TSCP_S2CUINT x)
				(TSCP_S2CUINT y))))
   x y))

(define (make-wrapper class)
  (let ((wrapper (make-vector (+ wrapper-length 2))))
    (do ((i 0 (+ i 1)))
	((= i wrapper-length))
      (vector-set! wrapper i (get-wrapper-cache-number)))
    (vector-set! wrapper wrapper-length class)
    (vector-set! wrapper (+ wrapper-length 1) wrapper-tag)
    wrapper))

(define wrapper-tag
  (list 'WRAPPER-TAG))

(define (wrapper-ref wrapper index)
  (wrapper-ref wrapper index))

(define-in-line (wrapper-set! wrapper index cache-number)
  (vector-set! wrapper index cache-number))

(define (wrapper-class wrapper)
  (wrapper-class wrapper))

(define-in-line (next-wrapper-index index)
  (let ((index (+ index 1)))
    (and (< index wrapper-length)
	 index)))

(define-in-line (wrapper-invalid? wrapper)
  (or (not wrapper)
      (= (wrapper-ref wrapper 0) 0)))

(define-constant wrapper-cache-number-adds-ok
  ;; This constant controls the number of non-zero bits wrapper cache
  ;; numbers will have.
  ;;
  ;; The value of this constant is the number of wrapper cache numbers
  ;; which can be added and still be certain the result will be a
  ;; fixnum.  This is used by all the code that computes primary cache
  ;; locations from multiple wrappers.
  ;;
  ;; The value of this constant is used to derive the next two which
  ;; are the forms of this constant which it is more convenient for
  ;; the runtime code to use.
  4)

(define-constant wrapper-cache-number-supremum
  ;; Correct value of this constant is
  ;;   (quotient (expt 2 29) wrapper-cache-number-adds-ok)
  ;; but this can't be computed using fixnum arithmetic.
  (expt 2 27))

(define-constant wrapper-cache-number-mask
  ;; Correct value is
  ;;   (- wrapper-cache-number-supremum 1)
  ;; but compiler won't hack constant in right-hand side.
  (- (expt 2 27) 1))

(define (get-wrapper-cache-number)
  (let ((n
	 (random wrapper-cache-number-supremum
		 wrapper-cache-number-state)))
    (if (= n 0)
	(get-wrapper-cache-number)
	n)))

(define wrapper-cache-number-state
  (make-random-state))

(define-structure (cache (constructor %make-cache))
  wrapper-index
  mask
  limit
  wrappers
  values
  overflow)

(define (new-cache)
  (make-cache 0 4))

(define (make-cache wrapper-index length)
  ;; LENGTH is assumed to be a power of two.
  (%make-cache wrapper-index
	       (- length 1)
	       (cond ((<= length 4) 1)
		     ((<= length 16) 4)
		     (else 6))
	       (make-vector length #f)
	       (make-vector length #f)
	       '()))

(define-in-line (cache-length cache)
  (vector-length (cache-wrappers cache)))

(define-in-line (cache-line-wrappers cache line)
  (vector-ref (cache-wrappers cache) line))

(define-in-line (set-cache-line-wrappers! cache line wrappers)
  (vector-set! (cache-wrappers cache) line wrappers))

(define-in-line (cache-line-value cache line)
  (vector-ref (cache-values cache) line))

(define-in-line (set-cache-line-value! cache line value)
  (vector-set! (cache-values cache) line value))

(define-in-line (cache-next-line cache line)
  (let ((line (+ line 1)))
    (if (= line (cache-length cache))
	0
	line)))

(define-in-line (cache-line-separation cache line line*)
  (let ((n (- line* line)))
    (if (< n 0)
	(+ n (cache-length cache))
	n)))

(define (probe-cache cache wrappers)
  (let ((line (compute-primary-cache-line cache wrappers)))
    (and line
	 (let ((limit (cache-limit cache)))
	   (letrec
	       ((search-lines
		 (lambda (line i)
		   (cond ((let ((wrappers (cache-line-wrappers cache line)))
			    (and wrappers
				 (match wrappers)))
			  (cache-line-value cache line))
			 ((= i limit)
			  (search-overflow (cache-overflow cache)))
			 (else
			  (search-lines (cache-next-line cache line)
					(+ i 1))))))
		(search-overflow
		 (lambda (overflow)
		   (and (not (null? overflow))
			(if (match (caar overflow))
			    (cdar overflow)
			    (search-overflow (cdr overflow))))))
		(match
		 (lambda (wrappers*)
		   (let loop ((w1 wrappers*) (w2 wrappers))
		     (and (eq? (car w1) (car w2))
			  (or (null? (cdr w1))
			      (loop (cdr w1) (cdr w2))))))))
	     (search-lines line 0))))))

(define (compute-primary-cache-line cache wrappers)
  (let ((index (cache-wrapper-index cache))
	(mask (cache-mask cache)))
    (let loop ((wrappers wrappers) (line 0) (n-adds 0))
      (cond ((null? wrappers)
	     (bit-and line mask))
	    ((wrapper-invalid? (car wrappers))
	     #f)
	    ((= n-adds wrapper-cache-number-adds-ok)
	     (loop (cdr wrappers)
		   (bit-and (+ line (wrapper-ref (car wrappers) index))
			    wrapper-cache-number-mask)
		   1))
	    (else
	     (loop (cdr wrappers)
		   (+ line (wrapper-ref (car wrappers) index))
		   (+ n-adds 1)))))))

(define (cache-entry-reusable? wrappers wrappers*)
  ;; True iff WRAPPERS is (1) empty, (2) contains a wrapper that is
  ;; invalid, or (3) has the same wrappers as WRAPPERS*.
  (or (not wrappers)
      (let loop ((wrappers wrappers) (wrappers* wrappers*))
	(or (null? wrappers)
	    (wrapper-invalid? (car wrappers))
	    (and (eq? (car wrappers) (car wrappers*))
		 (loop (cdr wrappers) (cdr wrappers*)))))))

(define (cache-count cache)
  (let ((length (cache-length cache)))
    (do ((line 0 (+ line 1))
	 (count 0
		(if (let ((wrappers (cache-line-wrappers cache line)))
		      (and wrappers
			   (let loop ((wrappers wrappers))
			     (or (null? wrappers)
				 (and (not (wrapper-invalid? (car wrappers)))
				      (loop (cdr wrappers)))))))
		    (+ count 1)
		    count)))
	((= line length) count))))

(define-smacro (probe-cache-body hash match)
  `(LET ((LIMIT (CACHE-LIMIT CACHE)))
     (LET LOOP ((LINE (BIT-AND ,hash (CACHE-MASK CACHE))) (I 0))
       (COND ((LET ((WRAPPERS (CACHE-LINE-WRAPPERS CACHE LINE)))
		(AND WRAPPERS
		     ,match))
	      (CACHE-LINE-VALUE CACHE LINE))
	     ((= I LIMIT)
	      (LET LOOP ((ENTRIES (CACHE-OVERFLOW CACHE)))
		(AND (NOT (NULL? ENTRIES))
		     (IF (LET ((WRAPPERS (CAR (CAR ENTRIES))))
			   ,match)
			 (CDR (CAR ENTRIES))
			 (LOOP (CDR ENTRIES))))))
	     (ELSE
	      (LOOP (CACHE-NEXT-LINE CACHE LINE) (+ I 1)))))))

(define (probe-cache-1 cache w1)
  (let ((h1 (wrapper-ref w1 (cache-wrapper-index cache))))
    (and (not (= h1 0))
	 (probe-cache-body h1 (eq? w1 (car wrappers))))))

(define (probe-cache-2 cache w1 w2)
  (let ((index (cache-wrapper-index cache)))
    (let ((h1 (wrapper-ref w1 index))
	  (h2 (wrapper-ref w2 index)))
      (and (not (= h1 0))
	   (not (= h2 0))
	   (probe-cache-body (+ h1 h2)
			     (and (eq? w1 (car wrappers))
				  (eq? w2 (car (cdr wrappers)))))))))

(define (probe-cache-3 cache w1 w2 w3)
  (let ((index (cache-wrapper-index cache)))
    (let ((h1 (wrapper-ref w1 index))
	  (h2 (wrapper-ref w2 index))
	  (h3 (wrapper-ref w3 index)))
      (and (not (= h1 0))
	   (not (= h2 0))
	   (not (= h3 0))
	   (probe-cache-body (+ h1 h2 h3)
			     (and (eq? w1 (car wrappers))
				  (eq? w2 (car (cdr wrappers)))
				  (eq? w3 (car (cdr (cdr wrappers))))))))))

(define (probe-cache-4 cache w1 w2 w3 w4)
  (let ((index (cache-wrapper-index cache)))
    (let ((h1 (wrapper-ref w1 index))
	  (h2 (wrapper-ref w2 index))
	  (h3 (wrapper-ref w3 index))
	  (h4 (wrapper-ref w4 index)))
      (and (not (= h1 0))
	   (not (= h2 0))
	   (not (= h3 0))
	   (not (= h4 0))
	   (probe-cache-body
	    (+ h1 h2 h3 h4)
	    (and (eq? w1 (car wrappers))
		 (eq? w2 (car (cdr wrappers)))
		 (eq? w3 (car (cdr (cdr wrappers))))
		 (eq? w4 (car (cdr (cdr (cdr wrappers)))))))))))

(define (fill-cache cache wrappers value)
  (or (fill-cache-if-possible cache wrappers value)
      (and (< (* (cache-count cache) 5) (* (cache-length cache) 4))
	   (adjust-cache cache wrappers value))
      (expand-cache cache wrappers value)))

(define (fill-cache-if-possible cache wrappers value)
  (let ((primary (compute-primary-cache-line cache wrappers)))
    (if primary
	(let ((free (find-free-cache-line cache primary wrappers)))
	  (and free
	       (begin
		 (set-cache-line-wrappers! cache free wrappers)
		 (set-cache-line-value! cache free value)
		 cache)))
	;; If WRAPPERS contains an invalid wrapper, then do nothing
	;; and return CACHE -- the fill is no longer needed.  While
	;; other logic tries to eliminate this case, it can still
	;; happen when one of the wrappers is GCed during complex
	;; cache operations.
	cache)))

(define (adjust-cache cache wrappers value)
  ;; Try to rehash the cache.  If that fails, try rehashing with
  ;; different wrapper indexes.  Fail only when all of the wrapper
  ;; indexes have been tried and none has worked.
  (let ((length (cache-length cache)))
    (let ((new-cache (make-cache (cache-wrapper-index cache) length)))
      (letrec
	  ((fill-lines
	    (lambda (line)
	      (cond ((= line length)
		     (fill-overflow (cache-overflow cache)))
		    ((try-entry (cache-line-wrappers cache line)
				(cache-line-value cache line))
		     (fill-lines (+ line 1)))
		    (else
		     (try-next-wrapper-index)))))
	   (fill-overflow
	    (lambda (entries)
	      (cond ((null? entries)
		     (or (fill-cache-if-possible new-cache wrappers value)
			 (try-next-wrapper-index)))
		    ((try-entry (caar entries) (cdar entries))
		     (fill-overflow (cdr entries)))
		    (else
		     (try-next-wrapper-index)))))
	   (try-entry
	    (lambda (wrappers* value)
	      (or (cache-entry-reusable? wrappers* wrappers)
		  (fill-cache-if-possible new-cache wrappers* value))))
	   (try-next-wrapper-index
	    (lambda ()
	      (let ((index
		     (next-wrapper-index (cache-wrapper-index new-cache))))
		(and index
		     (begin
		       (set-cache-wrapper-index! new-cache index)
		       (fill-lines 0)))))))
	(fill-lines 0)))))

(define (expand-cache cache wrappers value)
  ;; Create a new cache that is twice the length of CACHE, rehash the
  ;; contents of CACHE into the new cache, and make the new entry.
  ;; Permits overflows to occur in the new cache.
  (let ((length (cache-length cache)))
    (letrec
	((fill-lines
	  (lambda (new-cache line)
	    (if (= line length)
		(fill-overflow new-cache (cache-overflow cache))
		(fill-lines (maybe-do-fill new-cache
					   (cache-line-wrappers cache line)
					   (cache-line-value cache line))
			    (+ line 1)))))
	 (fill-overflow
	  (lambda (new-cache overflow)
	    (if (null? overflow)
		(do-fill new-cache wrappers value)
		(fill-overflow (maybe-do-fill new-cache
					      (caar overflow)
					      (cdar overflow))
			       (cdr overflow)))))
	 (maybe-do-fill
	  (lambda (cache wrappers* value)
	    (if (cache-entry-reusable? wrappers* wrappers)
		cache
		(do-fill cache wrappers* value))))
	 (do-fill
	  (lambda (cache wrappers value)
	    (let ((primary (compute-primary-cache-line cache wrappers)))
	      (if primary
		  (let ((free (find-free-cache-line cache primary wrappers)))
		    (if free
			(begin
			  (set-cache-line-wrappers! cache free wrappers)
			  (set-cache-line-value! cache free value)
			  cache)
			(or (adjust-cache cache wrappers value)
			    (begin
			      (set-cache-overflow!
			       cache
			       (cons (cons (cache-line-wrappers cache primary)
					   (cache-line-value cache primary))
				     (cache-overflow cache)))
			      (set-cache-line-wrappers! cache primary wrappers)
			      (set-cache-line-value! cache primary value)
			      cache))))
		  cache)))))
      (fill-lines (make-cache (cache-wrapper-index cache)
			      (+ length length))
		  0))))

(define (find-free-cache-line cache primary wrappers)
  ;; This procedure searches cache for a free line to hold an entry
  ;; with the given PRIMARY cache number and WRAPPERS.  Since the
  ;; entry can only be stored within (CACHE-LIMIT CACHE) lines of
  ;; PRIMARY, we either have to find a free line within that limit, or
  ;; we have to find a line with a larger primary which can be
  ;; displaced to another free line within *its* limit.
  (if (cache-entry-reusable? (cache-line-wrappers cache primary) wrappers)
      primary
      (let ((limit (cache-limit cache)))
	;; Find a line for an entry whose primary cache number is P.
	;; LINES is the sequence of entries that is waiting to be
	;; displaced into the line if we find it.
	(let pri-loop
	    ((line (cache-next-line cache primary))
	     (p primary)
	     (wrappers wrappers)
	     (lines '()))
	  (let sec-loop
	      ((line line)
	       (nsep (cache-line-separation cache p line)))
	    (cond ((= line primary)
		   ;; We've scanned through the entire cache without
		   ;; finding a usable line.
		   #f)
		  ((let ((wrappers* (cache-line-wrappers cache line)))
		     (and (not (cache-entry-reusable? wrappers* wrappers))
			  (compute-primary-cache-line cache wrappers*)))
		   =>
		   (lambda (lp)
		     (let ((osep (cache-line-separation cache lp line)))
		       (cond ((>= osep limit)
			      ;; This line contains an entry that is
			      ;; displaced to the limit.  [**** For
			      ;; some reason I don't understand, this
			      ;; terminates the search.]
			      #f)
			     ((or (> nsep osep)
				  (and (= nsep osep)
				       (= 0 (random 2))))
			      ;; The entry we're trying to place is
			      ;; further from its primary than the
			      ;; entry currently stored in this line.
			      ;; So now let's look for somewhere to
			      ;; displace the entry in this line.
			      (pri-loop (cache-next-line cache line)
					lp
					(cache-line-wrappers cache line)
					(cons line lines)))
			     (else
			      (sec-loop (cache-next-line cache line)
					(+ nsep 1)))))))
		  (else
		   ;; Found a free line.  First perform all of the
		   ;; entry displacements, then return the subsequent
		   ;; free line.
		   (let loop ((free-line line) (lines lines))
		     (if (null? lines)
			 (begin
			   (set-cache-line-wrappers! cache free-line #f)
			   (set-cache-line-value! cache free-line #f)
			   free-line)
			 (let ((line (car lines)))
			   (set-cache-line-wrappers!
			    cache
			    free-line
			    (cache-line-wrappers cache line))
			   (set-cache-line-value!
			    cache
			    free-line
			    (cache-line-value cache line))
			   (loop line (cdr lines))))))))))))