;;; -*-Scheme-*-
;;;
;;;	$Header: numunp.scm,v 13.42 87/07/07 20:28:32 GMT cph Rel $
;;;
;;;	Copyright (c) 1987 Massachusetts Institute of Technology
;;;
;;;	This material was developed by the Scheme project at the
;;;	Massachusetts Institute of Technology, Department of
;;;	Electrical Engineering and Computer Science.  Permission to
;;;	copy this software, to redistribute it, and to use it for any
;;;	purpose is granted, subject to the following restrictions and
;;;	understandings.
;;;
;;;	1. Any copy made of this software must include this copyright
;;;	notice in full.
;;;
;;;	2. Users of this software agree to make their best efforts (a)
;;;	to return to the MIT Scheme project any improvements or
;;;	extensions that they make, so that these may be included in
;;;	future releases; and (b) to inform MIT of noteworthy uses of
;;;	this software.
;;;
;;;	3. All materials developed as a consequence of the use of this
;;;	software shall duly acknowledge such use, in accordance with
;;;	the usual standards of acknowledging credit in academic
;;;	research.
;;;
;;;	4. MIT has made no warrantee or representation that the
;;;	operation of this software will be error-free, and MIT is
;;;	under no obligation to provide any services, by way of
;;;	maintenance, update, or otherwise.
;;;
;;;	5. In conjunction with products arising from the use of this
;;;	material, there shall be no use of the name of the
;;;	Massachusetts Institute of Technology nor of any adaptation
;;;	thereof in any advertising, promotional, or sales literature
;;;	without prior written consent from MIT in each case.
;;;

;;;; Number Unparser

(declare (usual-integrations))

(define number->string)

(define number-unparser-package
  (make-environment

(define *radix*)
(define *precision*)
(define *precision-error* 12)

;;; This is to handle special floating point numbers such as those
;;; generated by the 68881 chip on the SUNs. e.g. NAN, INF...  The
;;; function either returns #F or a list of characters to be printed.

(define *special-float-unparser-hook*
  (lambda (number)
    false))

(set! number->string
  (named-lambda (number->string number #!optional format)
    (cond ((unassigned? format)
	   (default-number->string number *unparser-radix*))
	  ((pair? format)
	   (case (car format)
	     ((HEUR)
	      (parse-format-tail (cdr format)
		(lambda (exactness-expressed radix radix-expressed)
		  (unparse-number-heuristically
		   number
		   (case radix ((B) 2) ((O) 8) ((#F D) 10) ((X) 16))
		   (case radix-expressed (#F 'HEUR) ((E) true) ((S) false))
		   (eq? exactness-expressed 'E)))))
	     ((INT RAT FIX FLO SCI RECT POLAR)
	      (error "Unimplemented format type" format))
	     (else
	      (error "Illegal format type" format))))
	  ((integer? format)
	   (default-number->string number format))
	  (else
	   (error "Illegal format" format)))))

(define (default-number->string number radix)
  (if (memv radix '(2 8 10 16))
      (unparse-number-heuristically number radix 'HEUR false)
      (error "Illegal unparser radix" radix)))

(define (parse-format-tail tail receiver)
  (define (loop tail exactness-expressed radix radix-expressed)
    (if (null? tail)
	(receiver exactness-expressed radix radix-expressed)
	(let ((modifier (car tail))
	      (tail (cdr tail)))
	  (let ((specify-modifier
		 (lambda (old)
		   (if old
		       (error "Respecification of format modifier"
			      (cadr modifier))
		       (cadr modifier)))))
	    (cond ((and (pair? modifier)
			(eq? (car modifier) 'EXACTNESS)
			(pair? (cdr modifier))
			(memq (cadr modifier) '(E S))
			(null? (cddr modifier)))
		   (loop tail
			 (specify-modifier exactness-expressed)
			 radix
			 radix-expressed))
		  ((and (pair? modifier)
			(eq? (car modifier) 'RADIX)
			(pair? (cdr modifier))
			(memq (cadr modifier) '(B O D X))
			(or (null? (cddr modifier))
			    (pair? (cddr modifier))
			    (memq (caddr modifier) '(E S))
			    (null? (cdddr modifier))))
		   (loop tail
			 exactness-expressed
			 (specify-modifier radix)
			 (if (null? (cddr modifier)) 'E (caddr modifier))))
		  (else
		   (error "Illegal format modifier" modifier)))))))
  (loop tail false false false))

(define (mantissa-precision radix)
  (round
   (/ (- (access floating-mantissa-bits implementation-dependencies)
	 *precision-error*)
      (/ (log radix) (log 2)))))

(define (unparse-number-heuristically number radix radix-expressed?
				      exactness-expressed?)
  (let ((exactness-prefix
	 (if exactness-expressed?
	     (prefix (if (exact? number) #\e #\i))
	     identity-procedure)))

    (define (unparse-signed-real real)
      (if (negative? real)
	  (cons #\- (unparse-unsigned-real (- real)))
	  (unparse-unsigned-real real)))

    (define (unparse-unsigned-real ureal)
      (exactness-prefix
       ((if (or (eq? radix-expressed? true)
		(and (eq? radix-expressed? 'HEUR)
		     (not (= radix 10))
		     (or (not (integer? ureal))
			 (>= (abs ureal) (min radix 10)))))
	    (prefix (case radix ((2) #\b) ((8) #\o) ((10) #\d) ((16) #\x)))
	    identity-procedure)
	((if (integer? ureal)
	     unparse-unsigned-integer
	     unparse-float)
	 ureal))))

    (list->string
     (fluid-let ((*radix* radix)
		 (*precision* (mantissa-precision radix)))
       (cond ((real? number)
	      (unparse-signed-real number))
	     ((complex? number)
	      (append (unparse-signed-real (real-part number))
		      (let ((im (imag-part number)))
			(if (negative? im)
			    (cons #\- (unparse-unsigned-real (- im)))
			    (cons #\+ (unparse-unsigned-real im))))
		      (list imaginary-unit-name)))
	     (else
	      (error "NUMBER->STRING: Not a number")))))))

(define ((prefix char) chars)
  (cons* #\# char chars))

(define imaginary-unit-name
  #\i)

(define (unparse-signed-integer integer)
  (if (negative? integer)
      (cons #\- (unparse-unsigned-integer (- integer)))
      (unparse-unsigned-integer integer)))

(define unparse-unsigned-integer
  (let ((bignum-type (microcode-type 'BIG-FIXNUM))
	(listify-bignum (make-primitive-procedure 'LISTIFY-BIGNUM)))
    (named-lambda (unparse-unsigned-integer integer)
      (cond ((zero? integer)
	     (list #\0))
	    ((primitive-type? bignum-type integer)
	     (map char-value (listify-bignum integer *radix*)))
	    (else
	     (unparse-positive-fixnum integer '()))))))

(define (unparse-positive-fixnum n tail)
  (if (zero? n)
      tail
      (let ((q (integer-divide n *radix*)))
	(unparse-positive-fixnum
	 (integer-divide-quotient q)
	 (cons (char-value (integer-divide-remainder q))
	       tail)))))

(define (char-value digit)
  (digit->char digit *radix*))

;;; Clever algorithm for generating a list of digits to be printed,
;;; suppressing trailing zeros.  It turns 3.499999 into 3.5 as well.  

(define (unparse-float x)
  (or (*special-float-unparser-hook* x)
      (normalize x
	(lambda (x initial-exponent)
	  (floating-digits x
	    (lambda (exponent-increment digits)
	      (let ((exponent (+ initial-exponent exponent-increment)))
		(cond ((and (>= exponent 0) ;type out as 100.3
			    (< exponent *precision*))
		       (spit-out-big digits exponent))
		      ((and (< exponent 0) ;type out as .0015
			    (<= (- (length digits) (1+ exponent))
				*precision*))
		       (cons #\.
			     (let spit-out-small ((exponent (1+ exponent)))
			       (if (zero? exponent)
				   (map char-value digits)
				   (cons #\0
					 (spit-out-small (1+ exponent)))))))
		      (else		;type out as 9.2E16
		       (cons*
			(char-value (car digits)) #\.
			(append!
			 (map char-value (cdr digits))
			 (cons #\e
			       (unparse-signed-integer exponent)))))))))))))

(define (spit-out-big digits exponent)
  (cons (char-value (car digits))
	(cond ((null? (cdr digits))
	       (let fill-out-with-zeros ((exponent exponent))
		 (if (zero? exponent)
		     '(#\.)
		     (cons #\0 (fill-out-with-zeros (-1+ exponent))))))
	      ((zero? exponent)
	       (cons #\. (map char-value (cdr digits))))
	      (else
	       (spit-out-big (cdr digits) (-1+ exponent))))))

(define (normalize x receiver)
  (define (loop x count)
    (cond ((< x 1) (loop (* x *radix*) (-1+ count)))
	  ((>= x *radix*) (loop (/ x *radix*) (1+ count)))
	  (else (receiver x count))))
  (loop x 0))

;;; Returns (exponent-increment digits) -- a list of digits flagged if
;;; the exponent is to be incremented.

(define (floating-digits x receiver)
  (define (clip indigits outdigits carry)
    (if (null? indigits)
	(if (null? outdigits)
	    (if (= carry 1)
		(receiver 1 (list 1))
		(error "Clip failure -- floating-printer -- call GJS" carry))
	    (receiver carry outdigits))
	(let ((ndigit (+ carry (car indigits))))
	  (cond ((zero? ndigit)
		 (clip (cdr indigits) outdigits 0))
		((= *radix* ndigit)
		 (clip (cdr indigits) outdigits 1))
		((< *radix* ndigit)
		 (clip (cdr indigits)
		       (cons (- ndigit *radix*) outdigits)
		       1))
		(else
		 (receiver 0
			   (append (reverse (cdr indigits))
				   (cons ndigit outdigits))))))))

  (define (make-digits x p digits)
    (if (= p 1)
	(cons (round x) digits)
	(let ((digit (floor x)))
	  (make-digits (* (- x digit) *radix*)
		       (-1+ p)
		       (cons digit digits)))))

  (clip (make-digits x *precision* '()) '() 0))

;;; end NUMBER-UNPARSER-PACKAGE
))
