/* -*-C-*-

Copyright (c) 1987 Massachusetts Institute of Technology

This material was developed by the Scheme project at the Massachusetts
Institute of Technology, Department of Electrical Engineering and
Computer Science.  Permission to copy this software, to redistribute
it, and to use it for any purpose is granted, subject to the following
restrictions and understandings.

1. Any copy made of this software must include this copyright notice
in full.

2. Users of this software agree to make their best efforts (a) to
return to the MIT Scheme project any improvements or extensions that
they make, so that these may be included in future releases; and (b)
to inform MIT of noteworthy uses of this software.

3. All materials developed as a consequence of the use of this
software shall duly acknowledge such use, in accordance with the usual
standards of acknowledging credit in academic research.

4. MIT has made no warrantee or representation that the operation of
this software will be error-free, and MIT is under no obligation to
provide any services, by way of maintenance, update, or otherwise.

5. In conjunction with products arising from the use of this material,
there shall be no use of the name of the Massachusetts Institute of
Technology nor of any adaptation thereof in any advertising,
promotional, or sales literature without prior written consent from
MIT in each case. */

/* $Header: Sgraph.c,v 1.1 87/05/28 00:52:07 GMT jinx Rel $

   Simple graphics for HP 9000 series 300 machines. 
*/

#include "scheme.h"
#include "primitive.h"
#include "flonum.h"
#include "Sgraph.h"

int screen_handle;
long replacement_rule = DEFAULT_REPLACEMENT_RULE;
float xposition, yposition;

void
C_Init_Graphics()
{
  screen_handle = gopen( "/dev/crt", OUTDEV, STARBASE_DRIVER_NAME, 0);
  if ( screen_handle == -1 )
    Primitive_Error( ERR_EXTERNAL_RETURN);
  vdc_extent( screen_handle,
	      STARBASE_XMIN, STARBASE_YMIN, STARBASE_ZMIN,
	     STARBASE_XMAX, STARBASE_YMAX, STARBASE_ZMAX);
  clip_rectangle( screen_handle,
		  STARBASE_XMIN, STARBASE_XMAX,
		  STARBASE_YMIN, STARBASE_YMAX);
  clear_control( screen_handle, CLEAR_CLIP_RECTANGLE);
  drawing_mode( screen_handle, DEFAULT_REPLACEMENT_RULE);
  replacement_rule = DEFAULT_REPLACEMENT_RULE;
  text_alignment( screen_handle, TA_NORMAL_HORIZONTAL, TA_NORMAL_VERTICAL,
		 0.0, 0.0);
  interior_style( screen_handle, INT_HOLLOW, 1);
  perimeter_color_index( screen_handle, 1);
  return;
}

Define_Primitive( Prim_Graphics_Initialize, 0, "GRAPHICS-INITIALIZE")
{
  Primitive_0_Args();

  C_Init_Graphics();
  PRIMITIVE_RETURN(NIL);
}

/* Uses the Starbase CLEAR_VIEW_SURFACE procedure.
   Clears the Starbase default area. */

C_Clear_Graphics()
{
  xposition = 0.0;
  yposition = 0.0;
  move2d( screen_handle, xposition, yposition);
  clear_view_surface( screen_handle);
  make_picture_current( screen_handle);
  return;
}

/* (GRAPHICS-CLEAR)
   Clear the graphics section of the screen.
   An internal C procedure does the real work.
*/
Define_Primitive( Prim_Graphics_Clear, 0, "GRAPHICS-CLEAR")
{
  Primitive_0_Args();

  C_Clear_Graphics();
  PRIMITIVE_RETURN(NIL);
}

/* (GRAPHICS-MOVE X Y)
   Uses the Starbase routine MOVE2D to pick up the pen and move
   to the position (X, Y).  Both must be Scheme FIXNUMs or FLONUMs. */

Define_Primitive( Prim_Graphics_Move, 2, "GRAPHICS-MOVE")
{
  long fixnum;
  Primitive_2_Args();

  Make_Flonum( Arg1, xposition, fixnum, ERR_ARG_1_WRONG_TYPE);
  Make_Flonum( Arg2, yposition, fixnum, ERR_ARG_2_WRONG_TYPE);
  move2d( screen_handle, xposition, yposition);
  make_picture_current( screen_handle);

  PRIMITIVE_RETURN(NIL);
}

/* (GRAPHICS-LINE X Y)
   Uses the Starbase routine DRAW2D to first make sure the current
   pen is down and uses the current line type to draw to position
   indicated by X and Y. */

Define_Primitive( Prim_Graphics_Line, 2, "GRAPHICS-LINE")
{
  long fixnum;
  Primitive_2_Args();

  Make_Flonum( Arg1, xposition, fixnum, ERR_ARG_1_WRONG_TYPE);
  Make_Flonum( Arg2, yposition, fixnum, ERR_ARG_2_WRONG_TYPE);
  draw2d( screen_handle, xposition, yposition);
  make_picture_current( screen_handle);

  PRIMITIVE_RETURN(NIL);
}

/* (GRAPHICS-PIXEL X Y)
   This routine plots one point on the screen at (X, Y).
   Again, X and Y must FIXNUMs or FLONUMs. */

Define_Primitive( Prim_Graphics_Pixel, 2, "GRAPHICS-PIXEL")
{
  long fixnum;
  Primitive_2_Args();

  Make_Flonum( Arg1, xposition, fixnum, ERR_ARG_1_WRONG_TYPE);
  Make_Flonum( Arg2, yposition, fixnum, ERR_ARG_2_WRONG_TYPE);
  move2d( screen_handle, xposition, yposition);
  draw2d( screen_handle, xposition, yposition);
  make_picture_current( screen_handle);

  PRIMITIVE_RETURN(NIL);
}

/* (GRAPHICS-SET-LINE-STYLE STYLE)
   Used to change the style of the line to dashes or dots and
   dashes or whatever.  Uses Starbase `line_type' procedure. */

Define_Primitive( Prim_Graphics_Set_Line_Style, 1, "GRAPHICS-SET-LINE-STYLE")
{
  long type;
  Primitive_1_Arg();

  Arg_1_Type( TC_FIXNUM);
  line_type( screen_handle, Get_Integer( Arg1));

  PRIMITIVE_RETURN(NIL);
}

/* (GRAPHICS-SET-DRAWING-MODE MODE)
   Used to change the replacment rule when drawing. */

Define_Primitive( Prim_Graphics_Set_Drawing_Mode, 1, "GRAPHICS-SET-DRAWING-MODE")
{
  long rule;
  Primitive_1_Arg();

  Arg_1_Type( TC_FIXNUM);
  rule = Get_Integer(Arg1);
  if (rule != replacement_rule)
  {
    replacement_rule = rule;
    drawing_mode( screen_handle, replacement_rule);
  }

  PRIMITIVE_RETURN(NIL);
}

/* (SET-CLIP-RECTANGLE X2 Y2)
   Restrict the graphics drawing primitives to the area between the
   pen and the given point (X2, Y2).*/

Define_Primitive( Prim_set_clip_rectangle, 2, "SET-CLIP-RECTANGLE")
{
  long fixnum;
  float x2, y2;
  Primitive_2_Args();

  Make_Flonum( Arg1, x2, fixnum, ERR_ARG_1_WRONG_TYPE);
  Make_Flonum( Arg2, y2, fixnum, ERR_ARG_2_WRONG_TYPE);
  clip_rectangle( screen_handle, xposition, x2, yposition, y2);
  make_picture_current( screen_handle);

  PRIMITIVE_RETURN(NIL);
}

/* Graphics Text */

/* (GRAPHICS-LABEL STRING)
   Prints a string label at the current pen position.  The label is
   written according to the current letter type as defined by
   `graphics-set-letter'. */

Define_Primitive(Prim_Graphics_Label, 1, "GRAPHICS-LABEL")
{
  Primitive_1_Arg();

  Arg_1_Type(TC_CHARACTER_STRING);

  text2d( screen_handle, xposition, yposition,
	 Scheme_String_To_C_String( Arg1), VDC_TEXT, FALSE);
  make_picture_current( screen_handle);

  PRIMITIVE_RETURN(NIL);
}

/* (GRAPHICS-SET-LETTER HEIGHT ASPECT SLANT)
   This routine will change the way in which letters are drawn by Starbase. */

Define_Primitive( Prim_Graphics_Set_Letter, 3, "GRAPHICS-SET-LETTER")
{
  fast float height, aspect, slant;
  fast long intemp;
  Primitive_3_Args();

  Make_Flonum( Arg1, height, intemp, ERR_ARG_1_WRONG_TYPE);
  Make_Flonum( Arg2, aspect, intemp, ERR_ARG_2_WRONG_TYPE);
  Make_Flonum( Arg3, slant, intemp, ERR_ARG_3_WRONG_TYPE);
  character_height( screen_handle, height);
  character_expansion_factor( screen_handle, aspect);
  character_slant( screen_handle, slant);

  PRIMITIVE_RETURN(NIL);
}

/* (GRAPHICS-SET-ROTATION ANGLE)
   Sets the character path of graphics text. */

Define_Primitive( Prim_Graphics_Set_Rotation, 1 , "GRAPHICS-SET-ROTATION")
{
  fast float angle;
  fast long intemp;
  fast int path_style;
  Primitive_1_Arg();

  Make_Flonum( Arg1, angle, intemp, ERR_ARG_1_WRONG_TYPE);
  if ((angle > 315.0) || (angle <=  45.0))
    path_style = PATH_RIGHT;
  else if ((angle > 45.0) && (angle <= 135.0))
    path_style = PATH_DOWN;
  else if ((angle > 135.0) && (angle <= 225.0))
    path_style = PATH_LEFT;
  else if ((angle > 225.0) && (angle <= 315.0))
    path_style = PATH_UP;
  text_path( screen_handle, path_style);

  PRIMITIVE_RETURN(NIL);
}

/* Graphics Screen Dump */

static char rasres[]    = "\033*t100R";
static char rastop[]    = "\033&l2E";
static char raslft[]    = "\033&a2L";
static char rasbeg[]    = "\033*r0A";
static char raslen[]    = "\033*b96W";
static char rasend[]    = "\033*rB";

void
C_Print_Graphics( dumpname)
     char *dumpname;
{
  int dumpfile;
  unsigned char rasdata[96];
  unsigned char pixdata[16*768];
  int n, nn, x;
  int err, col;
  register unsigned char c, *p, *r;

  dumpfile = creat( dumpname, 438);
  if (dumpfile == -1)
  {
    fprintf( stderr, "\nunable to create graphics dump file.");
    Primitive_Error(ERR_EXTERNAL_RETURN);
  }
  dumpfile = open( dumpname, OUTINDEV);
  if (dumpfile == -1)
  {
    fprintf( stderr, "\nunable to open graphics dump file.");
    Primitive_Error(ERR_EXTERNAL_RETURN);
  }

  write( dumpfile, rasres, strlen( rasres));
  write( dumpfile, rastop, strlen( rastop));
  write( dumpfile, raslft, strlen( raslft));
  write( dumpfile, rasbeg, strlen( rasbeg));

  for (col = 1008; col >= 0; col = (col - 16))
    {
      for (n = 0; n < 12288; n++)
	pixdata[n] = 0;
      dcblock_read( screen_handle, col, 0, 16, 768, pixdata, 0);
      for (x = 15; x >= 0; x--)
	{
	  p = (pixdata + x);
	  for (r = rasdata, n = 0; n < 96; n++)
	    {
	      for (c = 0, nn = 0; nn < 8; nn++)
		{
		  c <<= 1;
		  if (*p)
		    c |= 01;
		  p += 16;
		}
	      *r++ = c;
	    }
	  err = write( dumpfile, raslen, strlen( raslen));
	  err = write( dumpfile, rasdata, 96);
	}
    }
  write( dumpfile, rasend, strlen( rasend));
  close( dumpfile);
  return;
}

/* (PRINT-GRAPHICS FILENAME)
   Write a file containing an image of the screen, in a format
   suitable for printing on a laser printer. */

Define_Primitive( Prim_print_graphics, 1, "PRINT-GRAPHICS")
{
  Primitive_1_Arg();

  Arg_1_Type( TC_CHARACTER_STRING);

  C_Print_Graphics( Scheme_String_To_C_String( Arg1));

  PRIMITIVE_RETURN(NIL);
}
