;;; -*- Mode: Scheme; Syntax: Scheme; Package: (SCHEME :USE (PSEUDOSCHEME)) -*-

;;;
;;;	$Header$
;;;
;;;	Copyright (c) 1986, 1987 Massachusetts Institute of Technology
;;;     Initial implementation due to Ken Haase (KWH@AI.AI.MIT.EDU)
;;;
;;;	This material was developed by the Scheme project at the
;;;	Massachusetts Institute of Technology, Department of
;;;	Electrical Engineering and Computer Science.  Permission to
;;;	copy this software, to redistribute it, and to use it for any
;;;	purpose is granted, subject to the following restrictions and
;;;	understandings.
;;;
;;;	1. Any copy made of this software must include this copyright
;;;	notice in full.
;;;
;;;	2. Users of this software agree to make their best efforts (a)
;;;	to return to the MIT Scheme project any improvements or
;;;	extensions that they make, so that these may be included in
;;;	future releases; and (b) to inform MIT of noteworthy uses of
;;;	this software.
;;;
;;;	3. All materials developed as a consequence of the use of this
;;;	software shall duly acknowledge such use, in accordance with
;;;	the usual standards of acknowledging credit in academic
;;;	research.
;;;
;;;	4. MIT has made no warrantee or representation that the
;;;	operation of this software will be error-free, and MIT is
;;;	under no obligation to provide any services, by way of
;;;	maintenance, update, or otherwise.
;;;
;;;	5. In conjunction with products arising from the use of this
;;;	material, there shall be no use of the name of the
;;;	Massachusetts Institute of Technology nor of any adaptation
;;;	thereof in any advertising, promotional, or sales literature
;;;	without prior written consent from MIT in each case.
;;;

(declare (usual-integrations))
(declare (integrate-external "/u/kwh/programs/utility/plus")
	 (integrate-external "/u/kwh/programs/utility/mutable"))


;;;; Type Descriptions

;;;  A type description is a list of six elements: the first is the 
;;; predicate function describing the type, the second is a unique
;;; numeric id for the type, the third is a list of generalizations
;;; of the type, the fourth is a list of specializations of the
;;; type, the fifth is a unique s-expression identifying the type, and
;;; the sixth is an a-list of incidental properties of the type.

;;; This is an association list of predicates and the types they define.
(define type-list ())
;;; This is a counter used for giving new types unique numeric ids.
(define type-count 0)

;;; This is the procedure for printing out type descriptions. 
(define type-printer)

;;; The fields of the type description are used as follows:
;;;   PREDICATE:        The SCHEME procedure for determining satisfaction of
;;;                     the type.
;;;   INDEX:            A unique integer id attached to each type.
;;;   GENERALIZATIONS:  The minimal generalizations of the type in the lattice.
;;;   SPECIALIZATIONS:  The maximal specializations of the type in the lattice.
;;;   SPECIFICATION:    The combinator and arguments which generated the type.
;;;   PROPERTIES:       Arbitrary properties of the type (a mutable procedure).
;;;   LATTICE-CACHE:    A bit string used as a cache for lattice subsumption.
;;;   TYPE-CACHE:       A pair of bit strings used as a cache for
;;;                     satisfaction information by TYPICALs indexer.
(define-structure type-description
  (predicate index 
	     generalizations specializations
	     specification properties
	     lattice-cache
	     type-cache)
  (type-printer type-description))

;;; This constructs a type description with a new numeric id,
;;; and with the PREDICATE and SPECIFICATION fields filled in.
(define (make-type-description predicate specification)
  (let* ((new-td (cons-type-description 
		  predicate type-count () ()
		  (apply list specification) (make-mutable)
		  (bit-string-modify empty-bit-string type-count #T)
		  (make-empty-type-cache 200))))
    (set! type-count (1+ type-count))
    (set! type-list (cons new-td type-list))
    new-td))

;;; This tries to coerce its argument into a type.
(define (ftd x) 
  (if (and (integer? x) (< x type-count))
      (list-ref type-list (- type-count x 1))
      (error "X is not a type description: " x)))

(definline (->td thing) (if (type-description? thing) thing (ftd thing)))
(define (td-list . list) (map ->td list))
(definline (should-be-td thing) thing)

; When debugging:
;(definline (should-be-td thing)
;  (if (type-description? thing) thing
;      (error "Not a type description!" thing)))


;;;; More Type Description properties

;;; These are abbreviated versions of the structure accessors for type
;;; descriptions. 
(definline (td-predicate x) 
  (type-description-predicate (should-be-td x)))
(definline (td-id x)
  (type-description-index (should-be-td x)))
(definline (td-generalizations x)
  (type-description-generalizations (should-be-td x)))
(definline (td-specializations x)
  (type-description-specializations (should-be-td x)))
(definline (td-specification x)
  (type-description-specification (should-be-td x)))
(definline (td-properties x)
  (type-description-properties (should-be-td x)))
(definline (td-lattice-cache x)
  (type-description-lattice-cache (should-be-td x)))

;;; These allow the construction of mutable procedures which store
;;; values per-type rather than per-function.
(definline (td-modifier x) (modifier (td-properties x)))
(define (td-property name . type)
  (define (get-property from-x) ((td-properties (->td from-x)) name))
  (define (mutate-property object f)
    ((mutator (td-properties (->td object))) name f))
  (define (enumerate-property procedure)
    (define (apply-for-type type)
      (let ((value ((td-properties (->td type)) name)))
	(if (defined? value) (procedure type value))))
    (for-each apply-for-type type-list))
  ((modifier mutator) get-property mutate-property)
  ((modifier enumerator) get-property enumerate-property)
  (if type (declare-function! get-property types (first type) name)
      (procedure! get-property name "Gets the " name
		  " property from a type description."))
  get-property)


;;;; Naming and printing types.

;;; Pretty names for types are like pretty names for functions.
(define pretty-name (td-property 'pretty-name))
;;; This is a mutable mapping various combinators to ways of printing
;;; their results.
(define combinator-printer (make-mutable))

;;; If something is known to be a type, this PRINTOUT command just
;;; prints out its pretty name without the surrounding brackets and
;;; index of normal type printing.
(define ($type x)
  (let ((name (pretty-name x)))
    (if (undefined? name) x
	(if (procedure? name) (execute-token name x)
	    (if (string? name) name ($printout name))))))

(define (rename-type! type name)
  ;; This gives a type a new name even if it already has a special name.
  ((modifier (td-properties type)) 'pretty-name name)
  type)
(define (name-type! type name)
  ;; This names a type only if it hasn't been specially named yet.
  ((mutator (td-properties type)) 'pretty-name
   (lambda (v) (if (defined? v) v name)))
  type)
(define (type name generator . args)
  ;; This generates a type and names it in one expression.
  (name-type! (apply generator args) name))
(define (type! name generator . args)
  ;; This generates a type and names it in one expression.
  (rename-type! (apply generator args) name))

;;; This is the printer for types:
;;;  If a type has a `pretty-name', it is used:
;;;    if the name is a string or symbol, it is DISPLAYed
;;;    if the name is a list, it is PRINTOUTed,
;;;    if the name is a procedure, it is called
;;;    otherwise, it is WRITEn
;;;  If the type has no pretty name, but its combinator has a special
;;;   printer, then this special printer is called.
;;;  If the type's combinator has no special printer, then the type is
;;;   just printed out by its specification: the combinator invocation
;;;   which generated it.
(define (type-printer type)
  (define (print-specification spec) 
    (if (defined? (combinator-printer (first spec)))
	((combinator-printer (first spec)) type)
	(printout (procedure-name (car spec))
		  "(" ($comma-list (cdr spec)) ")")))
  (let ((pretty-name ((td-properties type) 'pretty-name)))
    (display "#[") (display (td-id type)) (display ":")
    (if (undefined? pretty-name)
	(print-specification (td-specification type))
	(cond ((string? pretty-name) (display pretty-name))
	      ((symbol? pretty-name) (display pretty-name))
	      ((list? pretty-name) (apply printout pretty-name))
	      ((procedure? pretty-name) (pretty-name))
	      (ELSE (write pretty-name))))
    (display "]")))

(define (declare-combinator-printer combinator printer)
  ;; This declares a combinator printer.
  ((modifier combinator-printer) combinator printer))
  
(define and-string "<and>")
(define or-string "<or>")


;;;; Accessing the lattice.

(define (td-subsumed-by? type1 type2)
  ;; This checks subsumption using the lattice cache.
  (check-bit (td-lattice-cache type1) (td-id type2)))

(define (td-subsumed-in-lattice? type1 type2)
  ;; This checks subsumption by combing the lattice.
  (cond ((eq? type1 type2) #T)
	((some? (td-generalizations type1) 
		(lambda (type) (subsumed-in-lattice? type type2)))
	 #T)
	(else #F)))

;;; This returns the generalizations or specializations of a type.
(define (generalizations t) (td-generalizations (->td t)))
(define (specializations t) (td-specializations (->td t)))

;;; Aliases 

(define genzns generalizations)
(define speczns specializations)
(define td-<<? td-subsumed-by?)
(define (subsumed-by? x y) (td-subsumed-by? (->td x) (->td y)))
(define <<? subsumed-by?)
(define <<  subsumed-by?)
(define (subsumed-in-lattice? x y) (td-subsumed-in-lattice? (->td x) (->td y)))
(define *<<* subsumed-in-lattice?)


;;;; Minimal and maximal type sets.

(define (minimal-type-set types)
  (do ((set types (cdr set))
       (into ()))
      ((null? set) into)
    (let ((head (first set)))
      (if (not (or (memq head into)
		   (some? types (lambda (x) (if (eq? x head) #F (td-<<? x head))))))
	  (set! into (cons head into))))))
(define (maximal-type-set types)
  (do ((set types (cdr set))
       (into ()))
      ((null? set) into)
    (let ((head (first set)))
      (if (not (or (memq head into)
		   (some? types (lambda (x) (if (eq? x head) #F (td-<<? head x))))))
	  (set! into (cons head into))))))


;;;; Extending the lattice.

;;; Always use these to ensure that the lattice goes in both directions;
;;; They update both generalizations and specializations.
(definline (%remove-generalization-link! s g)
  (begin (set-type-description-generalizations!
            s (delq g (td-generalizations s)))
	 (set-type-description-specializations!
            g (delq s (td-specializations g)))))
(definline (%add-generalization-link! s g)
  (begin (set-type-description-generalizations!
            s (cons g (td-generalizations s)))
	 (set-type-description-specializations!
            g (cons s (td-specializations g)))))

(define (add-td-generalization! s g)
  (cond ((td-<<? s g) #F)
	((td-<<? g s) 
         (ERROR "Lattice circularity! S is already below G" (list s g)))
	((disjoint? s g)
         (ERROR "Disjoint generalizations??? S is disjoint from G" (list s g)))
	(else
	 ;; Remove superfulous links to maintain minimal spanning tree.
         ;;  When putting in a link between two types, a superfulous link
         ;;  is a link which is IMPLIED by the link being added.
	 (for-each (lambda (t)
	              (if (td-<<? g t) (%remove-generalization-link! s t)))
		   (td-generalizations s))
	 (for-each (lambda (t)
		      (if (td-<<? t s) (%remove-generalization-link! t g)))
		   (td-specializations g))
	 ;; Add the actual link.
	 (%add-generalization-link! s g)
	 (let ((cache (td-lattice-cache g)))
	   (maptree (lambda (t)
	              (set-type-description-lattice-cache!
                         t (bit-string-or (td-lattice-cache t) cache)))
		    (list s) td-specializations))
	 #T)))
(define (add-td-specialization! g s) (add-td-generalization! s g))


;;;; Disjoint types.

;;; Types are declared disjoint by using the type property
;;; DISJOINT-FROM.  We can discover if two types are disjoint by
;;; climing the lattice and seeing if any of the generalizations are
;;; disjoint. 

(define disjoint-from (td-property 'disjoint-from))

(define (make-pair-disjoint! t1 t2)
  (define (place-link! t1 t2)
    ;; This places the disjointness link in one direction.
    (define (add-t2-to-value value)
      (if (undefined? value) (list t2)
	  (if (memq t2 value) value (cons t2 value))))
    ((mutator disjoint-from) t1 add-t2-to-value))
  ;; We catch attempts to make subtypes disjoint and, if everything is
  ;; cool, we actually place the disjointness link in both direction.
  (cond ((td-<<? t1 t2)
	 (ERROR "Trying to make specialization T1 of T2 disjoint from T2" (list t1 t2)))
	((td-<<? t2 t1)
	 (ERROR "Trying to make specialization T2 of T1 disjoint from T1" (list t1 t2)))
	(else (place-link! t1 t2) (place-link! t2 t1))))

(define (declare-disjoint! types)
  ;; Declares a list of types mutually disjoint.
  (let ((types (map ->td types)))
    (for-each (lambda (t1)
		(for-each (lambda (t2)
		            (if (not (eq? t1 t2)) (make-pair-disjoint! t1 t2)))
			  types))
	      types)))
(define (make-disjoint! . x)
  ;; Declares TYPES mutually disjoint.
  (declare-disjoint! x))

(define (disjoint? t1 t2)
  (let ((t1 (->td t1))
	(t2 (->td t2))
	(marks (make-empty-bit-string type-count)))
    (define (find-disjoin with-head disjoins)
      (if (null? disjoins) #F
	  (if (<<? t2 (first disjoins))
	      (list with-head (first disjoins))
	      (find-disjoin with-head (rest disjoins)))))
    (define (mapper nodes)
      (if (null? nodes) #F
	  (let* ((head (first nodes)) (disjoints (disjoint-from head)))
	    (if (undefined? disjoints)
		(extend-search (td-generalizations head) (rest nodes))
		(or (find-disjoin head disjoints)
		    (extend-search (td-generalizations head) (rest nodes)))))))
    (define (extend-search genzns nodes)
      (if (null? genzns) (mapper nodes)
	  (let ((new (first genzns)))
	    (if (check-bit marks (td-id new))
		(extend-search (rest genzns) nodes)
		(sequence (bit-string-modify! marks (td-id new) #T)
			  (extend-search (rest genzns) (cons new nodes)))))))
    (mapper (list t1))))


;;;; Defining new type generators

;;; A type generator is defined by three component functions: a
;;; predicate generator, a generalization generator, and a
;;; specialization generator.  The higher order procedure
;;; TYPE-GENERATOR combines these procedures into a procedure which
;;; calls the generalization and specialization generators to get a
;;; types position in the lattice.  If this position is EXACTLY an
;;; existing type, it is returned; otherwise, the predicate generator
;;; is called and an appropriate type description constructed.  This
;;; type description is then established in the lattice by calls to
;;; %ADD-TD-GENERALIZATION. 

(define (type-generator generator generalizer specializer)
  (let ((constructor ()))
    (define (generate-instance . x)
      (construct-type (apply generator x) (cons constructor x)
		      (map ->td (apply generalizer x))
		      (map ->td (apply specializer x))))
    (set! constructor (canonical-cache generate-instance))
    constructor))

;;; Most of the work of procedures generated by TYPE-GENERATOR is done
;;; by CONSTRUCT-TYPE.
(define (construct-type
	 predicate specification generalizations specializations)
  (let ((generalizations (minimal-type-set generalizations))
	(specializations (maximal-type-set specializations)))
    (if (and (not (null? generalizations))
	     (equal? generalizations specializations))
	(first specializations)
	(let ((type (make-type-description predicate specification)))
	  (for-each (lambda (g) (add-td-generalization! type g))
		    generalizations)
	  (for-each (lambda (s) (add-td-specialization! type s))
		    specializations)
	  type))))


;;;; Satisfaction functions.

(define (satisfies? object predicate)
  ((td-predicate predicate) object))

;;; We use the `undefined' token from mutables to represent
;;; indeterminancy of types.  (This might be a bad idea.)
(definline (defined-true? x)
  (let ((result x)) (if (undefined? result) #F result)))

;;; This returns #T only if the instance definitely satisfies the type
;;; in question.
(define (in? x type) (defined-true? (satisfies? x type)))


;;;; The top of the lattice.

;;; The top of the lattice is made directly by CONSTRUCT-TYPE.
(define lattice-top (construct-type return-true (list construct-type) () ()))
;;; And is named "ToP"
(name-type! lattice-top "ToP")

;;; This is a simple function which maps a procedure down the lattice,
;;;  calling and expanding only those types which satisfy an object.
;;;  MAPTYPES is a considerably hairier version of this which
;;;  incorporates guarantees about execution order as well as tricks
;;;  for efficent implementation.
(define (for-types map-fcn object)
  (define (mapper type)
    (cond ((satisfies? object type) (map-fcn type) #T)
	  (else #F)))
  (maptree mapper (list lattice-top) td-specializations))

